package com.elitesland.tw.tw5.server.demo.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.demo.payload.DemoSimplePayload;
import com.elitesland.tw.tw5.api.demo.query.DemoSimpleQuery;
import com.elitesland.tw.tw5.api.demo.vo.DemoSimpleVO;
import com.elitesland.tw.tw5.server.demo.entity.DemoSimpleDO;
import com.elitesland.tw.tw5.server.demo.repo.DemoSimpleRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.demo.entity.QDemoSimpleDO;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.List;

/**
 * 标准样例dao
 *
 * @author zhangyongqiang
 */
@Repository
@RequiredArgsConstructor
public class DemoSimpleDAO {

    private final JPAQueryFactory jpaQueryFactory;

    private final DemoSimpleRepo repo;
    private final QDemoSimpleDO qdo = QDemoSimpleDO.demoSimpleDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<DemoSimpleVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(DemoSimpleVO.class,
                qdo.id,
                qdo.objectName,
                qdo.objectNo,
                qdo.createUserId,
                qdo.createTime,
                qdo.objectStatus,
                qdo.procInstId,
                qdo.procInstStatus,
                qdo.submitTime,
                qdo.approvedTime,
                qdo.attribute1,
                qdo.attribute2,
                qdo.attribute3,
                qdo.attribute4,
                qdo.attribute5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<DemoSimpleVO> getJpaQueryWhere(DemoSimpleQuery query) {
        JPAQuery<DemoSimpleVO> jpaQuery = getJpaQuerySelect();

        if (!ObjectUtils.isEmpty(query.getObjectNo())) {
            jpaQuery.where(qdo.objectNo.like(SqlUtil.toSqlLikeString(query.getObjectNo())));
        }
        if (!ObjectUtils.isEmpty(query.getObjectName())) {
            jpaQuery.where(qdo.objectName.like(SqlUtil.toSqlLikeString(query.getObjectName())));
        }

        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public DemoSimpleVO queryByKey(Long id) {
        JPAQuery<DemoSimpleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param ids 主键
     * @return 结果
     */
    public List<DemoSimpleVO> queryByKeys(List<Long> ids) {
        JPAQuery<DemoSimpleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<DemoSimpleVO> queryListDynamic(DemoSimpleQuery query) {
        JPAQuery<DemoSimpleVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<DemoSimpleVO> queryPaging(DemoSimpleQuery query) {
        JPAQuery<DemoSimpleVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<DemoSimpleVO> result = jpaQuery.offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize()).fetchResults();
        return PagingVO.<DemoSimpleVO>builder().records(result.getResults()).total(result.getTotal()).build();
    }


    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public DemoSimpleDO save(DemoSimpleDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<DemoSimpleDO> saveAll(List<DemoSimpleDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(DemoSimplePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        if (payload.getObjectNo() != null) {
            update.set(qdo.objectNo, payload.getObjectNo());
        }
        if (payload.getObjectName() != null) {
            update.set(qdo.objectName, payload.getObjectName());
        }
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        if (payload.getSubmitTime() != null) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        if (payload.getObjectStatus() != null) {
            update.set(qdo.objectStatus, payload.getObjectStatus());
        }
        if (payload.getAttribute1() != null) {
            update.set(qdo.attribute1, payload.getAttribute1());
        }
        if (payload.getAttribute2() != null) {
            update.set(qdo.attribute2, payload.getAttribute2());
        }
        if (payload.getAttribute3() != null) {
            update.set(qdo.attribute3, payload.getAttribute3());
        }
        if (payload.getAttribute4() != null) {
            update.set(qdo.attribute4, payload.getAttribute4());
        }
        if (payload.getAttribute5() != null) {
            update.set(qdo.attribute5, payload.getAttribute5());
        }


        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            System.out.println(qdo.objectName.toString());
            if (nullFields.contains("objectNo")) {
                update.setNull(qdo.objectNo);
            }
            if (nullFields.contains("objectName")) {
                update.setNull(qdo.objectName);
            }
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


}
