package com.elitesland.tw.tw5.server.prd.office.util.file;

import com.elitesland.tw.tw5.server.prd.office.models.enums.DocumentType;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@Component
@Qualifier("default")
public class DefaultFileUtility implements FileUtility {
    @Value("${office.filesize-max:5242880}")
    private String filesizeMax;

    @Value("${office.files.docservice.viewed-docs:.pdf|.djvu|.xps|.oxps}")
    private String docserviceViewedDocs;

    @Value("${office.files.docservice.edited-docs:.docx|.xlsx|.csv|.pptx|.txt|.docxf}")
    private String docserviceEditedDocs;

    @Value("${office.files.docservice.convert-docs:.docm|.dotx|.dotm|.dot|.doc|.odt|.fodt|.ott|.xlsm|.xlsb|.xltx|.xltm|.xlt|.xls|.ods|.fods|.ots|.pptm|.ppt|.ppsx|.ppsm|.pps|.potx|.potm|.pot|.odp|.fodp|.otp|.rtf|.mht|.html|.htm|.xml|.epub|.fb2}")
    private String docserviceConvertDocs;

    @Value("${office.files.docservice.fillforms-docs:.oform|.docx}")
    private String docserviceFillDocs;

    // document extensions
    private List<String> ExtsDocument = Arrays.asList(
                            ".doc", ".docx", ".docm",
                            ".dot", ".dotx", ".dotm",
                            ".odt", ".fodt", ".ott", ".rtf", ".txt",
                            ".html", ".htm", ".mht", ".xml",
                            ".pdf", ".djvu", ".fb2", ".epub", ".xps", ".oform");

    // spreadsheet extensions
    private List<String> ExtsSpreadsheet = Arrays.asList(
                            ".xls", ".xlsx", ".xlsm", ".xlsb",
                            ".xlt", ".xltx", ".xltm",
                            ".ods", ".fods", ".ots", ".csv");

    // presentation extensions
    private List<String> ExtsPresentation = Arrays.asList(
                            ".pps", ".ppsx", ".ppsm",
                            ".ppt", ".pptx", ".pptm",
                            ".pot", ".potx", ".potm",
                            ".odp", ".fodp", ".otp");

    /**
     * 得到文档类型
     *
     * @param fileName 文件名称
     * @return {@link DocumentType}
     */
    @Override
    public DocumentType getDocumentType(String fileName)
    {
        String ext = getFileExtension(fileName).toLowerCase();  // get file extension from its name
        // word type for document extensions
        if (ExtsDocument.contains(ext)) {
            return DocumentType.word;
        }

        // cell type for spreadsheet extensions
        if (ExtsSpreadsheet.contains(ext)) {
            return DocumentType.cell;
        }

        // slide type for presentation extensions
        if (ExtsPresentation.contains(ext)) {
            return DocumentType.slide;
        }

        // default file type is word
        return DocumentType.word;
    }

    /**
     * 获取文件名称
     *
     * @param url url
     * @return {@link String}
     */
    @Override
    public String getFileName(String url)
    {
        if (url == null) {
            return "";
        }

        // get file name from the last part of URL
        String fileName = url.substring(url.lastIndexOf('/') + 1);
        fileName = fileName.split("\\?")[0];
        return fileName;
    }

    // get file name without extension
    @Override
    public String getFileNameWithoutExtension(String url)
    {
        String fileName = getFileName(url);
        if (fileName == null) {
            return null;
        }
        String fileNameWithoutExt = fileName.substring(0, fileName.lastIndexOf('.'));
        return fileNameWithoutExt;
    }

    // get file extension from URL
    @Override
    public String getFileExtension(String url)
    {
        String fileName = getFileName(url);
        if (fileName == null) {
            return null;
        }
        String fileExt = fileName.substring(fileName.lastIndexOf("."));
        return fileExt.toLowerCase();
    }

    // get an editor internal extension
    @Override
    public String getInternalExtension(DocumentType type)
    {
        // .docx for word file type
        if (type.equals(DocumentType.word)) {
            return ".docx";
        }

        // .xlsx for cell file type
        if (type.equals(DocumentType.cell)) {
            return ".xlsx";
        }

        // .pptx for slide file type
        if (type.equals(DocumentType.slide)) {
            return ".pptx";
        }

        // the default file type is .docx
        return ".docx";
    }

    @Override
    public List<String> getFillExts()
    {
        return Arrays.asList(docserviceFillDocs.split("\\|"));
    }

    // get file extensions that can be viewed
    @Override
    public List<String> getViewedExts()
    {
        return Arrays.asList(docserviceViewedDocs.split("\\|"));
    }

    // get file extensions that can be edited
    @Override
    public List<String> getEditedExts()
    {
        return Arrays.asList(docserviceEditedDocs.split("\\|"));
    }

    // get file extensions that can be converted
    @Override
    public List<String> getConvertExts()
    {
        return Arrays.asList(docserviceConvertDocs.split("\\|"));
    }

    // get all the supported file extensions
    @Override
    public List<String> getFileExts() {
        List<String> res = new ArrayList<>();

        res.addAll(getViewedExts());
        res.addAll(getEditedExts());
        res.addAll(getConvertExts());
        res.addAll(getFillExts());

        return res;
    }

    // generate the file path from file directory and name
    @Override
    public Path generateFilepath(String directory, String fullFileName){
        String fileName = getFileNameWithoutExtension(fullFileName);  // get file name without extension
        String fileExtension = getFileExtension(fullFileName);  // get file extension
        Path path = Paths.get(directory+fullFileName);  // get the path to the files with the specified name

        for(int i = 1; Files.exists(path); i++){  // run through all the files with the specified name
            fileName = getFileNameWithoutExtension(fullFileName) + "("+i+")";  // get a name of each file without extension and add an index to it
            path = Paths.get(directory+fileName+fileExtension);  // create a new path for this file with the correct name and extension
        }

        path = Paths.get(directory+fileName+fileExtension);
        return path;
    }

    // get maximum file size
    @Override
    public long getMaxFileSize(){
        long size = Long.parseLong(filesizeMax);
        return size > 0 ? size : 5 * 1024 * 1024;
    }
}
