package com.elitesland.tw.tw5.server.common;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.elitesland.tw.tw5.server.common.util.JavaMailPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.mail.MailProperties;
import org.springframework.core.io.InputStreamResource;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import javax.mail.BodyPart;
import javax.mail.Flags;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMultipart;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * 邮件发送工具类
 *
 * @author wangding
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class JavaMailUtil {

    private final MailProperties mailProperties;
    private final JavaMailSender mailSender;


    @Value("${tw5.mail.enabled}")
    private Boolean enabled = false;


    /**
     * 获取MimeMessageHelper
     *
     * @param withAttachments 是否有附件
     * @return MimeMessageHelper
     * @throws MessagingException 邮件异常
     */
    private MimeMessageHelper messageHelper(boolean withAttachments) throws MessagingException {
        MimeMessageHelper msg = new MimeMessageHelper(this.mailSender.createMimeMessage(), withAttachments);
        msg.setSentDate(Calendar.getInstance().getTime());
        String senderMail = mailProperties.getUsername();
        msg.setFrom(senderMail);
        msg.setReplyTo(senderMail);
        return msg;
    }

    /**
     * 添加附件
     *
     * @param messageHelper messageHelper
     * @param attachment    attachment
     * @throws MessagingException MessagingException
     */
    public void addAttachement(MimeMessageHelper messageHelper, Path attachment) throws MessagingException {
        try {
            InputStream attachmentStream = Files.newInputStream(attachment);
            Throwable var4 = null;

            try {
                messageHelper.addAttachment(attachment.getFileName().toString(), new InputStreamResource(attachmentStream));
            } catch (Throwable var14) {
                var4 = var14;
                throw var14;
            } finally {
                if (attachmentStream != null) {
                    if (var4 != null) {
                        try {
                            attachmentStream.close();
                        } catch (Throwable var13) {
                            var4.addSuppressed(var13);
                        }
                    } else {
                        attachmentStream.close();
                    }
                }

            }

        } catch (IOException var16) {
            throw new MessagingException("[CORE-IO] read attachment FAILED: " + attachment, var16);
        }
    }

    public void sendMessage(MimeMessageHelper messageHelper) {
        if (enabled) {
            this.mailSender.send(messageHelper.getMimeMessage());
        } else {
            log.warn("邮件未发送，原因：邮件服务已关闭。请设置'el.mail.enabled: true' 打开");
        }

    }

    /**
     * 发送邮件
     *
     * @param payload 邮件消息
     */
    public void sendEmailByAdmin(com.elitesland.tw.tw5.server.common.util.JavaMailPayload payload) {
        log.info("【MESSAGE-EMAIL】 开始发送邮件： {}...", payload);
        try {
            MimeMessageHelper helper = messageHelper(payload.isFileContentFlag());
            List<String> receiveEmails = new ArrayList<>();
            if (payload.getReceiveEmails() != null && payload.getReceiveEmails().size() > 0) {
                receiveEmails.addAll(payload.getReceiveEmails());
            }
            if (payload.getReceiveEmail() != null) {
                receiveEmails.add(payload.getReceiveEmail());
            }
            if (receiveEmails.size() < 1) {
                throw new RuntimeException("邮件发送失败，收件人邮箱不能为空!");
            }
            String[] toArray = new String[receiveEmails.size()];
            helper.setTo(receiveEmails.toArray(toArray));
            helper.setSubject(payload.getSubject());
            helper.setText(payload.getText(), payload.isHtmlContentFlag());
            List<String> ccEmails = new ArrayList<>();
            if (payload.getCcEmails() != null && payload.getCcEmails().size() > 0) {
                ccEmails.addAll(payload.getCcEmails());
            }
            if (payload.getCcEmail() != null) {
                ccEmails.add(payload.getCcEmail());
            }
            String[] ccArray = new String[ccEmails.size()];
            helper.setCc(ccArray);
            sendMessage(helper);
            log.info("【MESSAGE-EMAIL】 邮件发送成功");
        } catch (Exception e) {
            log.error("【MESSAGE-EMAIL】 邮件发送失败", e);
            throw new RuntimeException("发送到【{" + payload.getReceiveEmail() + "}】的邮件发送失败!");
        }
    }

    /**
     * 异步发送邮件
     *
     * @param payload 邮件消息
     */
    @Async()
    public void sendEmailAsyncByAdmin(JavaMailPayload payload) {
        sendEmailByAdmin(payload);
    }

//
//    /**
//     * 判断此邮件是否包含附件
//     */
//    public boolean isContainAttach(Part part) throws Exception {
//        boolean attachable = false;
//        PDDocument document = null;
//        String text = null;
//        if (part.isMimeType("multipart/*")) {
//            Multipart mp = (Multipart) part.getContent();
//            for (int i = 0; i < mp.getCount(); i++) {
//                BodyPart mPart = mp.getBodyPart(i);
//                String disposition = mPart.getDisposition();
//                boolean flag = (disposition != null) && ((disposition.equalsIgnoreCase(Part.ATTACHMENT)) || (disposition.equalsIgnoreCase(Part.INLINE)));
//                if (flag) {
//                    attachable = true;
//                } else if (mPart.isMimeType("multipart/*")) {
//                    attachable = isContainAttach((Part) mPart);
//                } else {
//                    String contentType = mPart.getContentType();
//                    if (contentType.toLowerCase().contains("application")) {
//                        attachable = true;
//                    }
//                    if (contentType.toLowerCase().contains("name")) {
//                        attachable = true;
//                    }
//                }
//            }
//        } else if (part.isMimeType("message/rfc822")) {
//            attachable = isContainAttach((Part) part.getContent());
//        }
//        return attachable;
//    }

//    /**
//     * 定时连接邮箱
//     *
//     * @return IMAPStore
//     */
//    public IMAPStore getConnect(PrjEmailDTO dto) {
//        Session session;
//        IMAPStore store = null;
//        Properties props = new Properties();
//        props.setProperty("mail.imap.ssl.enable", dto.getSecureLink().equals(1) ? "true" : "false");
//        props.setProperty("mail.store.protocol", "imap");
//        props.setProperty("mail.imap.auth", "true");
//        props.setProperty("mail.imap.host", dto.getServerUrl());
//        props.setProperty("mail.imap.port", dto.getPort());
//
////        PasswordAuthentication authentication = new PasswordAuthentication(dto.getUserName(), dto.getPassword());
//        try {
////            session = Session.getDefaultInstance(props, null);
//            session = Session.getInstance(props, null);
//            // 通过session得到store对象
//            store = (IMAPStore) session.getStore("imap");
//            // 连接邮件服务器
//            store.connect(dto.getUserName(), dto.getPassword());
//        } catch (Exception e) {
//            log.error("邮箱连接失败", e);
//            e.printStackTrace();
//        }
//
//        return store;
//    }

    /**
     * 判断邮件是否未读
     *
     * @param message msg
     * @return 邮件是否未读
     * @throws MessagingException 异常
     */
    public boolean isNew(Message message) throws MessagingException {
        boolean isNewFlag = false;
        Flags flags = message.getFlags();
        Flags.Flag[] flagsArr = flags.getSystemFlags();
        log.info("邮箱状态：" + JSONObject.toJSONString(flagsArr));
        for (Flags.Flag flag : flagsArr) {
            if (flag == Flags.Flag.SEEN) {
                isNewFlag = true;
                log.info("当前邮件为未读状态");
                break;
            }
        }
        return isNewFlag;
    }

    /**
     * 检查当前邮件是否大于一天，大于则返回true
     *
     * @param msg 信息
     * @return 布尔值
     * @throws MessagingException
     */
    public boolean outDays(Message msg) throws MessagingException {
        Date date = DateUtil.offsetDay(new Date(), 1);
        Date receivedDate = msg.getReceivedDate();
        return receivedDate.getTime() - date.getTime() < 0;
    }

    /**
     * 获取邮件内容
     *
     * @param msg 信息
     * @return 布尔值
     */
    public String getMessageContent(Message msg) {
        if (msg == null) {
            return "";
        }
        StringBuilder result = new StringBuilder("");
        String contentType;
        try {
            contentType = msg.getContentType();
            if (contentType.toLowerCase().startsWith("text")) {
                result.append(msg.getContent().toString());
            }
            if (contentType.toLowerCase().startsWith("multipart")) {
                MimeMultipart mimeMultipart = (MimeMultipart) msg.getContent();
                parseMultipartText(result, mimeMultipart);
            }
        } catch (MessagingException | IOException e) {
            e.printStackTrace();
        }

        return result.toString();


    }

    /**
     * 解析复合类型的文本
     *
     * @param text          解析的文本存放处
     * @param mimeMultipart 复合类型对象
     * @throws MessagingException exception
     * @throws IOException        exception
     */
    private void parseMultipartText(StringBuilder text, MimeMultipart mimeMultipart) throws MessagingException, IOException {
        int count = mimeMultipart.getCount();
        for (int i = 0; i < count; i++) {
            BodyPart bodyPart = mimeMultipart.getBodyPart(i);
            String type = bodyPart.getContentType();
            if ("multipart/mixed".equals(type)) {
                // 复合的mime类型
                MimeMultipart mimeMultipart1 = (MimeMultipart) bodyPart.getContent();
                parseMultipartText(text, mimeMultipart1);
            }
            if (type.toLowerCase().startsWith("text")) {
                text.append(bodyPart.getContent());
            }

        }

    }

}
