package com.elitesland.tw.tw5.server.common;

import lombok.Data;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 不在线异常
 * 登录过期
 *
 * @author duwh
 * @date 2023/02/11
 */
@Data
public class NotOnlineException extends RuntimeException {

    /**
     * 操作返回码
     */
    private String code;


    /**
     * 上下文环境
     */
    private Object data;


    /**
     * 警告信息
     */
    private List<TwMessage> warns = new ArrayList<>();

    /**
     * 错误信息
     */
    private List<TwMessage> errors = new ArrayList<>();

    private NotOnlineException() {

    }

    /**
     * 根据错误码构造一个异常对象
     *
     * @param code 错误code
     * @return 失败异常
     */
    public static NotOnlineException exception(String code) {
        NotOnlineException exception = new NotOnlineException();
        if(ObjectUtils.isEmpty(code)){
            code = "401";
        }
        exception.code = code;
        return exception;
    }

    /**
     * 只有一个错误信息异常
     *
     * @param code 错误code
     * @param msg  错误提示
     * @return 失败异常
     */
    public static NotOnlineException error(String code, String msg) {
        if (ObjectUtils.isEmpty(code)) {
            code = "401";
        }
        NotOnlineException exception = new NotOnlineException();
        exception.code = code;
        exception.errors.add(TwMessage.of(msg));
        return exception;
    }

    /**
     * 只有一个警告信息异常
     *
     * @param code 错误code
     * @param msg  警告提示
     * @return 失败异常
     */
    public static NotOnlineException warn(String code, String msg) {
        NotOnlineException exception = new NotOnlineException();
        exception.code = code;
        exception.errors.add(TwMessage.of(msg));
        return exception;
    }

    /**
     * 只有一个错误信息异常(异常信息使用国际化)
     *
     * @param code       错误code
     * @param localeCode 国际化码
     * @param defaultMsg 默认提示
     * @param data       上下文数据
     * @return 失败异常
     */
    public static NotOnlineException error(String code, String localeCode, String defaultMsg, Object data) {
        NotOnlineException exception = new NotOnlineException();
        exception.code = code;
        exception.errors.add(TwMessage.of(localeCode, defaultMsg, data));
        return exception;
    }

    /**
     * 只有一个警告信息异常(异常信息使用国际化)
     *
     * @param code       错误code
     * @param localeCode 国际化码
     * @param defaultMsg 默认提示
     * @param data       上下文数据
     * @return 失败异常
     */
    public static NotOnlineException warn(String code, String localeCode, String defaultMsg, Object data) {
        NotOnlineException exception = new NotOnlineException();
        exception.code = code;
        exception.warns.add(TwMessage.of(localeCode, defaultMsg, data));
        return exception;
    }


    /**
     * 添加错误信息
     *
     * @param msg 错误信息
     * @return 失败异常
     */
    public NotOnlineException addError(String msg) {
        this.errors.add(TwMessage.of(msg));
        return this;
    }

    /**
     * 添加错误信息(异常信息使用国际化)
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认提示信息
     * @param data       上下文数据
     * @return 失败异常
     */
    public NotOnlineException addError(String localeCode, String defaultMsg, Object data) {
        this.errors.add(TwMessage.of(localeCode, defaultMsg, data));
        return this;
    }

    /**
     * 添加错误信息
     *
     * @param msg 错误信息
     * @return 失败异常
     */
    public NotOnlineException addWarn(String msg) {
        this.warns.add(TwMessage.of(msg));
        return this;
    }

    /**
     * 添加警告信息(异常信息使用国际化)
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认提示信息
     * @param data       上下文数据
     * @return 失败异常
     */
    public NotOnlineException addWarn(String localeCode, String defaultMsg, Object data) {
        this.warns.add(TwMessage.of(localeCode, defaultMsg, data));
        return this;
    }


}
