package com.elitesland.tw.tw5.server.common.log.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.common.log.query.ComLogQuery;
import com.elitesland.tw.tw5.api.common.log.vo.ComLogVO;
import com.elitesland.tw.tw5.server.common.log.entity.ComLogDO;
import com.elitesland.tw.tw5.server.common.log.entity.QComLogDO;
import com.elitesland.tw.tw5.server.common.log.repo.ComLogRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtilPlus;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 通用日志信息
 *
 * @author carl.wang
 * @date 2023-03-06
 */
@Repository
@RequiredArgsConstructor
public class ComLogDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ComLogRepo repo;
    private final QComLogDO qdo = QComLogDO.comLogDO;

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ComLogDO save(ComLogDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ComLogDO> saveAll(List<ComLogDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ComLogVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ComLogVO.class,
                qdo.id,
                qdo.createTime,
                qdo.creator,
                qdo.createUserId,
                qdo.objectId,
                qdo.logType,
                qdo.logContent,
                qdo.versionNo,
                qdo.remark,
                qdo.extString1,
                qdo.extString2,
                qdo.extString3,
                qdo.extString4,
                qdo.extString5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ComLogVO> getJpaQueryWhere(ComLogQuery query) {
        JPAQuery<ComLogVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));

        // 动态排序
        jpaQuery.orderBy(SqlUtilPlus.getSortedColumn(query.getOrders(), qdo));
        return jpaQuery;
    }


    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ComLogQuery query) {
        List<Predicate> list = new ArrayList<>();

        if (!ObjectUtils.isEmpty(query.getLogType())) {
            list.add(qdo.logType.eq(query.getLogType()));
        }
        if (!ObjectUtils.isEmpty(query.getObjectId())) {
            list.add(qdo.objectId.eq(query.getObjectId()));
        }
        if (!ObjectUtils.isEmpty(query.getVersionNo())) {
            list.add(qdo.versionNo.eq(query.getVersionNo()));
        }
        if (!ObjectUtils.isEmpty(query.getExtString1())) {
            list.add(qdo.extString1.eq(query.getExtString1()));
        }
        if (!ObjectUtils.isEmpty(query.getExtString2())) {
            list.add(qdo.extString2.eq(query.getExtString2()));
        }
        if (!ObjectUtils.isEmpty(query.getExtString3())) {
            list.add(qdo.extString3.eq(query.getExtString3()));
        }
        if (!ObjectUtils.isEmpty(query.getExtString4())) {
            list.add(qdo.extString4.eq(query.getExtString4()));
        }
        if (!ObjectUtils.isEmpty(query.getLogContent())) {
            list.add(qdo.logContent.like("%" + query.getLogContent() + "%"));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ComLogVO> queryListDynamic(ComLogQuery query) {
        JPAQuery<ComLogVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ComLogVO> queryPaging(ComLogQuery query) {
        JPAQuery<ComLogVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<ComLogVO> result = jpaQuery.offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize()).fetchResults();
        return PagingVO.<ComLogVO>builder().records(result.getResults()).total(result.getTotal()).build();
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ComLogVO queryByKey(Long id) {
        JPAQuery<ComLogVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }
}

