package com.elitesland.tw.tw5.server.common.util;

import cn.hutool.json.JSONUtil;
import com.elitesland.tw.tw5.server.common.HttpUtil;
import com.elitesland.tw.tw5.server.common.dto.gaode.GaoDeRoot;
import lombok.RequiredArgsConstructor;
import org.gavaghan.geodesy.Ellipsoid;
import org.gavaghan.geodesy.GeodeticCalculator;
import org.gavaghan.geodesy.GlobalCoordinates;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 高德地图工具类
 *
 * @author carl
 */
@Component
@RequiredArgsConstructor
public class GeodesyUtil {
    @Value("${tw5.gps.web_key:446978e9792758ea17d5dd71b26a30c0}")
    private String web_key;

    private final HttpUtil httpUtil;

    /**
     * 根据经纬度获取两个位置距离
     *
     * @param longitudeFrom 经度1
     * @param latitudeFrom  纬度1
     * @param longitudeTo   经度2
     * @param latitudeTo    纬度2
     * @return 公里数
     */
    public double getDistance(double longitudeFrom, double latitudeFrom, double longitudeTo, double latitudeTo) {
        GlobalCoordinates source = new GlobalCoordinates(latitudeFrom, longitudeFrom);
        GlobalCoordinates target = new GlobalCoordinates(latitudeTo, longitudeTo);
        double ellipsoidalDistance = new GeodeticCalculator().calculateGeodeticCurve(Ellipsoid.WGS84, source, target).getEllipsoidalDistance();
        return Math.round(ellipsoidalDistance) / 1000d;
    }

    /**
     * 根据地址获取经纬度
     *
     * @param address 详细地址
     * @return
     */
    public String getLongitudeAndLatitudeByAddress(String address) {
        String url = "https://restapi.amap.com/v3/geocode/geo";

        Map<String, Object> map = new HashMap<>();
        map.put("address", address);
        map.put("key", web_key);
        map.put("output", "JSON");

        Map result = httpUtil.sendGetCallMap(url, map);

        if (result.get("status").toString().equals("1")) {
            List<Map> geocodes = (List<Map>) result.get("geocodes");
            if (geocodes.size() > 0) {
                return geocodes.get(0).get("location").toString();
            }

        }
        return "";
    }

    /**
     * 根据经纬度获取地址
     *
     * @param location
     * @return
     */
    public String getAddressByLocation(String location) {
        String url = "https://restapi.amap.com/v3/geocode/regeo";

        Map<String, Object> map = new HashMap<>();
        map.put("location", location);
        map.put("key", web_key);
        map.put("output", "JSON");
        map.put("radius", 1000);
        map.put("extensions", "base");
        map.put("batch", "false");
        map.put("roadlevel", 1);

        Map result = httpUtil.sendGetCallMap(url, map);

        if (result.get("status").toString().equals("1")) {
            Map regeocode = (Map) result.get("regeocode");
            if (!ObjectUtils.isEmpty(regeocode)) {
                final List<String> formatted_address = (List<String>) regeocode.get("formatted_address");
                if (!CollectionUtils.isEmpty(formatted_address)) {
                    return formatted_address.toString();//大致地址不精确
                }
//                String address = addressComponent.get("province").toString();//省
//                if (!ObjectUtils.isEmpty(addressComponent.get("city"))) {
//                    address += addressComponent.get("city").toString();//市
//                }
//                address += addressComponent.get("district").toString();//区
//
//                Map streetNumber = (Map) addressComponent.get("streetNumber");
//                address += streetNumber.get("street").toString();//z
//
//                return geocodes.get(0).get("location").toString();
            }

        }
        return "";
    }

    /**
     * 周边搜索 (POI 高德)
     * https://lbs.amap.com/api/webservice/guide/api/search/#around
     *
     * @param location 必填 中心点坐标 规则： 经度和纬度用","分割，经度在前，纬度在后，经纬度小数点后不得超过6位
     * @param keywords 查询关键字 规则： 多个关键字用“|”分割
     * @param city     查询城市
     * @param sortrule 排序规则 规定返回结果的排序规则。按距离排序：distance；综合排序：weight  默认 distance
     * @param offset   每页记录数据 强烈建议不超过25，若超过25可能造成访问报错
     * @param page     当前页数
     * @param radius   查询半径 取值范围:0-50000。规则：大于50000按默认值，单位：米 默认 5000
     * @param types    查询POI类型
     * @return {@link String}
     */
    public GaoDeRoot around(String location, String keywords, String city, String sortrule, Integer offset, Integer page, Integer radius, String types) {
        String url = "https://restapi.amap.com/v3/place/around?parameters";
        Map<String, Object> map = new HashMap<>();
        map.put("location", location);
        map.put("keywords", keywords);
        map.put("city", city);
        map.put("key", web_key);
        map.put("output", "JSON");
        // 取值范围:0-50000。规则：大于50000按默认值，单位：米
        map.put("radius", radius);
        // 排序规则
        //规定返回结果的排序规则。
        //按距离排序：distance；综合排序：weight
        map.put("sortrule", sortrule);
        // 每页记录数据 强烈建议不超过25，若超过25可能造成访问报错
        map.put("offset", offset);
        // 当前页数
        map.put("page", page);
        map.put("types", types);
        String result = cn.hutool.http.HttpUtil.get(url, map);
        GaoDeRoot root = JSONUtil.toBean(result, GaoDeRoot.class);
        return root;
    }

    /**
     * 关键字搜索（POI 高德）
     * https://lbs.amap.com/api/webservice/guide/api/search/#text
     *
     * @param keywords  查询关键字 规则： 多个关键字用“|”分割
     * @param city      查询城市
     * @param citylimit 仅返回指定城市数据     可选值：true/false
     * @param children  是否按照层级展示子POI数据    可选值：children=1 当为0的时候，子POI都会显示。 当为1的时候，子POI会归类到父POI之中。仅在extensions=all的时候生效  默认0
     * @param offset    每页记录数据 强烈建议不超过25，若超过25可能造成访问报错
     * @param page      当前页数
     * @param types     查询POI类型
     * @return {@link String}
     */
    public GaoDeRoot text(String keywords, String city, Boolean citylimit, String children, Integer offset, Integer page, String types) {
        String url = "https://restapi.amap.com/v3/place/text?parameters";
        Map<String, Object> map = new HashMap<>();
        map.put("citylimit", citylimit);
        map.put("children", children);
        map.put("keywords", keywords);
        map.put("city", city);
        map.put("key", web_key);
        map.put("output", "JSON");
        // 每页记录数据 强烈建议不超过25，若超过25可能造成访问报错
        map.put("offset", offset);
        // 当前页数
        map.put("page", page);
        map.put("types", types);
        String result = cn.hutool.http.HttpUtil.get(url, map);
        GaoDeRoot root = JSONUtil.toBean(result, GaoDeRoot.class);
        return root;
    }


}
