package com.elitesland.fin.application.service.saleinv;

import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.messenger.Messenger;
import com.elitescloud.cloudt.messenger.model.ByteMessageAttachmentVO;
import com.elitescloud.cloudt.messenger.model.MessageAccountVO;
import com.elitescloud.cloudt.messenger.model.MessageAttachmentVO;
import com.elitesland.fin.application.convert.saleinv.SaleInvConvert;
import com.elitesland.fin.application.facade.param.arorder.ArOrderDtlRecordSaveParam;
import com.elitesland.fin.application.facade.param.arorder.ArOrderRecordSaveParam;
import com.elitesland.fin.application.facade.param.saleinv.SaleInvDtlParam;
import com.elitesland.fin.application.facade.param.saleinv.SaleInvEmailParam;
import com.elitesland.fin.application.facade.param.saleinv.SaleInvParam;
import com.elitesland.fin.application.facade.vo.saleinv.*;
import com.elitesland.fin.application.service.arorder.ArOrderService;
import com.elitesland.fin.application.service.workflow.WorkFlowDefKey;
import com.elitesland.fin.common.FinFlexFieldCodeConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.saleinv.SaleInv;
import com.elitesland.fin.domain.entity.saleinv.SaleInvDO;
import com.elitesland.fin.domain.entity.saleinv.SaleInvDtlDO;
import com.elitesland.fin.domain.entity.saleinv.SaleInvdDtlDO;
import com.elitesland.fin.domain.param.saleinv.SaleInvAppPageParam;
import com.elitesland.fin.domain.param.saleinv.SaleInvDtlPageParam;
import com.elitesland.fin.domain.param.saleinv.SaleInvPageParam;
import com.elitesland.fin.domain.service.saleinv.SaleInvDomainService;
import com.elitesland.fin.domain.service.saleinv.SaleInvDtlDomainService;
import com.elitesland.fin.infinity.utils.MessageUtil;
import com.elitesland.fin.infr.dto.saleinv.*;
import com.elitesland.fin.infr.repo.saleinv.SaleInvDtlRepo;
import com.elitesland.fin.infr.repo.saleinv.SaleInvRepo;
import com.elitesland.fin.infr.repo.saleinv.SaleInvdDtlRepo;
import com.elitesland.fin.rpc.workflow.WorkflowRpcService;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.WorkflowConstant;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.application.service.saleinv
 * @date 2022/5/6 15:02
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class SaleInvServiceImpl implements SaleInvService {

    public final SaleInvDomainService saleInvDomainService;
    public final SaleInvDtlDomainService saleInvDtlDomainService;
    public final WorkflowRpcService workflowRpcService;
    public final ArOrderService arOrderService;
    private final TaskExecutor taskExecutor;
    private final FlexFieldUtilService flexFieldUtilService;
    private final SaleInvDtlRepo saleInvDtlRepo;
    private final SaleInvdDtlRepo saleInvdDtlRepo;
    private final SaleInvRepo saleInvRepo;
    private final UdcProvider udcProvider;

    @Override
    @SysCodeProc
    public PagingVO<SaleInvVO> page(SaleInvPageParam param) {
        PagingVO<SaleInvDTO> page = saleInvDomainService.page(param);
        PagingVO<SaleInvVO> res = SaleInvConvert.INSTANCE.convertPage(page);
        if (CollectionUtils.isNotEmpty(res.getRecords())) {
            flexFieldUtilService.handleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.SALE_INV, res.getRecords());
        }

        Map<String, String> orderType3 = udcProvider.getValueMapByUdcCode("yst-order", "ORDER_TYPE3");
        res.getRecords().forEach(record -> {
            record.setDocType3Name(orderType3.get(record.getDocType3()));
        });
        return res;
    }

    @Override
    @SysCodeProc
    public PagingVO<SaleInvAppVO> appPage(SaleInvAppPageParam param) {
        PagingVO<SaleInvAppDTO> page = saleInvDomainService.appPage(param);
        PagingVO<SaleInvAppVO> res = SaleInvConvert.INSTANCE.convertAppPage(page);
        if (res.isNotEmpty()) {
            List<SaleInvAppVO> records = res.getRecords();
            List<Long> invIds = records.stream().map(SaleInvAppVO::getId).collect(Collectors.toList());
            List<SaleInvDtlDO> details = saleInvDtlDomainService.findByMasIds(invIds);
            Map<Long, List<SaleInvDtlDO>> dtlMap = details.stream().collect(Collectors.groupingBy(SaleInvDtlDO::getMasId));

            records.forEach(record -> {

                List<SaleInvDtlDO> saleInvDtlDOS = dtlMap.get(record.getId());
                // 获取saleInvDtlDOS 数组中的 totalAmt
                record.setTotalAmt(saleInvDtlDOS.stream().map(SaleInvDtlDO::getTotalAmt).reduce(BigDecimal.ZERO, BigDecimal::add));

                // 查找最早的rootDocTime，如果没有找到则使用主表创建时间
                Optional<LocalDateTime> earliestRootDocTime = saleInvDtlDOS.stream()
                        .map(SaleInvDtlDO::getRootDocTime)
                        .filter(Objects::nonNull)
                        .min(LocalDateTime::compareTo);
                record.setRootDocTimeFrom(earliestRootDocTime.orElse(record.getCreateTime()));


                // 查找最晚的rootDocTime，如果没有找到则使用主表创建时间
                Optional<LocalDateTime> latestRootDocTime = saleInvDtlDOS.stream()
                        .map(SaleInvDtlDO::getRootDocTime)
                        .filter(Objects::nonNull)
                        .max(LocalDateTime::compareTo);
                record.setRootDocTimeTo(latestRootDocTime.orElse(record.getCreateTime()));
            });
        }

        return res;
    }

    @Override
    @SysCodeProc
    public SaleInvSumVO selectListSum(SaleInvPageParam param){
        SaleInvSumDTO saleInvSumDTO = saleInvDomainService.selectListSum(param);
        SaleInvSumVO saleInvSumVO = SaleInvConvert.INSTANCE.sumDtoConvertSumVo(saleInvSumDTO);
        return saleInvSumVO;
    }

    @Override
    @SysCodeProc
    public List<SaleInvDtlVO> getList(Long masId) {
        List<SaleInvDtlDTO> list = saleInvDtlDomainService.getList(masId);
        List<SaleInvDtlVO> res = SaleInvConvert.INSTANCE.convertListVO(list);
        return res;
    }

    @Override
    public List<SaleInvDtlVO> getSummaryList(List<SaleInvDtlParam> saleInvDtlParamList) {
        return saleInvDtlDomainService.getSummaryList(saleInvDtlParamList);
    }

    @Override
    @SysCodeProc
    public SaleInvVO get(Long id) {
        SaleInvDTO saleInvDTO = saleInvDomainService.get(id);
        SaleInvVO res = SaleInvConvert.INSTANCE.convert(saleInvDTO);
        flexFieldUtilService.handleSingleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.SALE_INV,res);
        return res;
    }

    @Override
    public Long save(SaleInvParam saleInvParam) {
        SaleInv saleInv = SaleInvConvert.INSTANCE.convert(saleInvParam);
        Long res = saleInvDomainService.save(saleInv);
        return res;
    }

    @Override
    public Long del(List<Long> ids) {
        Long res = saleInvDomainService.del(ids);
        return res;
    }

    @Override
    public Long update(SaleInvParam saleInvParam) {
        SaleInv saleInv = SaleInvConvert.INSTANCE.convert(saleInvParam);
        Long res = saleInvDomainService.update(saleInv);
        return res;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = {Exception.class})
    public Long submit(SaleInvParam saleInvParam) {
        SaleInv saleInv = SaleInvConvert.INSTANCE.convert(saleInvParam);
        Long resId = saleInvDomainService.submit(saleInv);
        // 启动工作流
        if (saleInv.getProcInstId() == null
                || WorkflowConstant.CAN_START_PROC_STATUSES.contains(saleInv.getProcInstStatus())) {
            // 启动流程
            String procInstName = "销售发票审核-" + saleInv.getInvRegNo();
            String procKey = WorkFlowDefKey.FIN_SALE_INV.name();
            ProcessInfo processInfo = workflowRpcService.startProcess(procKey, procInstName, resId.toString(), new HashMap<>());
            saleInvDomainService.updateWorkInfo(processInfo, resId);
        }
        return resId;
    }

    @Override
    @SysCodeProc
    public List<SaleInvdDtlVO> getInvdLists(Long masId) {
        List<SaleInvdDtlDTO> list = saleInvDomainService.getInvdLists(masId);
        List<SaleInvdDtlVO> res = SaleInvConvert.INSTANCE.convertInvdListVO(list);
        return res;
    }

    @Override
    public Long updateInvInfo(SaleInvParam saleInvParam) {
        Long res = saleInvDomainService.updateInvInfo(saleInvParam);
        return res;
    }

    @Override
    public PagingVO<SaleInvDtlVO> dtlPage(SaleInvDtlPageParam saleInvDtlPageParam) {
        PagingVO<SaleInvDtlDTO> res = saleInvDtlDomainService.dtlPage(saleInvDtlPageParam);
        return SaleInvConvert.INSTANCE.convertDtlPage(res);
    }

    @Override
    public SaleInvVO queryAmt() {
       return saleInvDomainService.queryAmt();
    }



    /**
     * 金税开票-自动生成应收单
     *
     * @param saleInvParam 销售发票ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void autoCreateArOrder(SaleInvParam saleInvParam){

        Long id = saleInvParam.getId();
        //1.查询发票和销售发票明细
        SaleInvDTO saleInvDTO = saleInvDomainService.get(id);
        List<SaleInvDtlDTO> saleInvDtlDTOList = saleInvDTO.getSaleInvDtlDTOList();
        //2.查询销售已开发票明细
        //List<SaleInvdDtlDTO> saleInvdDtlDTOList = saleInvDomainService.getInvdLists(id);

        //List<SaleInvDtlDTO> list = saleInvDtlDomainService.getList(id);
        //TODO 补充缺少字段
        if (saleInvDTO.getArFlag()) {
            throw new BusinessException("应收单已生成，请勿重复点击");
        }


        ArOrderRecordSaveParam arOrderRecordSaveParam = new ArOrderRecordSaveParam();
        arOrderRecordSaveParam.setSourceDocType(UdcEnum.DOC_TYPE_SAINV.getValueCode());
        arOrderRecordSaveParam.setSourceDocStatus(UdcEnum.DOC_STATUS_SAINV.getValueCode());
        arOrderRecordSaveParam.setSourceNo(saleInvDTO.getApplyNo());
        arOrderRecordSaveParam.setCreateMode(UdcEnum.DOC_CLS_SAINV.getValueCode());
        arOrderRecordSaveParam.setOuCode(saleInvDTO.getOuCode());
        arOrderRecordSaveParam.setOuId(saleInvDTO.getOuId());
        arOrderRecordSaveParam.setOuName(saleInvDTO.getOuName());

        LocalDateTime drawDate = Objects.isNull(saleInvDTO.getDrawDate()) ? LocalDateTime.now() : saleInvDTO.getDrawDate();
        arOrderRecordSaveParam.setBuDate(drawDate);

        arOrderRecordSaveParam.setCurrCode(saleInvDTO.getCurrCode());
        arOrderRecordSaveParam.setCurrName(saleInvDTO.getCurrName());
        arOrderRecordSaveParam.setLocalCurrCode(saleInvDTO.getLocalCurrCode());
        arOrderRecordSaveParam.setLocalCurrName(saleInvDTO.getLocalCurrName());
        arOrderRecordSaveParam.setExchangeRate(saleInvDTO.getExchangeRate());
        arOrderRecordSaveParam.setCustCode(saleInvDTO.getInvCustCode());
        arOrderRecordSaveParam.setCustId(saleInvDTO.getInvCustId());
        arOrderRecordSaveParam.setCustName(saleInvDTO.getInvCustName());
        //arOrderRecordSaveParam.setInOutCust(null);
        //arOrderRecordSaveParam.setRelevanceOuCode(null);
        arOrderRecordSaveParam.setBuCode(null);
        arOrderRecordSaveParam.setBuId(null);
        arOrderRecordSaveParam.setBuName(null);
        arOrderRecordSaveParam.setPayMentCode(null);
        arOrderRecordSaveParam.setPayMentId(null);
        arOrderRecordSaveParam.setPayMentName(null);
        arOrderRecordSaveParam.setProtocolCode(null);
        arOrderRecordSaveParam.setSaleUserId(saleInvDtlDTOList.get(0).getAgentEmpId());
        arOrderRecordSaveParam.setSaleUserCode(null);
        arOrderRecordSaveParam.setSaleUser(saleInvDtlDTOList.get(0).getAgentName());
        //arOrderRecordSaveParam.setDocType2();
        arOrderRecordSaveParam.setEs1(saleInvDTO.getMainCustCode());
        arOrderRecordSaveParam.setEs2(null);
        arOrderRecordSaveParam.setEs3(null);
        arOrderRecordSaveParam.setEs4(null);
        //arOrderRecordSaveParam.setEs5();
        arOrderRecordSaveParam.setEs6(null);



        List<ArOrderDtlRecordSaveParam> arOrderDtlRecordSaveParams = saleInvDtlDTOList.stream().map(saleInvDtlDTO -> {
            ArOrderDtlRecordSaveParam arOrderDtlRecordSaveParam = new ArOrderDtlRecordSaveParam();
            arOrderDtlRecordSaveParam.setSourceNo(saleInvDtlDTO.getSourceNo());
            arOrderDtlRecordSaveParam.setSourceLine(saleInvDtlDTO.getSourceLine());
            arOrderDtlRecordSaveParam.setItemCode(saleInvDtlDTO.getItemCode());
            arOrderDtlRecordSaveParam.setItemName(saleInvDtlDTO.getItemName());
            arOrderDtlRecordSaveParam.setItemType(saleInvDtlDTO.getItemType());
            arOrderDtlRecordSaveParam.setSmallCateCode(null);
            arOrderDtlRecordSaveParam.setSmallCateName(null);
            arOrderDtlRecordSaveParam.setUom(saleInvDtlDTO.getUom());
            arOrderDtlRecordSaveParam.setUomName(saleInvDtlDTO.getUomName());
            arOrderDtlRecordSaveParam.setQty(saleInvDtlDTO.getQty());

            //结算单价 没有就在sale_inv_dtl表新增，可开票含税金额/数量
            BigDecimal settlePrice = saleInvDtlDTO.getInvAmt().divide(saleInvDtlDTO.getQty(), 6, RoundingMode.HALF_UP);
            //结算未税单价 没有就在sale_inv_dtl表新增，用结算单价/(1+税率)
            BigDecimal settleNetPrice = settlePrice.divide((BigDecimal.ONE.add(saleInvDtlDTO.getTaxRate())), 6, RoundingMode.HALF_UP);

            arOrderDtlRecordSaveParam.setExclTaxPrice(settleNetPrice);
            arOrderDtlRecordSaveParam.setPrice(settlePrice);
            arOrderDtlRecordSaveParam.setTaxRate(saleInvDtlDTO.getTaxRate());
            arOrderDtlRecordSaveParam.setTotalAmt(saleInvDtlDTO.getInvAmt());

            //可开票未税金额 没有就在sale_inv_dtl表新增，可开票含税金额/(1+税率)
            BigDecimal invNetAmt = saleInvDtlDTO.getInvAmt().divide(BigDecimal.ONE.add(saleInvDtlDTO.getTaxRate()), 2, RoundingMode.HALF_UP);

            arOrderDtlRecordSaveParam.setExclTaxAmt(invNetAmt);

            //可开票税额 没有就在sale_inv_dtl表新增，可开票含税金额-可开票未税金额
            BigDecimal invTaxAmt = saleInvDtlDTO.getInvAmt().subtract(invNetAmt);
            arOrderDtlRecordSaveParam.setTaxAmt(invTaxAmt);


            // 含税金额(本位币)=含税金额*汇率
            BigDecimal totalCurAmt = arOrderDtlRecordSaveParam.getTotalAmt().multiply(arOrderRecordSaveParam.getExchangeRate()).setScale(2, RoundingMode.HALF_UP);
            arOrderDtlRecordSaveParam.setTotalCurAmt(totalCurAmt);
            // 不含税金额(本位币)=不含税金额*汇率
            BigDecimal exclTaxCurAmt = arOrderDtlRecordSaveParam.getExclTaxAmt().multiply(arOrderRecordSaveParam.getExchangeRate()).setScale(2, RoundingMode.HALF_UP);
            arOrderDtlRecordSaveParam.setExclTaxCurAmt(exclTaxCurAmt);
            // 税额(本位币) =含税金额(本位币)-不含税金额(本位币)
            BigDecimal taxCurAmt = totalCurAmt.subtract(exclTaxCurAmt).setScale(2, RoundingMode.HALF_UP);
            arOrderDtlRecordSaveParam.setTaxCurAmt(taxCurAmt);
            //arOrderDtlRecordSaveParam.setTotalCurAmt(null);
            //arOrderDtlRecordSaveParam.setExclTaxCurAmt(null);
            //arOrderDtlRecordSaveParam.setTaxCurAmt(null);
            arOrderDtlRecordSaveParam.setBuCode(null);
            arOrderDtlRecordSaveParam.setBuId(null);
            arOrderDtlRecordSaveParam.setBuName(null);
            arOrderDtlRecordSaveParam.setEs11(saleInvDtlDTO.getSourceNo());
            arOrderDtlRecordSaveParam.setEs12(saleInvDtlDTO.getSourceId());
            arOrderDtlRecordSaveParam.setEs13(saleInvDTO.getMainCustCode());
            arOrderDtlRecordSaveParam.setEs14(saleInvDTO.getCustCode());
            arOrderDtlRecordSaveParam.setEs15(null);
            arOrderDtlRecordSaveParam.setEs16(saleInvDtlDTO.getTaxCode());
            arOrderDtlRecordSaveParam.setEs17(saleInvDtlDTO.getInvNo());
            arOrderDtlRecordSaveParam.setEs18(saleInvDtlDTO.getInvType());
            arOrderDtlRecordSaveParam.setEs19(null);
            arOrderDtlRecordSaveParam.setProtocolCode(null);
            //saleInvDtlDTO.getAgentEmpId() 业务员ID
            if (Objects.nonNull(saleInvDtlDTO.getAgentEmpId())){
                arOrderDtlRecordSaveParam.setEs20(String.valueOf(saleInvDtlDTO.getAgentEmpId()));
            }
            arOrderDtlRecordSaveParam.setEs21(null);
            arOrderDtlRecordSaveParam.setEs22(null);
            arOrderDtlRecordSaveParam.setEs23(null);
            arOrderDtlRecordSaveParam.setEs24(null);

            return arOrderDtlRecordSaveParam;
        }).collect(Collectors.toList());

        //明细含税金额汇总
        BigDecimal totalAmtSum = arOrderDtlRecordSaveParams.stream().map(ArOrderDtlRecordSaveParam::getTotalAmt).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
        //明细不含税金额汇总
        BigDecimal exclTaxAmtSum = arOrderDtlRecordSaveParams.stream().map(ArOrderDtlRecordSaveParam::getExclTaxAmt).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
        //明细税额汇总
        BigDecimal taxAmtSum = arOrderDtlRecordSaveParams.stream().map(ArOrderDtlRecordSaveParam::getTaxAmt).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);

        //明细含税金额汇总(本位币)
        BigDecimal totalCurAmtSum = arOrderDtlRecordSaveParams.stream().map(ArOrderDtlRecordSaveParam::getTotalCurAmt).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
        //明细不含税金额汇总(本位币)
        BigDecimal exclTaxCurAmtSum = arOrderDtlRecordSaveParams.stream().map(ArOrderDtlRecordSaveParam::getExclTaxCurAmt).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
        //明细税额汇总(本位币)
        BigDecimal taxCurAmtSum = arOrderDtlRecordSaveParams.stream().map(ArOrderDtlRecordSaveParam::getTaxCurAmt).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);


        arOrderRecordSaveParam.setTotalAmt(totalAmtSum);
        arOrderRecordSaveParam.setExclTaxAmt(exclTaxAmtSum);
        arOrderRecordSaveParam.setTaxAmt(taxAmtSum);
        arOrderRecordSaveParam.setTotalCurAmt(totalCurAmtSum);
        arOrderRecordSaveParam.setExclTaxCurAmt(exclTaxCurAmtSum);
        arOrderRecordSaveParam.setTaxCurAmt(taxCurAmtSum);

        arOrderRecordSaveParam.setArOrderDtlRecordSaveParams(arOrderDtlRecordSaveParams);

        CompletableFuture.runAsync(() ->{
            arOrderService.autoCreate(arOrderRecordSaveParam);
        }, taskExecutor);

    }

    @Override
    @SysCodeProc
    public SaleInvAppVO appGet(Long id) {
        SaleInvAppDTO saleInvAppDTO = saleInvDomainService.appGet(id);
        SaleInvAppVO saleInvAppVO = SaleInvConvert.INSTANCE.convertAppPage(saleInvAppDTO);
        List<SaleInvDtlDO> dtlList = saleInvDtlRepo.findAllByMasIdIn(List.of(id));
        Map<String, List<SaleInvDtlDO>> dtlGroup = dtlList.stream().collect(Collectors.groupingBy(d -> getDtlKey(d.getSourceNo(), d.getRootDocNo())));
        saleInvAppVO.setDocNum(dtlGroup.size());
        List<SaleInvDtlAppVO> detailList = new ArrayList<>();
        dtlGroup.values().forEach(list -> {
            SaleInvDtlAppVO invDtlAppVO = new SaleInvDtlAppVO();
            SaleInvDtlDO saleInvDtlDO = list.get(0);
            invDtlAppVO.setSourceNo(saleInvDtlDO.getSourceNo());
            invDtlAppVO.setSourceId(saleInvDtlDO.getSourceId());
            invDtlAppVO.setSourceTime(saleInvDtlDO.getSourceTime());
            invDtlAppVO.setRootDocNo(saleInvDtlDO.getRootDocNo());
            invDtlAppVO.setRootDocId(saleInvDtlDO.getRootDocId());
            invDtlAppVO.setRootDocTime(saleInvDtlDO.getRootDocTime());
            invDtlAppVO.setRecApplyTime(saleInvAppVO.getRecApplyTime());
            invDtlAppVO.setTotalAmt(list.stream().map(SaleInvDtlDO::getTotalAmt).reduce(BigDecimal.ZERO, BigDecimal::add));
            List<SaleInvDtlAppVO.SaleInvDtlItemAppVO> itemList = list.stream().map(d -> {
                SaleInvDtlAppVO.SaleInvDtlItemAppVO item = new SaleInvDtlAppVO.SaleInvDtlItemAppVO();
                item.setItemCode(d.getItemCode());
                item.setItemName(d.getItemName());
                item.setQty(d.getQty());
                return item;
            }).toList();
            invDtlAppVO.setDetailList(itemList);
            detailList.add(invDtlAppVO);
        });
        saleInvAppVO.setDocDetails(detailList);

        // 发票图片
        List<SaleInvdDtlDO> allById = saleInvdDtlRepo.findAllByMasId(id);
        if (CollectionUtils.isNotEmpty(allById)) {
            List<String> list = allById.stream().map(SaleInvdDtlDO::getInvPdfBase64).filter(Objects::nonNull).toList();
            saleInvAppVO.setInvPics(list);
            saleInvAppVO.setInvNum(list.size());
        } else {
            saleInvAppVO.setInvNum(0);
        }
        return saleInvAppVO;
    }

    @Override
    public void sendEmail(SaleInvEmailParam param) {
        log.info("开始发送邮件,入参:{}", JSON.toJSON(param));
        List<Long> idList = param.getIds();
        String email = param.getEmail();
        List<SaleInvDO> allById = saleInvRepo.findAllById(idList);
        if (CollectionUtils.isEmpty(allById)) {
            throw new BusinessException("发票不存在");
        }

        Map<Long, SaleInvDO> saleInvDOMap = allById.stream().collect(Collectors.toMap(SaleInvDO::getId, Function.identity()));

        // 获取发票
        List<SaleInvdDtlDO> saleInvdDtlDOList = saleInvdDtlRepo.findAllByMasIdIn(saleInvDOMap.keySet());
        if (CollectionUtils.isEmpty(saleInvdDtlDOList)) {
            throw new BusinessException("发票详情不存在");
        }
        Map<Long, List<SaleInvdDtlDO>> invdMap = saleInvdDtlDOList.stream().collect(Collectors.groupingBy(SaleInvdDtlDO::getMasId));

        // 获取发票申请单
        List<SaleInvDtlDO> allByMasIdIn = saleInvDtlRepo.findAllByMasIdIn(idList);
        Map<Long, List<SaleInvDtlDO>> invDtlMap;
        if (CollectionUtils.isNotEmpty(allByMasIdIn)) {
            invDtlMap = allByMasIdIn.stream().collect(Collectors.groupingBy(SaleInvDtlDO::getMasId));
        } else {
            invDtlMap = new HashMap<>();
        }

        for (SaleInvDO saleInvDO : allById) {
            List<SaleInvdDtlDO> saleInvdDtlDOS = invdMap.get(saleInvDO.getId());
            if (CollectionUtils.isEmpty(saleInvdDtlDOS)) {
                log.error("发票详情为空:{}", saleInvDO.getId());
                continue;
            }
            SaleInvdDtlDO saleInvdDtlDO = saleInvdDtlDOS.get(0);
            if (!saleInvdDtlDO.getInvState().equals(UdcEnum.INV_STATE_SUCCESS.getValueCode())) {
                log.error("只有开票状态为开票成功才能发送邮件:{}", saleInvDO.getId());
                continue;
            }
            if (email == null && saleInvDO.getInvEmail() == null) {
                log.error("收件邮箱为空:{}", saleInvDO.getId());
                continue;
            }

            MultipartFile file = MessageUtil.convert(saleInvdDtlDO.getInvPdfBase64(), saleInvdDtlDO.getInvNo() + saleInvDO.getCustName() + ".pdf", "application/pdf");
            List<MessageAttachmentVO> attachmentVOList;
            try {
                attachmentVOList = file.isEmpty() ? Collections.emptyList() :
                        List.of(new ByteMessageAttachmentVO(file.getBytes(), file.getOriginalFilename(), file.getContentType()));
            } catch (IOException e) {
                log.error("获取文件内容失败:{}", e.getMessage(), e);
                continue;
            }

            try {
                // 日期格式转换String 年月日时分秒
                String invDateFormat = "";
                if (saleInvdDtlDO.getInvDate() != null) {
                    invDateFormat = saleInvdDtlDO.getInvDate().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
                }
                // 时间段+门店名称,时间段取选取订单的最小时间与最大时间。比如:20250601-20250630XX门店 + 流水号
                List<SaleInvDtlDO> details = invDtlMap.get(saleInvDO.getId());
                LocalDateTime minDate = null;
                LocalDateTime maxDate = null;
                for (SaleInvDtlDO detail : details) {
                    if (detail.getConfirmTime() != null) {
                        if (minDate == null || detail.getConfirmTime().isBefore(minDate)) {
                            minDate = detail.getConfirmTime();
                        }
                        if (maxDate == null || detail.getConfirmTime().isAfter(maxDate)) {
                            maxDate = detail.getConfirmTime();
                        }
                    }
                }

                String minDateStr = null;
                String maxDateStr = null;
                if (minDate != null) {
                    DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
                    minDateStr = minDate.format(formatter);
                    maxDateStr = maxDate.format(formatter);
                }

                // 金额去除结尾多余的0
                String totalAmt = saleInvdDtlDO.getTotalAmt().stripTrailingZeros().toPlainString();
                Messenger.richTextEmail()
                        //开票方公司名称+"发票"+发票号码'
                        .setSubject("爷爷不泡茶电子发票[" + saleInvDO.getCustName() + "]([" + saleInvDO.getStoreCode() + "])")
                        .setContent("<!DOCTYPE html>\n" +
                                "<html lang=\"zh-CN\">\n" +
                                "<head>\n" +
                                "    <meta charset=\"UTF-8\">\n" +
                                "    <style>\n" +
                                "        body {\n" +
                                "            font-family: Arial, sans-serif;\n" +
                                "            font-size: 14px;\n" +
                                "            line-height: 1.5;\n" +
                                "            color: #333333;\n" +
                                "        }\n" +
                                "\n" +
                                "        a {\n" +
                                "            color: #0073aa;\n" +
                                "        }\n" +
                                "    </style>\n" +
                                "</head>\n" +
                                "<body>\n" +
                                "<div>\n" +
                                "    <p>尊敬的[" + saleInvDO.getInvCustName() + "]</p>\n" +
                                "    <p>感谢您对爷爷不泡茶的支持。</p>\n" +
//                                "    <p>我们已为您开具了[" + saleInvDO.getCustName() + "]张电子发票，总金额" + saleInvDO.getTotalAmt() + "，涵盖[" + saleInvDO.getTotalAmt() + "]至[" + saleInvDO.getTotalAmt() + "]期间的款项。为方便您的管理,发票已随本邮件附上。</p>\n" +
                                "    <p>我们已为您开具了[1]张电子发票，总金额" + totalAmt + "，涵盖[" + minDateStr + "]至[" + maxDateStr + "]期间的款项。为方便您的管理，发票已随本邮件附上。</p>\n" +
                                //详细清单如下:
//                                "    <p>发票号码:" + saleInvdDtlDO.getInvNo() + "</p>\n" +
                                "\n" +
                                "\n" +
                                "    <!--签名-->\n" +
                                "    <div style=\"margin-top:50px;\">\n" +
                                "        <div style=\"border-bottom:1px dashed #e5e5e5;\"></div>\n" +
                                "        <div style=\"padding:8px;color:#999999; font-size:12px\">疑问咨询:如有疑问,请联系督导处理。</div>\n" +
//                                "        <div style=\"padding:8px;color:#999999; font-size:15px\">您已收到数电发票。其法律效力、基本用途等与现有纸质发票相同。</div>\n" +
                                "    </div>\n" +
                                "</div>\n" +
                                "</body>\n" +
                                "</html>")
                        .setReceiverList(List.of(new MessageAccountVO(email == null ? saleInvDO.getEmail() : email)))
                        .setAttachments(attachmentVOList)
                        .build()
                        .send();
            } catch (Exception e) {
                log.error("发送邮件失败:{}", e.getMessage(), e);
            }
        }
    }

    public String getDtlKey(String sourceNo, String rootDocNo) {
        return rootDocNo + "_" + sourceNo;
    }

}
