package com.elitesland.fin.application.service.invoice.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.el.coordinator.core.common.exception.BusinessException;
import com.elitescloud.boot.mq.MessageQueueTemplate;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitesland.fin.Application;
import com.elitesland.fin.application.convert.invoice.InvoiceConvert;
import com.elitesland.fin.application.convert.saleinv.SaleInvConvert;
import com.elitesland.fin.application.facade.dto.mq.DemoMqMessageDTO;
import com.elitesland.fin.application.facade.param.invoice.InvoiceApplyParam;
import com.elitesland.fin.application.facade.param.invoice.InvoiceApplySaveParam;
import com.elitesland.fin.application.facade.vo.invoice.InvoiceAwaitRespVO;
import com.elitesland.fin.application.facade.vo.invoice.InvoiceSaveVO;
import com.elitesland.fin.application.service.invoice.InvoiceAwaitService;
import com.elitesland.fin.application.service.invoice.InvoiceSaveService;
import com.elitesland.fin.application.service.saleinv.SaleInvService;
import com.elitesland.fin.application.service.workflow.WorkFlowDefKey;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinFlexFieldCodeConstant;
import com.elitesland.fin.common.PaymentRecordTypeEnum;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.saleinv.*;
import com.elitesland.fin.domain.service.saleinv.SaleInvDomainService;
import com.elitesland.fin.infinity.aisino.enums.RedStateEnum;
import com.elitesland.fin.infinity.aisino.service.AisinoPayloadService;
import com.elitesland.fin.infinity.aisino.service.AisinoService;
import com.elitesland.fin.infinity.aisino.vo.param.AisinoBlueInvoiceApplyParam;
import com.elitesland.fin.infinity.aisino.vo.param.AisinoCoverParamPayload;
import com.elitesland.fin.infinity.aisino.vo.param.InvoiceRequestSerialNoParam;
import com.elitesland.fin.infinity.aisino.vo.resp.AisinoResultResp;
import com.elitesland.fin.infinity.aisino.vo.resp.BlueInvoiceApplyRespVO;
import com.elitesland.fin.infr.dto.invoice.InvoiceApplyDTO;
import com.elitesland.fin.infr.repo.saleinv.*;
import com.elitesland.fin.param.saleinv.InvoiceDetailSaveParam;
import com.elitesland.fin.repo.invoice.InvoiceAwaitRepoProc;
import com.elitesland.fin.rpc.workflow.WorkflowRpcService;
import com.elitesland.fin.utils.SysUtils;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.WorkflowConstant;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
public abstract class AbstractInvoiceSaveServiceImpl implements InvoiceSaveService, InitializingBean {
    protected static final Map<PaymentRecordTypeEnum, AbstractInvoiceSaveServiceImpl> SERVICE_MAP = new HashMap<>();
    @Autowired
    protected SaleInvRepoProc saleInvRepoProc;
    @Autowired
    @Lazy
    protected AisinoService aisinoService;
    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    @Lazy
    private InvoiceAwaitService invoiceAwaitService;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private FlexFieldUtilService flexFieldUtilService;
    @Autowired
    private SaleInvRepo saleInvRepo;
    @Autowired
    private SaleInvDtlRepoProc saleInvDtlRepoProc;
    @Autowired
    protected InvoiceAwaitRepoProc invoiceAwaitRepoProc;
    @Autowired
    private SaleInvDomainService saleInvDomainService;
    @Autowired
    private WorkflowRpcService workflowRpcService;
    @Autowired
    private MessageQueueTemplate messageQueueTemplate;
    @Autowired
    private SaleInvDtlRepo saleInvDtlRepo;
    @Autowired
    private SaleInvdDtlRepo saleInvdDtlRepo;
    @Autowired
    private AisinoPayloadService aisinoPayloadService;
    @Autowired
    private SaleInvService saleInvService;


    @Override
    public void afterPropertiesSet() {
        SERVICE_MAP.put(getSourceType(), this);
    }

    /**
     * 子类设置单据类型
     *
     * @return
     */
    protected abstract PaymentRecordTypeEnum getSourceType();

    /**
     * 每种单据处理完成后的回调
     *
     * @param records
     */
    protected abstract void callBackMethod(List<InvoiceAwaitRespVO> records, List<InvoiceApplyDTO> applyList);

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<InvoiceSaveVO> save(InvoiceApplyParam param) {
        log.info("开始保存开票单,入参为{}", JSON.toJSONString(param));
        // 校验1
        // 获取待开发票详情
        List<InvoiceAwaitRespVO> records = invoiceAwaitService.checkAndQueryInvoiceAwait(param);
        // 校验2
        String sourceType = getSourceType(records);
        PaymentRecordTypeEnum strategyEnum;
        try {
            strategyEnum = PaymentRecordTypeEnum.valueOf(sourceType);
        } catch (IllegalArgumentException e) {
            throw new BusinessException("来源单据类型未定义");
        }

        // 获取子类
        AbstractInvoiceSaveServiceImpl abstractInvoiceSaveService = SERVICE_MAP.get(strategyEnum);

        if (Objects.isNull(abstractInvoiceSaveService)) {
            throw new BusinessException("未定义此来源单据类型的实现类，类型为" + strategyEnum);
        }
        // 组装保存参数
        List<InvoiceApplySaveParam> saveParamList = invoiceAwaitService.getInvoiceApplySaveParams(records, param);

        // 保存创建开票申请单
        log.info("开始调用开票单创建方法,参数为{}", JSON.toJSONString(saveParamList));
        // 保存开票申请单
        List<InvoiceApplyDTO> applyList = new ArrayList<>();
        saveParamList.forEach(saveParam -> {
            InvoiceApplyDTO apply = saveAndSubmitInvoice(saveParam);
            if (apply == null || apply.getApplyId() == null) {
                log.error("开票申请单创建结果为空，参数为{}", JSON.toJSONString(saveParam));
                throw new BusinessException("开票数据异常，请联系管理员");
            }
            saveParam.setApplyNo(apply.getApplyNo());
            saveParam.setId(apply.getApplyId());
            applyList.add(apply);
        });

        // 每种单据处理完成后的回调
        abstractInvoiceSaveService.callBackMethod(records, applyList);

        // 返回数据
        List<InvoiceSaveVO> resultList = new ArrayList<>();
        for (InvoiceApplyDTO invoiceApplyDTO : applyList) {
            InvoiceSaveVO result = new InvoiceSaveVO();
            result.setApplyId(invoiceApplyDTO.getApplyId());
            result.setApplyNo(invoiceApplyDTO.getApplyNo());
            resultList.add(result);
        }
        log.info("开票申请单创建成功，结果为{}", JSON.toJSONString(resultList));
        return resultList;
    }


    public InvoiceApplyDTO saveAndSubmitInvoice(InvoiceApplySaveParam param) {
        SaleInv saleInv = InvoiceConvert.INSTANCE.toSaleInv(param);
        // 开票状态 UDC[yst-supp:APPLY_STATUS]
        saleInv.setOrderState(UdcEnum.INVOICE_STATUS_DRAFT.getValueCode());
        param.setInvoiceAwaitStatus(UdcEnum.INVOICE_AWAIT_STATUS_ING.getValueCode());
        // 业务字段校验
        saleInv.check();
        saleInv.checkDtl();
        log.info("[YST-FIN] saleInv: {}", saleInv);
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        SaleInvDO execute = transactionTemplate.execute(transactionStatus -> {
            try {
                // 保存表头
                SaleInvDO convertDO = SaleInvConvert.INSTANCE.convert(saleInv);
                log.info("[YST-FIN] convertDO: {}", convertDO);
                flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.SALE_INV, convertDO);
                SaleInvDO save = saleInvRepo.save(convertDO);
                // 保存明细
                saleInvDtlRepoProc.delByMasId(List.of(save.getId()));
                List<SaleInvDtl> saleInvDtls = saleInv.getSaleInvDtls();
                List<SaleInvDtlDO> details = SaleInvConvert.INSTANCE.convertListDO(saleInvDtls);
                log.info("[YST-FIN] convert details: {}", details);
                details.forEach(row -> {
                    row.setMasId(save.getId());
                    saleInvDtlRepo.save(row);
                });
                return save;
            } catch (Exception e) {
                log.error("待开发票生成销售发票出错:{}", e.getMessage(), e);
                //回滚
                transactionStatus.setRollbackOnly();
                throw new com.elitescloud.boot.exception.BusinessException(ApiCode.FAIL, "待开发票生成销售发票保存时出错");
            }
        });
        Set<Long> awaitId = param.getDetails().stream().map(InvoiceDetailSaveParam::getInvoiceAwaitId).filter(Objects::nonNull).collect(Collectors.toSet());

        // 不是自动审批的就走审批流
//        if (ObjUtil.equals(param.getAutoReview(), Boolean.FALSE)) {
//            if (execute != null) {
//                workFlow(execute);
//            }
//        }
        InvoiceApplyDTO result = new InvoiceApplyDTO();
        result.setApplyId(execute.getId());
        result.setApplyNo(execute.getApplyNo());
        result.setInvoiceAwaitIds(awaitId);
        return result;
    }

    private BigDecimal add(List<InvoiceDetailSaveParam> details, Function<InvoiceDetailSaveParam, BigDecimal> func) {
        return details.stream().map(func).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 开票申请接口
     *
     * @param saveParam
     */
    public Boolean invoiceApply(InvoiceApplySaveParam saveParam, SaleInvdDtlDO saleInvdDtlDO) {
        Map<String, List<InvoiceDetailSaveParam>> collect = saveParam.getDetails().stream().collect(Collectors.groupingBy(InvoiceDetailSaveParam::getItemCode));
        List<InvoiceDetailSaveParam> result = new ArrayList<>();

        String technologyCostsItemCode = SysUtils.getTechnologyCosts();
        String marketingCostsItemCode = SysUtils.getMarketingCosts();
        String operatingCostsItemCode = SysUtils.getOperatingCosts();

        List<String> costsItemCodes = List.of(technologyCostsItemCode, marketingCostsItemCode, operatingCostsItemCode);
        log.info("costsItemCodes22: {}", costsItemCodes);
        collect.forEach((itemCode, details) -> {
            InvoiceDetailSaveParam row = details.get(0);
            InvoiceDetailSaveParam curr = new InvoiceDetailSaveParam();
            curr.setInvNature(row.getInvNature());
            curr.setItemId(row.getItemId());
            curr.setItemCode(row.getItemCode());
            curr.setItemName(row.getItemName());
            curr.setTaxCode(row.getTaxCode());
            curr.setTaxName(row.getTaxName());
            curr.setItemType(row.getItemType());
            curr.setUom(row.getUom());
            curr.setUomName(row.getUomName());

            // 如果是三个费用明细行，则数量默认为1
            if (costsItemCodes.contains(row.getItemCode())) {
                curr.setQty(BigDecimal.ONE);
            } else {
                curr.setQty(add(details, InvoiceDetailSaveParam::getQty));
            }

            BigDecimal priceTotal = add(details, InvoiceDetailSaveParam::getTotalAmt);
            curr.setPrice(priceTotal.divide(curr.getQty(), 2, RoundingMode.HALF_UP));
            curr.setTotalAmt(add(details, InvoiceDetailSaveParam::getTotalAmt));
            curr.setTaxRate(row.getTaxRate());
            curr.setTaxAmt(add(details, InvoiceDetailSaveParam::getTaxAmt));
            curr.setExclTaxAmt(add(details, InvoiceDetailSaveParam::getExclTaxAmt));
            curr.setFlDeductionAmt(add(details, InvoiceDetailSaveParam::getFlDeductionAmt));
            curr.setInvDiscountAmt(add(details, InvoiceDetailSaveParam::getInvDiscountAmt));
            curr.setDiscountAmt(add(details, InvoiceDetailSaveParam::getDiscountAmt));
            curr.setInvAmt(add(details, InvoiceDetailSaveParam::getInvAmt));
            result.add(curr);
        });
        if (CollectionUtil.isNotEmpty(result)) {
            saveParam.setDetails(result);
        }
        // 原始入参
        AisinoBlueInvoiceApplyParam applyParam = aisinoPayloadService.blueInvoiceApplyInitial(saveParam, saleInvdDtlDO, costsItemCodes);
        String initialParam = JSON.toJSONString(applyParam);
        // 获取结构入参
        AisinoCoverParamPayload finialParam = aisinoPayloadService.assembleAisinoPayload(initialParam, saveParam.getSaleTaxNo());
        // 调用航信接口
        BlueInvoiceApplyRespVO response = null;
        String failCause = null;
        try {
            response = aisinoService.blueInvoiceApply(finialParam, initialParam);
        } catch (Exception e) {
            log.error("蓝票申请失败:{}", e.getMessage(), e);
            failCause = e.getMessage();
        }
        if (saleInvdDtlDO == null) {
            saleInvdDtlDO = new SaleInvdDtlDO();
            saleInvdDtlDO.setCreator(saveParam.getInvUser());
        }

        // 失败原因
        if (Strings.isNotBlank(failCause)) {
            saleInvdDtlDO.setInvFailCause(failCause);
        }
        // 蓝票
        saleInvdDtlDO.setRedBlueType(UdcEnum.INV_RED_BLUE_TYPE_BLUE.getValueCode());

        saleInvdDtlDO.setOriginalContent(initialParam);
        // 开票申请表头id
        saleInvdDtlDO.setMasId(saveParam.getId());
        // 流水号
        saleInvdDtlDO.setFlowNo(applyParam.getFPQQLSH());
        // 开票金额
        BigDecimal totalAmt = new BigDecimal(applyParam.getREQUEST().getJSHJ());
        saleInvdDtlDO.setTotalAmt(totalAmt);
        // 销方税号
        saleInvdDtlDO.setSaleTaxNo(saveParam.getSaleTaxNo());
        return saveSaleInvDDtl(saleInvdDtlDO, response, failCause);
    }


    /**
     * 保存发票开票信息
     *
     * @param saleInvdDtlDO
     * @param response
     * @return
     */
    private Boolean saveSaleInvDDtl(SaleInvdDtlDO saleInvdDtlDO, BlueInvoiceApplyRespVO response, String failCause) {
        // 开票日期
        saleInvdDtlDO.setInvDate(LocalDateTime.now());
        String orderState = null;
        boolean result = false;
        if (response != null && Arrays.asList("0000", "200").contains(response.getCODE())) {
            BlueInvoiceApplyRespVO.Data data = response.getDATA();
            // 开票状态
            saleInvdDtlDO.setInvState(RedStateEnum.BLUE_APPLY_SUCCESS.getCode());
            saleInvdDtlDO.setInvNo(data.getFPHM());
            saleInvdDtlDO.setInvFailCause(null);
            saleInvdDtlDO.setRedState(UdcEnum.RED_STATE_NO.getValueCode());
            if (Strings.isNotBlank(data.getKPRQ())) {
                // 开票日期
                LocalDateTime invoiceDate = LocalDateTime.parse(data.getKPRQ(), DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
                saleInvdDtlDO.setInvDate(invoiceDate);
            }
            switch (data.getKPZT()) {
                case "1" -> {
                    saleInvdDtlDO.setInvState(UdcEnum.INV_STATE_SUCCESS.getValueCode());
                    orderState = UdcEnum.INV_STATE_SUCCESS.getValueCode();
                    saleInvdDtlDO.setInvPdfUrl(data.getFPURL());
                    saleInvdDtlDO.setInvFailCause(data.getKPZT_MSG());
                    result = true;
                }
                case "2" -> {
                    saleInvdDtlDO.setInvState(UdcEnum.INV_STATE_FAIL.getValueCode());
                    // 开票失败原因
                    orderState = UdcEnum.INV_STATE_FAIL.getValueCode();
                    saleInvdDtlDO.setInvFailCause(data.getKPZT_MSG());
                }
                case "0" -> {
                    saleInvdDtlDO.setInvState(UdcEnum.INV_STATE_ING.getValueCode());
                    orderState = UdcEnum.INV_STATE_ING.getValueCode();
                }
            }
        } else {
            // 开票状态
            saleInvdDtlDO.setInvState(UdcEnum.INV_STATE_FAIL.getValueCode());
            // 开票失败原因
            saleInvdDtlDO.setInvFailCause(response != null ? response.getRETURNMESSAGE() : Strings.isNotBlank(failCause) ? failCause : "开票申请返回数据为空");
            orderState = UdcEnum.INV_STATE_FAIL.getValueCode();
        }
        saleInvdDtlRepo.save(saleInvdDtlDO);
        saleInvRepo.updateOrderStateById(orderState, saleInvdDtlDO.getMasId());
        return result;
    }


    /**
     * 重新申请发票
     *
     * @param saleInvdDtlDO
     */
    public Boolean invoiceReApply(SaleInvdDtlDO saleInvdDtlDO) {
        // 原始入参
        String initialParam = saleInvdDtlDO.getOriginalContent();
        // 获取结构入参
        AisinoCoverParamPayload finialParam = aisinoPayloadService.assembleAisinoPayload(initialParam, saleInvdDtlDO.getSaleTaxNo());
        // 调用航信接口
        BlueInvoiceApplyRespVO response = null;
        String failCause = null;
        try {
            response = aisinoService.blueInvoiceApply(finialParam, initialParam);
        } catch (Exception e) {
            log.error("蓝票申请失败:{}", e.getMessage(), e);
            failCause = e.getMessage();
        }
        // 保存开票信息
        return saveSaleInvDDtl(saleInvdDtlDO, response, failCause);
    }


    /**
     * 根据流水号查询发票信息
     *
     * @param saleInvdDtlDO
     */
    public Boolean getInvoiceInfoByFlowNo(SaleInvdDtlDO saleInvdDtlDO) {
        String failCause = null;
        BlueInvoiceApplyRespVO response = null;
        try {
            // 原始入参
            InvoiceRequestSerialNoParam applyParam = aisinoPayloadService.getInvoiceInfoByFlowNo(saleInvdDtlDO);
            String initialParam = JSON.toJSONString(applyParam);
            // 获取结构入参
            AisinoCoverParamPayload finialParam = aisinoPayloadService.assembleAisinoPayload(initialParam, saleInvdDtlDO.getSaleTaxNo());
            // 调用航信接口
            response = aisinoService.getInvoiceInfoByFlowNo(finialParam, initialParam);
        } catch (Exception e) {
            String msg = "获取开票结果失败:" + e.getMessage();
            log.error(msg, e);
            failCause = msg;
        }
        // 保存开票信息
        return saveSaleInvDDtl(saleInvdDtlDO, response, failCause);
    }

    public void getBlueResult(SaleInvdDtlDO saleInvDO) {
        if (saleInvDO.getInvNo() != null && saleInvDO.getInvDate() != null) {
            String blueResultParam = aisinoPayloadService.getBlueResult(saleInvDO.getInvNo(), saleInvDO.getInvDate());
            // 加密秘钥
            AisinoCoverParamPayload payload = aisinoPayloadService.assembleAisinoPayload(blueResultParam, saleInvDO.getSaleTaxNo());
            // 更新上传结果
            AisinoResultResp aisinoResultResp = aisinoService.blueInvoiceResult(payload, blueResultParam);
            if (Objects.equals(aisinoResultResp.getCODE(), "0000")) {
                AisinoResultResp.Data data1 = aisinoResultResp.getDATA();
                saleInvDO.setInvState(RedStateEnum.BLUE_RESULT_SUCCESS.getCode());
                saleInvDO.setRedState(RedStateEnum.RED_PUNCH.getCode());
                saleInvDO.setInvPdfUrl(data1.getDATA());
            } else {
                log.error("[YST-FIN] 获取蓝票开票结果失败:{}", aisinoResultResp.getRETURNMESSAGE());
                saleInvDO.setInvState(RedStateEnum.BLUE_RESULT_FAIL.getCode());
                saleInvDO.setInvFailCause(aisinoResultResp.getRETURNMESSAGE());
            }
        }
    }

    public Boolean getBase64Result(SaleInvdDtlDO saleInvDO) {
        if (saleInvDO.getInvNo() != null && saleInvDO.getInvDate() != null) {
            String param = aisinoPayloadService.getBase64Result(saleInvDO.getInvNo(), saleInvDO.getInvDate());
            // 加密秘钥
            AisinoCoverParamPayload payload = aisinoPayloadService.assembleAisinoPayload(param, saleInvDO.getSaleTaxNo());
            // 更新上传结果
            AisinoResultResp aisinoResultResp = aisinoService.blueInvoiceResultBase64(payload, param);
            if (Objects.equals(aisinoResultResp.getCODE(), "200")) {
                AisinoResultResp.Data data1 = aisinoResultResp.getDATA();
                if (data1.getDATA() != null) {
                    saleInvDO.setInvPdfBase64(data1.getDATA());
                    return true;
                }
            }
        }
        return false;
    }

    @NotNull
    private String getSourceType(List<InvoiceAwaitRespVO> records) {
        List<String> sourceType = records.stream().map(InvoiceAwaitRespVO::getOptDocCls).distinct().toList();
        if (CollUtil.isEmpty(sourceType)) {
            throw new BusinessException("开票失败：所选单据来源为空");
        }
        if (sourceType.size() > 1) {
            throw new BusinessException("开票失败：多种来源单据不能一起开票");
        }
        return sourceType.get(0);
    }

    private void workFlow(SaleInvDO saleInvDO) {
        String applyNo = saleInvDO.getApplyNo();
        // 开启工作流
        /************************************************************************
         *                              工作流开始                                *
         ************************************************************************/

        if (saleInvDO.getProcInstId() == null
                || WorkflowConstant.CAN_START_PROC_STATUSES.contains(saleInvDO.getProcInstStatus())) {
            /**
             * 为什么要加上面3个判断，是因为流程已经启动了,就没必要再启动流程(场景:在'审批中'的过程中，做保存操作)
             * 工作流规则:
             *  1,一个业务单据在一个流程没结束前是不能再起的同一个工作流
             *  2,一个业务单据可以同时有2个"不同类型"的流程再跑
             */
            String procInstName = "开票申请(" + applyNo + ")审批";
            String businessKey = saleInvDO.getId() + "";
            ProcessInfo processInfo = workflowRpcService.startProcess(
                    WorkFlowDefKey.FIN_SALE_INV.name(),
                    procInstName,
                    businessKey,
                    new HashMap<>());
            // 修改业务审批数据
            saleInvDomainService.updateWorkInfo(processInfo, saleInvDO.getId());
            log.info("待开发票申请走工作流,申请销售发票单号 {}, 工作流返回 {}", applyNo, processInfo);
            /************************************************************************
             *                              工作流结束                                *
             ************************************************************************/
        }
    }

    private void sendMessage(SaleInvDO saleInvDO) {
        // 发送消息
        DemoMqMessageDTO messageDTO = new DemoMqMessageDTO();
        messageDTO.setId(saleInvDO.getId());
        messageDTO.setCode(saleInvDO.getApplyNo());
        messageDTO.setInterfaceType(FinConstant.ZT_TO_NC_SALINV);
        messageDTO.setDomainCode(FinConstant.FIN);
        try {
            messageQueueTemplate.publishMessage(Application.NAME, "yst-fin", messageDTO);
        } catch (Exception e) {
            log.error("发送mq消息，同步销售开票信息到NC失败,同步内容{}, 失败原因：{}", messageDTO, e.getMessage());
            throw new com.elitescloud.boot.exception.BusinessException("同步销售开票信息到NC失败");
        }

    }

}
