package com.elitesland.fin.application.service.invoice.impl;

import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.mq.MessageQueueListener;
import com.elitescloud.cloudt.common.util.RedLockUtils;
import com.elitesland.fin.application.facade.dto.mq.InvoiceOpeningProcessMqMessageDTO;
import com.elitesland.fin.application.service.invoice.InvoiceService;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.saleinv.SaleInvdDtlDO;
import com.elitesland.fin.infr.repo.saleinv.SaleInvdDtlRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.redisson.RedissonRedLock;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

@Component
@Slf4j
@RequiredArgsConstructor
public class InvoiceOpeningProcessListener implements MessageQueueListener<InvoiceOpeningProcessMqMessageDTO> {

    private final RedLockUtils redLockUtils;
    private final InvoiceService invoiceService;
    private final SaleInvdDtlRepo saleInvdDtlRepo;

    @Override
    public @NotEmpty String[] channels() {
        return new String[]{InvoiceOpeningProcessMqMessageDTO.SUBMIT_CHANNEL};
    }

    @Override
    public void onConsume(@NotBlank String s, @NotNull InvoiceOpeningProcessMqMessageDTO messageDTO) {
        log.info("获取开票结果提交：{}", JSON.toJSONString(messageDTO));
        RedissonRedLock redLock = null;
        try {
            // 分布式锁
            redLock = redLockUtils.getRedLock("invoice_opening_process" + messageDTO.getBusinessKey());
            boolean lockFlag = redLock.tryLock(3, 5, TimeUnit.SECONDS);
            if (!lockFlag) {
                log.error("发票已在处理中,获取锁失败");
                return;
            }
            // 幂等，判断发票状态
            Long id = messageDTO.getId();
            Optional<SaleInvdDtlDO> byId = saleInvdDtlRepo.findById(id);
            if (byId.isEmpty()) {
                log.error("未找到开票中发票");
                return;
            }
            SaleInvdDtlDO saleInvdDtlDO = byId.get();
            if (!saleInvdDtlDO.getInvState().equals(UdcEnum.INV_STATE_ING.getValueCode())) {
                log.error("发票不处于开票中状态，跳过处理");
                return;
            }
            if (Objects.equals(saleInvdDtlDO.getRedBlueType(), UdcEnum.INV_RED_BLUE_TYPE_BLUE.getValueCode())) {
                invoiceService.getBlueInvoiceResult(List.of(id));
            } else if (Objects.equals(saleInvdDtlDO.getRedBlueType(), UdcEnum.INV_RED_BLUE_TYPE_RED.getValueCode())) {
                invoiceService.getRedInvoiceResult(List.of(id));
            } else {
                log.error("发票红蓝票类型错误");
            }
        } catch (Exception e) {
            log.error("获取开票结果发生异常,发票流水号:{}", messageDTO.getBusinessKey(), e);
        } finally {
            if (redLock != null) {
                redLock.unlock();
                log.info("获取开票结果完成，释放锁成功");
            }
        }
    }
}
