package com.elitesland.fin.repo.push;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitesland.fin.application.facade.vo.push.FinPushResultVO;
import com.elitesland.fin.common.FinPushInter;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.push.FinPushResultDO;
import com.elitesland.fin.entity.push.QFinPushResultDO;
import com.querydsl.core.types.Projections;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2025/7/26 周六
 */
@Repository
public class FinPushResultRepoProc extends BaseRepoProc<FinPushResultDO> {
    private static final QFinPushResultDO QDO = QFinPushResultDO.finPushResultDO;

    public FinPushResultRepoProc() {
        super(QDO);
    }

    /**
     * 获取已推送记录
     *
     * @param interCode
     * @param docId
     * @return
     */
    public Long getExistsRecord(@NotBlank String interCode, long docId) {
        return super.getIdByPredicate(QDO.docId.eq(docId).and(QDO.interCode.eq(interCode)));
    }

    /**
     * 获取已推送记录
     *
     * @param interCode
     * @param docId
     * @return
     */
    public FinPushResultDO getRecord(@NotBlank String interCode, long docId) {
        return super.getOne(QDO.docId.eq(docId).and(QDO.interCode.eq(interCode)));
    }

    /**
     * 设置无需推送
     *
     * @param id
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public void updatePushNoop(long id, String reason) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.failReason, reason)
                .set(QDO.pushTimeLast, LocalDateTime.now())
                .set(QDO.pushResult, UdcEnum.COM_PUSH_RESULT_NOOP.getValueCode())
                .where(QDO.id.eq(id).and(QDO.pushResult.ne(UdcEnum.COM_PUSH_RESULT_SUCCESS.getValueCode())))
                .execute();
    }

    /**
     * 更新推送失败原因
     *
     * @param id
     * @param failReason
     */
    @Transactional(rollbackFor = Exception.class)
    public void updatePushFail(long id, String failReason) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.failReason, failReason)
                .set(QDO.pushTimeLast, LocalDateTime.now())
                .set(QDO.pushResult, UdcEnum.COM_PUSH_RESULT_FAIL.getValueCode())
                .where(QDO.id.eq(id).and(QDO.pushResult.ne(UdcEnum.COM_PUSH_RESULT_SUCCESS.getValueCode())))
                .execute();
    }

    /**
     * 推送成功
     *
     * @param id
     */
    @Transactional(rollbackFor = Exception.class)
    public void updatePushSuccess(long id) {
        super.jpaQueryFactory.update(QDO)
                .setNull(QDO.failReason)
                .set(QDO.pushTimeLast, LocalDateTime.now())
                .set(QDO.pushResult, UdcEnum.COM_PUSH_RESULT_SUCCESS.getValueCode())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新推送记录ID
     *
     * @param id
     * @param pushRecordId
     */
    @Transactional(rollbackFor = Exception.class)
    public void updatePushRecordId(long id, Long pushRecordId) {
        super.updateValue(QDO.pushRecordId, pushRecordId, id);
    }

    /**
     * 根据单据ID查询
     *
     * @param docIds
     * @return
     */
    public Map<Long, FinPushResultVO> queryByDocId(@NotEmpty Collection<Long> docIds, @NotBlank FinPushInter inter) {
        var qbean = Projections.bean(FinPushResultVO.class, QDO.docId, QDO.docNo, QDO.interCode, QDO.pushResult, QDO.pushTimeFirst, QDO.pushTimeLast,
                QDO.failReason, QDO.pushRecordId);
        return super.getList(qbean, QDO.docId.in(docIds).and(QDO.interCode.eq(inter.getInterCode()))).stream()
                .collect(Collectors.toMap(FinPushResultVO::getDocId, v -> v, (v1, v2) -> v1));
    }
}
