package com.elitesland.fin.application.service.creditaccountflow;

import cn.hutool.core.lang.Assert;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.core.base.SeqNumProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.Application;
import com.elitesland.fin.application.convert.creditaccountflow.CreditAccountFlowConvert;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountDTO;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountIocDTO;
import com.elitesland.fin.application.facade.dto.creditaccountflow.CreditAccountFlowDTO;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountIocParam;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountRuleConfigQueryParam;
import com.elitesland.fin.application.facade.param.creditaccountflow.CreditAccountFlowParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowPageParam;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountRuleConfigPageVO;
import com.elitesland.fin.application.facade.vo.creditaccountflow.CreditAccountFlowVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.service.creditaccount.CreditAccountIocService;
import com.elitesland.fin.application.service.creditaccount.CreditAccountRuleConfigService;
import com.elitesland.fin.application.service.creditaccount.CreditAccountService;
import com.elitesland.fin.application.service.unionpay.entity.constant.Constants;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinRedisConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.creditaccount.CreditAccountDO;
import com.elitesland.fin.entity.creditaccount.CreditAccountRuleConfigDO;
import com.elitesland.fin.entity.creditaccountflow.CreditAccountFlowDO;
import com.elitesland.fin.repo.account.AccountRuleConfigRepo;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepo;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepoProc;
import com.elitesland.fin.repo.creditaccount.CreditAccountRuleConfigRepo;
import com.elitesland.fin.repo.creditaccountflow.CreditAccountFlowRepo;
import com.elitesland.fin.repo.creditaccountflow.CreditAccountFlowRepoProc;
import com.elitesland.fin.rocketmq.channel.FinSalMqProcessor;
import com.elitesland.fin.rpc.order.RmiOrderRpcService;
import com.elitesland.fin.rpc.sale.RmiSaleRpcService;
import com.elitesland.fin.rpc.ystsupp.RmiOrgOuRpcServiceService;
import com.elitesland.support.provider.org.param.OrgOuRpcDtoParam;
import io.swagger.annotations.ApiModelProperty;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.integration.support.MessageBuilder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @Author gyj
 * @Date 2023/02/25
 */
@Service
@Slf4j
public class CreditAccountFlowServiceImpl implements CreditAccountFlowService {

    @Autowired
    private CreditAccountFlowRepo creditAccountFlowRepo;

    @Autowired
    private CreditAccountFlowRepoProc creditAccountFlowRepoProc;

    @Autowired
    private CreditAccountRepoProc creditAccountRepoProc;

    @Autowired
    private CreditAccountRepo creditAccountRepo;

    @Autowired
    private RedissonClient redissonClient;

    @Autowired
    private RmiOrgOuRpcServiceService rmiOrgOuRpcServiceService;

    @Autowired
    private AccountRuleConfigRepo accountRuleConfigRepo;

    @Autowired
    private RmiOrderRpcService rmiOrderRpcService;

    @Autowired
    private CreditAccountService creditAccountService;

    @Autowired
    private FinSalMqProcessor finSalMqProcessor;

    @Autowired
    private CreditAccountIocService creditAccountIocService;

    @Autowired
    private SeqNumProvider seqNumProvider;

    @Autowired
    private CreditAccountRuleConfigService creditAccountRuleConfigService;

    @Autowired
    private RmiSaleRpcService rmiSaleRpcService;

    @Autowired
    private CreditAccountRuleConfigRepo creditAccountRuleConfigRepo;


    private static final BigDecimal ONE_HUNDRED_BIGDECIMAL = new BigDecimal("100");


    @Override
    @SysCodeProc
    public PagingVO<CreditAccountFlowDTO> page(CreditAccountFlowParam creditAccountFlowParam) {

        PagingVO<CreditAccountFlowDTO> pagingVO = creditAccountFlowRepoProc.page(creditAccountFlowParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<CreditAccountFlowVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        //根据公司编码、加盟商编码、规则编码获取名称
        List<CreditAccountFlowDTO> creditAccountFlowDTOList = pagingVO.getRecords();

        OrgOuRpcDtoParam orgOuRpcDtoParam = new OrgOuRpcDtoParam();
        orgOuRpcDtoParam.setOuCodes(creditAccountFlowDTOList.stream().map(CreditAccountFlowDTO::getOuCode).collect(Collectors.toList()));

        List<CreditAccountRuleConfigDO> creditAccountRuleConfigDOList = creditAccountRuleConfigRepo.findAll();

        creditAccountFlowDTOList.stream().forEach(item -> {

            CreditAccountRuleConfigDO creditAccountRuleConfigDO = creditAccountRuleConfigDOList.stream()
                    .filter(accountRuleConfigDO -> StringUtils.equals(item.getCreditAccountRuleCode(), accountRuleConfigDO.getRuleCode()))
                    .findFirst()
                    .orElse(null);
            if (creditAccountRuleConfigDO != null) {
                item.setCreditAccountRuleName(creditAccountRuleConfigDO.getRuleName());
            }
        });

        return PagingVO.<CreditAccountFlowDTO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void approve(CreditAccountFlowParam creditAccountFlowParam) {
        checkApproveParam(creditAccountFlowParam);
        approve(creditAccountFlowParam, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode(), true);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void reject(CreditAccountFlowParam creditAccountFlowParam) {
        checkApproveParam(creditAccountFlowParam);
        approve(creditAccountFlowParam, UdcEnum.APPLY_STATUS_REJECTED.getValueCode(), false);
    }


    @Override
    public void accountFlowCallBack(String doc, CreditAccountFlowDO creditAccountFlowDO, boolean isApprove) {

        if (UdcEnum.DOC_CLS_SVO.getValueCode().equals(doc) || UdcEnum.DOC_CLS_AO.getValueCode().equals(doc)) {
            return;
        }
//        OrderAccountFlowRpcParam orderAccountFlowRpcParam = new OrderAccountFlowRpcParam();
//        //销售订单id
//        orderAccountFlowRpcParam.setSoId(creditAccountFlowDO.getSourceId());
//        //订单类别
//        orderAccountFlowRpcParam.setDocCls(creditAccountFlowDO.getSourceDoc());
//        //流水号
//        orderAccountFlowRpcParam.setReceiptSerial(creditAccountFlowDO.getFlowNo());
//        //交易类型
//        orderAccountFlowRpcParam.setTransType(creditAccountFlowDO.getTransactionType());
//        //发生金额
//        orderAccountFlowRpcParam.setReceiptAmt(creditAccountFlowDO.getAmount());
//        //交易日期
//        orderAccountFlowRpcParam.setReceiptDate(creditAccountFlowDO.getTransactionTime());
//        //账户编码
//        orderAccountFlowRpcParam.setPayAccNo(creditAccountFlowDO.getAccountCode());
//        //账户名称
//        orderAccountFlowRpcParam.setPayAccName(creditAccountFlowDO.getAccountName());
//        //账户名称
//        orderAccountFlowRpcParam.setIsApprove(isApprove);
//
//        rmiOrderRpcService.orderAccountFlow(orderAccountFlowRpcParam);
    }


    private void checkApproveParam(CreditAccountFlowParam param) {
        Assert.notNull(param, "参数不能为空");
        Assert.notNull(param.getId(), "id不能为空");
        Assert.notEmpty(param.getCreditAccountCode(), "账户编码不能为空");
        Assert.notEmpty(param.getTransactionType(), "交易类型不能为空");
        Assert.notNull(param.getAmount(), "发生金额不能为空");
    }

    private void approve(CreditAccountFlowParam creditAccountFlowParam, String state, boolean isApprove) {

        Optional<CreditAccountFlowDO> creditAccountFlowDOOptional = creditAccountFlowRepo.findById(creditAccountFlowParam.getId());
        Assert.isTrue(creditAccountFlowDOOptional.isPresent(), "查询不到单据");

        CreditAccountFlowDO creditAccountFlowDO = creditAccountFlowDOOptional.get();
        Assert.equals(UdcEnum.APPLY_STATUS_DOING.getValueCode(), creditAccountFlowDO.getOrderState(), "只有审批中状态单据才可审批");
        if(UdcEnum.CREDIT_IO_TYPE_T.getValueCode().equals(creditAccountFlowParam.getTransactionType())){
            updateAccountAmountFor_CREDIT_IO_TYPE_T(creditAccountFlowParam, state);
        }else{
            //更新账户金额
            updateAccountAmount(creditAccountFlowParam, state);
        }
        //更新流水单状态
        SysUserDTO user = getUser();

        creditAccountFlowParam.setOrderState(state);
        creditAccountFlowParam.setAuditDate(LocalDateTime.now());
        creditAccountFlowParam.setAuditUserName(FinConstant.SYSTEM);

        if (user != null) {
            creditAccountFlowParam.setAuditUserId(user.getId());
            creditAccountFlowParam.setAuditUserName(user.getUsername());
        }

        creditAccountFlowRepoProc.approveAccountFlow(creditAccountFlowParam);

    }

    private void updateAccountAmount(CreditAccountFlowParam creditAccountFlowParam, String state) {
        if (UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(state)) {

            CreditAccountDO creditAccountDO = creditAccountRepoProc.findByCreditAccountCode(creditAccountFlowParam.getCreditAccountCode());

            BigDecimal creditAccountLimit = creditAccountDO.getCreditAccountLimit();
            BigDecimal creditAccountUsedLimit = creditAccountDO.getCreditAccountUsedLimit();
            BigDecimal creditAccountOccupancyLimit = creditAccountDO.getCreditAccountOccupancyLimit();
            BigDecimal creditAccountAvailableLimit;

            Assert.equals(creditAccountDO.getStatus(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "信用账户已经被禁用");

            if (UdcEnum.CREDIT_IO_TYPE_I.getValueCode().equals(creditAccountFlowParam.getTransactionType())) {
                creditAccountLimit = creditAccountLimit.add(creditAccountFlowParam.getAmount());
            }

            if (UdcEnum.CREDIT_IO_TYPE_O.getValueCode().equals(creditAccountFlowParam.getTransactionType())) {
                creditAccountLimit = creditAccountLimit.subtract(creditAccountFlowParam.getAmount());
            }

            if (UdcEnum.CREDIT_IO_TYPE_T.getValueCode().equals(creditAccountFlowParam.getTransactionType())) {
                creditAccountOccupancyLimit = creditAccountOccupancyLimit.add(creditAccountFlowParam.getAmount());
            }

            if (UdcEnum.CREDIT_IO_TYPE_R.getValueCode().equals(creditAccountFlowParam.getTransactionType())) {
                creditAccountOccupancyLimit = creditAccountOccupancyLimit.subtract(creditAccountFlowParam.getAmount());
            }

            if (UdcEnum.CREDIT_IO_TYPE_U.getValueCode().equals(creditAccountFlowParam.getTransactionType())) {
                creditAccountUsedLimit = creditAccountUsedLimit.add(creditAccountFlowParam.getAmount());
            }

            if (UdcEnum.CREDIT_IO_TYPE_D.getValueCode().equals(creditAccountFlowParam.getTransactionType())) {
                creditAccountUsedLimit = creditAccountUsedLimit.subtract(creditAccountFlowParam.getAmount());
            }

            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountDO.setCreditAccountLimit(creditAccountLimit);
            creditAccountDO.setCreditAccountUsedLimit(creditAccountUsedLimit);
            creditAccountDO.setCreditAccountOccupancyLimit(creditAccountOccupancyLimit);
            creditAccountDO.setCreditAccountAvailableLimit(creditAccountAvailableLimit);

            //更新审批中的流水的状态
            creditAccountFlowParam.setCreditAccountLimit(creditAccountLimit);
            creditAccountFlowParam.setCreditAccountUsedLimit(creditAccountUsedLimit);
            creditAccountFlowParam.setCreditAccountOccupancyLimit(creditAccountOccupancyLimit);
            creditAccountFlowParam.setCreditAccountAvailableLimit(creditAccountAvailableLimit);
            creditAccountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            creditAccountFlowRepoProc.updateCreditAccountFlowAmount(creditAccountFlowParam);
        }
    }
    private void updateAccountAmountFor_CREDIT_IO_TYPE_T(CreditAccountFlowParam creditAccountFlowParam, String state) {
        if (UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(state)) {
            CreditAccountDO creditAccountDO = creditAccountRepoProc.findByCreditAccountCode(creditAccountFlowParam.getCreditAccountCode());
            Assert.equals(creditAccountDO.getStatus(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "信用账户已经被禁用");
            int res= creditAccountRepo.updateCreditAccountOccupancyLimitAndCreditAccountAvailableLimitById(creditAccountDO.getId(), creditAccountFlowParam.getAmount());
            log.info("信用账户流水生成，更新记录数：{}",res);
            if(res<1){
                log.error("信用账户流水生成，更新信用账户失败，可用金额不足");
                throw new BusinessException("信用账户流水生成，更新信用账户失败，可用金额不足");
            }
            CreditAccountDTO creditAccountDTO = creditAccountRepoProc.get(creditAccountDO.getId());
            if(creditAccountDTO==null){
                log.error("信用账户流水生成，查询信用账户失败");
                throw new BusinessException("信用账户流水生成，查询信用账户失败");
            }
            BigDecimal creditAccountLimit = creditAccountDTO.getCreditAccountLimit();
            BigDecimal creditAccountUsedLimit = creditAccountDTO.getCreditAccountUsedLimit();
            BigDecimal creditAccountOccupancyLimit = creditAccountDTO.getCreditAccountOccupancyLimit();
            BigDecimal creditAccountAvailableLimit=creditAccountDTO.getCreditAccountAvailableLimit();
            //更新审批中的流水的状态
            creditAccountFlowParam.setCreditAccountLimit(creditAccountLimit);
            creditAccountFlowParam.setCreditAccountUsedLimit(creditAccountUsedLimit);
            creditAccountFlowParam.setCreditAccountOccupancyLimit(creditAccountOccupancyLimit);
            creditAccountFlowParam.setCreditAccountAvailableLimit(creditAccountAvailableLimit);
            creditAccountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            creditAccountFlowRepoProc.updateCreditAccountFlowAmount(creditAccountFlowParam);
        }
    }
    private SysUserDTO getUser() {
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (userDetails == null) {
            return null;
        }
        return userDetails.getUser();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public String generateCreditAccountFlow(CreditAccountFlowParam param) {
        //参数检查
        checkGenerateCreditAccountFlowParam(param);

        CreditAccountFlowParam creditAccountFlowParam = CreditAccountFlowConvert.INSTANCE.copyParam(param);

        //组装财务域udc
        buildCreditAccountFlowParam(creditAccountFlowParam);

        //获取账户信息
        CreditAccountDO creditAccountDO = getCreditAccountMsg(creditAccountFlowParam);

        //幂等
        checkIdempotent(creditAccountDO,creditAccountFlowParam);

        //获取账户规则
        List<CreditAccountRuleConfigPageVO> accountRuleConfigPageVOList = getCreditAccountRuleMsg(creditAccountDO, creditAccountFlowParam);

        //获取账户事务码
        List<CreditAccountIocDTO> accountIocDTOList = getAccountIocMsg(accountRuleConfigPageVOList);

        //根据优先级排序
        accountRuleConfigPageVOList = accountRuleConfigPageVOList.stream()
                .sorted(Comparator.comparing(CreditAccountRuleConfigPageVO::getPriorityNo))
                .collect(Collectors.toList());

        //校验账户余额
        checkCreditAccountLimit(accountRuleConfigPageVOList, creditAccountDO, creditAccountFlowParam, accountIocDTOList);

        //生成账户流水
        List<CreditAccountFlowParam> creditAccountFlowParamList = generateCreditAccountFlow(accountRuleConfigPageVOList,
                creditAccountFlowParam,
                creditAccountDO,
                accountIocDTOList);

        //流水审批
        creditAccountFlowReview(creditAccountFlowParamList);

        return creditAccountFlowParamList.get(0).getFlowNo();
    }

    private void creditAccountFlowReview(List<CreditAccountFlowParam> creditAccountFlowParamList) {

        creditAccountFlowParamList.stream().forEach(item -> {
            if (item.isAutomaticReview()) {
                approve(item);
            }
        });
    }

    private void sendCreditAccountFlowReviewMsg(List<CreditAccountFlowParam> creditAccountFlowParamList) {
        creditAccountFlowParamList.stream().forEach(item -> {
            if (item.isAutomaticReview()) {
                //发送流水审批消息
                try {
                    boolean result = finSalMqProcessor.approveCreditAccountFlowOutput().send(MessageBuilder.withPayload(item).build());
                    if (!result) {
                        log.error("流水自动审批消息发送失败");
                    }
                } catch (Exception e) {
                    log.error("流水自动审批消息发送失败，失败原因：" + e.getMessage(), e);
                }
            }
        });
    }

    private List<CreditAccountFlowParam> generateCreditAccountFlow(List<CreditAccountRuleConfigPageVO> accountRuleConfigPageVOList,
                                                                   CreditAccountFlowParam creditAccountFlowParam,
                                                                   CreditAccountDO creditAccountDO,
                                                                   List<CreditAccountIocDTO> accountIocDTOList) {
        List<CreditAccountRuleConfigPageVO> finalAccountRuleConfigPageVOList = accountRuleConfigPageVOList;
        List<CreditAccountFlowParam> creditAccountFlowParamList = new ArrayList<>();

        //生成流水单
        finalAccountRuleConfigPageVOList.stream().forEach(accountRuleConfigPageVO -> {

            CreditAccountFlowParam param = buildCreditAccountFlowParam(creditAccountFlowParam,
                    creditAccountDO,
                    accountIocDTOList,
                    accountRuleConfigPageVO);

            //创建信用账户流水
            CreditAccountFlowDO creditAccountFlowDO = CreditAccountFlowConvert.INSTANCE.param2DO(param);
            creditAccountFlowRepo.save(creditAccountFlowDO);

            param.setFlowNo(creditAccountFlowDO.getFlowNo());
            param.setId(creditAccountFlowDO.getId());
            creditAccountFlowParamList.add(param);
        });
        return creditAccountFlowParamList;
    }

    @Override
    public Boolean checkCreditAccountBalance(CreditAccountFlowParam param) {
        //参数检查
        checkGenerateCreditAccountFlowParam(param);

        CreditAccountFlowParam creditAccountFlowParam = CreditAccountFlowConvert.INSTANCE.copyParam(param);

        //组装财务域udc
        buildCreditAccountFlowParam(creditAccountFlowParam);

        //获取账户信息
        CreditAccountDO creditAccountDO = getCreditAccountMsg(creditAccountFlowParam);

        //幂等
        checkIdempotent(creditAccountDO,creditAccountFlowParam);

        //获取账户规则
        List<CreditAccountRuleConfigPageVO> creditAccountRuleConfigPageVOList = getCreditAccountRuleMsg(creditAccountDO, creditAccountFlowParam);

        //获取账户事务码
        List<CreditAccountIocDTO> creditAccountIocDTOList = getAccountIocMsg(creditAccountRuleConfigPageVOList);

        //根据优先级排序
        creditAccountRuleConfigPageVOList = creditAccountRuleConfigPageVOList.stream()
                .sorted(Comparator.comparing(CreditAccountRuleConfigPageVO::getPriorityNo))
                .collect(Collectors.toList());

        //校验账户余额
        checkCreditAccountLimit(creditAccountRuleConfigPageVOList, creditAccountDO, creditAccountFlowParam, creditAccountIocDTOList);

        return true;
    }

    private void buildCreditAccountFlowParam(CreditAccountFlowParam creditAccountFlowParam) {
        creditAccountFlowParam.setSourceDocType(creditAccountFlowParam.getSourceDoc()
                .concat(FinConstant.UNDERLINE)
                .concat(creditAccountFlowParam.getSourceDocType()));

        creditAccountFlowParam.setSourceDocStatus(creditAccountFlowParam.getSourceDoc()
                .concat(FinConstant.UNDERLINE)
                .concat(creditAccountFlowParam.getSourceDocStatus()));
    }

    private void checkCreditAccountLimit(List<CreditAccountRuleConfigPageVO> accountRuleConfigPageVOList,
                                         CreditAccountDO creditAccountDO,
                                         CreditAccountFlowParam creditAccountFlowParam,
                                         List<CreditAccountIocDTO> accountIocDTOList) {

        accountRuleConfigPageVOList.stream().forEach(item -> {
            //交易类型
            CreditAccountIocDTO accountIocDTO = accountIocDTOList.stream().filter(accountIocRespVO -> accountIocRespVO.getIoCode()
                    .equals(item.getIoCode()))
                    .findFirst()
                    .orElse(null);

            Assert.notNull(accountIocDTO, "没有查询到启用的信用账户事务");

            if (UdcEnum.CREDIT_IO_TYPE_T.getValueCode().equals(accountIocDTO.getIoType())) {
                Assert.isTrue(creditAccountDO.getCreditAccountAvailableLimit().compareTo(creditAccountFlowParam.getSourceDocAmount()
                        .multiply(item.getCalculatePercent())
                        .divide(ONE_HUNDRED_BIGDECIMAL)) >= 0, "信用账户可用余额不足");
            }
        });
    }

    private void checkIdempotent(CreditAccountDO creditAccountDO,CreditAccountFlowParam creditAccountFlowParam) {
        creditAccountFlowParam.setCreditAccountType(creditAccountDO.getCreditAccountType());

        Assert.isTrue(CollectionUtils.isEmpty(creditAccountFlowRepoProc.queryCreditAccountFlowVO(creditAccountFlowParam)),
                "订单已经生成过流水");
    }

    private List<CreditAccountIocDTO> getAccountIocMsg(List<CreditAccountRuleConfigPageVO> accountRuleConfigPageVOList) {
        Set<String> ioCodeSet = new HashSet<>();
        accountRuleConfigPageVOList.stream().forEach(item -> ioCodeSet.add(item.getIoCode()));

        CreditAccountIocParam accountIocParam = new CreditAccountIocParam();
        accountIocParam.setIoCodeSet(ioCodeSet);
        accountIocParam.setState(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<CreditAccountIocDTO> accountIocDTOList = creditAccountIocService.queryByAccountIocParam(accountIocParam);
        Assert.notEmpty(accountIocDTOList, "查询账户事务码失败");
        return accountIocDTOList;
    }

    public CreditAccountFlowParam buildCreditAccountFlowParam(CreditAccountFlowParam creditAccountFlowParam,
                                                              CreditAccountDO creditAccountDO,
                                                              List<CreditAccountIocDTO> accountIocDTOList,
                                                              CreditAccountRuleConfigPageVO accountRuleConfigPageVO) {

        CreditAccountFlowParam result = new CreditAccountFlowParam();
        //公司编码
        result.setOuCode(creditAccountDO.getOuCode());
        //公司名称
        result.setOuName(creditAccountDO.getOuName());
        //对象编码
        result.setObjectCode(creditAccountDO.getObjectCode());
        //对象名称
        result.setObjectName(creditAccountDO.getObjectName());
        //对象类型
        result.setObjectType(creditAccountDO.getObjectType());
        //流水号
        result.setFlowNo(seqNumProvider.generateCode(Application.NAME, FinConstant.ACCOUNT_FLOW_NO, Collections.emptyList()));
        //交易类型
        CreditAccountIocDTO accountIocDTO = accountIocDTOList.stream().filter(accountIocRespVO -> accountIocRespVO.getIoCode()
                .equals(accountRuleConfigPageVO.getIoCode()))
                .findFirst()
                .orElse(null);

        Assert.notNull(accountIocDTO, "没有查询到启用的账户事务");

        result.setTransactionType(accountIocDTO.getIoType());

        //发生金额
        result.setAmount(creditAccountFlowParam.getSourceDocAmount()
                .multiply(accountRuleConfigPageVO.getCalculatePercent())
                .divide(ONE_HUNDRED_BIGDECIMAL));

        //交易日期
        result.setTransactionTime(LocalDateTime.now());
        //账户类型
        result.setCreditAccountType(creditAccountDO.getCreditAccountType());
        //账户编码
        result.setCreditAccountCode(creditAccountDO.getCreditAccountCode());
        //账户名称
        result.setCreditAccountName(creditAccountDO.getCreditAccountName());
        //来源单据
        result.setSourceDoc(creditAccountFlowParam.getSourceDoc());
        //来源单号
        result.setSourceNo(creditAccountFlowParam.getSourceNo());
        //来源单据id
        result.setSourceId(creditAccountFlowParam.getSourceId());

        //规则编码
        result.setCreditAccountRuleCode(accountRuleConfigPageVO.getRuleCode());
        //规则优先级
        result.setPriorityNo(accountRuleConfigPageVO.getPriorityNo());

        //单据状态
        result.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());

        //适用单据状态
        result.setSourceDocStatus(creditAccountFlowParam.getSourceDocStatus());

        result.setAutomaticReview(accountRuleConfigPageVO.isAutomaticReview());

        result.setCreditAccountLimit(creditAccountDO.getCreditAccountLimit());

        result.setCreditAccountUsedLimit(creditAccountDO.getCreditAccountUsedLimit());

        result.setCreditAccountOccupancyLimit(creditAccountDO.getCreditAccountOccupancyLimit());

        result.setCreditAccountAvailableLimit(creditAccountDO.getCreditAccountAvailableLimit());

        return result;
    }

    private List<CreditAccountRuleConfigPageVO> getCreditAccountRuleMsg(CreditAccountDO creditAccountDO, CreditAccountFlowParam creditAccountFlowParam) {
        CreditAccountRuleConfigQueryParam accountRuleConfigQueryParam = new CreditAccountRuleConfigQueryParam();

        //适用单据
        accountRuleConfigQueryParam.setOptDoc(creditAccountFlowParam.getSourceDoc());
        //适用单据类型
        accountRuleConfigQueryParam.setOptDocType(creditAccountFlowParam.getSourceDocType());
        //适用单据状态
        accountRuleConfigQueryParam.setOptDocStatus(creditAccountFlowParam.getSourceDocStatus());
        //适用账户类型
        accountRuleConfigQueryParam.setOptAccountType(creditAccountDO.getCreditAccountType());
        //指定账户
        accountRuleConfigQueryParam.setSpecificAccount(creditAccountDO.getCreditAccountCode());
        accountRuleConfigQueryParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        accountRuleConfigQueryParam.setSize(Constants.PAGE_SIZE);
        PagingVO<CreditAccountRuleConfigPageVO> accountRuleConfigPageVOPagingVO = creditAccountRuleConfigService.searchPage(accountRuleConfigQueryParam);

        Assert.notEmpty(accountRuleConfigPageVOPagingVO.getRecords(), "没有匹配到信用账户规则");
        return accountRuleConfigPageVOPagingVO.getRecords();

    }

    private CreditAccountDO getCreditAccountMsg(CreditAccountFlowParam creditAccountFlowParam) {
        return creditAccountRepoProc.findByCreditAccountCode(creditAccountFlowParam.getCreditAccountCode());
    }

    private void checkGenerateCreditAccountFlowParam(CreditAccountFlowParam creditAccountFlowParam) {
        Assert.notNull(creditAccountFlowParam, "参数不能为空");

        Assert.notEmpty(creditAccountFlowParam.getSourceDoc(), "来源单据不能为空");
        Assert.notEmpty(creditAccountFlowParam.getSourceDocType(), "来源单据类型不能为空");
        Assert.notEmpty(creditAccountFlowParam.getSourceDocStatus(), "来源单据状态不能为空");
        Assert.notNull(creditAccountFlowParam.getSourceDocAmount(), "来源单据金额不能为空");
        Assert.notNull(creditAccountFlowParam.getSourceId(), "来源单据id不能为空");
        Assert.notEmpty(creditAccountFlowParam.getSourceNo(), "来源单号不能为空");
        Assert.notEmpty(creditAccountFlowParam.getCreditAccountCode(), "信用账户编码不能为空");
    }

    @Override
    @SysCodeProc
    public List<CreditAccountFlowVO> selectListByQueryParam(CreditAccountFlowParam queryParam) {
        List<CreditAccountFlowVO> flowVOList = creditAccountFlowRepoProc.selectListByQueryParam(queryParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }

    @Override
    @SysCodeProc
    public List<CreditAccountFlowVO> selectRepairAfterPage(CreditAccountFlowParam pageParam) {
        List<CreditAccountFlowVO> flowVOList = creditAccountFlowRepoProc.selectRepairAfterPage(pageParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }

}
