package com.elitesland.fin.application.service.arorder;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.system.dto.resp.SysCurrencyRespDTO;
import com.elitesland.fin.application.convert.arorder.ArOrderConvert;
import com.elitesland.fin.application.convert.arorder.ArOrderDtlRecordConvert;
import com.elitesland.fin.application.convert.arorder.ArOrderRecordConvert;
import com.elitesland.fin.application.facade.param.arorder.ArOrderDtlRecordSaveParam;
import com.elitesland.fin.application.facade.param.arorder.ArOrderRecordSaveParam;
import com.elitesland.fin.application.facade.param.arorder.ArOrderSaveParam;
import com.elitesland.fin.application.facade.vo.arorder.ArOrderVO;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.arorder.ArOrder;
import com.elitesland.fin.domain.entity.arorder.ArOrderDtl;
import com.elitesland.fin.domain.param.arorder.ArOrderPageParam;
import com.elitesland.fin.domain.param.arorder.ArOrderRecordParam;
import com.elitesland.fin.domain.param.artype.ArTypePageParam;
import com.elitesland.fin.domain.service.arorder.ArOrderDomainService;
import com.elitesland.fin.domain.service.arorder.ArOrderDtlDomainService;
import com.elitesland.fin.domain.service.arorder.ArOrderRecordDomainService;
import com.elitesland.fin.domain.service.artype.ArTypeDomainService;
import com.elitesland.fin.domain.service.artype.ArTypeOuDomainService;
import com.elitesland.fin.infr.dto.arorder.ArOrderRecordDTO;
import com.elitesland.fin.infr.dto.artype.ArTypeDTO;
import com.elitesland.fin.infr.dto.artype.ArTypeOuDTO;
import com.elitesland.fin.rpc.system.SystemRpcService;
import com.elitesland.fin.rpc.workflow.WorkflowRpcService;
import com.elitesland.fin.infr.dto.arorder.ArOrderDTO;
import com.elitesland.fin.application.service.workflow.WorkFlowDefKey;
import com.elitesland.order.service.SalDoRpcService;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import com.elitesland.support.provider.org.dto.OrgOuRpcDTO;
import com.elitesland.support.provider.org.param.OrgOuRpcDtoParam;
import com.elitesland.support.provider.org.service.OrgBuRpcService;
import com.elitesland.support.provider.org.service.OrgOuRpcService;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.WorkflowConstant;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.security.util.SecurityUtil;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.application.service.arorder
 * @date 2022/4/12 15:46
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ArOrderServiceImpl implements ArOrderService {

    private final ArOrderDomainService arOrderDomainService;
    private final ArOrderDtlDomainService arOrderDtlDomainService;
    private final ArTypeDomainService arTypeDomainService;
    private final SystemRpcService systemRpcService;
    private final WorkflowRpcService workflowRpcService;
    private final ArTypeOuDomainService arTypeOuDomainService;
    private final ArOrderRecordDomainService arOrderRecordDomainService;
    @Autowired
    private TaskExecutor taskExecutor;
    private final TransactionTemplate transactionTemplate;
    private final SalDoRpcService salDoRpcService;
    private final ItmItemRpcService itmItemRpcService;
    private final OrgBuRpcService orgBuRpcService;
    private final OrgOuRpcService orgOuRpcService;
    private final UdcProvider udcProvider;

    @SysCodeProc
    @Override
    public PagingVO<ArOrderVO> page(ArOrderPageParam arOrderPageParam) {
        PagingVO<ArOrderDTO> page = arOrderDomainService.page(arOrderPageParam);
        return ArOrderConvert.INSTANCE.convertPage(page);
    }

    @SysCodeProc
    @Override
    public ArOrderVO get(Long id) {
        ArOrderDTO arOrderDTO = arOrderDomainService.get(id);
        ArOrderVO res = ArOrderConvert.INSTANCE.convert(arOrderDTO);
        return res;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> audit(Long id) {

        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> refush(List<Long> ids, String content) {

        SysUserDTO user = SecurityUtil.getUser().getUser();
        Long res = arOrderDomainService.audit(ids, content, user);
        return ApiResult.ok(res);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> del(List<Long> ids) {
        Long res = arOrderDomainService.del(ids);
        return ApiResult.ok(res);
    }

    @SysCodeProc
    @Override
    public ArOrderVO getArOrderAndDtl(Long id) {
        ArOrderDTO arOrderAndDtl = arOrderDomainService.getArOrderAndDtl(id);
        ArOrderVO res = ArOrderConvert.INSTANCE.convert(arOrderAndDtl);
        return res;
    }


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> save(ArOrderSaveParam apOrderSaveParam) {
        // 新增只用于单据来源为手工的
        if (!apOrderSaveParam.getCreateMode().equals(UdcEnum.FIN_AR_DOC_CLS_MANU.getValueCode())) {
            throw new BusinessException("新增的单据来源需为手工");
        }
        ArOrder apOrder = ArOrderConvert.INSTANCE.convert(apOrderSaveParam);
        Long res = arOrderDomainService.save(apOrder);
        return ApiResult.ok(res);
    }


    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = {Exception.class})
    public ApiResult<Long> commit(ArOrderSaveParam apOrderSaveParam) {
        ArOrder apOrder = ArOrderConvert.INSTANCE.convert(apOrderSaveParam);
        Boolean isAuto = arTypeDomainService.queryIsAuto(apOrderSaveParam.getArTypeId());
        if (isAuto) {
            apOrder.setAuditDate(LocalDateTime.now());
            apOrder.setAuditUser("admin");
            SysUserDTO user = systemRpcService.getUserByName("admin");
            apOrder.setAuditUserId(user.getId());
        }
        Long resId = arOrderDomainService.commit(apOrder, isAuto);
        if (!isAuto) {
            // 启动工作流
            startWorkFlow(apOrder, resId);
        }
        return ApiResult.ok(resId);
    }

    private void startWorkFlow(ArOrder apOrder, Long resId) {
        if (apOrder.getProcInstId() == null
                || WorkflowConstant.CAN_START_PROC_STATUSES.contains(apOrder.getProcInstStatus())) {
            // 启动流程
            String procInstName = "应收单审核-" + apOrder.getArOrderNo();
            String procKey = WorkFlowDefKey.FIN_AR_ORDER.name();
            ProcessInfo processInfo = workflowRpcService.startProcess(procKey, procInstName, resId.toString(), new HashMap<>());
            arOrderDomainService.updateWorkInfo(processInfo, resId);
        }
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> update(ArOrderSaveParam apOrderSaveParam) {

        ArOrder apOrder = ArOrderConvert.INSTANCE.convert(apOrderSaveParam);
        apOrder.checkUpdate();
        Long res = arOrderDomainService.save(apOrder);
        return ApiResult.ok(res);
    }


    /**
     * 自动生成应收单:命中配置项设置规则的业务单据则可以生成
     * <p>
     * 业务方提供公司，来源单据，来源单据类型，来源单据状态等入参，调用该接口时判断是否命中应收单类型的配置规则，
     * 命中规则的数据按接口样式传输生成对应的应收单，状态是待审核；根据配置项判断是否自动审核。
     * 已生成应收单的业务单据需要打上标记ar_flag，1表示已经生成；0表示未生成。
     *（注意调用规则时判断该公司是否分配该规则，如果没分配的话给出提示未分配公司）
     *
     * @param arOrderRecordSaveParam 入参
     * @return 出参
     */
    @Override
    public ApiResult<Void> autoCreate(ArOrderRecordSaveParam arOrderRecordSaveParam){
        log.info("自动生成应收单入参: {}", JSON.toJSONString(arOrderRecordSaveParam));

        //1.保存到记录表
        Long arOrderRecordId = saveArOrderRecord(arOrderRecordSaveParam);

        //2.查询应收单类型的配置规则
        ArTypeDTO arTypeDTO = selectArType(arOrderRecordSaveParam,arOrderRecordId);
        //3.生成应收单
        autoCreateArOrder(arOrderRecordSaveParam,arTypeDTO,arOrderRecordId);

        return ApiResult.ok();
    }

    /**
     * 查询应收单类型的配置规则
     *
     * @param arOrderRecordSaveParam 入参
     * @param arOrderRecordId 记录表ID
     * @return
     */
    private ArTypeDTO selectArType(ArOrderRecordSaveParam arOrderRecordSaveParam,Long arOrderRecordId){
        checkRequired(arOrderRecordSaveParam,arOrderRecordId);

        //1.来源单据，来源单据类型，来源单据状态等入参，调用该接口时判断是否命中应收单类型的配置规则
        ArTypePageParam arTypePageParam = new ArTypePageParam();
        arTypePageParam.setSourceDoc(arOrderRecordSaveParam.getCreateMode());
        arTypePageParam.setSourceDocType(arOrderRecordSaveParam.getSourceDocType());
        arTypePageParam.setSourceDocStatus(arOrderRecordSaveParam.getSourceDocStatus());
        List<ArTypeDTO> arTypeDTOList = arTypeDomainService.arOrderAutoSelectMatchByParam(arTypePageParam);
        if (CollectionUtil.isEmpty(arTypeDTOList)){
            log.error("未匹配到应收单类型的配置规则, 来源单据: {},来源单据类型: {},来源单据状态:{}",
                    arOrderRecordSaveParam.getCreateMode(), arOrderRecordSaveParam.getSourceDocType(),arOrderRecordSaveParam.getSourceDocStatus());
            arOrderRecordDomainService.updateArFlagAndFailMsgById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_0.getValueCode(),"未匹配到应收单类型的配置规则");
            throw new BusinessException("未匹配到应收单类型的配置规则");
        }
        if (CollectionUtil.isNotEmpty(arTypeDTOList) && arTypeDTOList.size() > 1 ){
            log.error("匹配到多条应收单类型的配置规则, 来源单据: {},来源单据类型: {},来源单据状态:{}",
                    arOrderRecordSaveParam.getCreateMode(), arOrderRecordSaveParam.getSourceDocType(),arOrderRecordSaveParam.getSourceDocStatus());
            arOrderRecordDomainService.updateArFlagAndFailMsgById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_0.getValueCode(),"匹配到多条应收单类型的配置规则");
            throw new BusinessException("匹配到多条应收单类型的配置规则");
        }

        //2.注意调用规则时判断该公司是否分配该规则，如果没分配的话给出提示未分配公司
        //List<Long> arTypeIds = arTypeDTOList.stream().map(ArTypeDTO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        Long arTypeId = arTypeDTOList.get(0).getId();
        List<ArTypeOuDTO> arTypeOuDTOList = arTypeOuDomainService.queryByArTypeId(arTypeId);
        if (CollectionUtil.isEmpty(arTypeOuDTOList)){
            log.error("应收单的分配公司为空, 应收单编码: {}", arTypeDTOList.get(0).getArTypeCode());
            arOrderRecordDomainService.updateArFlagAndFailMsgById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_0.getValueCode(),"应收单的分配公司为空");
            throw new BusinessException("应收单的分配公司为空");
        }
        List<String> arTypeOuCodeList = arTypeOuDTOList.stream().map(ArTypeOuDTO::getOuCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (!arTypeOuCodeList.contains(arOrderRecordSaveParam.getOuCode())){
            log.error("应收单未分配该公司, 应收单编码: {},公司编码: {}", arTypeDTOList.get(0).getArTypeCode(),arOrderRecordSaveParam.getOuCode());
            arOrderRecordDomainService.updateArFlagAndFailMsgById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_0.getValueCode(),"应收单未分配该公司");
            throw new BusinessException("应收单未分配该公司");
        }

        return arTypeDTOList.get(0);
    }

    private void checkRequired(ArOrderRecordSaveParam arOrderRecordSaveParam,Long arOrderRecordId){
        if (StringUtils.isBlank(arOrderRecordSaveParam.getCreateMode())){
            arOrderRecordDomainService.updateArFlagAndFailMsgById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_0.getValueCode(),"单据来源为空");
            throw new BusinessException("单据来源为空");
        }
        if (StringUtils.isBlank(arOrderRecordSaveParam.getSourceDocType())){
            arOrderRecordDomainService.updateArFlagAndFailMsgById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_0.getValueCode(),"来源单据类型为空");
            throw new BusinessException("来源单据类型为空");
        }
        if (StringUtils.isBlank(arOrderRecordSaveParam.getSourceDocStatus())){
            arOrderRecordDomainService.updateArFlagAndFailMsgById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_0.getValueCode(),"来源单据状态为空");
            throw new BusinessException("来源单据状态为空");
        }
        if (Objects.isNull(arOrderRecordSaveParam.getOuId()) && StringUtils.isBlank(arOrderRecordSaveParam.getOuCode())){
            arOrderRecordDomainService.updateArFlagAndFailMsgById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_0.getValueCode(),"公司为空");
            throw new BusinessException("公司为空");
        }
    }

    /**
     * 保存应收单记录表信息
     *
     * @param arOrderRecordSaveParam 入参
     * @return 记录表ID
     */
    @Transactional(rollbackFor = {Exception.class})
    public Long saveArOrderRecord(ArOrderRecordSaveParam arOrderRecordSaveParam){
        Long arOrderRecordId = arOrderRecordDomainService.save(arOrderRecordSaveParam);

        return arOrderRecordId;
    }

    /**
     * 生成应收单
     *
     * @param arOrderRecordSaveParam 入参
     * @param arTypeDTO 应收单类型的配置
     * @param arOrderRecordId 记录表ID
     * @return
     */
    public void autoCreateArOrder(ArOrderRecordSaveParam arOrderRecordSaveParam,ArTypeDTO arTypeDTO,Long arOrderRecordId){

        //设置事务传播行为
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionTemplate.execute(trans -> {
            Long arOrderId = null;
            try {
                //是否自动审核
                Boolean isAuto = arTypeDTO.getAutoAudit();
                ArOrder arOrder = ArOrderConvert.INSTANCE.convertRecord(arOrderRecordSaveParam);
                arOrder.setArTypeCode(arTypeDTO.getArTypeCode());
                arOrder.setArTypeId(arTypeDTO.getId());
                arOrder.setArTypeName(arTypeDTO.getArTypeName());

                autoCreateArOrderHandle(arOrder);
                //自动审核
                if (isAuto) {
                    arOrder.setAuditDate(LocalDateTime.now());
                    arOrder.setAuditUser("admin");
                    SysUserDTO user = systemRpcService.getUserByName("admin");
                    arOrder.setAuditUserId(user.getId());
                    arOrderId = arOrderDomainService.commit(arOrder, isAuto);
                }else {
                    //新增为草稿
                    arOrderId = arOrderDomainService.save(arOrder);
                }

                //更新已生成
                arOrderRecordDomainService.updateArFlagById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_1.getValueCode());
                ArOrderRecordDTO arOrderRecordDTO = arOrderRecordDomainService.selectById(arOrderRecordId);
                salDoRpcService.updateArOrderFlag(Collections.singletonList(arOrderRecordDTO.getSourceNo()));
                arOrderRecordDomainService.softDeleteByIds(Collections.singletonList(arOrderRecordId));
            } catch (Exception e) {
                log.error("单据: {}自动生成应收单发生错误,错误原因: {}", arOrderRecordSaveParam.getSourceNo(), e.getMessage());
                //更新生成失败
                arOrderRecordDomainService.updateArFlagAndFailMsgById(arOrderRecordId,UdcEnum.AR_FLAG_TYPE_2.getValueCode(),e.getMessage());
                trans.setRollbackOnly();
            }
            return arOrderId;
        });
    }

    private void autoCreateArOrderHandle(ArOrder arOrder){
        arOrder.setTaxFlag(true);
        arOrder.setInitFlag(true);


        //公司信息
        List<OrgOuRpcDTO> orgOuRpcDTOList = selectOu(arOrder);
        if (CollectionUtil.isEmpty(orgOuRpcDTOList)){
            throw new BusinessException("未查询到公司信息");
        }
        OrgOuRpcDTO orgOuRpcDTO = orgOuRpcDTOList.get(0);
        arOrder.setOuName(orgOuRpcDTO.getOuName());
        arOrder.setOuCode(orgOuRpcDTO.getOuCode());
        arOrder.setOuId(orgOuRpcDTO.getId());
        //币种信息
        Map<String, String> currMap = selectCurr(arOrder);
        arOrder.setCurrName(currMap.getOrDefault(arOrder.getCurrCode(), null));
        arOrder.setLocalCurrName(currMap.getOrDefault(arOrder.getLocalCurrCode(), null));



        List<ArOrderDtl> dtlList = arOrder.getDtlList();


        dtlList.forEach(arOrderDtl -> {

            // 含税金额(本位币)=含税金额*汇率
            BigDecimal totalCurAmt = arOrderDtl.getTotalAmt().multiply(arOrder.getExchangeRate()).setScale(2, RoundingMode.HALF_UP);
            arOrderDtl.setTotalCurAmt(totalCurAmt);
            // 不含税金额(本位币)=不含税金额*汇率
            BigDecimal exclTaxCurAmt = arOrderDtl.getExclTaxAmt().multiply(arOrder.getExchangeRate()).setScale(2, RoundingMode.HALF_UP);
            arOrderDtl.setExclTaxCurAmt(exclTaxCurAmt);
            // 税额(本位币) =含税金额(本位币)-不含税金额(本位币)
            BigDecimal taxCurAmt = totalCurAmt.subtract(exclTaxCurAmt).setScale(2, RoundingMode.HALF_UP);
            arOrderDtl.setTaxCurAmt(taxCurAmt);
        });
    }


    /**
     * 定时任务自动生成应收单
     *
     * @param param 入参
     * @return 出参
     */
    @Override
    public void jobAutoCreate(String param){

        ArOrderRecordParam arOrderRecordParam = new ArOrderRecordParam();
        if (StringUtils.isNotBlank(param)) {
            JSONObject jsonObject = JSONObject.parseObject(param);
            ArOrderRecordParam arOrderRecordJobParam = JSONObject.toJavaObject(jsonObject, ArOrderRecordParam.class);
            if (Objects.nonNull(arOrderRecordJobParam)) {
                ArOrderRecordConvert.INSTANCE.oldToNewParam(arOrderRecordJobParam,arOrderRecordParam);
            }
        }

        jobAutoCreateArOrder(arOrderRecordParam);
    }

    @Override
    public void jobAutoCreateArOrder(ArOrderRecordParam arOrderRecordParam) {
        log.info("登录人信息：{}", SecurityContextUtil.currentUser());
        log.info("自动生成应收单定时任务参数:{}", JSON.toJSONString(arOrderRecordParam));

        ArrayList arFlagList = new ArrayList<>();
        arFlagList.add(UdcEnum.AR_FLAG_TYPE_0.getValueCode());
        arOrderRecordParam.setArFlagList(arFlagList);
        List<ArOrderRecordDTO> arOrderRecordDTOList = arOrderRecordDomainService.selectDetailsByParam(arOrderRecordParam);

        if (CollectionUtil.isEmpty(arOrderRecordDTOList)){
            log.info("查询待生成应收单记录数据为空");
            return;
        }

        arOrderRecordDTOList.forEach(arOrderRecordDTO -> {

            //1.数据转换
            Long arOrderRecordId = arOrderRecordDTO.getId();
            ArOrderRecordSaveParam arOrderRecordSaveParam = ArOrderRecordConvert.INSTANCE.dtoToSaveParam(arOrderRecordDTO);
            List<ArOrderDtlRecordSaveParam> arOrderDtlRecordSaveParams = ArOrderDtlRecordConvert.INSTANCE.dtoToSaveParamBatch(arOrderRecordDTO.getArOrderDtlRecordDTOS());
            arOrderRecordSaveParam.setArOrderDtlRecordSaveParams(arOrderDtlRecordSaveParams);
            //2.校验
            //2.查询应收单类型的配置规则
            //设置事务传播行为
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            ArTypeDTO arTypeDTO = transactionTemplate.execute(trans -> {
                ArTypeDTO arTypeDto = null;
                try {
                    arTypeDto = selectArType(arOrderRecordSaveParam,arOrderRecordId);
                } catch (Exception e) {
                    trans.setRollbackOnly();
                }
                return arTypeDto;
            });
            //当查询的应付单类型配置不为空时才可继续生成应收单
            if (Objects.nonNull(arTypeDTO)){
                //3.生成应收单
                autoCreateArOrder(arOrderRecordSaveParam,arTypeDTO,arOrderRecordId);
            }

        });

    }


    private List<OrgOuRpcDTO> selectOu(ArOrder arOrder){
        List<OrgOuRpcDTO> orgOuAllList = new ArrayList<>();
        if (Objects.nonNull(arOrder.getOuId()) && StringUtils.isBlank(arOrder.getOuCode())){
            OrgOuRpcDtoParam orgOuByIdParam = new OrgOuRpcDtoParam();
            orgOuByIdParam.setOuIds(Collections.singletonList(arOrder.getOuId()));
            List<OrgOuRpcDTO> orgOuRpcDTOList = orgOuRpcService.findOuDtoByParam(orgOuByIdParam);
            if (CollectionUtil.isNotEmpty(orgOuRpcDTOList)){
                orgOuAllList.addAll(orgOuRpcDTOList);
            }
        }
        if (StringUtils.isNotBlank(arOrder.getOuCode())){
            OrgOuRpcDtoParam orgOuByCodeParam = new OrgOuRpcDtoParam();
            orgOuByCodeParam.setOuCodes(Collections.singletonList(arOrder.getOuCode()));
            List<OrgOuRpcDTO> orgOuRpcDTOList = orgOuRpcService.findOuDtoByParam(orgOuByCodeParam);
            if (CollectionUtil.isNotEmpty(orgOuRpcDTOList)){
                orgOuAllList.addAll(orgOuRpcDTOList);
            }
        }

        return orgOuAllList;
    }

    private Map<String, String> selectCurr(ArOrder arOrder){
        Set<String> currCodes = new HashSet<>();
        currCodes.add(arOrder.getCurrCode());
        currCodes.add(arOrder.getLocalCurrCode());
        List<SysCurrencyRespDTO> currencyRespDTOS = systemRpcService.findCurrByCodes(currCodes);
        if (CollectionUtil.isEmpty(currencyRespDTOS)){
            return new HashMap<>();
        }
        return currencyRespDTOS.stream().collect(Collectors.toMap(i -> i.getCurrCode(), i -> i.getCurrName(), (o, n) -> n));
    }

}
