package com.elitesland.fin.application.service.flow;

import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitesland.fin.application.convert.flow.AccountFlowConvert;
import com.elitesland.fin.application.facade.param.flow.AccountFlowAccAmtParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowPageParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowParam;
import com.elitesland.fin.application.facade.vo.account.AccountVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowAccAmtVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.service.account.AccountService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.SysNumEnum;
import com.elitesland.fin.common.SysNumberGenerator;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.flow.AccountFlowDO;
import com.elitesland.fin.repo.flow.AccountFlowRepo;
import com.elitesland.fin.repo.flow.AccountFlowRepoProc;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.temporal.TemporalAdjusters;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明: 账户流水共同的ServiceImpl
 * </p>
 *
 * @Author Darren
 * @Date 2023/02/27
 * @Version 1.0
 * @Content:
 */
@Service
public class AccountFlowCommonServiceImpl implements AccountFlowCommonService {

    @Autowired
    private AccountFlowRepo accountFlowRepo;
    @Autowired
    private AccountFlowRepoProc accountFlowRepoProc;

    @Autowired
    private AccountService accountService;
    @Autowired
    private SysNumberGenerator sysNumberGenerator;

    /**
     * 根据账户编码查询上一条流水的信息:只查询数据库数据
     * <p>
     * 按照创建时间进行倒叙查询并取第一条数据
     *
     * @param accCode 账户编码
     * @return 账户流水信息
     */
    @Override
    @SysCodeProc
    public Optional<AccountFlowVO> selectByAccCode(String accCode) {
        if (StringUtils.isEmpty(accCode)) {
            return Optional.empty();
        }
        AccountFlowVO accountFlowVO = accountFlowRepoProc.selectByAccCode(accCode);

        return Optional.ofNullable(accountFlowVO);
    }

    /**
     * 查询上一条流水的信息:查询数据库或查询账户里的数据
     * <p>
     * 初始化第一次查不到流水记录的时候需要查询账户里的信息
     * 查到上一条流水数据时使用上一条的数据
     *
     * @param accCode 账户编码
     * @return 组装后的账户流水账户金额、账户占用金额计算处理返回共用对象
     */
    @Override
    public AccountFlowAccAmtVO selectPreviousData(String accCode) {
        boolean previousFlag = true;
        boolean accFlag = true;
        AccountFlowAccAmtVO accountFlowAccAmtVO = new AccountFlowAccAmtVO();
        val accountFlowVOOptional = this.selectByAccCode(accCode);
        if (accountFlowVOOptional.isPresent()) {
            previousFlag = false;
            accFlag = false;
            AccountFlowVO accountFlowVO = accountFlowVOOptional.get();
            accountFlowAccAmtVO.setAccAmt(accountFlowVO.getAccAmt());
            accountFlowAccAmtVO.setAccOccAmt(accountFlowVO.getAccOccAmt());
        } else {
            //初始化第一次查不到的时候 查询账户里的信息
            previousFlag = false;
            accFlag = false;

            AccountVO accountVO = this.selectAccountByAccCode(accCode);

            accountFlowAccAmtVO.setAccAmt(accountVO.getAccAmt());
            accountFlowAccAmtVO.setAccOccAmt(accountVO.getAccOccAmt());
        }

        if (previousFlag) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到上一条流水金额数据!");
        }
        if (accFlag) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到账户金额数据!");
        }

        return accountFlowAccAmtVO;
    }

    /**
     * 根据账户编码查询账户信息
     *
     * @param accCode 账户编码
     * @return 账户信息对象
     */
    @Override
    public AccountVO selectAccountByAccCode(String accCode) {
        if (StringUtils.isBlank(accCode)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户编码为空!");
        }
        AccountVO accountVO = accountService.getByCode(accCode);
        if (Objects.isNull(accountVO)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到账户信息!");
        }
        return accountVO;
    }

    /**
     * 共用入参对象赋值、传值
     *
     * @param amount           本条发生金额
     * @param accAmt           上一条的账户金额
     * @param accOccAmt        上一条的账户占用金额
     * @param accHandleType    账户金额处理类型
     * @param accOccHandleType 账户占用金额处理类型
     * @return 共用入参对象
     */
    @Override
    public AccountFlowAccAmtParam accAmtParamPassValue(BigDecimal amount, BigDecimal accAmt, BigDecimal accOccAmt
            , String accHandleType, String accOccHandleType) {
        return AccountFlowAccAmtParam.builder()
                .amount(amount)
                .accAmt(accAmt)
                .accOccAmt(accOccAmt)
                .accHandleType(accHandleType)
                .accOccHandleType(accOccHandleType).build();
    }

    /**
     * 根据账户处理类型处理填充账户金额、账户占用金额,进行计算赋值：
     * <p>
     * 本条账户金额 = 上一条的账户金额
     * 本条账户金额 = 上一条的账户金额 + 本条发生金额
     * 本条账户金额 = 上一条的账户金额 - 本条发生金额
     * 本条账户金额 = 赋值为空
     * <p>
     * 本条账户占用金额 = 上一条的账户占用金额
     * 本条账户占用金额 = 上一条的账户占用金额 + 本条发生金额
     * 本条账户占用金额 = 上一条的账户占用金额 - 本条发生金额
     * 本条账户占用金额 = 赋值为空
     * <p>
     * 账户流水账户金额、账户占用金额处理类型-相加、相减、相同、值为空
     *
     * @param accAmtParam 账户金额、账户占用金额计算处理入参共用对象
     * @return 账户金额、账户占用金额计算处理返回共用对象
     */
    @Override
    public AccountFlowAccAmtVO handleAccAmt(AccountFlowAccAmtParam accAmtParam) {
        if (StringUtils.isBlank(accAmtParam.getAccHandleType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户金额处理类型为空!");
        }
        if (StringUtils.isBlank(accAmtParam.getAccOccHandleType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户占用金额处理类型为空!");
        }
        AccountFlowAccAmtVO accountFlowAccAmtVO = new AccountFlowAccAmtVO();
        //本条发生金额
        BigDecimal amount = Objects.isNull(accAmtParam.getAmount()) ? BigDecimal.ZERO : accAmtParam.getAmount();
        //上一条的账户金额
        BigDecimal accAmt = Objects.isNull(accAmtParam.getAccAmt()) ? BigDecimal.ZERO : accAmtParam.getAccAmt();
        //上一条的账户占用金额
        BigDecimal accOccAmt = Objects.isNull(accAmtParam.getAccOccAmt()) ? BigDecimal.ZERO : accAmtParam.getAccOccAmt();
        //1.处理计算账户金额
        switch (accAmtParam.getAccHandleType()) {
            case FinConstant.ACC_HANDLE_TYPE_EQU:
                //本条账户金额 = 上一条的账户金额
                accountFlowAccAmtVO.setAccAmt(accAmt);
                break;
            case FinConstant.ACC_HANDLE_TYPE_ADD:
                //本条账户金额 = 上一条的账户金额 + 本条发生金额
                accountFlowAccAmtVO.setAccAmt(accAmt.add(amount));
                break;
            case FinConstant.ACC_HANDLE_TYPE_SUB:
                //本条账户金额 = 上一条的账户金额 - 本条发生金额
                accountFlowAccAmtVO.setAccAmt(accAmt.subtract(amount));
                break;
            case FinConstant.ACC_HANDLE_TYPE_EMP:
                //本条账户金额 = 赋值为空
                accountFlowAccAmtVO.setAccAmt(BigDecimal.ZERO);
                break;
            default:
                break;
        }
        //2.处理计算账户占用金额
        switch (accAmtParam.getAccOccHandleType()) {
            case FinConstant.ACC_HANDLE_TYPE_EQU:
                //本条账户占用金额 = 上一条的账户占用金额
                accountFlowAccAmtVO.setAccOccAmt(accOccAmt);
                break;
            case FinConstant.ACC_HANDLE_TYPE_ADD:
                //本条账户占用金额 = 上一条的账户占用金额 + 本条发生金额
                accountFlowAccAmtVO.setAccOccAmt(accOccAmt.add(amount));
                break;
            case FinConstant.ACC_HANDLE_TYPE_SUB:
                //本条账户占用金额 = 上一条的账户占用金额 - 本条发生金额
                accountFlowAccAmtVO.setAccOccAmt(accOccAmt.subtract(amount));
                break;
            case FinConstant.ACC_HANDLE_TYPE_EMP:
                //本条账户占用金额 = 赋值为空
                accountFlowAccAmtVO.setAccOccAmt(BigDecimal.ZERO);
                break;
            default:
                break;
        }


        return accountFlowAccAmtVO;
    }

    /**
     * 账户流水批量保存：
     *
     * @param doList 账户流水入参
     * @return 账户流水ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public List<Long> saveAccountFlowBatch(List<AccountFlowDO> doList) {
        return accountFlowRepo.saveAll(doList).stream().map(AccountFlowDO::getId).collect(Collectors.toList());
    }

    /**
     * 账户流水保存：
     *
     * @param accountFlowDO 账户流水入参
     * @return 账户流水ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveAccountFlow(AccountFlowDO accountFlowDO) {
        return accountFlowRepo.save(accountFlowDO).getId();
    }

    /**
     * 账户流水保存并更新账户金额信息：
     *
     * @param accountFlowParam 账户流水入参
     * @return 账户流水ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveAccountAndFlow(AccountFlowParam accountFlowParam) {
        AccountFlowDO accountFlowDO = AccountFlowConvert.INSTANCE.paramToDo(accountFlowParam);
        Long id = accountFlowRepo.save(accountFlowDO).getId();
        accountService.updateAmtByCode(accountFlowDO.getAccountCode(), accountFlowDO.getAccountAmount(), accountFlowDO.getAccountOccupancyAmount());
        return id;
    }


    /**
     * 账户流水Param批量保存：对Param进行转换DO并保存
     *
     * @param paramList 入参
     * @return 账户流水ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public List<Long> saveParamBatch(List<AccountFlowParam> paramList) {
        val accountFlowDOList = paramList.stream().map(AccountFlowConvert.INSTANCE::paramToDo).collect(Collectors.toList());
        return accountFlowRepo.saveAll(accountFlowDOList).stream().map(AccountFlowDO::getId).collect(Collectors.toList());
    }

    /**
     * 默认赋值
     *
     * @param accountFlowParam 入参
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void defaultAssignment(AccountFlowParam accountFlowParam) {
        //流水号
        String flowNo = sysNumberGenerator.generate(SysNumEnum.FIN_FLOW_NO.getCode());
        accountFlowParam.setFlowNo(flowNo);
        //账户进出方式
        String accIoType = accIoTypeAssignment(accountFlowParam);
        accountFlowParam.setAccIoType(accIoType);
        //来源平台
        if (StringUtils.isEmpty(accountFlowParam.getSourcePlatform())) {
            accountFlowParam.setSourcePlatform(UdcEnum.FIN_SOURCE_PLATFORM_TYPE_OTHER.getValueCode());
        }

    }

    /**
     * 账户进出方式默认赋值
     *
     * @param accountFlowParam 入参
     * @return 账户进出方式
     */
    private String accIoTypeAssignment(AccountFlowParam accountFlowParam) {
        //默认无
        String accIoType = FinConstant.ACC_IO_TYPE_NO;
        //可以根据交易类型直接进行区分：发生金额为增加的交易类型编码的集合
        if (FinConstant.ADD_TRANSACTION_TYPE.contains(accountFlowParam.getTransactionType())) {
            return FinConstant.ACC_IO_TYPE_ADD;
        }
        //可以根据交易类型直接进行区分：发生金额为扣减的交易类型编码的集合
        if (FinConstant.SUB_TRANSACTION_TYPE.contains(accountFlowParam.getTransactionType())) {
            return FinConstant.ACC_IO_TYPE_SUB;
        }
        //可以根据交易类型直接进行区分：发生金额为无-增加的交易类型编码的集合
        if (FinConstant.NO_ADD_TRANSACTION_TYPE.contains(accountFlowParam.getTransactionType())) {
            return FinConstant.ACC_IO_TYPE_NO_ADD;
        }
        //可以根据交易类型直接进行区分：发生金额为无-扣减的交易类型编码的集合
        if (FinConstant.NO_SUB_TRANSACTION_TYPE.contains(accountFlowParam.getTransactionType())) {
            return FinConstant.ACC_IO_TYPE_NO_SUB;
        }
        //调整单、调剂单的交易类型是调整原因并后续可能增加，故不适用通过交易类型区分，
        //数据来源:调整类型=调出的调整单审核通过
        if (Objects.equals(accountFlowParam.getDataSource(), UdcEnum.FIN_DATA_SOURCE_TYPE_TZD03.getValueCode())) {
            //使用数据来源+交易类型区分,当调整单释放时为无，其它默认为减
            if (Objects.equals(accountFlowParam.getTransactionType(), UdcEnum.FIN_TRANSACTION_TYPE_TZDSF.getValueCode())) {
                accIoType = FinConstant.ACC_IO_TYPE_NO_SUB;
            } else {
                //账户类型为统筹账户时也是
                accIoType = FinConstant.ACC_IO_TYPE_SUB;
            }
            return accIoType;
        }
        //数据来源:调整类型=调进的调整单审核通过
        if (Objects.equals(accountFlowParam.getDataSource(), UdcEnum.FIN_DATA_SOURCE_TYPE_TZD04.getValueCode())) {
            //账户类型为统筹账户时也是
            return FinConstant.ACC_IO_TYPE_ADD;
        }
        //数据来源:调剂单审核通过
        if (Objects.equals(accountFlowParam.getDataSource(), UdcEnum.FIN_DATA_SOURCE_TYPE_TJD03.getValueCode())) {
            //使用数据来源+交易类型或拆分类型区分,当调剂单释放时为无，从账户时减,至账户时加
            if (Objects.equals(accountFlowParam.getTransactionType(), UdcEnum.FIN_TRANSACTION_TYPE_TJDSF.getValueCode())) {
                accIoType = FinConstant.ACC_IO_TYPE_NO_SUB;
            } else if (Objects.equals(accountFlowParam.getSplitType(), FinConstant.SPLIT_TYPE_ADJUST_APPROVE_FROM)) {
                //账户类型为统筹账户时也是
                accIoType = FinConstant.ACC_IO_TYPE_SUB;
            } else if (Objects.equals(accountFlowParam.getSplitType(), FinConstant.SPLIT_TYPE_ADJUST_APPROVE_TO)) {
                //账户类型为统筹账户时也是
                accIoType = FinConstant.ACC_IO_TYPE_ADD;
            }
            return accIoType;
        }

        return accIoType;
    }

    /**
     * 根据账户进出方式对发生金额进行正负转换
     *
     * @param accIoType 账户进出方式
     * @return -1 或 1
     */
    @Override
    public BigDecimal plusOrMinus(String accIoType) {
        if (StringUtils.isBlank(accIoType)) {
            return BigDecimal.ONE;
        }
        if (FinConstant.ACC_IO_TYPE_ADD_LIST.contains(accIoType)) {
            return BigDecimal.ONE;
        } else if (FinConstant.ACC_IO_TYPE_SUB_LIST.contains(accIoType)) {
            return BigDecimal.ONE.negate();
        }
        return BigDecimal.ONE;
    }

    /**
     * 根据账户进出方式判断是否为负数(扣减)
     *
     * @param accIoType 账户进出方式
     * @return true 或 false
     */
    @Override
    public Boolean plusOrMinus02(String accIoType) {
        if (StringUtils.isBlank(accIoType)) {
            return false;
        }
        if (FinConstant.ACC_IO_TYPE_ADD_LIST.contains(accIoType)) {
            return false;
        } else if (FinConstant.ACC_IO_TYPE_SUB_LIST.contains(accIoType)) {
            return true;
        }
        return false;
    }

    /**
     * 判断过滤出账户类型为整车的
     *
     * @param accType 账户类型
     * @return 属于 true 否则 false
     */
    @Override
    public boolean judgeZcAccType(String accType) {
        if (Objects.equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCCZZH.getValueCode(), accType)
                || Objects.equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCFLZH.getValueCode(), accType)) {
            return true;
        }
        return false;
    }

    /**
     * 判断过滤出账户类型为配件的
     *
     * @param accType 账户类型
     * @return 属于 true 否则 false
     */
    @Override
    public boolean judgePjAccType(String accType) {
        if (Objects.equals(UdcEnum.FIN_ACCOUNT_TYPE_PJCZZH.getValueCode(), accType)
                || Objects.equals(UdcEnum.FIN_ACCOUNT_TYPE_PJFLZH.getValueCode(), accType)) {
            return true;
        }
        return false;
    }

    /**
     * 判断过滤出账户类型为附件包的
     *
     * @param accType 账户类型
     * @return 属于 true 否则 false
     */
    @Override
    public boolean judgeFjbAccType(String accType) {
        if (Objects.equals(UdcEnum.FIN_ACCOUNT_TYPE_FJCZZH.getValueCode(), accType)) {
            return true;
        }
        return false;
    }

    /**
     * 判断过滤出账户类型为统筹的
     *
     * @param accType 账户类型
     * @return 属于 true 否则 false
     */
    @Override
    public boolean judgeTcAccType(String accType) {
        if (Objects.equals(UdcEnum.FIN_ACCOUNT_TYPE_TCZH.getValueCode(), accType)) {
            return true;
        }
        return false;
    }

    /**
     * 把年月格式yyyy-MM字符串转换拼接成年月日格式的日期
     *
     * @param yearMonthStr 年月格式yyyy-MM字符串入参
     * @return 年月日格式的日期
     */
    @Override
    public LocalDate transitionYearMonthStr(String yearMonthStr) {
        if (StringUtils.isBlank(yearMonthStr)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "年月格式字符串为空!");
        }
        String[] yearMonthParams = yearMonthStr.split(FinConstant.LINE_SPLIT);
        if (yearMonthParams.length != FinConstant.SPLIT_ARRAY_LENGTH) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "年月格式字符串格式不正确!");
        }
        int year = Integer.parseInt(yearMonthParams[FinConstant.ARRAY_INDEX_0]);
        int month = Integer.parseInt(yearMonthParams[FinConstant.ARRAY_INDEX_1]);
        //把年月拼接成年月日
        return LocalDate.of(year, month, FinConstant.FIRST_DAY_OF_MONTH);
    }

    /**
     * 根据年月日格式的日期拼装入参里的交易起始时间、交易截止时间
     * 根据传入的当前年月日获取其所在月份的第一天最小值和最后一天最大值
     *
     * @param localDate 年月日格式的日期
     * @param pageParam 查询入参
     */
    @Override
    public void splicTransactionTimeSection(LocalDate localDate, AccountFlowPageParam pageParam) {
        if (Objects.isNull(localDate)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "年月日格式的日期为空!");
        }
        // 获取第一天的最小时间
        LocalDateTime minTime = LocalDateTime.of(localDate.with(TemporalAdjusters.firstDayOfMonth()), LocalTime.MIN);
        // 获取最后一天的最大时间
        LocalDateTime maxTime = LocalDateTime.of(localDate.with(TemporalAdjusters.lastDayOfMonth()), LocalTime.MAX);
        pageParam.setTransactionTimeS(minTime);
        pageParam.setTransactionTimeE(maxTime);
    }


    /**
     * 调剂单审核通过时对入参的拆分类型做判断：必须有一条，是表明是从账户的，用于拆分使用
     *
     * @param paramList 入参
     */
    @Override
    public void checkAdjustApprove(List<AccountFlowParam> paramList) {
        paramList.forEach(param -> {
            if (StringUtils.isBlank(param.getSplitType())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "拆分类型为空!");
            }
        });
    }

    /**
     * 调剂单审核通过时:判断是否需要拆分一条交易类型为调剂单释放的数据
     *
     * @param birthOrderType 生单类型
     * @return true 是  false 否
     */
    @Override
    public boolean judgeAdjustRelease(String birthOrderType) {
        //判断生单类型是否是系统自动生单，自动生单时不需要拆分
        if (Objects.equals(birthOrderType, FinConstant.BIRTH_ORDER_TYPE_ADJUST_AUTO)) {
            return false;
        }
        return true;
    }

}
