package com.elitesland.fin.application.service.flowrepair;

import com.alibaba.fastjson.JSON;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.fin.application.convert.creditaccount.CreditAccountRuleConfigDtlConvert;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountRuleConfigDTO;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountRuleConfigDtlDTO;
import com.elitesland.fin.application.facade.dto.flowrepair.AccountFlowRepairHandleDTO;
import com.elitesland.fin.application.facade.param.flow.AccountFlowPageParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowQueryParam;
import com.elitesland.fin.application.facade.param.flowrepair.AccountFlowRepairPageParam;
import com.elitesland.fin.application.facade.param.flowrepair.AccountFlowRepairParam;
import com.elitesland.fin.application.facade.param.flowrepair.AccountFlowRepairRedoParam;
import com.elitesland.fin.application.facade.vo.account.AccountRuleConfigQueryVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.facade.vo.flowrepair.AccountFlowRepairVO;
import com.elitesland.fin.application.service.flow.AccountFlowService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinRedisConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.account.AccountDO;
import com.elitesland.fin.entity.creditaccount.CreditAccountRuleConfigDtlDO;
import com.elitesland.fin.entity.flowrepair.AccountFlowRepairDO;
import com.elitesland.fin.repo.account.AccountRepo;
import com.elitesland.fin.repo.account.AccountRepoProc;
import com.elitesland.fin.repo.flow.AccountFlowRepoProc;
import com.elitesland.fin.repo.flowrepair.AccountFlowRepairRepo;
import com.elitesland.fin.repo.flowrepair.AccountFlowRepairRepoProc;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RBucket;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Example;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2023/10/25
 * @Version 1.0
 * @Content:
 */
@Service
@Slf4j
@AllArgsConstructor
public class AccountFlowRepairServiceImpl implements AccountFlowRepairService{

    @Autowired
    private AccountFlowRepairRepo accountFlowRepairRepo;
    @Autowired
    private AccountFlowRepairRepoProc accountFlowRepairRepoProc;
    @Autowired
    private AccountFlowService accountFlowService;
    @Autowired
    private AccountFlowRepoProc accountFlowRepoProc;
    @Autowired
    private AccountRepoProc accountRepoProc;
    @Autowired
    private AccountRepo accountRepo;

    private final RedissonClient redissonClient;

    /**
     * 账户流水修复重算分页查询
     *
     * @param pageParam 入参
     * @return 账户流水修复重算信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountFlowRepairVO> page(AccountFlowRepairPageParam pageParam){

        PagingVO<AccountFlowRepairVO> pagingVO = accountFlowRepairRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AccountFlowRepairVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        return PagingVO.<AccountFlowRepairVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }


    /**
     * 账户流水修复重算保存
     *
     * @param param 入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void save(AccountFlowRepairParam param){

        if (StringUtils.isBlank(param.getFlowNo())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "流水号不能为空!");
        }
        if (Objects.isNull(param.getActualAmount())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "实际金额不能为空!");
        }

        AccountFlowVO flowVO = selectFlowByFlowNo(param.getFlowNo());

        if (Objects.isNull(param.getId())){
            AccountFlowRepairDO flowRepairDO = new AccountFlowRepairDO();
            assembleFlowRepair(flowRepairDO,param,flowVO);
            accountFlowRepairRepo.save(flowRepairDO);
        }else {
            Optional<AccountFlowRepairDO> flowRepairOptional = accountFlowRepairRepo.findById(param.getId());
            if (flowRepairOptional.isEmpty()){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到此条流水修复数据!");
            }
            AccountFlowRepairDO flowRepairDO = flowRepairOptional.get();
            if (!Objects.equals(flowRepairDO.getRepairStatus(),UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode())){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "草稿状态才可修改!");
            }
            assembleFlowRepair(flowRepairDO,param,flowVO);
            accountFlowRepairRepo.save(flowRepairDO);
        }

    }

    private void assembleFlowRepair(AccountFlowRepairDO flowRepairDO,
                               AccountFlowRepairParam param,AccountFlowVO flowVO){
        flowRepairDO.setFlowNo(param.getFlowNo());
        flowRepairDO.setActualAmount(param.getActualAmount());
        flowRepairDO.setAmount(flowVO.getAmount());
        flowRepairDO.setOpenAccountEntityCode(null);
        flowRepairDO.setOpenAccountEntityName(flowVO.getAccountHolderName());
        flowRepairDO.setAccountType(flowVO.getAccountType());
        flowRepairDO.setAccountCode(flowVO.getAccountCode());
        flowRepairDO.setAccountName(flowVO.getAccountName());
        flowRepairDO.setSourceNo(flowVO.getSourceNo());
        flowRepairDO.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());
    }

    private AccountFlowVO selectFlowByFlowNo(String flowNo){
        if (StringUtils.isBlank(flowNo)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "流水号为空!");
        }
        AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
        flowQueryParam.setFlowNo(flowNo);
        List<AccountFlowVO> flowVOList = accountFlowService.selectListByQueryParam(flowQueryParam);
        if (CollectionUtils.isEmpty(flowVOList)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到此流水号("+flowNo+")的流水信息!");
        }
        if (CollectionUtils.isNotEmpty(flowVOList) && flowVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "此流水号("+flowNo+")的存在多条流水信息!");
        }

        return flowVOList.get(0);
    }

    /**
     * 根据账户流水修复重算ID批量删除
     *
     * @param ids ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteBatch(List<Long> ids) {
        AccountFlowRepairPageParam flowRepairPageParam = new AccountFlowRepairPageParam();
        flowRepairPageParam.setIds(ids);
        List<AccountFlowRepairVO> respVOList = accountFlowRepairRepoProc.selectListByParam(flowRepairPageParam);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        //针对‘待发布’状态的任务（主表）
        List<AccountFlowRepairVO> voList = respVOList.stream().filter(respVO -> !Objects.equals(respVO.getRepairStatus(), UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(voList)) {
            String checkResult = voList.stream().map(vo ->
                    "流水号:" + vo.getFlowNo() + ""
            ).collect(Collectors.joining(";", "只有草稿可删除,[", "], 请检查"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        //逻辑删除
        accountFlowRepairRepoProc.updateDeleteFlagBatch(1, ids);

    }


    /**
     * 账户流水修复重算
     *
     * @param repairRedoParam 重算入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void repairRedo(AccountFlowRepairRedoParam repairRedoParam){
        //选中或者不选(默认全部)的草稿状态的才可重算
        AccountFlowRepairPageParam flowRepairPageParam = new AccountFlowRepairPageParam();
        flowRepairPageParam.setIds(repairRedoParam.getIds());
        flowRepairPageParam.setFlowNoList(repairRedoParam.getFlowNoList());
        flowRepairPageParam.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());
        List<AccountFlowRepairVO> repairFlowVOList = accountFlowRepairRepoProc.selectListByParam(flowRepairPageParam);
        if (CollectionUtils.isEmpty(repairFlowVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "无符合重算的账户流水数据!");
        }

        val flowNoCheckList = repairFlowVOList.stream().filter(repairParam -> !StringUtils.isBlank(repairParam.getFlowNo()))
                // 获得元素出现频率的 Map，键为元素，值为元素出现的次数
                .map(AccountFlowRepairVO::getFlowNo).collect(Collectors.toMap(e -> e, e -> 1, (a, b) -> a + b))
                // 所有 entry 对应的 Stream
                .entrySet().stream()
                // 过滤出元素出现次数大于 1 的 entry
                .filter(entry -> entry.getValue() > 1)
                // 获得 entry 的键（重复元素）对应的 Stream
                .map(entry -> entry.getKey())
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(flowNoCheckList)) {
            String checkResult = flowNoCheckList.stream().map(Objects::toString).collect(Collectors.joining("、", "账户流水号[", "], 存在重复数据,请检查!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        List<String> repairFlowNoList = repairFlowVOList.stream().map(AccountFlowRepairVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(repairFlowNoList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重算的账户流水数据的流水号为空!");
        }
        /*AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
        flowQueryParam.setFlowNoList(repairFlowNoList);
        log.info("修复重算账户流水查询数据,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowQueryParam));
        List<AccountFlowVO> flowVOList = accountFlowService.selectListByQueryParam(flowQueryParam);
        log.info("修复重算账户流水查询数据,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));

        if (CollectionUtils.isEmpty(flowVOList)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到账户流水信息!");
        }*/

        List<String> repairFlowNoCacheList = new ArrayList<>();
        repairFlowNoList.forEach(flowNo -> {
            if (StringUtils.isNotBlank(getFlowNoCache(flowNo))){
                repairFlowNoCacheList.add(flowNo);
            }else {
                saveFlowNoCache(flowNo);
            }
        });
        if (CollectionUtils.isNotEmpty(repairFlowNoCacheList)) {
            String checkResult = repairFlowNoCacheList.stream().map(Objects::toString).collect(Collectors.joining("、", "账户流水号[", "], 修复重算中,请稍后再次重算!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        if (CollectionUtils.isNotEmpty(repairFlowVOList)){
            processingCalculation(repairFlowVOList,repairFlowNoList);
        }

    }

    @Override
    public void removeFlowNoCacheBatch(List<String> flowNoList) {
        flowNoList.forEach(flowNo -> {
            removeFlowNoCache(flowNo);
        });
    }

    @Override
    public void removeFlowNoCache(String flowNo) {
        RBucket<String> bucket = redissonClient.getBucket(FinRedisConstant.ACCOUNT_FLOW_REPAIR_KEY+flowNo);
        if (bucket.isExists()) {
            bucket.delete();
        }
    }

    @Override
    public String getFlowNoCache(String flowNo) {
        RBucket<String> bucket = redissonClient.getBucket(FinRedisConstant.ACCOUNT_FLOW_REPAIR_KEY+flowNo);
        if (bucket.isExists()) {
            return bucket.get();
        }
        return null;
    }

    @Override
    public void saveFlowNoCache(String flowNo){
        RBucket<String> bucket = redissonClient.getBucket(FinRedisConstant.ACCOUNT_FLOW_REPAIR_KEY+flowNo);
        bucket.set(flowNo);
    }



    private void processingCalculation(List<AccountFlowRepairVO> repairFlowVOList,List<String> repairFlowNoList){

        List<String> accCodeList = new ArrayList<>();

        repairFlowVOList.forEach(flowRepairVO -> {
            //Optional<AccountFlowVO> flowOptional = flowVOList.stream().filter(flowVO -> Objects.equals(flowRepairVO.getFlowNo(), flowVO.getFlowNo())).findFirst();
            AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
            flowQueryParam.setFlowNo(flowRepairVO.getFlowNo());
            log.info("修复重算账户流水查询数据,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowQueryParam));
            List<AccountFlowVO> flowVOList = accountFlowService.selectListByQueryParam(flowQueryParam);
            log.info("修复重算账户流水查询数据,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));

            if (CollectionUtils.isNotEmpty(flowVOList) && flowVOList.size() == 1 ){
                if (Objects.nonNull(flowVOList.get(0))) {
                    AccountFlowVO flowVO = flowVOList.get(0);

                    if (Objects.equals(flowVO.getAccountType(),UdcEnum.ACCOUNT_TYPE_STORE.getValueCode()) &&
                            FinConstant.ACCOUNT_FLOW_REPAIR_TRANSACTION_TYPE.contains(flowVO.getTransactionType())){
                        repairFlow(flowRepairVO,flowVO,accCodeList);
                    }else if (Objects.equals(flowVO.getAccountType(),UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode()) &&
                            FinConstant.ACCOUNT_FLOW_REPAIR_TRANSACTION_TYPE.contains(flowVO.getTransactionType())){
                        repairFlow(flowRepairVO,flowVO,accCodeList);

                    }else {
                        log.info("未修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
                    }

                } else {
                    log.info("未查询到修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowRepairVO));
                }
            }else {
                log.info("修复账户流水数据为空或者有多条,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));
            }

        });

        //更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户金额、账户占用金额、账户可用金额
        if (CollectionUtils.isNotEmpty(accCodeList)) {
            List<String> accCodes = accCodeList.stream().filter(Objects::nonNull).distinct().collect(Collectors.toList());
            updateAccountAmt(accCodes,repairFlowNoList);
        }

        repairFlowNoList.forEach(flowNo -> {
            removeFlowNoCache(flowNo);
        });
    }


    @Transactional(rollbackFor = Exception.class)
    public void repairFlow(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,List<String> accCodeList) {
        repairCurrentFlow(flowRepairVO,flowVO,accCodeList);


    }

    /**
     * 根据账户编码更新账户金额：更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户金额、账户占用金额、账户可用金额
     *
     * @param accCodeList 账户编码
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateAccountAmt(List<String> accCodeList,List<String> repairFlowNoList) {
        accCodeList.forEach(accCode -> {
            //根据账户编码查询,按照创建时间进行倒叙查询并取第一条数据
            AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
            flowQueryParam.setAccountCode(accCode);
            AccountFlowVO flowVO = accountFlowRepoProc.selectPreviousRepairAmtByParam(flowQueryParam);
            //根据编码更新账户金额
            AccountDO accountDO = accountRepoProc.findByAccountCode(accCode);
            if (Objects.nonNull(flowVO) && Objects.nonNull(accountDO)){
                BigDecimal accountAmount = flowVO.getAccountAmount();
                BigDecimal accountOccupancyAmount = flowVO.getAccountOccupancyAmount();

                accountDO.setAccountAmount(accountAmount);
                accountDO.setAccountOccupancyAmount(accountOccupancyAmount);
                accountDO.setAccountAvailableAmount(accountAmount.subtract(accountOccupancyAmount));
                accountRepo.save(accountDO);
            }else if (Objects.isNull(accountDO)){
                repairFlowNoList.forEach(flowNo -> {
                    removeFlowNoCache(flowNo);
                });
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到账户("+accCode+")信息!");

            }


        });
    }

    /**
     * 修复重算当前此条账户流水表的信息和修复流水记录表的信息
     *
     * @param flowRepairVO   修复流水表的此条数据
     * @param flowVO   要修复的此条账户流水数据
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairCurrentFlow(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,List<String> accCodeList) {
        AccountFlowRepairHandleDTO flowRepairHandleDTO = assembleRepairHandle(flowRepairVO,flowVO);
        if (Objects.nonNull(flowRepairHandleDTO)){
            repairCalculationFlow(flowRepairHandleDTO,accCodeList);
            accountFlowRepairRepoProc.updateAmountAndRepairStatusById(
                    flowRepairHandleDTO.getAmount(),UdcEnum.FLOW_REPAIR_STATUS_FIXED.getValueCode(),flowRepairHandleDTO.getFlowRepairId());
            repairAfterFlow(flowRepairVO,flowVO,flowRepairHandleDTO,accCodeList);
        }

    }

    /**
     * 修复重算当前此条之后的账户流水表的信息
     *
     * @param flowRepairVO   修复流水表的此条数据
     * @param flowVO   要修复的此条账户流水数据
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairAfterFlow(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,AccountFlowRepairHandleDTO flowRepairCurrentHandleDTO,List<String> accCodeList) {
        List<AccountFlowRepairHandleDTO> flowRepairHandleList = selectRepairAfterFlow(flowRepairVO,flowVO,flowRepairCurrentHandleDTO);
        if (CollectionUtils.isNotEmpty(flowRepairHandleList)){
            flowRepairHandleList.forEach(flowRepairHandleDTO -> {
                repairCalculationFlowAfter(flowRepairHandleDTO,accCodeList);
            });
        }
    }

    /**
     * 查询某条账户流水以及之后账户流水的数据-相同账户编码的
     *
     * @param flowRepairVO   修复流水表的此条数据
     * @param flowVO   要修复的此条账户流水数据
     * @param flowRepairCurrentHandleDTO   当前修复的此条账户流水的处理数据
     * @return
     */
    public List<AccountFlowRepairHandleDTO> selectRepairAfterFlow(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,AccountFlowRepairHandleDTO flowRepairCurrentHandleDTO) {
        int size = 500;
        int page = 1;

        List<AccountFlowRepairHandleDTO>  flowRepairHandleList= new ArrayList<>();

        //查询要修复的该条账户流水及之后的相同账户编码的全部流水的
        AccountFlowPageParam flowPageParam = new AccountFlowPageParam();
        flowPageParam.setAccountCode(flowVO.getAccountCode());
        flowPageParam.setRepairTime(flowVO.getCreateTime());
        //之后的账户流水不做全部查出，要分批查出循环处理
        while (true) {
            flowPageParam.setCurrent(page++);
            flowPageParam.setSize(size);
            log.info("查询修复账户流水及之后流水,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowPageParam));
            List<AccountFlowVO> flowAfterList = accountFlowService.selectRepairAfterPage(flowPageParam);
            log.info("查询修复账户流水及之后流水,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowAfterList));
            if (CollectionUtils.isEmpty(flowAfterList)) {
                break;
            }

            flowAfterList.forEach(flowAfterVO -> {
                AccountFlowRepairHandleDTO flowRepairHandleDTO = assembleRepairAfterHandle(flowRepairVO,flowAfterVO,flowRepairCurrentHandleDTO);
                if (Objects.nonNull(flowRepairHandleDTO)){
                    flowRepairHandleList.add(flowRepairHandleDTO);
                }else {
                    log.info("按照交易类型未匹配到账户流水,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowAfterList));
                }
            });

            if (flowAfterList.size() < size) {
                // 少于要查询的数量，则说明已无数据
                break;
            }
        }

        return flowRepairHandleList;
    }

    private AccountFlowRepairHandleDTO assembleRepairHandle(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO){
        if (Objects.isNull(flowRepairVO.getActualAmount()) || Objects.isNull(flowVO.getAccountAmount()) ||
                Objects.isNull(flowVO.getAccountOccupancyAmount()) || Objects.isNull(flowVO.getAmount())){
            log.info("未修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
            return null;
        }
        AccountFlowRepairHandleDTO flowRepairHandleDTO = new AccountFlowRepairHandleDTO();
        flowRepairHandleDTO.setFlowRepairId(flowRepairVO.getId());
        flowRepairHandleDTO.setFlowRepairFlowNo(flowRepairVO.getFlowNo());
        flowRepairHandleDTO.setFlowId(flowVO.getId());
        flowRepairHandleDTO.setFlowNo(flowVO.getFlowNo());
        flowRepairHandleDTO.setActualAmount(flowRepairVO.getActualAmount());
        flowRepairHandleDTO.setAmount(flowVO.getAmount());
        //变动金额=实际交易金额-原先错误流水交易金额
        BigDecimal variableAmount = flowRepairVO.getActualAmount().subtract(flowVO.getAmount());
        flowRepairHandleDTO.setVariableAmount(variableAmount);
        flowRepairHandleDTO.setAccountType(flowVO.getAccountType());
        flowRepairHandleDTO.setAccountCode(flowVO.getAccountCode());
        flowRepairHandleDTO.setAccountName(flowVO.getAccountName());
        flowRepairHandleDTO.setTransactionType(flowVO.getTransactionType());

        flowRepairHandleDTO.setAccountAmount(flowVO.getAccountAmount());
        flowRepairHandleDTO.setAccountOccupancyAmount(flowVO.getAccountOccupancyAmount());
        flowRepairHandleDTO.setAccountAvailableAmount(flowVO.getAccountAvailableAmount());
        return flowRepairHandleDTO;
    }

    private AccountFlowRepairHandleDTO assembleRepairAfterHandle(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,AccountFlowRepairHandleDTO flowRepairCurrentHandleDTO){
        if (Objects.isNull(flowRepairVO.getActualAmount()) || Objects.isNull(flowVO.getAccountAmount()) ||
                Objects.isNull(flowVO.getAccountOccupancyAmount()) || Objects.isNull(flowVO.getAmount())){
            log.info("未修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
            return null;
        }
        AccountFlowRepairHandleDTO flowRepairHandleDTO = new AccountFlowRepairHandleDTO();
        flowRepairHandleDTO.setFlowRepairId(flowRepairVO.getId());
        flowRepairHandleDTO.setFlowRepairFlowNo(flowRepairVO.getFlowNo());
        flowRepairHandleDTO.setFlowId(flowVO.getId());
        flowRepairHandleDTO.setFlowNo(flowVO.getFlowNo());
        flowRepairHandleDTO.setActualAmount(flowRepairVO.getActualAmount());
        flowRepairHandleDTO.setAmount(flowVO.getAmount());
        //变动金额=实际交易金额-原先错误流水交易金额
        //BigDecimal variableAmount = flowRepairVO.getActualAmount().subtract(flowVO.getAmount());
        BigDecimal variableAmount = flowRepairCurrentHandleDTO.getVariableAmount();
        flowRepairHandleDTO.setVariableAmount(variableAmount);
        flowRepairHandleDTO.setAccountType(flowVO.getAccountType());
        flowRepairHandleDTO.setAccountCode(flowVO.getAccountCode());
        flowRepairHandleDTO.setAccountName(flowVO.getAccountName());
        flowRepairHandleDTO.setTransactionType(flowRepairCurrentHandleDTO.getTransactionType());

        flowRepairHandleDTO.setAccountAmount(flowVO.getAccountAmount());
        flowRepairHandleDTO.setAccountOccupancyAmount(flowVO.getAccountOccupancyAmount());
        flowRepairHandleDTO.setAccountAvailableAmount(flowVO.getAccountAvailableAmount());
        return flowRepairHandleDTO;
    }
    /**
     * 修复的计算处理逻辑
     *
     * @param flowRepairHandleDTO 账户流水修复重算处理计算DTO
     *
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairCalculationFlow(AccountFlowRepairHandleDTO flowRepairHandleDTO,List<String> accCodeList) {
        //账户金额
        BigDecimal accountAmount = flowRepairHandleDTO.getAccountAmount();
        //账户占用金额
        BigDecimal accountOccupancyAmount = flowRepairHandleDTO.getAccountOccupancyAmount();
        //账户可用金额 = 账户金额-账户占用金额
        BigDecimal accountAvailableAmount = BigDecimal.ZERO;
        //变动金额
        BigDecimal variableAmount = flowRepairHandleDTO.getVariableAmount();
        //实际金额
        BigDecimal actualAmount = flowRepairHandleDTO.getActualAmount();

        if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_R.getValueCode())) {
            //交易类型=释放占用时，该条流水及之后的相同账户编码的全部流水的账户占用金额改成原账户占用金额 - 变动金额，
            accountOccupancyAmount = accountOccupancyAmount.subtract(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);
            accountFlowRepoProc.updateOccupancyAndAvailableAndAmountById(
                    accountOccupancyAmount,accountAvailableAmount,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());

        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_T.getValueCode())) {
            //交易类型=账户占用时，该条流水及之后的相同账户编码的全部流水的账户占用金额改成原账户占用金额 + 变动金额，
            accountOccupancyAmount = accountOccupancyAmount.add(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateOccupancyAndAvailableAndAmountById(
                    accountOccupancyAmount,accountAvailableAmount,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_O.getValueCode())) {
            //交易类型=账户扣减时，该条流水之后的相同账户编码的全部流水的账户金额改成原账户金额 - 变更金额，
            accountAmount = accountAmount.subtract(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateAccountAndAvailableAndAmountById(
                    accountAmount,accountAvailableAmount,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_I.getValueCode())) {
            //交易类型=账户增加时，该条流水之后的相同账户编码的全部流水的账户金额改成原账户金额 + 变更金额，
            accountAmount = accountAmount.add(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateAccountAndAvailableAndAmountById(
                    accountAmount,accountAvailableAmount,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        }

    }

    /**
     * 之后流水的修复的计算处理逻辑
     *
     * @param flowRepairHandleDTO 账户流水修复重算处理计算DTO
     *
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairCalculationFlowAfter(AccountFlowRepairHandleDTO flowRepairHandleDTO,List<String> accCodeList) {
        //账户金额
        BigDecimal accountAmount = flowRepairHandleDTO.getAccountAmount();
        //账户占用金额
        BigDecimal accountOccupancyAmount = flowRepairHandleDTO.getAccountOccupancyAmount();
        //账户可用金额 = 账户金额-账户占用金额
        BigDecimal accountAvailableAmount = BigDecimal.ZERO;
        //变动金额
        BigDecimal variableAmount = flowRepairHandleDTO.getVariableAmount();
        //实际金额
        //BigDecimal actualAmount = flowRepairHandleDTO.getActualAmount();

        if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_R.getValueCode())) {
            //交易类型=释放占用时，该条流水及之后的相同账户编码的全部流水的账户占用金额改成原账户占用金额 - 变动金额，
            accountOccupancyAmount = accountOccupancyAmount.subtract(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);
            accountFlowRepoProc.updateOccupancyAndAvailableById(
                    accountOccupancyAmount,accountAvailableAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());

        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_T.getValueCode())) {
            //交易类型=账户占用时，该条流水及之后的相同账户编码的全部流水的账户占用金额改成原账户占用金额 + 变动金额，
            accountOccupancyAmount = accountOccupancyAmount.add(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateOccupancyAndAvailableById(
                    accountOccupancyAmount,accountAvailableAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_O.getValueCode())) {
            //交易类型=账户扣减时，该条流水之后的相同账户编码的全部流水的账户金额改成原账户金额 - 变更金额，
            accountAmount = accountAmount.subtract(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateAccountAndAvailableById(
                    accountAmount,accountAvailableAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_I.getValueCode())) {
            //交易类型=账户增加时，该条流水之后的相同账户编码的全部流水的账户金额改成原账户金额 + 变更金额，
            accountAmount = accountAmount.add(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateAccountAndAvailableById(
                    accountAmount,accountAvailableAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        }

    }



}
