package com.elitesland.fin.domain.service.arorder;

import cn.hutool.core.text.CharSequenceUtil;
import com.alibaba.nacos.shaded.com.google.common.collect.Lists;
import com.elitescloud.cloudt.core.seq.SeqNumProvider;
import com.elitesland.fin.application.convert.arorder.ArOrderConvert;
import com.elitesland.fin.application.convert.arorder.ArOrderDtlConvert;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.arorder.ArOrder;
import com.elitesland.fin.domain.entity.arorder.ArOrderDO;
import com.elitesland.fin.domain.entity.arorder.ArOrderDtl;
import com.elitesland.fin.domain.entity.arorder.ArOrderDtlDO;
import com.elitesland.fin.domain.param.arorder.ArOrderPageParam;
import com.elitesland.fin.domain.param.arorder.ArOrderParam;
import com.elitesland.fin.infr.dto.arorder.ArOrderDTO;
import com.elitesland.fin.infr.dto.arorder.ArOrderDtlDTO;
import com.elitesland.fin.infr.dto.common.ArVerDTO;
import com.elitesland.fin.infr.factory.arorder.ArOrderFactory;
import com.elitesland.fin.infr.repo.arorder.ArOrderDtlRepo;
import com.elitesland.fin.infr.repo.arorder.ArOrderDtlRepoProc;
import com.elitesland.fin.infr.repo.arorder.ArOrderRepo;
import com.elitesland.fin.infr.repo.arorder.ArOrderRepoProc;
import com.elitesland.workflow.ProcessInfo;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.domain.service.arorder
 * @date 2022/4/12 16:18
 */
@Service
@RequiredArgsConstructor
public class ArOrderDomainServiceImpl implements ArOrderDomainService {

    private final ArOrderRepo arOrderRepo;
    private final ArOrderRepoProc arOrderRepoProc;
    private final ArOrderDtlRepoProc arOrderDtlRepoProc;
    private final ArOrderDtlRepo arOrderDtlRepo;
    private final ArOrderFactory arOrderFactory;

    //发号器生成付款单号
    private final SeqNumProvider sysNumberRuleService;


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long save(ArOrder arOrder) {
        if (arOrder.getVerAmt() == null) {
            //默认核销状态
            arOrder.setDef();
        }
        // 单头信息输入的校验
        arOrder.check();
        // 校验明细行
        arOrder.checkDtlList();
        // 来源为销售订单
        arOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        if (arOrder.getCreateMode().equals(UdcEnum.FIN_AR_DOC_CLS_SO.getValueCode())) {
            arOrder.setSoDef();
            arOrder.countBySo();
        }
        // 数据的新增
        ArOrderDO apOrderRes = saveOrUpdate(arOrder);
        return apOrderRes.getId();
    }


    /**
     * 对 应收单主表/明细表/汇总明细表
     * 进行数据库新增
     *
     * @param arOrder
     * @return
     */
    @Transactional(rollbackFor = {Exception.class})
    public ArOrderDO saveOrUpdate(ArOrder arOrder) {
        if (CharSequenceUtil.isBlank(arOrder.getArOrderNo())) {
            // 发号器-生成应付单号
            String apOrderNo = sysNumberRuleService.generateCode(FinConstant.FIN,FinConstant.YSD, null);
            arOrder.setArOrderNo(apOrderNo);
        }
        ArOrderDO apOrderDO = ArOrderConvert.INSTANCE.convert(arOrder);
        //新增时锁版本默认值0
        if (apOrderDO.getId() == null) {
            apOrderDO.setAuditDataVersion(0);
        }
        ArOrderDO apOrderRes = arOrderRepo.save(apOrderDO);
        // 应收单明细单的新增
        // 根据应收单id删除明细行
        arOrderDtlRepoProc.delByMasId(Lists.newArrayList(apOrderRes.getId()));
        List<ArOrderDtl> apOrderDtlList = arOrder.getDtlList();
        List<ArOrderDtlDO> apOrderDtlDOS = ArOrderDtlConvert.INSTANCE.convert(apOrderDtlList);
        apOrderDtlDOS.stream().forEach(x -> {
            x.setMasId(apOrderRes.getId());
            arOrderDtlRepo.save(x);
        });
        return apOrderRes;
    }


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long commit(ArOrder arOrder, Boolean auto) {
        // 校验单头与明细
        arOrder.check();
        arOrder.checkDtlList();
        // 校验明细金额
        arOrder.checkAmt();
        // 金额数据校验(单头，明细，汇总)金额的校验
        arOrder.checkAmtSum();
        // 进行数据的新增或保存
        if (arOrder.getVerAmt() == null) {
            //默认核销状态
            arOrder.setDef();
        }
        //是否自动审核
        if (auto) {
            arOrder.setOrderState(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
            arOrder.setProposedStatus(UdcEnum.DOC_PROPOSED_STATUS_DRAFT.getValueCode());
        } else {
            arOrder.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
        }
        ArOrderDO arOrderDO = saveOrUpdate(arOrder);
        return arOrderDO.getId();
    }

    @Override
    public ArOrderDTO getArOrderAndDtl(Long id) {
        ArOrderDTO apOrderDTO = get(id);
        // 新增明细列表
        List<ArOrderDtlDTO> apOrderDtlDOS = arOrderDtlRepoProc.listByMisId(id);
        // 新增汇总列表
        apOrderDTO.setArOrderDtlDTOList(apOrderDtlDOS);
        return apOrderDTO;
    }

    @Override
    public Boolean queryByArTypeId(Long arTypeId) {
        return arOrderRepoProc.queryByArTypeId(arTypeId);
    }

    @Override
    public List<ArOrderDTO> getArOrderList(ArOrderParam arOrderParam) {
        List<ArOrderDTO> res = arOrderRepoProc.getArOrderList(arOrderParam);
        return res;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void updateVerAmt(Long id, BigDecimal amt) {
        ArVerDTO arVerDTO = arOrderRepoProc.queryVerAmtById(id);
        BigDecimal totalAmt = arVerDTO.getTotalAmt();
        if (arVerDTO.getVerAmt().add(amt).compareTo(totalAmt) > 0) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前核销金额大于应收单金额，请刷新页面重新输入");
        }
        arVerDTO.setAmtAndVerState(amt);
        long rows = arOrderRepoProc.updateVerAmt(arVerDTO);
        if (rows == 0) {
            throw new BusinessException("应付单更新已核销金额失败");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateWorkInfo(ProcessInfo processInfo, Long resId) {
        arOrderRepoProc.updateWorkInfo(processInfo, resId);
    }

    @Override
    public PagingVO<ArOrderDTO> page(ArOrderPageParam arOrderPageParam) {
        PagingVO<ArOrderDTO> res = arOrderFactory.page(arOrderPageParam);
        return res;
    }

    @Override
    public ArOrderDTO get(Long id) {
        ArOrderDTO arOrderDTO = arOrderRepoProc.get(id);
        return arOrderDTO;
    }


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long audit(List<Long> ids, String content, SysUserDTO user) {
        Long res = arOrderRepoProc.audit(ids, content, user);
        return res;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long del(List<Long> ids) {
        // 非草稿状态不能删除
        List<ArOrderDTO> arOrderDTOS = arOrderRepoProc.queryByIds(ids);
        arOrderDTOS.stream().forEach(x -> {
            if (!x.getOrderState().equals(UdcEnum.APPLY_STATUS_DRAFT.getValueCode())) {
                throw new BusinessException("非草稿状态不能删除");
            }
        });
        Long res = arOrderRepoProc.del(ids);
        // 删除明细信息
        arOrderDtlRepoProc.delByMasId(ids);
        return res;
    }

}
