package com.elitesland.fin.repo.payment;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.facade.dto.payment.PaymentRuleConfigDtlDTO;
import com.elitesland.fin.application.facade.param.payment.PaymentRuleConfigParam;
import com.elitesland.fin.application.facade.vo.payment.PaymentRuleConfigPageVO;
import com.elitesland.fin.entity.payment.PaymentRuleConfigDO;
import com.elitesland.fin.entity.payment.QPaymentRuleConfigDO;
import com.elitesland.fin.entity.payment.QPaymentRuleConfigDtlDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.List;

/**
 * @author gyj
 * @date 2023/05/24 18:23
 */
@Slf4j
@Component
@AllArgsConstructor
public class PaymentRuleConfigRepoProc {

    private final JPAQueryFactory jpaQueryFactory;

    private final QPaymentRuleConfigDO jpaQDo = QPaymentRuleConfigDO.paymentRuleConfigDO;
    private final QPaymentRuleConfigDtlDO jpaDQDo = QPaymentRuleConfigDtlDO.paymentRuleConfigDtlDO;

    public void updateDynamically(PaymentRuleConfigDO param) {
        JPAUpdateClause updateClause = jpaQueryFactory.update(jpaQDo);

        //编码
        if (StringUtils.isNotBlank(param.getRuleCode())) {
            updateClause.set(jpaQDo.ruleCode, param.getRuleCode());
        }

        //名称
        if (StringUtils.isNotBlank(param.getRuleName())) {
            updateClause.set(jpaQDo.ruleName, param.getRuleName());
        }

        //适用单据
        if (StringUtils.isNotBlank(param.getOptDoc())) {
            updateClause.set(jpaQDo.optDoc, param.getOptDoc());
        }

        //启用状态
        if (StringUtils.isNotBlank(param.getStatus())) {
            updateClause.set(jpaQDo.status, param.getStatus());
        }

        updateClause.set(jpaQDo.modifyTime, LocalDateTime.now());
        updateClause.where(jpaQDo.id.eq(param.getId())).execute();
    }


    public List<PaymentRuleConfigDtlDTO> searchByPaymentRuleConfigParam(PaymentRuleConfigParam paymentRuleConfigParam) {
        val jpaQuery = select(PaymentRuleConfigDtlDTO.class);
        jpaQuery.where(where(paymentRuleConfigParam));
        return jpaQuery.fetch();
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(
                Projections.bean(
                        cls,
                        jpaDQDo.id,
                        jpaQDo.ruleCode,
                        jpaQDo.ruleName,
                        jpaQDo.optDoc,
                        jpaDQDo.optDocType,
                        jpaDQDo.optDocStatus,
                        jpaDQDo.calculatePercent,
                        jpaQDo.status,
                        jpaDQDo.masId,
                        jpaQDo.createUserId,
                        jpaQDo.creator,
                        jpaQDo.createTime,
                        jpaQDo.modifyUserId,
                        jpaQDo.updater,
                        jpaQDo.modifyTime,
                        jpaQDo.deleteFlag,
                        jpaQDo.tenantId,
                        jpaQDo.auditDataVersion,
                        jpaDQDo.remark,
                        jpaDQDo.priorityNo,
                        jpaDQDo.optAccountType
                )
        ).from(jpaDQDo).leftJoin(jpaQDo).on(jpaQDo.id.eq(jpaDQDo.masId));
    }


    public PagingVO<PaymentRuleConfigPageVO> searchPage(PaymentRuleConfigParam param) {
        val jpaQuery = select(PaymentRuleConfigPageVO.class);
        jpaQuery.where(where(param));
        param.fillOrders(jpaQuery, jpaQDo);
        param.setPaging(jpaQuery);

        return PagingVO.<PaymentRuleConfigPageVO>builder()
                .total(jpaQuery.fetchCount())
                .records(jpaQuery.fetch())
                .build();
    }

    private Predicate where(PaymentRuleConfigParam param) {
        Predicate predicate = jpaQDo.isNotNull();

        //账户编码/名称
        if (StringUtils.isNotBlank(param.getMultiKeywords())) {
            predicate = ExpressionUtils.and(predicate, ExpressionUtils.or(
                    jpaQDo.ruleCode.like("%" + param.getMultiKeywords() + "%"),
                    jpaQDo.ruleName.like("%" + param.getMultiKeywords() + "%")
            ));
        }
        if (StringUtils.isNotBlank(param.getRuleCodeName())) {
            predicate = ExpressionUtils.and(predicate, ExpressionUtils.or(
                    jpaQDo.ruleCode.like("%" + param.getRuleCodeName() + "%"),
                    jpaQDo.ruleName.like("%" + param.getRuleCodeName() + "%")
            ));
        }

        //适用单据
        if (StringUtils.isNotEmpty(param.getOptDoc())) {
            predicate = ExpressionUtils.and(predicate, jpaQDo.optDoc.eq(param.getOptDoc()));
        }

        //适用单据类型
        if (StringUtils.isNotEmpty(param.getOptDocType())) {
            predicate = ExpressionUtils.and(predicate, jpaDQDo.optDocType.contains(param.getOptDocType()));
        }

        //适用单据状态
        if (StringUtils.isNotEmpty(param.getOptDocStatus())) {
            predicate = ExpressionUtils.and(predicate, jpaDQDo.optDocStatus.eq(param.getOptDocStatus()));
        }

        //适用账户类型
        if (StringUtils.isNotEmpty(param.getOptAccountType())) {
            predicate = ExpressionUtils.and(predicate, jpaDQDo.optAccountType.eq(param.getOptAccountType()));
        }

        //启用状态
        if (StringUtils.isNotEmpty(param.getStatus())) {
            predicate = ExpressionUtils.and(predicate, jpaQDo.status.eq(param.getStatus()));
        }

        predicate = ExpressionUtils.and(predicate, jpaQDo.deleteFlag.eq(0).or(jpaQDo.deleteFlag.isNull()));

        return predicate;
    }

}