package com.elitesland.fin.application.service.workflow.limitadjustorder;

import cn.hutool.core.util.StrUtil;
import com.elitesland.fin.application.facade.param.creditaccountflow.CreditAccountFlowParam;
import com.elitesland.fin.application.service.creditaccountflow.CreditAccountFlowService;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.limitadjustorder.LimitAdjustOrderDO;
import com.elitesland.fin.entity.limitadjustorder.QLimitAdjustOrderDO;
import com.elitesland.workflow.CommentInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;

/**
 * @author Jason.zhao
 * @date 2022/5/7 10:23
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class LimitAdjustOrderProcessServiceImpl implements LimitAdjustOrderProcessService {


    private final JPAQueryFactory jpaQueryFactory;

    private final CreditAccountFlowService creditAccountFlowService;

    QLimitAdjustOrderDO qLimitAdjustOrderDo = QLimitAdjustOrderDO.limitAdjustOrderDO;
    /**
     * 第一个用户任务节点的KEY
     */
    public static String FIRST_TASK_DEF_KEY = "createrSubmit";

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void processStatusChange(String businessKey, ProcInstStatus procInstStatus, CommentInfo commentInfo) {
        LimitAdjustOrderDO limitAdjustOrderDO = getLimitAdjustOrderDO(businessKey);
        log.info("开始财务调整单工作流回调，流程状态为:" + procInstStatus.getDesc());

        // 设置工作流状态
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qLimitAdjustOrderDo)
                .set(qLimitAdjustOrderDo.workflowProcInstStatus, procInstStatus)
                .set(qLimitAdjustOrderDo.auditUser, commentInfo.getUserName())
                .set(qLimitAdjustOrderDo.auditTime, commentInfo.getTime())
                .where(qLimitAdjustOrderDo.id.eq(limitAdjustOrderDO.getId()));
        // 单据状态
        switch (procInstStatus) {
            case NOTSUBMIT:
                // 未提交 将单据状态变成"草稿"
                jpaUpdateClause.set(qLimitAdjustOrderDo.docState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                break;
            case INTERRUPT:
                // 中断执行 将单据状态变成"草稿",并且将单据上的"流程实例状态"，"流程实例ID"清成null(不是空字符串)
                jpaUpdateClause.set(qLimitAdjustOrderDo.docState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                jpaUpdateClause.setNull(qLimitAdjustOrderDo.workflowProcInstId);
                break;

            case INVALID:
                // 作废 一般情况将单据状态变成"作废" ，直接删除单据
                jpaUpdateClause.set(qLimitAdjustOrderDo.docState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode())
                        .setNull(qLimitAdjustOrderDo.workflowProcInstId);
                break;
            case REJECTED:
                // 审批拒绝 一般情况将单据状态变成草稿
                // 工作流驳回到创建人节点/工作流审核拒绝 将单据状态设置为审批拒绝
                jpaUpdateClause.set(qLimitAdjustOrderDo.docState, UdcEnum.APPLY_STATUS_REJECTED.getValueCode());
                jpaUpdateClause.set(qLimitAdjustOrderDo.auditRejectReason, commentInfo.getComment());
                break;
            case APPROVING:
                // 审批中 一般情况将单据状态变成"审批中
                jpaUpdateClause.set(qLimitAdjustOrderDo.docState, UdcEnum.APPLY_STATUS_DOING.getValueCode())
                        .set(qLimitAdjustOrderDo.workflowSubmitTime, LocalDateTime.now());
                break;
            case APPROVED:
                // 最终节点审批通过 一般情况将单据状态变成"审批通过"
                jpaUpdateClause.set(qLimitAdjustOrderDo.docState, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode())
                        .set(qLimitAdjustOrderDo.workflowEndTime, commentInfo.getTime());
                break;
            default:
                break;
        }


        //生成账户流水参数
        CreditAccountFlowParam creditAccountFlowParam = null;
        // 调出单拒绝--生成释放占用的流水
        if (procInstStatus.equals(ProcInstStatus.REJECTED)
                && StrUtil.equals(limitAdjustOrderDO.getAdjustType(), UdcEnum.ADJUST_TYPE_2.getValueCode())) {
            creditAccountFlowParam = buildCreditAccountFlowParam(limitAdjustOrderDO,
                    UdcEnum.APPLY_STATUS_REJECTED.getValueCode());
        }

        // 调整单同意 当前时间在生效日期之后
        if (procInstStatus.equals(procInstStatus.APPROVED)
                && LocalDateTime.now().isAfter(limitAdjustOrderDO.getEffectiveTime())) {

            jpaUpdateClause.set(qLimitAdjustOrderDo.docState, UdcEnum.APPLY_STATUS_ACTIVE.getValueCode()).execute();
            // 1.创建增加的流水
            // 调出单 2.创建扣除的流水 账户规则配置状态为已完成就能够创建两条
            creditAccountFlowParam = buildCreditAccountFlowParam(limitAdjustOrderDO,
                    UdcEnum.APPLY_STATUS_ACTIVE.getValueCode());
        }

        if(creditAccountFlowParam!=null){
            creditAccountFlowService.generateCreditAccountFlow(creditAccountFlowParam);

        }
        jpaUpdateClause.execute();
    }

    private CreditAccountFlowParam buildCreditAccountFlowParam(LimitAdjustOrderDO limitAdjustOrderDO, String sourceDocStatus) {
        CreditAccountFlowParam creditAccountFlowParam = new CreditAccountFlowParam();
        creditAccountFlowParam.setSourceNo(limitAdjustOrderDO.getDocNo());
        creditAccountFlowParam.setSourceId(limitAdjustOrderDO.getId());
        creditAccountFlowParam.setSourceDoc(UdcEnum.DOC_CLS_AO.getValueCode());
        creditAccountFlowParam.setSourceDocType(limitAdjustOrderDO.getAdjustType());
        creditAccountFlowParam.setSourceDocStatus(sourceDocStatus);
        creditAccountFlowParam.setSourceDocAmount(limitAdjustOrderDO.getAdjustLimit());
        creditAccountFlowParam.setCreditAccountCode(limitAdjustOrderDO.getCreditAccountCode());

        return creditAccountFlowParam;
    }

    @Override
    public ArrayList<String> taskAssignee(String businessKey, String customParams) {
        return null;
    }



    /**
     * 查询单据
     */
    private LimitAdjustOrderDO getLimitAdjustOrderDO(String businessKey) {
        String docNo = businessKey.split("#")[0].trim();
        LimitAdjustOrderDO limitAdjustOrderDO = jpaQueryFactory.selectFrom(qLimitAdjustOrderDo)
                .where(qLimitAdjustOrderDo.docNo.eq(docNo))
                .fetchOne();
        if (limitAdjustOrderDO == null) {
            throw new RuntimeException("额度调整单(编码:" + docNo + ")不存在，无法审批");
        }

        return limitAdjustOrderDO;
    }
}
