package com.elitesland.fin.application.service.workflow.payorder;

import com.elitesland.fin.application.service.payorder.PayOrderService;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.payorder.QPayOrderDO;
import com.elitesland.fin.infr.dto.payorder.PayOrderDTO;
import com.elitesland.fin.infr.repo.payorder.PayOrderRepoProc;
import com.elitesland.workflow.CommentInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.system.param.SysUserIdFlowRoleRpcParam;
import com.elitescloud.cloudt.system.service.SysUserFlowRoleRpcService;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author zhiyu.he
 * @date 2022/5/7 13:26
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class PayOrderProcessServiceImpl implements PayOrderProcessService {

    private final SysUserFlowRoleRpcService sysUserFlowRoleRpcService;

    private final PayOrderRepoProc payOrderRepoProc;

    private final JPAQueryFactory jpaQueryFactory;

    private final PayOrderService payOrderService;

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void processStatusChange(long id, ProcInstStatus procInstStatus, CommentInfo commentInfo) {
        PayOrderDTO payOrderDTO = payOrderRepoProc.queryById(id);
        if (payOrderDTO == null) {
            throw new BusinessException("单据(id:" + id + ")不存在,无法审批");
        }
        QPayOrderDO payOrderDO = QPayOrderDO.payOrderDO;
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(payOrderDO)
                .set(payOrderDO.procInstStatus, procInstStatus)
                .where(payOrderDO.id.eq(id));

        //3,修改单据状态
        switch (procInstStatus) {
            // 未提交
            case NOTSUBMIT:
                jpaUpdateClause.set(payOrderDO.orderState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                break;
                // 中断执行 业务转会草稿
            case INTERRUPT:
                jpaUpdateClause.set(payOrderDO.orderState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode())
                        .set(payOrderDO.procInstId, (String) null);
                break;
                //审批中 业务依旧是草稿
            case APPROVING:
                jpaUpdateClause.set(payOrderDO.orderState, UdcEnum.APPLY_STATUS_DOING.getValueCode())
                        .set(payOrderDO.submitTime,LocalDateTime.now());
                break;
            // 审批拒绝 业务数据已拒绝
            case REJECTED:
                jpaUpdateClause.set(payOrderDO.orderState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode())
                        .set(payOrderDO.approvedTime, LocalDateTime.now())
                        .set(payOrderDO.auditDate, LocalDateTime.now())
                        .set(payOrderDO.auditUserId, Long.valueOf(commentInfo.getUserId()))
                        .set(payOrderDO.auditRejection, commentInfo.getComment())
                        .set(payOrderDO.auditUser, commentInfo.getUserName());
                break;
            // 作废
            case INVALID:
                jpaUpdateClause.set(payOrderDO.orderState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode())
                        .set(payOrderDO.approvedTime, LocalDateTime.now())
                        .set(payOrderDO.auditDate, LocalDateTime.now())
                        .set(payOrderDO.auditUserId, Long.valueOf(commentInfo.getUserId()))
                        .set(payOrderDO.auditRejection, commentInfo.getComment())
                        .set(payOrderDO.auditUser, commentInfo.getUserName())
                        .set(payOrderDO.procInstId, (String) null);
                break;
            // 审批通过 //已审批
            case APPROVED:
                jpaUpdateClause.set(payOrderDO.orderState, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode())
                        .set(payOrderDO.approvedTime, LocalDateTime.now())
                        .set(payOrderDO.auditDate, LocalDateTime.now())
                        .set(payOrderDO.auditUserId, Long.valueOf(commentInfo.getUserId()))
                        .set(payOrderDO.auditUser, commentInfo.getUserName());
                break;
            default:
                break;
        }
        //执行
        jpaUpdateClause.execute();
        //4,其他业务
        if (procInstStatus.equals(ProcInstStatus.APPROVED)) {
            //自动核销
            payOrderService.approved(id);
        }
    }

    @Override
    public ArrayList<String> taskAssignee(String businessKey, String customParams) {
        Long businessId = Long.valueOf(businessKey);
        PayOrderDTO payOrderDTO = payOrderRepoProc.queryById(businessId);
        if (payOrderDTO == null) {
            throw new BusinessException("单据(id:" + businessId + ")不存在,无法审批");
        }
        SysUserIdFlowRoleRpcParam param = SysUserIdFlowRoleRpcParam.builder()
                .ouIds(Collections.singletonList(payOrderDTO.getOuId()))
                .flowRoleCodes(Collections.singletonList(customParams))
                .build();
        List<Long> userIdsByFlowRoles = sysUserFlowRoleRpcService.findUserIdsByFlowRoles(param);
        return userIdsByFlowRoles.stream().map(e -> e + "").collect(Collectors.toCollection(ArrayList::new));
    }
}
