package com.elitesland.fin.utils.excel.convert;

import com.elitesland.fin.utils.excel.convert.annotation.ExcelConvert;
import com.elitesland.fin.application.facade.excel.convert.DefaultConverter;
import com.google.common.collect.Lists;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ExcelConvertUtils {
    private static final String GET = "get";
    
    private static final String SET = "set";
    
    public static <S, T extends Serializable> List<T> convert(List<S> sources, Class<T> targetClz) {
        if (sources.isEmpty()) {
            return List.of();
        }
        ArrayList<T> targets = new ArrayList<>(sources.size());
        
        for (S source : sources) {
            T target = null;
            try {
                target = targetClz.getDeclaredConstructor().newInstance();
            } catch (InstantiationException | IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                throw new RuntimeException("when create object of " + targetClz.getName() + " type had error: " + e.getMessage());
            }
            convert(source, target);
            targets.add(target);
        }
        return targets;
    }
    
    public static <S, T> void convert(S source, T target) {
        if(source == null || target == null) {
            return;
        }
        
        Map<String, MethodDescriptor> writeMethods = getWriteMethods(target);
        Map<String, Method> readMethods = getReadMethods(source);
        for (Map.Entry<String, MethodDescriptor> methodEntry : writeMethods.entrySet()) {
            MethodDescriptor writeMethodDescriptor = methodEntry.getValue();
            Method writeMethod = writeMethodDescriptor.method;
            Method readMethod = readMethods.get(writeMethodDescriptor.fieldName);
            if(readMethod == null) {
                continue;
            }
            try {
                Object invoked = readMethod.invoke(source);
                if (writeMethodDescriptor.converter != null
                        && writeMethodDescriptor.converter.getClass() != DefaultConverter.class) {
                    Object converted = writeMethodDescriptor.converter.convert(invoked);
                    writeMethod.invoke(target, converted);
                } else if (writeMethodDescriptor.defaultValue != null
                            && !writeMethodDescriptor.defaultValue.equals("")) {
                    writeMethod.invoke(target, writeMethodDescriptor.defaultValue);
                } else {
                    writeMethod.invoke(target, invoked);
                }
            } catch (IllegalAccessException | InvocationTargetException e) {
                throw new RuntimeException("Cannot convert source value to target, there had some error " + e.getMessage());
            }
        }
    }
    private static <D> Map<String, MethodDescriptor> getWriteMethods(D target) {
        Class<?> targetClass = target.getClass();
        List<Field> fields = getAllFields(target);
        
        if (fields.size() == 0) {
            return new HashMap<>();
        }
        
        HashMap<String, MethodDescriptor> map = new HashMap<>();
        ExcelConverterManager excelConverterManager = ExcelConverterManager.instance();
        for (Field field : fields) {
            String name = field.getName();
            Method method = null;
            String sourceFieldName = name;
            try {
                method = targetClass.getMethod(SET + capitalize(name), field.getType());
            } catch (NoSuchMethodException e) {
                continue;
            }
            ExcelConvert excelConvert = field.getAnnotation(ExcelConvert.class);
            Converter converter = null;
            String defaultValue = null;
            if(excelConvert == null) {
                map.put(name, new MethodDescriptor(method, converter, defaultValue, sourceFieldName));
                continue;
            }
            
            if (!excelConvert.value().equals("")) {
                defaultValue = excelConvert.value();
            }
            
            if (!excelConvert.name().equals("")) {
                sourceFieldName = excelConvert.name();
            }
            
            if (excelConvert.converter() != null) {
                try {
                    converter = excelConverterManager.getConverter(excelConvert.converter());
                } catch (InstantiationException | IllegalAccessException | InvocationTargetException |
                         NoSuchMethodException e) {
                    throw new RuntimeException(e);
                }
            }
            map.put(name, new MethodDescriptor(method, converter, defaultValue, sourceFieldName));
        }
        return map;
    }
    
    private static <D> Map<String, Method> getReadMethods(D source) {
        Class<?> sourceClass = source.getClass();
        List<Field> fields = getAllFields(source);
        
        if (fields.size() == 0) {
            return new HashMap<>();
        }
        
        HashMap<String, Method> map = new HashMap<>();
        for (Field field : fields) {
            String name = field.getName();
            Method method = null;
            try {
                method = sourceClass.getMethod(GET + capitalize(name), null);
            } catch (NoSuchMethodException e) {
                continue;
            }
            map.put(name, method);
        }
        return map;
    }
    
    private static String capitalize(final String str) {
        final int strLen = str == null ? 0 : str.length();
        if (strLen == 0) {
            return str;
        }
        
        final int firstCodepoint = str.codePointAt(0);
        final int newCodePoint = Character.toTitleCase(firstCodepoint);
        if (firstCodepoint == newCodePoint) {
            // already capitalized
            return str;
        }
        
        final int[] newCodePoints = new int[strLen]; // cannot be longer than the char array
        int outOffset = 0;
        newCodePoints[outOffset++] = newCodePoint; // copy the first codepoint
        for (int inOffset = Character.charCount(firstCodepoint); inOffset < strLen; ) {
            final int codepoint = str.codePointAt(inOffset);
            newCodePoints[outOffset++] = codepoint; // copy the remaining ones
            inOffset += Character.charCount(codepoint);
        }
        return new String(newCodePoints, 0, outOffset);
    }
    
    private static <D> List<Field> getAllFields(D obj) {
        ArrayList<Field> fields = Lists.newArrayList();
        Class<?> clazz = obj.getClass();
        while (clazz != Object.class) {
            fields.addAll(List.of(clazz.getDeclaredFields()));
            clazz = clazz.getSuperclass();
        }
        return fields;
    }
    
    
    public static class MethodDescriptor {
        public final Method method;
        
        public final Converter converter;
        
        public final String defaultValue;
        
        public final String fieldName;
        
        public MethodDescriptor (Method method, Converter converter, String defaultValue, String fieldName) {
            this.method = method;
            this.converter = converter;
            
            this.defaultValue = defaultValue;
            this.fieldName = fieldName;
        }
    }
}
