package com.elitesland.fin.application.service.accountingengine;

import cn.hutool.core.lang.Assert;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.accountingengine.*;
import com.elitesland.fin.application.facade.param.accountingengine.*;
import com.elitesland.fin.application.facade.vo.accountingengine.FinEventTableLineVO;
import com.elitesland.fin.application.facade.vo.accountingengine.FinEventTableVO;
import com.elitesland.fin.application.facade.vo.accountingengine.FinEventTableVO;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.domain.entity.accountingengine.FinAccountPeriodDO;
import com.elitesland.fin.domain.entity.accountingengine.FinEventTableDO;
import com.elitesland.fin.domain.entity.accountingengine.FinSetOfBookOuDO;
import com.elitesland.fin.repo.accountingengine.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.*;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author gyj
 * @date 2023/10/10
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class FinEventTableServiceImpl implements FinEventTableService {


    private final FinEventTableRepo finEventTableRepo;

    private final FinEventTableRepoProc finEventTableRepoProc;

    private final FinEventTableConditionRepo finEventTableConditionRepo;

    private final FinEventTableLineRepo finEventTableLineRepo;


    @SysCodeProc
    @Override
    public PagingVO<FinEventTableVO> page(FinEventTableParam finEventTableParam) {
        return FinEventTableConvert.INSTANCE.DTOToVO(finEventTableRepoProc.page(finEventTableParam));
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void enableOrDisable(FinEventTableParam finEventTableParam) {

        checkEnableOrDisableParam(finEventTableParam);

        List<FinEventTableDO> finEventTableDOList = finEventTableRepo.findAllById(finEventTableParam.getIds());
        finEventTableDOList.stream().forEach(item -> Assert.isFalse(finEventTableParam.getStatus().equals(item.getStatus()), "数据已经启用/禁用"));

        finEventTableDOList.stream().forEach(item -> item.setStatus(finEventTableParam.getStatus()));
    }

    @Override
    public List<FinEventTableLineVO> loadDataTemplate(FinEventTableParam finEventTableParam) {

        checkLoadDataTemplateParam(finEventTableParam);

        List<FinEventTableLineVO> finEventTableLineVOList = queryTableField(finEventTableParam, finEventTableParam.getMasTable());

        if (StringUtils.isNotEmpty(finEventTableParam.getTable())) {
            finEventTableLineVOList.addAll(queryTableField(finEventTableParam, finEventTableParam.getTable()));
        }

        return finEventTableLineVOList;

    }

    private List<FinEventTableLineVO> queryTableField(FinEventTableParam finEventTableParam, String tableName) {

        Connection connection = null;
        ResultSet columnResultSet = null;

        try {
            Class.forName(FinConstant.DRIVEN_NAME);

            connection = DriverManager.getConnection(String.format(FinConstant.CONNECTION_NAME,
                    finEventTableParam.getHost(),
                    finEventTableParam.getPort(),
                    finEventTableParam.getDatabase()),
                    finEventTableParam.getUser(),
                    finEventTableParam.getPassword());

            DatabaseMetaData metaData = connection.getMetaData();

            bulidFinEventTableLineVOList();

            columnResultSet = metaData.getColumns(finEventTableParam.getDatabase(),
                    finEventTableParam.getDatabase(),
                    tableName,
                    null);

            List<FinEventTableLineVO> finEventTableLineVOList = new ArrayList<>();
            while (columnResultSet.next()) {

                FinEventTableLineVO finEventTableLineVO = new FinEventTableLineVO();
                finEventTableLineVO.setTableName(tableName);
                finEventTableLineVO.setColumnName(columnResultSet.getString(FinConstant.COLUMN_NAME));
                finEventTableLineVO.setColumnType(columnResultSet.getString(FinConstant.TYPE_NAME));
                finEventTableLineVO.setColumnComment(columnResultSet.getString(FinConstant.REMARKS));

                finEventTableLineVOList.add(finEventTableLineVO);
            }

            return finEventTableLineVOList;

        } catch (Exception e) {
            throw new BusinessException("查询表结果异常", e);
        } finally {
            if (connection != null) {
                try {
                    connection.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }

            if (columnResultSet != null) {
                try {
                    columnResultSet.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void bulidFinEventTableLineVOList() {

    }

    private void checkLoadDataTemplateParam(FinEventTableParam finEventTableParam) {
        Assert.notEmpty(finEventTableParam.getHost(), "主机必填");
        Assert.notEmpty(finEventTableParam.getPort(), "端口必填");
        Assert.notEmpty(finEventTableParam.getUser(), "用户名必填");
        Assert.notEmpty(finEventTableParam.getPassword(), "密码必填");
        Assert.notEmpty(finEventTableParam.getDatabase(), "数据库名称必填");
        Assert.notEmpty(finEventTableParam.getMasTable(), "主表必填");
        Assert.notEmpty(finEventTableParam.getMasTableColumn(), "主表条件字段必填");
    }

    private void checkEnableOrDisableParam(FinEventTableParam finEventTableParam) {
        Assert.notEmpty(finEventTableParam.getIds(), "id必填");
        Assert.notEmpty(finEventTableParam.getStatus(), "状态必填");
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveOrUpdate(FinEventTableParam finEventTableParam) {

        //防止npe
        finEventTableParam.setFinEventTableLineDetailList(Optional.ofNullable(finEventTableParam.getFinEventTableLineDetailList()).orElse(Collections.emptyList()));
        finEventTableParam.setFinEventTableConditionDetailList(Optional.ofNullable(finEventTableParam.getFinEventTableConditionDetailList()).orElse(Collections.emptyList()));

        checkSaveOrUpdateParam(finEventTableParam);

        checkIdempotent(finEventTableParam);

        FinEventTableDO finEventTableDO = finEventTableRepo.save(FinEventTableConvert.INSTANCE.paramToDO(finEventTableParam));

        List<FinEventTableLineParam> finEventTableLineDetailList = finEventTableParam.getFinEventTableLineDetailList();
        List<FinEventTableConditionParam> finEventTableConditionDetailList = finEventTableParam.getFinEventTableConditionDetailList();

        if (CollectionUtils.isNotEmpty(finEventTableLineDetailList)) {
            finEventTableLineRepo.deleteAllByMasId(finEventTableDO.getId());
            finEventTableLineDetailList.stream().forEach(item -> item.setMasId(finEventTableDO.getId()));
            finEventTableLineRepo.saveAll(FinEventTableLineConvert.INSTANCE.paramToDO(finEventTableLineDetailList));
        }

        if (CollectionUtils.isNotEmpty(finEventTableConditionDetailList)) {
            finEventTableConditionRepo.deleteAllByMasId(finEventTableDO.getId());
            finEventTableConditionDetailList.stream().forEach(item -> item.setMasId(finEventTableDO.getId()));
            finEventTableConditionRepo.saveAll(FinEventTableConditionConvert.INSTANCE.paramToDO(finEventTableConditionDetailList));
        }

        return finEventTableDO.getId();
    }

    private void checkSaveOrUpdateParam(FinEventTableParam finEventTableParam) {
        Assert.notEmpty(finEventTableParam.getEventTable(), "事件表单名称必填");
        Assert.notEmpty(finEventTableParam.getHost(), "主机必填");
        Assert.notEmpty(finEventTableParam.getPort(), "端口必填");
        Assert.notEmpty(finEventTableParam.getUser(), "用户名必填");
        Assert.notEmpty(finEventTableParam.getPassword(), "密码必填");
        Assert.notEmpty(finEventTableParam.getDatabase(), "数据库名称必填");
        Assert.notEmpty(finEventTableParam.getMasTable(), "主表必填");
        Assert.notEmpty(finEventTableParam.getMasTableColumn(), "主表条件字段必填");

        List<FinEventTableLineParam> finEventTableLineDetailList = finEventTableParam.getFinEventTableLineDetailList();
        List<FinEventTableConditionParam> finEventTableConditionDetailList = finEventTableParam.getFinEventTableConditionDetailList();

        Assert.isTrue(CollectionUtils.isNotEmpty(finEventTableLineDetailList) || CollectionUtils.isNotEmpty(finEventTableConditionDetailList), "明细不能为空");

        finEventTableLineDetailList.stream().forEach(item -> {
            Assert.notEmpty(item.getTableName(), "表单名称必填");
            Assert.notEmpty(item.getColumnName(), "字段名称必填");
            Assert.notEmpty(item.getColumnType(), "字段类型必填");
            Assert.notEmpty(item.getColumnComment(), "字段含义必填");
        });

        finEventTableConditionDetailList.stream().forEach(item -> {
            Assert.notEmpty(item.getTableName(), "表单名称必填");
            Assert.notEmpty(item.getColumnName(), "字段名称必填");
            Assert.notEmpty(item.getConditionType(), "条件必填");
            Assert.notNull(item.getValueFrom(), "从值内容必填");
        });
    }

    private void checkIdempotent(FinEventTableParam finEventTableParam) {

        FinEventTableDO finEventTableDO = finEventTableRepoProc.findByEventTable(finEventTableParam.getEventTable());

        if (finEventTableDO != null) {
            Assert.equals(finEventTableParam.getId(), finEventTableDO.getId(), "事件表单名称已经存在");
        }
    }
}
