package com.elitesland.fin.application.service.accountingengine;

import cn.hutool.core.lang.Assert;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.accountingengine.FinSetOfBookConvert;
import com.elitesland.fin.application.convert.accountingengine.FinSetOfBookLineConvert;
import com.elitesland.fin.application.convert.accountingengine.FinSetOfBookOuConvert;
import com.elitesland.fin.application.facade.param.accountingengine.FinSetOfBookLineParam;
import com.elitesland.fin.application.facade.param.accountingengine.FinSetOfBookOuParam;
import com.elitesland.fin.application.facade.param.accountingengine.FinSetOfBookParam;
import com.elitesland.fin.application.facade.vo.accountingengine.FinSetOfBookOuVO;
import com.elitesland.fin.application.facade.vo.accountingengine.FinSetOfBookVO;
import com.elitesland.fin.domain.entity.accountingengine.*;
import com.elitesland.fin.repo.accountingengine.*;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author gyj
 * @date 2023/10/10
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class FinSetOfBookServiceImpl implements FinSetOfBookService {


    private final FinSetOfBookRepo finSetOfBookRepo;

    private final FinSetOfBookRepoProc finSetOfBookRepoProc;

    private final FinSetOfBookLineRepo finSetOfBookLineRepo;

    private final FinSetOfBookOuRepo finSetOfBookOuRepo;

    private final FinSetOfBookOuRepoProc finSetOfBookOuRepoProc;

    private final FinSobAccountPeriodRepo finSobAccountPeriodRepo;

    private final FinJournalRepo finJournalRepo;

    private final FinAccountPeriodRepoProc finAccountPeriodRepoProc;

    private final FinAccountPeriodLineRepo finAccountPeriodLineRepo;


    @SysCodeProc
    @Override
    public PagingVO<FinSetOfBookVO> page(FinSetOfBookParam finSetOfBookParam) {
        return FinSetOfBookConvert.INSTANCE.DTOToVO(finSetOfBookRepoProc.page(finSetOfBookParam));
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void enableOrDisable(FinSetOfBookParam finSetOfBookParam) {

        checkEnableOrDisableParam(finSetOfBookParam);

        List<FinSetOfBookDO> finSetOfBookDOList = finSetOfBookRepo.findAllById(finSetOfBookParam.getIds());
        finSetOfBookDOList.stream().forEach(item -> Assert.isFalse(finSetOfBookParam.getStatus().equals(item.getStatus()), "数据已经启用/禁用"));

        finSetOfBookDOList.stream().forEach(item -> item.setStatus(finSetOfBookParam.getStatus()));
    }

    @Override
    public void deleteFinSetOfBookOu(FinSetOfBookParam finSetOfBookParam) {
        checkDeleteFinSetOfBookOuParam(finSetOfBookParam);
        List<String> ouCodes = finSetOfBookParam.getOuDetailList().stream().map(FinSetOfBookOuParam::getOuCode).collect(Collectors.toList());

        List<FinSobAccountPeriodDO> finSobAccountPeriodDOList = finSobAccountPeriodRepo.findAllBySobCodeAndAccountPeriodCodeAndOuCodeIn(
                finSetOfBookParam.getSobCode(),
                finSetOfBookParam.getAccountPeriodCode(), ouCodes);

        if (CollectionUtils.isEmpty(finSobAccountPeriodDOList)) {
            return;
        }
        Assert.isTrue(CollectionUtils.isEmpty(finSobAccountPeriodDOList),
                "{}会计期间控制已经生成，不能删除", finSobAccountPeriodDOList.get(0).getOuCode());
    }

    @Override
    public void deleteFinSetOfBookLine(FinSetOfBookParam finSetOfBookParam) {
        Assert.notEmpty(finSetOfBookParam.getSobCode(), "账套编码不能为空");
        List<FinJournalRepo> finJournalRepoList = finJournalRepo.findAllBySobLedgerCode(finSetOfBookParam.getSobCode());
        Assert.isTrue(CollectionUtils.isEmpty(finJournalRepoList), "帐套已经生成分录,无法删除");
    }

    private void checkDeleteFinSetOfBookOuParam(FinSetOfBookParam finSetOfBookParam) {
        Assert.notEmpty(finSetOfBookParam.getSobCode(), "账套编码不能为空");
        Assert.notEmpty(finSetOfBookParam.getAccountPeriodCode(), "会计期间编码不能为空");
        Assert.notEmpty(finSetOfBookParam.getOuDetailList(), "公司明细不能为空");
        finSetOfBookParam.getOuDetailList().stream().forEach(item -> Assert.notEmpty(item.getOuCode(), "公司编码不能为空"));
    }

    private void checkEnableOrDisableParam(FinSetOfBookParam finSetOfBookParam) {
        Assert.notEmpty(finSetOfBookParam.getIds(), "id必填");
        Assert.notEmpty(finSetOfBookParam.getStatus(), "状态必填");
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveOrUpdate(FinSetOfBookParam finSetOfBookParam) {

        //防止npe
        finSetOfBookParam.setDimenDetailList(Optional.ofNullable(finSetOfBookParam.getDimenDetailList()).orElse(Collections.emptyList()));
        finSetOfBookParam.setOuDetailList(Optional.ofNullable(finSetOfBookParam.getOuDetailList()).orElse(Collections.emptyList()));

        checkSaveOrUpdateParam(finSetOfBookParam);

        checkIdempotent(finSetOfBookParam);

        FinSetOfBookDO finSetOfBookDO = finSetOfBookRepo.save(FinSetOfBookConvert.INSTANCE.paramToDO(finSetOfBookParam));

        List<FinSetOfBookLineParam> dimenDetailList = finSetOfBookParam.getDimenDetailList();
        List<FinSetOfBookOuParam> ouDetailList = finSetOfBookParam.getOuDetailList();

        if (CollectionUtils.isNotEmpty(dimenDetailList)) {
            finSetOfBookLineRepo.deleteAllByMasId(finSetOfBookDO.getId());
            dimenDetailList.stream().forEach(item -> item.setMasId(finSetOfBookDO.getId()));
            finSetOfBookLineRepo.saveAll(FinSetOfBookLineConvert.INSTANCE.paramToDO(dimenDetailList));
        }

        if (CollectionUtils.isNotEmpty(ouDetailList)) {
            finSetOfBookOuRepo.deleteAllByMasId(finSetOfBookDO.getId());
            ouDetailList.stream().forEach(item -> item.setMasId(finSetOfBookDO.getId()));
            finSetOfBookOuRepo.saveAll(FinSetOfBookOuConvert.INSTANCE.paramToDO(ouDetailList));
        }

        return finSetOfBookDO.getId();
    }

    private void checkSaveOrUpdateParam(FinSetOfBookParam finSetOfBookParam) {
        Assert.notEmpty(finSetOfBookParam.getSobCode(), "账套编码必填");
        Assert.notEmpty(finSetOfBookParam.getSobName(), "账套名称必填");
        Assert.notEmpty(finSetOfBookParam.getAccountPeriodCode(), "会计期间必填");
        Assert.notEmpty(finSetOfBookParam.getAccountPeriodName(), "会计期间名称必填");
        Assert.notEmpty(finSetOfBookParam.getCurrCode(), "本位币必填");
        Assert.notEmpty(finSetOfBookParam.getStatus(), "状态必填");


        List<FinSetOfBookLineParam> dimenDetailList = finSetOfBookParam.getDimenDetailList();
        List<FinSetOfBookOuParam> ouDetailList = finSetOfBookParam.getOuDetailList();

        Assert.isTrue(CollectionUtils.isNotEmpty(dimenDetailList) || CollectionUtils.isNotEmpty(ouDetailList), "明细不能为空");

        ouDetailList.stream().forEach(item -> {
            Assert.notEmpty(item.getOuCode(), "公司编码必填");
            Assert.notEmpty(item.getOuName(), "公司名称必填");
        });

        dimenDetailList.stream().forEach(item -> {
            Assert.notEmpty(item.getColumnName(), "字段名称必填");
            Assert.notEmpty(item.getAccountDimenCode(), "核算维度编码必填");
            Assert.notEmpty(item.getAccountDimenName(), "核算维度名称必填");
            Assert.notNull(item.getFlexibleFlag(), "是否是值集必填");

            if (Boolean.TRUE.equals(item.getFlexibleFlag())) {
                Assert.notEmpty(item.getFlexibleCode(), "值集编码必填");
                Assert.notEmpty(item.getFlexibleName(), "值集名称必填");
            }
        });
    }

    private void checkIdempotent(FinSetOfBookParam finSetOfBookParam) {

        FinSetOfBookDO finSetOfBookDO = finSetOfBookRepoProc.findBySobCode(finSetOfBookParam.getSobCode());

        if (finSetOfBookDO != null) {
            Assert.equals(finSetOfBookParam.getId(), finSetOfBookDO.getId(), "账套编码已经存在");
        }

        List<FinSetOfBookOuParam> ouDetailList = finSetOfBookParam.getOuDetailList();

        if (CollectionUtils.isNotEmpty(ouDetailList)) {

            List<String> ouCodes = ouDetailList.stream().map(FinSetOfBookOuParam::getOuCode).collect(Collectors.toList());
            List<FinSetOfBookOuDO> finSetOfBookOuDOList = finSetOfBookOuRepo.findAllByOuCodeIn(ouCodes);

            List<Long> ids = ouDetailList.stream().map(FinSetOfBookOuParam::getId).collect(Collectors.toList());
            finSetOfBookOuDOList.stream().forEach(item -> Assert.isTrue(ids.contains(item.getId()), "{}已经分配账套", item.getOuCode()));
        }
    }

    @Override
    public FinSetOfBookVO queryOuAndYear(FinSetOfBookParam finSetOfBookParam) {

        Assert.notBlank(finSetOfBookParam.getSobCode(),"账套编码不能为空");
        FinSetOfBookVO finSetOfBookVO = new FinSetOfBookVO();

        FinSetOfBookDO finSetOfBookDO = finSetOfBookRepoProc.findBySobCode(finSetOfBookParam.getSobCode());
        Assert.notNull(finSetOfBookDO,"账套不存在");
//
//        List<FinSetOfBookOuDO> finSetOfBookOuDOList = finSetOfBookOuRepo.findAllByMasIdIn(Lists.newArrayList(finSetOfBookDO.getId()));
//        Assert.notEmpty(finSetOfBookOuDOList,"该账套还没有分配公司");

        FinAccountPeriodDO finAccountPeriodDO = finAccountPeriodRepoProc.findByAccountPeriodCode(finSetOfBookDO.getAccountPeriodCode());
        Assert.notNull("该账套的会计期间设置不存在");

//        finSetOfBookVO.setOuDetailList(FinSetOfBookConvert.INSTANCE.dosToVo(finSetOfBookOuDOList));

        List<FinAccountPeriodLineDO> finAccountPeriodLineDOList = finAccountPeriodLineRepo.findAllByMasIdIn(Lists.newArrayList(finAccountPeriodDO.getId()));
        List<String> yearList = finAccountPeriodLineDOList.stream().map(FinAccountPeriodLineDO::getYear).distinct().collect(Collectors.toList());
        finSetOfBookVO.setYearList(yearList);

        return finSetOfBookVO;
    }

    @Override
    public PagingVO<FinSetOfBookOuVO> queryOu(FinSetOfBookOuParam finSetOfBookParam) {
        return finSetOfBookOuRepoProc.page(finSetOfBookParam);
    }
}
