package com.elitesland.fin.application.service.excel.imp;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitesland.fin.application.convert.adjustorder.AdjustOrderConvert;
import com.elitesland.fin.application.facade.dto.account.AccountDTO;
import com.elitesland.fin.application.facade.excel.adjust.AdjustOrderImportEntity;
import com.elitesland.fin.application.facade.param.account.AccountParam;
import com.elitesland.fin.application.facade.param.adjustorder.AdjustOrderParam;
import com.elitesland.fin.application.service.adjustorder.AdjustOrderServiceImpl;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.SysNumEnum;
import com.elitesland.fin.common.SysNumberGenerator;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.adjustorder.AdjustOrderDO;
import com.elitesland.fin.repo.account.AccountRepoProc;
import com.elitesland.fin.repo.adjustorder.AdjustOrderRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 调整单导入
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class AdjustOrderImportServiceImpl implements DataImport<AdjustOrderImportEntity> {

    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    private final AccountRepoProc accountRepoProc;

    private final AdjustOrderRepo adjustOrderRepo;

    private final UdcProvider udcProvider;

    private final SysNumberGenerator sysNumberGenerator;

    private final AdjustOrderServiceImpl adjustOrderService;

    @Override
    public String getTmplCode() {
        return "yst_fin_adjust_order_import";
    }

    @Override
    public List<String> executeImport(List<AdjustOrderImportEntity> dataList, int rowId) {

        if (CollectionUtil.isEmpty(dataList)) {
            return List.of();
        }

        // 查询账户信息
        Set<String> accountCodeList = dataList.stream().map(AdjustOrderImportEntity::getAccCode).filter(code->StrUtil.isNotBlank(code)).collect(Collectors.toSet());
        Map<String, AccountDTO> accountMap = new HashMap<>();
        if (CollectionUtil.isNotEmpty(accountCodeList)) {
            AccountParam accountParam = new AccountParam();
            accountParam.setAccountCodes(accountCodeList);
            List<AccountDTO> accountDTOList = accountRepoProc.getAccountByAccountParam(accountParam);
            accountMap = accountDTOList.stream().collect(Collectors.toMap(AccountDTO::getAccountCode, t -> t, (t1, t2) -> t1));
        }

        // 查询UDC
        Map<String, Map<String, String>> udcCodeMap = udcProvider.getValueMapByUdcCode((UdcEnum.ADJUST_TYPE_1.getModel()),
                Set.of(UdcEnum.ADJUST_TYPE_1.getCode(), UdcEnum.FIN_ADJUST_REASON_1.getCode()));
        Map<String, String> adjustTypeMap = udcCodeMap.get(UdcEnum.ADJUST_TYPE_1.getCode()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (v1, v2) -> v1));
        Map<String, String> adjustReasonMap = udcCodeMap.get(UdcEnum.FIN_ADJUST_REASON_1.getCode()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (v1, v2) -> v1));


        List<String> errorResult = new ArrayList<>();
        // 储存新导入的数据
        List<AdjustOrderParam> saveList = new ArrayList<>();

        // 保存调出总金额
        Map<String, BigDecimal> amtMap = new HashMap<>();
        for (AdjustOrderImportEntity entity : dataList) {
            if (StrUtil.isNotBlank(entity.getAccCode())
                    && NumberUtil.isNumber(entity.getTotalAmt())
                    && StrUtil.equals(entity.getAdjTypeName(), UdcEnum.ADJUST_TYPE_2.getValueCodeName())) {
                BigDecimal bigDecimal = new BigDecimal(entity.getTotalAmt());
                BigDecimal totalAmt = amtMap.get(entity.getAccCode()) == null ? bigDecimal : amtMap.get(entity.getAccCode()).add(bigDecimal);
                amtMap.put(entity.getAccCode(), totalAmt);
            }
        }

        // 储存已经存在的数据
        for (AdjustOrderImportEntity importEntity : dataList) {

            AdjustOrderParam adjustOrderParam = new AdjustOrderParam();
            List<String> errorList = new ArrayList<>();

            String docNo = sysNumberGenerator.generate(SysNumEnum.FIN_ADJ_ORDER.getCode());

            // 必填校验
            checkImportMandatoryField(importEntity, errorList, rowId);

            // 填写的类型是否匹配UDC
            checkUdc(rowId, importEntity, adjustOrderParam, errorList, adjustTypeMap, adjustReasonMap);

            // 调整金额的账户信息校验
            checkAccount(rowId, importEntity, adjustOrderParam, errorList, accountMap, amtMap);

            // 设置信息
            adjustOrderParam.setRemark(importEntity.getRemark());
            adjustOrderParam.setType(UdcEnum.ADJUST_ORDER_TYPE_YETZ.getValueCode());
            adjustOrderParam.setDocNo(docNo);
            adjustOrderParam.setState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());

            if (CollectionUtils.isNotEmpty(errorList)) {
                errorResult.add(StringUtils.join(errorList, FinConstant.WRAP));
            } else {
                // 不存在错误信息
                errorResult.add(null);
                saveList.add(adjustOrderParam);
            }
            // 行数自增
            rowId++;
        }

        if (CollectionUtils.isNotEmpty(saveList)) {
            saveData(saveList);
            return CollectionUtil.isNotEmpty(errorResult) ? errorResult : null;
        } else {
            return errorResult;
        }
    }

    private void checkUdc(int rowId, AdjustOrderImportEntity importEntity,
                                 AdjustOrderParam adjustOrderParam, List<String> errorList,
                                 Map<String, String> adjustTypeMap,
                                 Map<String, String> adjustReasonMap) {

        if (adjustReasonMap.containsKey(importEntity.getAdjReasonName())) {
            adjustOrderParam.setAdjReason(adjustReasonMap.get(importEntity.getAdjReasonName()));
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整原因", "填写不正确"));
        }

        if (adjustTypeMap.containsKey(importEntity.getAdjTypeName())) {
            adjustOrderParam.setAdjType(adjustTypeMap.get(importEntity.getAdjTypeName()));
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整类型", "填写不正确"));
        }
    }


    private void checkAccount(int rowId, AdjustOrderImportEntity importEntity,
                                     AdjustOrderParam adjustOrderParam, List<String> errorList,
                                     Map<String, AccountDTO> accountMap, Map<String, BigDecimal> amtMap) {

        // 校验金额
        BigDecimal amt = BigDecimal.ZERO;
        if (NumberUtil.isNumber(importEntity.getTotalAmt())) {

            Pattern pattern = Pattern.compile(FinConstant.TWO_DECIMAL_REGULAR);
            if (!pattern.matcher(importEntity.getTotalAmt()).matches()) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调整金额，最多两位小数"));
            }

            amt = new BigDecimal(importEntity.getTotalAmt());
            if (NumberUtil.isLess(amt, BigDecimal.ZERO)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调整金额，不能小于0"));
            }

        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调整金额格式不对"));
        }

        // 检验账户
        if (accountMap.containsKey(importEntity.getAccCode())) {
            AccountDTO accountDTO = accountMap.get(importEntity.getAccCode());

            if (!StrUtil.equals(accountDTO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户未激活"));
            }

            // 调出金额小于账户可用金额
            if (StrUtil.equals(adjustOrderParam.getAdjType(), UdcEnum.ADJUST_TYPE_2.getValueCode())
                    && amtMap.containsKey(importEntity.getAccCode())
                    && NumberUtil.isGreater(amtMap.get(importEntity.getAccCode()), accountDTO.getAccountAvailableAmount())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调出金额大于账户可用金额"));
            }
            adjustOrderParam.setAccCode(importEntity.getAccCode());
            adjustOrderParam.setAccName(accountDTO.getAccountName());
            adjustOrderParam.setTotalAmt(amt);
            adjustOrderParam.setAccType(accountDTO.getAccountType());
            adjustOrderParam.setAccountHolderName(accountDTO.getAccountHolderName());
            adjustOrderParam.setAccountHolderCode(accountDTO.getAccountHolderCode());

        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户不存在"));
        }
    }


    private void checkImportMandatoryField(AdjustOrderImportEntity importEntity, List<String> errorList, int rowId) {

        if (StringUtils.isBlank(importEntity.getAccCode())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户编码必填"));
        }
        if (StringUtils.isBlank(importEntity.getAdjTypeName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整类型", "调整类型必填"));
        }
        if (StringUtils.isBlank(importEntity.getTotalAmt())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调整金额必填"));
        }
        if (StringUtils.isBlank(importEntity.getAdjReasonName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整原因", "调整原因必填"));
        }
    }

    public void saveData(List<AdjustOrderParam> saveList) {

        List<AdjustOrderDO> doList = AdjustOrderConvert.INSTANCE.paramsToDOs(saveList);
        List<AdjustOrderDO> adjustOrderDOS = adjustOrderRepo.saveAll(doList);
        submit(adjustOrderDOS);

    }

    public void submit(List<AdjustOrderDO> doList) {

        doList.stream().forEach(adjustOrderDO -> {
            adjustOrderService.workflowAndGenerateAccountFlow(adjustOrderDO);
        });

    }

}
