package com.elitesland.fin.application.service.aporder;

import com.alibaba.excel.EasyExcelFactory;
import com.elitesland.fin.application.convert.aporder.ApOrderConvert;
import com.elitesland.fin.application.facade.param.aporder.ApOrderSaveParam;
import com.elitesland.fin.application.facade.vo.aporder.ApOrderVO;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.aporder.ApOrder;
import com.elitesland.fin.domain.param.aporder.ApOrderPageParam;
import com.elitesland.fin.domain.service.aporder.ApOrderDomainService;
import com.elitesland.fin.domain.service.aporder.ApOrderDtlDomainService;
import com.elitesland.fin.infr.dto.arorder.ArOrderDTO;
import com.elitesland.fin.rpc.workflow.WorkflowRpcService;
import com.elitesland.fin.infr.dto.aporder.ApOrderDTO;
import com.elitesland.fin.application.service.workflow.WorkFlowDefKey;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.WorkflowConstant;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.core.security.util.SecurityUtil;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import java.math.BigDecimal;
import java.util.Objects;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.application.service.aporder
 * @date 2022/3/16 16:06
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ApOrderServiceImpl implements ApOrderService {

    private final ApOrderDomainService apOrderDomainService;
    private final ApOrderDtlDomainService apOrderDtlDomainService;
    private final WorkflowRpcService workflowRpcService;


    @SysCodeProc
    @Override
    public PagingVO<ApOrderVO> page(ApOrderPageParam apOrderPageParam) {
        PagingVO<ApOrderDTO> page = apOrderDomainService.page(apOrderPageParam);
        PagingVO<ApOrderVO> res = ApOrderConvert.INSTANCE.convertPage(page);
        return res;
    }

    @SysCodeProc
    @Override
    public ApOrderVO get(Long id) {
        ApOrderDTO apOrderDTO = apOrderDomainService.get(id);
        ApOrderVO res = ApOrderConvert.INSTANCE.convert(apOrderDTO);
        return res;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> audit(Long id) {

        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> refush(List<Long> ids, String content) {

        SysUserDTO user = SecurityUtil.getUser().getUser();
        Long res = apOrderDomainService.audit(ids, content, user);
        return ApiResult.ok(res);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> del(List<Long> ids) {
        Long res = apOrderDomainService.del(ids);
        return ApiResult.ok(res);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> save(ApOrderSaveParam apOrderSaveParam) {
        // 新增只用于单据来源为手工的
        if (!apOrderSaveParam.getCreateMode().equals(UdcEnum.FIN_AR_DOC_CLS_MANU.getValueCode())) {
            throw new BusinessException("新增的单据来源需为手工");
        }
        ApOrder apOrder = ApOrderConvert.INSTANCE.convert(apOrderSaveParam);
        Long res = apOrderDomainService.save(apOrder);
        return ApiResult.ok(res);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> commit(ApOrderSaveParam apOrderSaveParam) {
        ApOrder apOrder = ApOrderConvert.INSTANCE.convert(apOrderSaveParam);
        Long resId = apOrderDomainService.commit(apOrder);
        // 启动工作流
        if (apOrder.getProcInstId() == null
                || WorkflowConstant.CAN_START_PROC_STATUSES.contains(apOrder.getProcInstStatus())) {
            // 启动流程
            String procInstName = "应付单审核-" + apOrder.getApOrderNo();
            String procKey = WorkFlowDefKey.FIN_AP_ORDER.name();
            ProcessInfo processInfo = workflowRpcService.startProcess(procKey, procInstName, resId.toString(), new HashMap<>());
            apOrderDomainService.updateWorkInfo(processInfo, resId);
        }
        return ApiResult.ok(resId);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> update(ApOrderSaveParam apOrderSaveParam) {
        ApOrder apOrder = ApOrderConvert.INSTANCE.convert(apOrderSaveParam);
        apOrder.checkUpdate();
        Long res = apOrderDomainService.save(apOrder);
        return ApiResult.ok(res);
    }

    @SysCodeProc
    @Override
    public ApOrderVO getApOrderAndDtl(Long id) {
        ApOrderDTO apOrderDTO = apOrderDomainService.getApOrderAndDtl(id);
        ApOrderVO res = ApOrderConvert.INSTANCE.convert(apOrderDTO);
        return res;
    }

    @SysCodeProc
    @Override
    public void exportExl(HttpServletResponse response, ApOrderPageParam apOrderPageParam) {
        PagingVO<ApOrderDTO> page = apOrderDomainService.page(apOrderPageParam);
        PagingVO<ApOrderVO> res = ApOrderConvert.INSTANCE.convertPage(page);
        try {
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            response.setCharacterEncoding("utf-8");
            // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
            String fileName = URLEncoder.encode("采购结算单", "UTF-8").replace("+", "%20");
            response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
            EasyExcelFactory.write(response.getOutputStream(), ApOrderVO.class).sheet("采购结算单").doWrite(res.getRecords());
        } catch (IOException e) {
            log.error("采购结算单导出异常信息:{}", e);
        }
    }

    @Override
    public ApiResult<Void> cancelApprove(List<Long> ids) {
        // 重置状态
        apOrderDomainService.restDocState(ids);
        return ApiResult.ok();
    }

    @Override
    public ApiResult<Long> redPunch(Long id) {
        ApOrderDTO apOrderDTO = apOrderDomainService.get(id);
        this.checkRedPunch(apOrderDTO);
        ApOrder apOrder = apOrderDomainService.redPunchCreate(id);
        apOrder.setCheck(false);
        return this.commit(ApOrderConvert.INSTANCE.convertParam(apOrder));
    }

    /**
     * 取消单据检查
     * @param apOrderDTO
     */
    void checkRedPunch(ApOrderDTO apOrderDTO) {
        if (Objects.isNull(apOrderDTO)){
            throw new BusinessException("单据不存在");
        }
        if (Boolean.TRUE.equals(apOrderDTO.getRedState())){
            throw new BusinessException("单据已红冲");
        }
        if (Objects.nonNull(apOrderDTO.getRedSourceNo())){
            throw new BusinessException("来源单据不可为红冲单据");
        }
        if (!(BigDecimal.ZERO.compareTo(apOrderDTO.getVerAmt())==0)){
            throw new BusinessException("已核销金额必须为0");
        }
        if (!UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(apOrderDTO.getOrderState())){
            throw new BusinessException("单据状态必须为审核通过");
        }

    }


}
