package com.elitesland.fin.application.service.workflow.aporder;

import com.elitesland.fin.application.service.aporder.ApOrderService;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.aporder.QApOrderDO;
import com.elitesland.fin.infr.dto.aporder.ApOrderDTO;
import com.elitesland.fin.infr.repo.aporder.ApOrderRepoProc;
import com.elitesland.workflow.CommentInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.system.param.SysUserIdFlowRoleRpcParam;
import com.elitescloud.cloudt.system.service.SysUserFlowRoleRpcService;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author Jason.zhao
 * @date 2022/5/7 10:23
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ApOrderProcessServiceImpl implements ApOrderProcessService {

    private final SysUserFlowRoleRpcService sysUserFlowRoleRpcService;

    private final ApOrderService apOrderService;

    private final ApOrderRepoProc apOrderRepoProc;

    private final JPAQueryFactory jpaQueryFactory;

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void processStatusChange(long id, ProcInstStatus procInstStatus, CommentInfo commentInfo) {
        ApOrderDTO apOrderDTO = apOrderRepoProc.get(id);
        if (apOrderDTO == null) {
            throw new BusinessException("单据(id:" + id + ")不存在,无法审批");
        }
        QApOrderDO apOrderDO = QApOrderDO.apOrderDO;
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(apOrderDO)
                .set(apOrderDO.procInstStatus, procInstStatus)
                .where(apOrderDO.id.eq(id));

        //单据状态
        switch (procInstStatus) {
            case NOTSUBMIT: // 未提交
                //一般情况将单据状态变成"草稿"
                jpaUpdateClause.set(apOrderDO.orderState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                break;
            case INTERRUPT: // 中断执行
                //一般情况将单据状态变成"草稿",并且将单据上的"流程实例状态"，"流程实例ID"清成null(不是空字符串)
                jpaUpdateClause.set(apOrderDO.orderState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode())
                        .set(apOrderDO.procInstId, (String) null);
                break;
            case REJECTED: // 审批拒绝
                //一般情况将单据状态变成"草稿"或"拒绝"
                jpaUpdateClause.set(apOrderDO.orderState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode())
                        .set(apOrderDO.approvedTime, LocalDateTime.now())
                        .set(apOrderDO.auditDate, LocalDateTime.now())
                        .set(apOrderDO.auditUserId, Long.valueOf(commentInfo.getUserId()))
                        .set(apOrderDO.auditRejection, commentInfo.getComment())
                        .set(apOrderDO.auditUser, commentInfo.getUserName());
                break;
            case INVALID: // 作废
                //一般情况将单据状态变成"作废" ，或直接删除单据
                jpaUpdateClause.set(apOrderDO.orderState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode())
                        .set(apOrderDO.auditDate, LocalDateTime.now())
                        .set(apOrderDO.auditUserId, Long.valueOf(commentInfo.getUserId()))
                        .set(apOrderDO.auditRejection, commentInfo.getComment())
                        .set(apOrderDO.auditUser, commentInfo.getUserName())
                        .set(apOrderDO.procInstId, (String) null);
                break;
            case APPROVING: // 审批中
                //一般情况将单据状态变成"审批中
                jpaUpdateClause.set(apOrderDO.orderState, UdcEnum.APPLY_STATUS_DOING.getValueCode())
                        .set(apOrderDO.submitTime,LocalDateTime.now());
                break;
            case APPROVED: // 最终节点审批通过
                //一般情况将单据状态变成"审批通过"
                jpaUpdateClause.set(apOrderDO.orderState, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode())
                    .set(apOrderDO.proposedStatus, UdcEnum.DOC_PROPOSED_STATUS_DRAFT.getValueCode())
                        .set(apOrderDO.approvedTime, LocalDateTime.now())
                        .set(apOrderDO.auditDate, LocalDateTime.now())
                        .set(apOrderDO.auditUserId, Long.valueOf(commentInfo.getUserId()))
                        .set(apOrderDO.auditUser, commentInfo.getUserName());
                break;
            default:
                break;
        }
        //执行
        jpaUpdateClause.execute();
    }

    @Override
    public ArrayList<String> taskAssignee(String businessKey, String customParams) {
        Long businessId = Long.valueOf(businessKey);
        ApOrderDTO apOrderDTO = apOrderRepoProc.get(businessId);
        if (apOrderDTO == null) {
            throw new BusinessException("单据(id:" + businessId + ")不存在,无法审批");
        }
        SysUserIdFlowRoleRpcParam param = SysUserIdFlowRoleRpcParam.builder()
                .ouIds(Collections.singletonList(apOrderDTO.getOuId()))
                .flowRoleCodes(Collections.singletonList(customParams))
                .build();
        List<Long> userIdsByFlowRoles = sysUserFlowRoleRpcService.findUserIdsByFlowRoles(param);
        return userIdsByFlowRoles.stream().map(e -> e + "").collect(Collectors.toCollection(ArrayList::new));

    }
}
