package com.elitesland.fin.application.service.payorder;


import cn.hutool.core.collection.CollUtil;
import com.elitesland.fin.application.convert.payorder.PayOrderConvert;
import com.elitesland.fin.application.convert.paytype.PayTypeConvert;
import com.elitesland.fin.application.facade.param.payorder.ApOrderToPaySaveParam;
import com.elitesland.fin.application.facade.param.payorder.PayOrderDtlSaveParam;
import com.elitesland.fin.application.facade.param.payorder.PayOrderSaveParam;
import com.elitesland.fin.application.facade.vo.payorder.PayOrderVO;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.apordertopay.ApOrderToPay;
import com.elitesland.fin.domain.entity.apverrec.ApVerRecDtl;
import com.elitesland.fin.domain.entity.payorder.PayOrder;
import com.elitesland.fin.domain.entity.payorder.PayOrderDO;
import com.elitesland.fin.domain.entity.payorder.PayOrderDtl;
import com.elitesland.fin.domain.entity.recorder.RecOrderDO;
import com.elitesland.fin.domain.param.payorder.PayOrderPageParam;
import com.elitesland.fin.domain.service.aporder.ApOrderDomainService;
import com.elitesland.fin.domain.service.apordertopay.ApOrderToPayDomainService;
import com.elitesland.fin.domain.service.apverrec.ApVerRecDomainService;
import com.elitesland.fin.domain.service.payorder.PayOrderDomainService;
import com.elitesland.fin.domain.service.payorder.PayOrderDtlDomainService;
import com.elitesland.fin.domain.service.paytype.PayTypeDomainService;
import com.elitesland.fin.rpc.workflow.WorkflowRpcService;
import com.elitesland.fin.infr.dto.aporder.ApOrderDTO;
import com.elitesland.fin.infr.dto.apordertopay.ApOrderToPayDTO;
import com.elitesland.fin.infr.dto.payorder.PayOrderDTO;
import com.elitesland.fin.infr.dto.payorder.PayOrderDtlDTO;
import com.elitesland.fin.infr.dto.paytype.PayTypeDTO;
import com.elitesland.fin.application.service.workflow.WorkFlowDefKey;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.WorkflowConstant;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import java.util.Objects;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author zhiyu.he
 * @date 2022/3/16 13:03
 */
@Service
@RequiredArgsConstructor
public class PayOrderServiceImpl implements PayOrderService {

    private final PayOrderDomainService payOrderDomainService;

    private final PayOrderDtlDomainService payOrderDtlDomainService;

    private final PayTypeDomainService payTypeDomainService;

    private final ApOrderToPayDomainService apOrderToPayDomainService;

    private final ApOrderDomainService apOrderDomainService;

    private final ApVerRecDomainService apVerRecDomainService;

    private final WorkflowRpcService workflowRpcService;

    @SysCodeProc
    @Override
    public PagingVO<PayOrderVO> page(PayOrderPageParam param) {
        PagingVO<PayOrderDTO> page = payOrderDomainService.page(param);
        PagingVO<PayOrderVO> payOrderVOPagingVO = PayOrderConvert.INSTANCE.convertPage(page);
        return payOrderVOPagingVO;
    }

    @SysCodeProc
    @Override
    public PayOrderVO queryById(Long id) {
        PayOrderDTO payOrderDTO = payOrderDomainService.queryById(id, false);
        PayOrderVO payOrderVO = PayOrderConvert.INSTANCE.dtoToVo(payOrderDTO);
        return payOrderVO;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<List<Long>> deleteByIds(List<Long> ids) {
        List<Long> deleteByIds = payOrderDomainService.deleteByIds(ids);
        //删除中间表应付单数据
        apOrderToPayDomainService.deleteByPayIds(ids);
        return ApiResult.ok(deleteByIds);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<Long> save(PayOrderSaveParam param) {
        PayOrder payOrder = PayOrderConvert.INSTANCE.convertParam(param);
        //如果修改的是应付单来源
        if (param.getId() != null) {
            deleteBeforePushDate(param);
        }
        Long id = payOrderDomainService.save(payOrder);
        return ApiResult.ok(id);
    }

    private void deleteBeforePushDate(PayOrderSaveParam param) {
        if (param.getCreateMode().equals(UdcEnum.FIN_PAY_DOC_CLS_AP.getValueCode())) {
            //修改数据之前需要检验金额是否满足修改条件
            checkApOrderMoney(param);
            //删除之前已推数据
            apOrderToPayDomainService.deleteByPayId(param.getId());
            //新增上之前关联数据
            saveApOrderToPay(param);
        }
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> submit(PayOrderSaveParam param) {
        PayOrder payOrder = PayOrderConvert.INSTANCE.convertParam(param);
        //如果修改的是应付单来源
        if (param.getId() != null) {
            deleteBefore(param);
        }
        Long id = payOrderDomainService.submit(payOrder);
        // 启动工作流
        if (payOrder.getProcInstId() == null
                || WorkflowConstant.CAN_START_PROC_STATUSES.contains(payOrder.getProcInstStatus())) {
            // 启动流程
            String procInstName = "付款单审核-" + payOrder.getPayOrderNo();
            String procKey = WorkFlowDefKey.FIN_PAY_ORDER.name();
            ProcessInfo processInfo = workflowRpcService.startProcess(procKey, procInstName, id.toString(), new HashMap<>());
            payOrderDomainService.updateWorkInfo(processInfo, id);
        }
        return ApiResult.ok(id);
    }

    private void deleteBefore(PayOrderSaveParam param) {
        if (param.getCreateMode().equals(UdcEnum.FIN_PAY_DOC_CLS_AP.getValueCode())) {
            //修改数据之前需要检验金额是否满足修改条件
            checkApOrderMoney(param);
            //删除之前已推数据
            apOrderToPayDomainService.deleteByPayId(param.getId());
            //新增上之前关联数据
            saveApOrderToPay(param);
        }
    }

    @Override
    public ApiResult<PayOrderVO> defaultValue() {
        PayTypeDTO payTypeDTO = payTypeDomainService.defaultValue();
        PayOrderVO payOrderVO = PayTypeConvert.INSTANCE.typeToOrder(payTypeDTO);
        if (payOrderVO == null) {
            return ApiResult.ok(new PayOrderVO());
        }
        return ApiResult.ok(payOrderVO);
    }

    @SysCodeProc
    @Override
    public PayOrderVO queryDetailsById(Long id) {
        PayOrderDTO payOrderDTO = payOrderDomainService.queryById(id, true);
        PayOrderVO payOrderVO = PayOrderConvert.INSTANCE.dtoToVo(payOrderDTO);
        return payOrderVO;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<String> saveByApOrder(ApOrderToPaySaveParam param) {
        List<ApOrderDTO> apOrderDTOS = apOrderDomainService.queryByIds(param.getIds());
        List<String> ouCode = new ArrayList<>();
        List<String> supCode = new ArrayList<>();
        List<String> curr = new ArrayList<>();
        List<String> apType = new ArrayList<>();
        //校验规则
        apOrderDTOS.forEach(dto -> {
            if (!dto.getOrderState().equals(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能选择审核通过的应付单据!");
            }
            if (dto.getTotalAmt().compareTo(BigDecimal.ZERO) == 0) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能选择应付单金额不为0的单据!");
            }
            ouCode.add(dto.getOuCode());
            supCode.add(dto.getSuppCode());
            curr.add(dto.getCurrCode());
            apType.add(dto.getApTypeCode());
        });
        //校验是否同一家公司
        List<String> ou = ouCode.stream().distinct().collect(Collectors.toList());
        if (ou.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择同一家公司的数据!");
        }
        //校验是否同一供应商
        List<String> sup = supCode.stream().distinct().collect(Collectors.toList());
        if (sup.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择同一家供应商的数据!");
        }
        //校验是否同一币种
        List<String> currCode = curr.stream().distinct().collect(Collectors.toList());
        if (currCode.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择同一币种的数据!");
        }
        //校验是否同一结算类型
        List<String> apTypeCode = apType.stream().distinct().collect(Collectors.toList());
        if (apTypeCode.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择同一应付单类型的数据!");
        }
        //根据应付单ID查询中间表数据
        Map<Long, List<ApOrderToPayDTO>> apOrderToPayMap = apOrderToPayDomainService.queryByApId(param.getIds()).stream().collect(Collectors.groupingBy(ApOrderToPayDTO::getApOrderId));

        HashMap<Long, BigDecimal> BigDecimalMap = new HashMap<>();
        //计算每个应付单的已退金额总和
        for (Map.Entry<Long, List<ApOrderToPayDTO>> entry : apOrderToPayMap.entrySet()) {
            BigDecimal reduce = entry.getValue().stream().map(ApOrderToPayDTO::getPayAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            BigDecimalMap.put(entry.getKey(), reduce);
        }

        // 可推付款单金额（剩余金额）=含税金额-（应付单列表已核销金额+已推付款单金额(未核销的付款单的金额)）
        //如果有推送记录得，计算可推金额放入totalAmt
        apOrderDTOS.forEach(order -> {
            if (BigDecimalMap.get(order.getId()) != null) {
                //减去核销金额
                BigDecimal payAmt = order.getTotalAmt().subtract(BigDecimalMap.get(order.getId())).subtract(order.getVerAmt());
                //对比获取到的付款单金额和应付单剩余可推金额  如果此刻推送的金额大于可推金额则校验
                if (payAmt.compareTo(BigDecimal.ZERO) <= 0) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "应付单号:" + order.getApOrderNo() + ",可推金额不足,请重新选择!");
                }
                //设置可推付款金额
                order.setTotalAmt(payAmt);
            }
        });
        //如果没有推送记录得直接取当前金额
        PayOrder payOrder = new PayOrder();
        getPayOrder(payOrder, param, apOrderDTOS);
        List<PayOrderDtl> dtl = getDtl(apOrderDTOS, param);
        payOrder.setDtlList(dtl);
        Long id = payOrderDomainService.save(payOrder);
        return ApiResult.ok(id.toString());
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void approved(Long id) {
        //获取来源单据为AP的付款单ID
        PayOrderDTO payOrderDTO = payOrderDomainService.queryById(id, false);
        if (payOrderDTO.getCreateMode().equals(UdcEnum.FIN_PAY_DOC_CLS_AP.getValueCode()) && !payOrderDTO.getApFlag()) {
            //获取对应付款信息行数据
            List<PayOrderDtlDTO> payOrderDtlDTOS = payOrderDtlDomainService.queryByMasId(CollUtil.newArrayList(payOrderDTO.getId()));
            List<Long> apIds = payOrderDtlDTOS.stream().map(PayOrderDtlDTO::getSourceId).collect(Collectors.toList());
            List<ApOrderDTO> apOrderDTOS = apOrderDomainService.queryByIds(apIds);
            ApOrderDTO apOrderDTO = apOrderDTOS.get(0);
            Map<Long, ApOrderDTO> apOrderDTOMap = apOrderDTOS.stream().collect(Collectors.toMap(ApOrderDTO::getId, t -> t));
            List<ApVerRecDtl> apVerRecDtls = new ArrayList<>();
            //组装付款单对应的核销明细
            List<ApVerRecDtl> payVerDtls = payOrderDtlDTOS.stream().map(t -> {
                ApVerRecDtl apVerRecDtl = new ApVerRecDtl();
                apVerRecDtl.setOuCode(apOrderDTO.getOuCode());
                apVerRecDtl.setOuName(apOrderDTO.getOuName());
                apVerRecDtl.setOuId(apOrderDTO.getOuId());
                apVerRecDtl.setSuppCode(apOrderDTO.getSuppCode());
                apVerRecDtl.setSuppName(apOrderDTO.getSuppName());
                apVerRecDtl.setSuppId(apOrderDTO.getSuppId());
                apVerRecDtl.setCurrCode(apOrderDTO.getCurrCode());
                apVerRecDtl.setCurrName(apOrderDTO.getCurrName());
                apVerRecDtl.setAmt(t.getRealPayAmt());
                apVerRecDtl.setOrderId(t.getMasId());
                apVerRecDtl.setOrderNo(payOrderDTO.getPayOrderNo());
                apVerRecDtl.setOrderType(FinConstant.PAY);
                apVerRecDtl.setOrderName("付款单");
                apVerRecDtl.setVerDate(LocalDateTime.now());
                apVerRecDtl.setBuDate(payOrderDTO.getBuDate());
                apVerRecDtl.setVerifyType(UdcEnum.FIN_VERIFY_TYPE_AUTO.getValueCode());
                return apVerRecDtl;
            }).collect(Collectors.toList());
            apVerRecDtls.addAll(payVerDtls);
            //组装应付单对应的核销明细
            List<ApVerRecDtl> apVerDtls = payOrderDtlDTOS.stream().map(t -> {
                ApVerRecDtl apVerRecDtl = new ApVerRecDtl();
                apVerRecDtl.setOuCode(apOrderDTO.getOuCode());
                apVerRecDtl.setOuName(apOrderDTO.getOuName());
                apVerRecDtl.setOuId(apOrderDTO.getOuId());
                apVerRecDtl.setSuppCode(apOrderDTO.getSuppCode());
                apVerRecDtl.setSuppName(apOrderDTO.getSuppName());
                apVerRecDtl.setSuppId(apOrderDTO.getSuppId());
                apVerRecDtl.setCurrCode(apOrderDTO.getCurrCode());
                apVerRecDtl.setCurrName(apOrderDTO.getCurrName());
                apVerRecDtl.setAmt(t.getRealPayAmt());
                apVerRecDtl.setOrderId(t.getSourceId());
                apVerRecDtl.setOrderNo(apOrderDTOMap.get(t.getSourceId()).getApOrderNo());
                apVerRecDtl.setOrderType(FinConstant.AP);
                apVerRecDtl.setOrderName("应付单");
                apVerRecDtl.setVerDate(LocalDateTime.now());
                apVerRecDtl.setBuDate(apOrderDTO.getBuDate());
                apVerRecDtl.setVerifyType(UdcEnum.FIN_VERIFY_TYPE_AUTO.getValueCode());
                return apVerRecDtl;
            }).collect(Collectors.toList());
            apVerRecDtls.addAll(apVerDtls);
            apVerRecDomainService.save(apVerRecDtls);
            apVerRecDtls.stream().forEach(t -> {
                if (t.getOrderType().equals(FinConstant.AP)) {
                    apOrderDomainService.updateVerAmt(t.getOrderId(), t.getAmt());
                } else if (t.getOrderType().equals(FinConstant.PAY)) {
                    payOrderDomainService.updateVerAmt(t.getOrderId(), t.getAmt());
                }
            });
        }
    }

    private List<PayOrderDtl> getDtl(List<ApOrderDTO> apOrderDTOS, ApOrderToPaySaveParam param) {
        return apOrderDTOS.stream().map(dto -> {
            PayOrderDtl dtl = new PayOrderDtl();
            dtl.setSourceId(dto.getId());
            dtl.setSourceNo(dto.getApOrderNo());
            dtl.setPayAccount(param.getPayAccount());
            dtl.setPayBank(param.getPayBank());
            dtl.setPayType(param.getPayType());
            dtl.setRecBank(param.getRecBank());
            dtl.setRecAccount(param.getRecAccount());
            dtl.setTotalAmt(dto.getTotalAmt());
            return dtl;
        }).collect(Collectors.toList());
    }

    private void getPayOrder(PayOrder payOrder, ApOrderToPaySaveParam param, List<ApOrderDTO> apOrderDTOS) {
        //状态
        payOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        //应付单
        payOrder.setCreateMode(UdcEnum.FIN_PAY_DOC_CLS_AP.getValueCode());
        //备注
        payOrder.setRemark(param.getRemark());
        ApOrderDTO apOrderDTO = apOrderDTOS.get(0);
        payOrder.setOuId(apOrderDTO.getOuId());
        payOrder.setOuCode(apOrderDTO.getOuCode());
        payOrder.setOuName(apOrderDTO.getOuName());
        payOrder.setSuppCode(apOrderDTO.getSuppCode());
        payOrder.setSuppName(apOrderDTO.getSuppName());
        payOrder.setSuppId(apOrderDTO.getSuppId());
        payOrder.setApTypeId(apOrderDTO.getApTypeId());
        payOrder.setApTypeName(apOrderDTO.getApTypeName());
        payOrder.setApTypeCode(apOrderDTO.getApTypeCode());
        payOrder.setApFlag(false);
        payOrder.setInitFlag(false);
        //本位币
        payOrder.setLocalCurrCode(apOrderDTO.getLocalCurrCode());
        payOrder.setLocalCurrName(apOrderDTO.getLocalCurrName());
        payOrder.setCurrCode(apOrderDTO.getCurrCode());
        payOrder.setCurrName(apOrderDTO.getCurrName());
        //税率
        payOrder.setExchangeRate(apOrderDTO.getExchangeRate());
        payOrder.setBuDate(LocalDateTime.now());
        payOrder.setApDate(LocalDateTime.now());
    }

    /**
     * 新增应付单和付款单关系
     *
     * @param param 新增参数
     */
    private void saveApOrderToPay(PayOrderSaveParam param) {
        List<ApOrderToPay> collect = param.getPayOrderDtlSaveParams().stream().map(dtl -> {
            ApOrderToPay apOrderToPay = new ApOrderToPay();
            //修改时获取的付款单单头ID
            apOrderToPay.setPayOrderId(param.getId());
            //明细信息行返回的数据 应付单ID和金额
            apOrderToPay.setApOrderId(dtl.getSourceId());
            apOrderToPay.setPayAmt(dtl.getTotalAmt());
            apOrderToPay.setSourceType(UdcEnum.FIN_PAY_DOC_CLS_AP.getValueCode());
            return apOrderToPay;
        }).collect(Collectors.toList());
        apOrderToPayDomainService.save(collect);
    }

    /**
     * 校验可推金额
     *
     * @param param 生成付款单参数
     */
    private void checkApOrderMoney(PayOrderSaveParam param) {
        //获取到修改的明细数据
        List<PayOrderDtlSaveParam> payOrderDtlSaveParams = param.getPayOrderDtlSaveParams();
        //获取到应付单ID
        List<Long> apOrderId = payOrderDtlSaveParams.stream().map(PayOrderDtlSaveParam::getSourceId).collect(Collectors.toList());
        //查询应付单数据
        List<ApOrderDTO> orderDTOS = apOrderDomainService.queryByIds(apOrderId);
        Map<Long, ApOrderDTO> apMap = orderDTOS.stream().collect(Collectors.toMap(ApOrderDTO::getId, t -> t));
        //根据应付单ID查询中间表数据
        Map<Long, List<ApOrderToPayDTO>> apOrderToPayMap = apOrderToPayDomainService.queryByApId(apOrderId).stream().collect(Collectors.groupingBy(ApOrderToPayDTO::getApOrderId));

        HashMap<Long, BigDecimal> BigDecimalMap = new HashMap<>();
        //计算每个应付单的已退金额总和
        for (Map.Entry<Long, List<ApOrderToPayDTO>> entry : apOrderToPayMap.entrySet()) {
            //计算时过滤掉修改数据之前得金额
            BigDecimal reduce = entry.getValue().stream().filter(order -> !order.getPayOrderId().equals(param.getId())).map(ApOrderToPayDTO::getPayAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            BigDecimalMap.put(entry.getKey(), reduce);
        }

        // 可推付款单金额（剩余金额）=含税金额-（应付单列表已核销金额+已推付款单金额(未核销的付款单的金额)）
        payOrderDtlSaveParams.forEach(order -> {
            if (BigDecimalMap.get(order.getSourceId()) != null) {
                payAmtCount(apMap, BigDecimalMap, order);

            }
        });
    }

    private static void payAmtCount(Map<Long, ApOrderDTO> apMap, HashMap<Long, BigDecimal> BigDecimalMap, PayOrderDtlSaveParam order) {
        if (apMap.get(order.getSourceId()) != null) {
            //获取应付单金额
            BigDecimal apOrderAmount = apMap.get(order.getSourceId()).getTotalAmt();
            //应付单的核销金额
            BigDecimal verAmount = apMap.get(order.getSourceId()).getVerAmt();
            //计算可推金额，如果修改值大于可推金额，则报错
            BigDecimal payAmt = apOrderAmount.subtract(verAmount).subtract(BigDecimalMap.get(order.getSourceId()));
            //对比获取到的付款单金额和应付单剩余可推金额  如果此刻推送的金额大于可推金额则校验
            if (order.getTotalAmt().compareTo(payAmt) > 0) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "应付单号:" + order.getSourceNo() + ",可推金额不足,请重新选择!");
            }
        }
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Void> cancelApprove(List<Long> ids) {
        List<PayOrderDO>  payOrderDOS = payOrderDomainService.queryByIds(ids);
        //检查
        checkCancelDoc(payOrderDOS);
        // 重置状态
        payOrderDOS.forEach(payOrderDTO -> {

            payOrderDTO.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
            payOrderDTO.setProposedStatus(null);
            payOrderDTO.setProcInstId(null);
            payOrderDTO.setProcInstStatus(null);
        });
        payOrderDomainService.saveAll(payOrderDOS);
        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = {RuntimeException.class, Exception.class})
    public ApiResult<Long> redPunch(Long id) {
        PayOrderDTO payOrderDTO = payOrderDomainService.queryById(id, false);
        //检查
        this.checkRedPunch(payOrderDTO);
        //红冲
        PayOrder payOrder = payOrderDomainService.redPunchCreate(id);
        PayOrderSaveParam param = PayOrderConvert.INSTANCE.convert(payOrder);
        return this.submit(param);
    }




    /**
     * 取消单据检查
     * @param payOrderDOS
     */
    void checkCancelDoc(List<PayOrderDO> payOrderDOS) {
        if (CollectionUtils.isEmpty(payOrderDOS)) {
            throw new com.elitescloud.boot.exception.BusinessException("单据不存在");
        }
        payOrderDOS.forEach(payOrderDO -> {
            String perfix = "单号：" + payOrderDO.getPayOrderNo() + "-";
            if (!UdcEnum.DOC_PROPOSED_STATUS_DRAFT.getValueCode().equals(payOrderDO.getProposedStatus()) && !UdcEnum.DOC_PROPOSED_STATUS_PROPOSED_FAIL.getValueCode()
                .equals(payOrderDO.getProposedStatus())) {
                throw new com.elitescloud.boot.exception.BusinessException(perfix + "拟定状态必须为草稿或拟定失败");
            }
            if (!(BigDecimal.ZERO.compareTo(payOrderDO.getVerAmt()) == 0)) {
                throw new com.elitescloud.boot.exception.BusinessException(perfix + "已核销金额必须为0");
            }
            if (!UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(payOrderDO.getOrderState())) {
                throw new com.elitescloud.boot.exception.BusinessException(perfix + "单据状态必须为审核通过");
            }
            if (Boolean.TRUE.equals(payOrderDO.getRedState())){
                throw new com.elitescloud.boot.exception.BusinessException(perfix + "单据已红冲");
            }
        });


    }

    /**
     * 红冲单据检查
     * @param payOrderDTO
     */
    void checkRedPunch(PayOrderDTO payOrderDTO) {
        if (Objects.isNull(payOrderDTO)){
            throw new com.elitescloud.boot.exception.BusinessException("单据不存在");
        }
        if (Boolean.TRUE.equals(payOrderDTO.getRedState())){
            throw new com.elitescloud.boot.exception.BusinessException("单据已红冲");
        }
        if (Objects.nonNull(payOrderDTO.getRedSourceNo())){
            throw new com.elitescloud.boot.exception.BusinessException("来源单据不可为红冲单据");
        }
        if (!(BigDecimal.ZERO.compareTo(payOrderDTO.getVerAmt())==0)){
            throw new com.elitescloud.boot.exception.BusinessException("已核销金额必须为0");
        }
        if (!UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(payOrderDTO.getOrderState())){
            throw new com.elitescloud.boot.exception.BusinessException("单据状态必须为审核通过");
        }

    }
}
