package com.elitesland.fin.domain.service.aporder;

import static com.elitesland.fin.domain.entity.arorder.QArOrderDO.arOrderDO;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.core.seq.SeqNumProvider;
import com.elitesland.fin.application.convert.aporder.ApOrderConvert;
import com.elitesland.fin.application.convert.aporder.ApOrderDtlConvert;
import com.elitesland.fin.application.convert.aporder.ApOrderDtlGroupConvert;
import com.elitesland.fin.application.convert.arorder.ArOrderConvert;
import com.elitesland.fin.application.convert.arorder.ArOrderDtlConvert;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.aporder.*;
import com.elitesland.fin.domain.entity.arorder.ArOrder;
import com.elitesland.fin.domain.entity.arorder.ArOrderDO;
import com.elitesland.fin.domain.entity.arorder.ArOrderDtlDO;
import com.elitesland.fin.domain.entity.saleinv.SaleInvDO;
import com.elitesland.fin.domain.param.aporder.ApOrderPageParam;
import com.elitesland.fin.domain.param.aporder.ApOrderParam;
import com.elitesland.fin.domain.service.apordertopay.ApOrderToPayDomainService;
import com.elitesland.fin.infr.dto.aporder.ApOrderDTO;
import com.elitesland.fin.infr.dto.aporder.ApOrderDtlDTO;
import com.elitesland.fin.infr.dto.aporder.ApOrderDtlGroupDTO;
import com.elitesland.fin.infr.dto.apordertopay.ApOrderToPayDTO;
import com.elitesland.fin.infr.dto.common.ApVerDTO;
import com.elitesland.fin.infr.factory.aporder.ApOrderFactory;
import com.elitesland.fin.infr.repo.aporder.*;
import com.elitesland.fin.rpc.order.RmiOrderRpcService;
import com.elitesland.fin.rpc.pur.RmiPurRpcService;
import com.elitesland.fin.utils.BeanUtils;
import com.elitesland.order.param.SalReconciliatInvDTO;
import com.elitesland.pur.dto.PurAccountDTO;
import com.elitesland.workflow.ProcessInfo;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.google.common.collect.Lists;
import java.util.HashMap;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.function.Function;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.domain.service.aporder
 * @date 2022/3/16 10:45
 */
@Service
@RequiredArgsConstructor
public class ApOrderDomainServiceImpl implements ApOrderDomainService {

    private final ApOrderRepo apOrderRepo;
    private final ApOrderRepoProc apOrderRepoProc;
    private final ApOrderDtlRepoProc apOrderDtlRepoProc;
    private final ApOrderDtlRepo apOrderDtlRepo;
    private final ApOrderFactory apOrderFactory;
    private final ApOrderDtlGroupRepo apOrderDtlGroupRepo;
    private final ApOrderDtlGroupRepoProc apOrderDtlGroupRepoProc;
    //发号器生成付款单号
    private final SeqNumProvider sysNumberRuleService;
    private final ApOrderToPayDomainService apOrderToPayDomainService;
    private final RmiPurRpcService rmiPurRpcService;


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long save(ApOrder apOrder) {
        if (apOrder.getVerAmt() == null) {
            // 设置默认
            apOrder.setDef();
        }
        // 单头信息输入的校验
        apOrder.check();
        // 校验明细行
        apOrder.checkDtlList();
        // 手工来源
        if (apOrder.getCreateMode().equals(UdcEnum.FIN_AP_DOC_CLS_MANU.getValueCode())) {
            // 校验明细金额
            apOrder.checkAmt();
            // 校验明细金额/单头/汇总金额
            apOrder.checkAmtSum();
        }
        // 对账单来源
        if (apOrder.getCreateMode().equals(UdcEnum.FIN_AP_DOC_CLS_PACCK.getValueCode())) {
            // 设置默认值
            apOrder.setPacckDef();
            // 进行统计的计算
            apOrder.countByPacck();
            // 汇总列表数据添加
            List<ApOrderDtlGroup> apOrderDtlGroups = this.genDtlGroup(apOrder);
            apOrder.setApOrderDtlGroupList(apOrderDtlGroups);
        }
        apOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        // 数据的新增
        ApOrderDO apOrderRes = saveOrUpdate(apOrder);
        return apOrderRes.getId();
    }

    /**
     * 根据明细生成汇总List
     *
     * @return
     */
    public List<ApOrderDtlGroup> genDtlGroup(ApOrder apOrder) {
        // 获取明细的List
        List<ApOrderDtl> res = apOrder.getApOrderDtlList();
        Map<String, List<ApOrderDtl>> collectMaps = res.stream().collect(Collectors.groupingBy(ApOrderDtl::getItemCode));
        List<ApOrderDtlGroup> resList = new ArrayList<>();
        for (Map.Entry<String,List<ApOrderDtl>> entry : collectMaps.entrySet()) {
            ApOrderDtlGroup apOrderDtlGroup = new ApOrderDtlGroup();
            List<ApOrderDtl> apOrderDtlList = entry.getValue();
            // 对集合中元素进行计算
            // 数量的总和
            BigDecimal qtySum = apOrderDtlList.stream().map(ApOrderDtl::getQty).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 税额的总和
            BigDecimal taxAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getTaxAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 税额（本位币）taxCurAmt
            BigDecimal taxCurAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getTaxCurAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 不含税金额 exclTaxAmt
            BigDecimal exclTaxAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getExclTaxAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 不含税金额（本位币）exclTaxCurAmt
            BigDecimal exclTaxCurAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getExclTaxCurAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 含税金额 totalAmt
            BigDecimal totalAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getTotalAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 含税金额（本位币）totalCurAmt
            BigDecimal totalCurAmtSum = apOrderDtlList.stream().map(ApOrderDtl::getTotalCurAmt).reduce(BigDecimal.ZERO, BigDecimal::add);

            // 拼接返回对象
            apOrderDtlGroup.setItemId(apOrderDtlList.get(0).getItemId());
            apOrderDtlGroup.setItemCode(apOrderDtlList.get(0).getItemCode());
            apOrderDtlGroup.setItemName(apOrderDtlList.get(0).getItemName());
            apOrderDtlGroup.setSmallCateCode(apOrderDtlList.get(0).getSmallCateCode());
            apOrderDtlGroup.setSmallCateName(apOrderDtlList.get(0).getSmallCateName());
            apOrderDtlGroup.setQty(qtySum);
            apOrderDtlGroup.setTotalAmt(totalAmtSum);
            apOrderDtlGroup.setExclTaxAmt(exclTaxAmtSum);
            apOrderDtlGroup.setTaxAmt(taxAmtSum);
            apOrderDtlGroup.setTotalCurAmt(totalCurAmtSum);
            apOrderDtlGroup.setExclTaxCurAmt(exclTaxCurAmtSum);
            apOrderDtlGroup.setTaxCurAmt(taxCurAmtSum);
            resList.add(apOrderDtlGroup);
            // 页面选择参数 (设置费用部门/费用类型)
        }
        return resList;
    }


    /**
     * 对 应付单主表/明细表/汇总明细表
     * 进行数据库新增
     *
     * @param apOrder
     * @return
     */
    @Transactional(rollbackFor = {Exception.class})
    public ApOrderDO saveOrUpdate(ApOrder apOrder) {
        if (CharSequenceUtil.isBlank(apOrder.getApOrderNo())) {
            // 发号器-生成应付单号
            String apOrderNo = sysNumberRuleService.generateCode(FinConstant.FIN,FinConstant.YFD, null);
            apOrder.setApOrderNo(apOrderNo);
        }
        ApOrderDO apOrderDO = ApOrderConvert.INSTANCE.convert(apOrder);
        //新增时锁版本默认值0
        if (apOrderDO.getId() == null) {
            apOrderDO.setAuditDataVersion(0);
        }

        ApOrderDO apOrderRes = apOrderRepo.save(apOrderDO);
        // 应付单明细单的新增
        // 根据应付单id删除明细行
        apOrderDtlRepoProc.delByMasId(Lists.newArrayList(apOrderRes.getId()));
        List<ApOrderDtl> apOrderDtlList = apOrder.getApOrderDtlList();
        List<ApOrderDtlDO> apOrderDtlDOS = ApOrderDtlConvert.INSTANCE.convert(apOrderDtlList);
        apOrderDtlDOS.stream().forEach(x -> {
            x.setMasId(apOrderRes.getId());
            apOrderDtlRepo.save(x);
        });
        // 应付单汇总数据新增
        // 根据应付单id删除汇总行
        apOrderDtlGroupRepoProc.delByMasId(Lists.newArrayList(apOrderRes.getId()));
        List<ApOrderDtlGroup> apOrderDtlGroups = apOrder.getApOrderDtlGroupList();
        if (null != apOrderDtlGroups) {
            List<ApOrderDtlGroupDO> apOrderDtlGroupDOS = ApOrderDtlGroupConvert.INSTANCE.convert(apOrderDtlGroups);
            apOrderDtlGroupDOS.stream().forEach(x -> {
                x.setMasId(apOrderRes.getId());
                apOrderDtlGroupRepo.save(x);
            });
        }
        return apOrderRes;
    }


    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = {Exception.class})
    public Long commit(ApOrder apOrder) {
        // 校验单头与明细
        apOrder.check();
        apOrder.checkDtlList();
        if (apOrder.getVerAmt() == null) {
            apOrder.setDef();
        }
        // 校验明细金额
        apOrder.checkAmt();
        // 金额数据校验(单头，明细，汇总)金额的校验
        apOrder.checkAmtSum();
        // 状态为待审批
        apOrder.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
        // 进行数据的新增或保存
        ApOrderDO apOrderRes = saveOrUpdate(apOrder);
        return apOrderRes.getId();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateWorkInfo(ProcessInfo processInfo, Long id) {
        apOrderRepoProc.updateWorkInfo(processInfo, id);
    }




    @Override
    public ApOrderDTO getApOrderAndDtl(Long id) {
        ApOrderDTO apOrderDTO = get(id);
        // 新增明细列表
        List<ApOrderDtlDTO> apOrderDtlDOS = apOrderDtlRepoProc.listByMisId(id);
        // 新增汇总列表
        apOrderDTO.setApOrderDtlDTOList(apOrderDtlDOS);
        List<ApOrderDtlGroupDTO> apOrderDtlGroupDTOS = apOrderDtlGroupRepoProc.listByMisId(id);
        apOrderDTO.setApOrderDtlGroupDTOList(apOrderDtlGroupDTOS);
        return apOrderDTO;
    }


    @Override
    public List<ApOrderDTO> queryByIds(List<Long> ids) {
        return apOrderRepoProc.queryByIds(ids);
    }

    @Override
    public List<ApOrderDTO> getApOrderList(ApOrderParam apOrderParam) {
        List<ApOrderDTO> res = apOrderRepoProc.getApOrderList(apOrderParam);
        return res;
    }

    @Override
    public Boolean queryByApTypeId(Long apTypeId) {
        return apOrderRepoProc.queryByApTypeId(apTypeId);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    @Retryable(value = {BusinessException.class}, maxAttempts = 5, backoff = @Backoff(value = 500))
    public void updateVerAmt(Long id, BigDecimal amt) {
        ApVerDTO apVerDTO = apOrderRepoProc.queryVerAmtById(id);
        if (Objects.isNull(apVerDTO)){
            throw new BusinessException("未查询到应付单信息");
        }
        BigDecimal totalAmt = Objects.isNull(apVerDTO.getTotalAmt()) ? BigDecimal.ZERO : apVerDTO.getTotalAmt();
        if (apVerDTO.getVerAmt().add(amt).compareTo(totalAmt) > 0) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前核销金额大于应付单金额，请刷新页面重新输入");
        }
        apVerDTO.setAmtAndVerState(amt);
        long rows = apOrderRepoProc.updateVerAmt(apVerDTO);
        if (rows == 0) {
            throw new BusinessException("应付单更新已核销金额失败");
        }
    }

    @Override
    public PagingVO<ApOrderDTO> page(ApOrderPageParam apOrderPageParam) {
        PagingVO<ApOrderDTO> res = apOrderFactory.page(apOrderPageParam);
        return res;
    }

    @Override
    public ApOrderDTO get(Long id) {
        ApOrderDTO apOrderDTO = apOrderRepoProc.get(id);
        return apOrderDTO;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long audit(List<Long> ids, String content, SysUserDTO user) {
        Long res = apOrderRepoProc.audit(ids, content, user);
        return res;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long del(List<Long> ids) {

        List<ApOrderDTO> apOrderDTOList = apOrderRepoProc.queryByIds(ids);
        HashMap<String, List<String>> createSourceMap = new HashMap<>();
        apOrderDTOList.stream().forEach(x -> {
            if (!x.getOrderState().equals(UdcEnum.APPLY_STATUS_DRAFT.getValueCode())) {
                throw new BusinessException("非草稿状态不能删除");
            }
            if (Objects.nonNull(x.getSourceNo()) && Objects.isNull(x.getRedSourceNo())) {
                List<String> sorceNos = createSourceMap.getOrDefault(x.getCreateMode(), new ArrayList<>());
                sorceNos.add(x.getSourceNo());
                createSourceMap.put(x.getCreateMode(), sorceNos);
            }
            //红冲原单处理
            if (Objects.nonNull(x.getRedSourceNo())) {
                apOrderRepoProc.setRedFlag(x.getRedSourceId(), false);
            }
        });

        for (Entry<String, List<String>> entry : createSourceMap.entrySet()) {
            switch (Optional.ofNullable(UdcEnum.getByValueCode("FIN", "AP_DOC_CLS", entry.getKey())).orElse(UdcEnum.DEFAUT)) {
                //对账单
                case FIN_AP_DOC_CLS_PACCK:
                    rmiPurRpcService.batchUpdateDocStatus(PurAccountDTO.builder().docStatus(UdcEnum.PUR_ACCOUNT_CHECK_STATUS_DONE.getValueCode()).docNos(entry.getValue()).build());
                    break;
                default:

            }
        }

        Long res = apOrderRepoProc.del(ids);
        // 删除明细信息
        apOrderDtlRepoProc.delByMasId(ids);
        // 删除汇总信息
        apOrderDtlGroupRepoProc.delByMasId(ids);
        return res;
    }

    @Override
    public void restDocState(List<Long> ids) {
        List<ApOrderDO> apOrderDOS = apOrderRepo.findAllById(ids);
        //检查
        checkCancelDoc(apOrderDOS);
        //重置
        apOrderDOS.forEach(apOrderDO -> {
            apOrderDO.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
            apOrderDO.setProposedStatus(null);
            apOrderDO.setProcInstId(null);
            apOrderDO.setProcInstStatus(null);
        });
        apOrderRepo.saveAll(apOrderDOS);
    }

    /**
     * 取消单据检查
     * @param apOrderDOS
     */
    void checkCancelDoc(List<ApOrderDO> apOrderDOS) {
        if (CollectionUtils.isEmpty(apOrderDOS)){
            throw new com.elitescloud.boot.exception.BusinessException("单据不存在");
        }
        List<Long> ids = apOrderDOS.stream().map(ApOrderDO::getId).collect(Collectors.toList());
        Map<Long, List<ApOrderToPayDTO>> apOrderPayMap = apOrderToPayDomainService.queryByApId(ids).stream().collect(Collectors.groupingBy(ApOrderToPayDTO::getApOrderId));
        apOrderDOS.forEach(apOrderDO -> {
            String perfix = "单号：" + apOrderDO.getApOrderNo() + "-";
            if (CollectionUtils.isNotEmpty(apOrderPayMap.get(apOrderDO.getId()))){
                throw new BusinessException(perfix + "已经生成付款单");
            }
            if (!UdcEnum.DOC_PROPOSED_STATUS_DRAFT.getValueCode().equals(apOrderDO.getProposedStatus()) && !UdcEnum.DOC_PROPOSED_STATUS_PROPOSED_FAIL.getValueCode()
                .equals(apOrderDO.getProposedStatus())) {
                throw new BusinessException(perfix + "拟定状态必须为草稿或拟定失败");
            }
            if (!(BigDecimal.ZERO.compareTo(apOrderDO.getVerAmt()) == 0)) {
                throw new BusinessException(perfix + "已核销金额必须为0");
            }
            if (!UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(apOrderDO.getOrderState())) {
                throw new BusinessException(perfix + "单据状态必须为审核通过");
            }
            if (Boolean.TRUE.equals(apOrderDO.getRedState())){
                throw new com.elitescloud.boot.exception.BusinessException(perfix + "单据已红冲");
            }
        });
    }



    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApOrder redPunchCreate(Long id) {
        ApOrderDO apOrderDO = apOrderRepo.findById(id).get();
        apOrderDO.setRedState(Boolean.TRUE);
        apOrderRepo.save(apOrderDO);
        ApOrder newApOrder = BeanUtils.toBean(apOrderDO, ApOrder.class);

        newApOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        newApOrder.setProcInstId(null);
        newApOrder.setProcInstStatus(null);
        newApOrder.setRedState(null);
        newApOrder.setRedSourceNo(apOrderDO.getApOrderNo());
        newApOrder.setRedSourceId(apOrderDO.getId());
        newApOrder.setTaxAmt(apOrderDO.getTaxAmt().negate());
        newApOrder.setTotalAmt(apOrderDO.getTotalAmt().negate());
        newApOrder.setExclTaxAmt(apOrderDO.getExclTaxAmt().negate());
        newApOrder.setTotalCurAmt(apOrderDO.getTotalCurAmt().negate());
        newApOrder.setTaxCurAmt(apOrderDO.getTaxCurAmt().negate());
        newApOrder.setExclTaxCurAmt(apOrderDO.getExclTaxCurAmt().negate());
        newApOrder.setId(null);
        newApOrder.setApOrderNo(null);
        newApOrder.setProposedStatus(null);
        newApOrder.setAuditUser(null);
        newApOrder.setApprovedTime(null);
        newApOrder.setAuditUserId(null);
        newApOrder.setRemark(null);

        List<ApOrderDtlDO> dtls = apOrderDtlRepo.findAllByMasId(id);
        List<ApOrderDtl> newDtls = dtls.stream().map(dtl -> {
            ApOrderDtl dtlDO = BeanUtils.toBean(dtl, ApOrderDtl.class);

            dtlDO.setTaxAmt(dtl.getTaxAmt().negate());
            dtlDO.setTotalAmt(dtl.getTotalAmt().negate());
            dtlDO.setExclTaxAmt(dtl.getExclTaxAmt().negate());
            dtlDO.setTotalCurAmt(dtl.getTaxCurAmt().negate());
            dtlDO.setTaxCurAmt(dtl.getTotalCurAmt().negate());
            dtlDO.setExclTaxCurAmt(dtl.getExclTaxCurAmt().negate());
            dtlDO.setQty(dtlDO.getQty().negate());
            return dtlDO;
        }).collect(Collectors.toList());
        newApOrder.setApOrderDtlList(newDtls);
        List<ApOrderDtlGroupDO> apOrderDtlGroupDOS = apOrderDtlGroupRepo.findAllByMasId(id);
        List<ApOrderDtlGroup> groupList = apOrderDtlGroupDOS.stream().map(dtl -> {
            ApOrderDtlGroup dtlDO = BeanUtils.toBean(dtl, ApOrderDtlGroup.class);
            dtlDO.setTaxAmt(dtl.getTaxAmt().negate());
            dtlDO.setTotalAmt(dtl.getTotalAmt().negate());
            dtlDO.setExclTaxAmt(dtl.getExclTaxAmt().negate());
            dtlDO.setTotalCurAmt(dtl.getTaxCurAmt().negate());
            dtlDO.setTaxCurAmt(dtl.getTotalCurAmt().negate());
            dtlDO.setExclTaxCurAmt(dtl.getExclTaxCurAmt().negate());
            dtlDO.setQty(dtlDO.getQty().negate());
            return dtlDO;
        }).collect(Collectors.toList());
        newApOrder.setApOrderDtlGroupList(groupList);

        return newApOrder;
    }

}
