package com.elitesland.fin.domain.service.arorder;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.map.MapUtil;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitesland.fin.application.convert.arorder.ArOrderDtlRecordConvert;
import com.elitesland.fin.application.convert.arorder.ArOrderRecordConvert;
import com.elitesland.fin.application.facade.param.arorder.ArOrderRecordSaveParam;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.arorder.ArOrderDtlRecordDO;
import com.elitesland.fin.domain.entity.arorder.ArOrderRecordDO;
import com.elitesland.fin.domain.param.arorder.ArOrderRecordParam;
import com.elitesland.fin.infr.dto.arorder.ArOrderDtlRecordDTO;
import com.elitesland.fin.infr.dto.arorder.ArOrderRecordDTO;
import com.elitesland.fin.infr.repo.arorder.ArOrderDtlRecordRepo;
import com.elitesland.fin.infr.repo.arorder.ArOrderDtlRecordRepoProc;
import com.elitesland.fin.infr.repo.arorder.ArOrderRecordRepo;
import com.elitesland.fin.infr.repo.arorder.ArOrderRecordRepoProc;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2024/02/28
 * @Version 1.0
 * @Content:
 */
@Service
@RequiredArgsConstructor
public class ArOrderRecordDomainServiceImpl implements ArOrderRecordDomainService {
    private final ArOrderRecordRepo arOrderRecordRepo;
    private final ArOrderRecordRepoProc arOrderRecordRepoProc;
    private final ArOrderDtlRecordRepo arOrderDtlRecordRepo;
    private final ArOrderDtlRecordRepoProc arOrderDtlRecordRepoProc;

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long save(ArOrderRecordSaveParam saveParam) {
        ArOrderRecordDO arOrderRecordDO = ArOrderRecordConvert.INSTANCE.paramConvertDo(saveParam);
        if (StringUtils.isBlank(arOrderRecordDO.getArFlag())){
            arOrderRecordDO.setArFlag(UdcEnum.AR_FLAG_TYPE_0.getValueCode());
        }
        ArOrderRecordDO newArOrderRecordDO = arOrderRecordRepo.save(arOrderRecordDO);
        Long arOrderRecordId = newArOrderRecordDO.getId();


        List<ArOrderDtlRecordDO> arOrderDtlRecordDOList = ArOrderDtlRecordConvert.INSTANCE.paramConvertDoBatch(saveParam.getArOrderDtlRecordSaveParams());
        arOrderDtlRecordDOList.forEach(arOrderDtlRecordDO -> {
            arOrderDtlRecordDO.setMasId(arOrderRecordId);
        });

        arOrderDtlRecordRepo.saveAll(arOrderDtlRecordDOList);

        return arOrderRecordId;
    }


    @Override
    @SysCodeProc
    public ArOrderRecordDTO selectById(Long id) {
        Optional<ArOrderRecordDO> optional = arOrderRecordRepo.findById(id);
        if (optional.isEmpty()){
            return null;
        }

        ArOrderRecordDTO arOrderRecordDTO = ArOrderRecordConvert.INSTANCE.doConvertDto(optional.get());
        return arOrderRecordDTO;
    }

    @Override
    @SysCodeProc
    public List<ArOrderRecordDTO> selectByParam(ArOrderRecordParam param) {
        List<ArOrderRecordDO> arOrderRecordDOS = arOrderRecordRepoProc.selectArOrderRecordByParam(param);
        if (CollectionUtil.isEmpty(arOrderRecordDOS)){
            return Collections.EMPTY_LIST;
        }

        List<ArOrderRecordDTO> arOrderRecordDTOList = ArOrderRecordConvert.INSTANCE.doConvertDtoBatch(arOrderRecordDOS);
        return arOrderRecordDTOList;
    }

    @Override
    @SysCodeProc
    public List<ArOrderRecordDTO> selectDetailsByParam(ArOrderRecordParam param) {
        List<ArOrderRecordDO> arOrderRecordDOS = arOrderRecordRepoProc.selectArOrderRecordByParam(param);
        if (CollectionUtil.isEmpty(arOrderRecordDOS)){
            return Collections.EMPTY_LIST;
        }
        List<ArOrderRecordDTO> arOrderRecordDTOList = ArOrderRecordConvert.INSTANCE.doConvertDtoBatch(arOrderRecordDOS);

        List<Long> arOrderRecordIds = arOrderRecordDOS.stream().map(ArOrderRecordDO::getId).filter(Objects::nonNull).collect(Collectors.toList());
        List<ArOrderDtlRecordDO> arOrderDtlRecordDOList = arOrderDtlRecordRepo.findAllByMasIdIn(arOrderRecordIds);
        List<ArOrderDtlRecordDTO> arOrderDtlRecordDTOS = ArOrderDtlRecordConvert.INSTANCE.doConvertDtoBatch(arOrderDtlRecordDOList);

        Map<Long, List<ArOrderDtlRecordDTO>> dtlMap = arOrderDtlRecordDTOS.stream().collect(Collectors.groupingBy(ArOrderDtlRecordDTO::getMasId));

        arOrderRecordDTOList.forEach(arOrderRecordDTO -> {
            if (MapUtil.isNotEmpty(dtlMap)){
                List<ArOrderDtlRecordDTO> dtlRecordDTOS = dtlMap.get(arOrderRecordDTO.getId());
                arOrderRecordDTO.setArOrderDtlRecordDTOS(dtlRecordDTOS);
            }
        });

        return arOrderRecordDTOList;
    }

    /**
     * 更新是否生成应收单标记
     * @param id 记录表ID
     * @param arFlag 是否生成应收单标记
     *
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateArFlagById(Long id, String arFlag) {
        arOrderRecordRepoProc.updateArFlagById(id,arFlag);
    }

    /**
     * 更新是否生成应收单标记和失败原因
     * @param id 记录表ID
     * @param arFlag 是否生成应收单标记
     * @param failMsg 失败原因
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateArFlagAndFailMsgById(Long id, String arFlag, String failMsg) {
        arOrderRecordRepoProc.updateArFlagAndFailMsgById(id,arFlag,failMsg);
    }


    /**
     * 根据来源单号批量查询记录表ID
     * @param sourceNos 来源单号
     * @return
     */
    @Override
    public List<Long> selectIdBySourceNo(List<String> sourceNos) {
        return arOrderRecordRepoProc.selectIdBySourceNo(sourceNos);
    }

    /**
     * 根据记录表ID批量逻辑删除
     * @param ids 记录表ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void softDeleteByIds(List<Long> ids) {
        arOrderRecordRepoProc.updateDeleteFlagById(ids);
        arOrderDtlRecordRepoProc.updateDeleteFlagByMasId(ids);
    }

    /**
     * 根据记录表ID批量物理删除
     * @param ids 记录表ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteByIds(List<Long> ids) {
        arOrderRecordRepo.deleteAllByIdIn(ids);
        arOrderDtlRecordRepo.deleteAllByMasIdIn(ids);
    }

    /**
     * 根据来源单号批量逻辑删除
     * @param sourceNos 来源单号
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void softDeleteBySourceNos(List<String> sourceNos) {
        List<Long> arOrderRecordIds = arOrderRecordRepoProc.selectIdBySourceNo(sourceNos);
        if (CollectionUtil.isNotEmpty(arOrderRecordIds)){
            arOrderRecordRepoProc.updateDeleteFlagById(arOrderRecordIds);
            arOrderDtlRecordRepoProc.updateDeleteFlagByMasId(arOrderRecordIds);
        }
    }

    /**
     * 根据来源单号批量物理删除
     * @param sourceNos 来源单号
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteBySourceNos(List<String> sourceNos) {
        List<Long> arOrderRecordIds = arOrderRecordRepoProc.selectIdBySourceNo(sourceNos);
        if (CollectionUtil.isNotEmpty(arOrderRecordIds)){
            arOrderRecordRepo.deleteAllByIdIn(arOrderRecordIds);
            arOrderDtlRecordRepo.deleteAllByMasIdIn(arOrderRecordIds);
        }
    }

}
