package com.elitesland.fin.utils.excel.convert;


import cn.hutool.core.collection.ConcurrentHashSet;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

public class ExcelConverterManager {
    // converter class and converter map
    private final Map<Class<? extends Converter>, Converter> converterMap = new ConcurrentHashMap<>();
    
    // The singleton object of ExcelConverterManager
    private static volatile ExcelConverterManager instance;
    
    // record which one converter cannot refresh
    private static final Set<Converter> notRequiredRefreshSet = new ConcurrentHashSet<>();
    
    // a volatile field to mark the converter whether refreshing
    private volatile boolean isRefreshing = false;
    
    
    public static ExcelConverterManager instance() {
        return instance(true);
    }
    
    public static ExcelConverterManager instance(boolean refresh) {
        if(instance == null) {
            synchronized (ExcelConverterManager.class) {
                if(instance == null) {
                    instance = new ExcelConverterManager();
                }
            }
        }
        if (refresh) {
            instance.isRefreshing = true;
            notRequiredRefreshSet.clear();
            return instance;
        }
        return instance;
    }
    
    public static void refresh() {
        if(instance == null || instance.isRefreshing) {
            return;
        }
        instance.isRefreshing = true;
        notRequiredRefreshSet.clear();
    }
    
    public Converter getConverter(Class<? extends Converter> clz) throws NoSuchMethodException, InvocationTargetException, InstantiationException, IllegalAccessException {
        if(converterMap.containsKey(clz)) {
            return refreshConverter(converterMap.get(clz));
        }
        
        Converter converter = initConverter(clz);
        converterMap.put(clz, converter);
        return converter;
    }
    
    private Converter initConverter(Class<? extends Converter> clz) throws NoSuchMethodException, InvocationTargetException, InstantiationException, IllegalAccessException {
        Converter converter = clz.getDeclaredConstructor().newInstance();
        converter.initBaseValue();
        return converter;
    }
    
    private Converter refreshConverter(Converter converter) {
        if(notRequiredRefreshSet.contains(converter)) {
            return converter;
        }
        
        converter.refreshBaseValue();
        notRequiredRefreshSet.add(converter);
        if(notRequiredRefreshSet.size() == converterMap.size()) {
            instance.isRefreshing = false;
        }
        return converter;
    }
}
