package com.elitesland.fin.application.service.accountingengine;

import cn.hutool.core.lang.Assert;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.application.facade.param.accountingengine.ManualProposedParam;
import com.elitesland.fin.common.FinAccountEngineConditionEnum;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.accountingengine.*;
import com.elitesland.fin.repo.accountingengine.*;
import com.elitesland.fin.utils.DataCompareUtil;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Service
@RequiredArgsConstructor
@Slf4j
public class JournalGenerateServiceImpl implements JournalGenerateService {
    private final FinAccountEngineRepo finAccountEngineRepo;

    private final FinAccountEngineRepoProc finAccountEngineRepoProc;

    private final FinAccountEngineDetailsRepo finAccountEngineDetailsRepo;

    private final FinAccEngDetConditionRepo finAccEngDetConditionRepo;

    private final FinAccEngDetDataRepo finAccEngDetDataRepo;

    private final FinAccEngDetDataLineRepo finAccEngDetDataLineRepo;

    private final FinEventTableRepo finEventTableRepo;

    private final FinEventTableLineRepo finEventTableLineRepo;

    private final FinEventTableConditionRepo finEventTableConditionRepo;

    private final FinJournalLogRepo finJournalLogRepo;

    private final FinSetOfBookOuRepo finSetOfBookOuRepo;

    private final FinSetOfBookRepo finSetOfBookRepo;

    private final FinSetOfBookLineRepo finSetOfBookLineRepo;

    private final FinFastCodeLineRepoProc finFastCodeLineRepoProc;

    private final FinFastCodeRepoProc finFastCodeRepoProc;

    private final FinFlexibleRepoProc finFlexibleRepoProc;

    private final FinFlexibleValueRepoProc finFlexibleValueRepoProc;

    private final FinSobAccountPeriodRepo finSobAccountPeriodRepo;

    private final TenantClientProvider tenantClientProvider;

    private final FinAccountEngineJdbcService finAccountEngineJdbcService;
    @Value("${fin.journal.batchSaveSize:500}")
    private int batchSize = 500;

    @Override
    public void generateJournal(List<Long> engineIds) {
        //参数校验
        //checkGenerateJournalParam(finAccountEngineParam);
        List<FinAccountEngineDO> finAccountEngineDOS = finAccountEngineRepo.findAllByIdInAndStatus(engineIds,
                UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        if (CollectionUtils.isEmpty(finAccountEngineDOS)) {
            log.error("自动拟定，查不到启用状态的会计引擎");
            saveFinJournalLog(null, "自动拟定，查不到启用状态的会计引擎");
            throw new BusinessException("查不到启用状态的会计引擎");
        }
        List<Long> eventTableIds = finAccountEngineDOS.stream().map(v -> v.getEventTableId()).collect(Collectors.toList());
        //事件表单
        List<FinEventTableDO> eventTableDOS = finEventTableRepo.findAllByIdInAndStatus(eventTableIds,
                UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        if (CollectionUtils.isEmpty(eventTableDOS)) {
            log.error("自动拟定失败，查不到启用状态的事件表单，会计引擎IDS：{}", engineIds);
            saveFinJournalLog(null, String.format("自动拟定，查不到启用状态的事件表单,会计引擎IDS：{}", engineIds));
            throw new BusinessException("查不到启用状态的事件表单");
        }
        Map<Long, FinEventTableDO> eventTableMap = eventTableDOS.stream().collect(Collectors.toMap(FinEventTableDO::getId, v -> v, (v1, v2) -> v2));
        for (FinAccountEngineDO finAccountEngineDO : finAccountEngineDOS) {
            FinEventTableDO finEventTableDO = eventTableMap.get(finAccountEngineDO.getEventTableId());
            if (finEventTableDO == null) {
                log.error("自动拟定失败，查不到启用状态的事件表单，会计引擎ID：{}", finAccountEngineDO.getId());
                saveFinJournalLog(null, String.format("自动拟定，查不到启用状态的事件表单,事件表单：{}", finAccountEngineDO.getEventTable()));
                continue;
            }
            try {
                generateJournalInner(finAccountEngineDO, finEventTableDO, false, null);
            } catch (Exception e) {
                log.error("自动拟定失败,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), e);
            }
        }
    }

    @Override
    public void manualProposed(ManualProposedParam manualProposedParam) {
        Assert.notNull(manualProposedParam, "参数为空");
        Assert.notBlank(manualProposedParam.getEventTableName(), "表名为空");
        Assert.notNull(manualProposedParam.getDocIds(), "单据ID为空");
        Assert.notNull(manualProposedParam.getDocNums(), "单据编码为空");
        Assert.isTrue(CollectionUtils.isNotEmpty(manualProposedParam.getDocIds()), "单据ID为空");
        Assert.isTrue(CollectionUtils.isNotEmpty(manualProposedParam.getDocNums()), "单据编码为空");
        Assert.equals(manualProposedParam.getDocIds().size(), manualProposedParam.getDocNums().size(), "单据ID和单据编码不匹配");
        //根据表名查询事件表单
        //事件表单
        FinEventTableDO finEventTableDO = finEventTableRepo.findByMasTableAndStatus(manualProposedParam.getEventTableName(),
                UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        if (finEventTableDO == null) {
            log.error("手动拟定失败，查不到启用状态的事件表单");
            saveFinJournalLog(null, "自动拟定，查不到启用状态的事件表单");
            throw new BusinessException("查不到启用状态的事件表单");
        }
        //根据事件表单查询会计引擎
        //会计引擎
        List<FinAccountEngineDO> finAccountEngineDOs = finAccountEngineRepo.findByEventTableAndStatus(finEventTableDO.getEventTable(),
                UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        if (CollectionUtils.isEmpty(finAccountEngineDOs)) {
            log.error("手动拟定失败，查不到启用状态的会计引擎");
            saveFinJournalLog(null, "手动拟定，查不到启用状态的会计引擎");
            throw new BusinessException("查不到启用状态的会计引擎");
        }
        //处理每一个会计引擎
        for (FinAccountEngineDO finAccountEngineDO : finAccountEngineDOs) {
            try {
                generateJournalInner(finAccountEngineDO, finEventTableDO, true, manualProposedParam);
            } catch (Exception e) {
                log.error("手动拟定失败,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), e);
            }
        }
    }

    private void generateJournalInner(FinAccountEngineDO finAccountEngineDO, FinEventTableDO finEventTableDO, boolean manualFlag, ManualProposedParam manualProposedParam) {
        List<Map> saveMapList = new ArrayList<>();
        Set<Long> proposingDocIds = new HashSet<>();
        Set<Long> successProposingDocIds = new HashSet<>();
        //查询配置
        FinAccountEngineConfigDO finAccountEngineConfigDO = null;
        Connection eventTableConnection = null;
        Connection journalConnection = null;
        try {
            try {
                finAccountEngineConfigDO = queryFinAccountEngineConfigData(finAccountEngineDO, finEventTableDO);
            } catch (Exception e) {
                log.error("查询会计引擎配置数据异常,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), e);
                throw new BusinessException("查询会计引擎配置异常," + e.getMessage());
            }
            finAccountEngineConfigDO.setFinAccountEngineDO(finAccountEngineDO);
            finAccountEngineConfigDO.setFinEventTableDO(finEventTableDO);
            finAccountEngineConfigDO.setManualFlag(manualFlag);
            if (manualFlag) {
                finAccountEngineConfigDO.setDocIds(manualProposedParam.getDocIds());
                finAccountEngineConfigDO.setDocNums(manualProposedParam.getDocNums());
            }
            //获取事件表单数据库链接
            try {
                eventTableConnection = finAccountEngineJdbcService.getEventTableConnection(finAccountEngineConfigDO.getFinEventTableDO());
            } catch (Exception e) {
                log.error("获取事件表单数据库连接异常,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), e);
                throw new BusinessException("获取事件表单数据库连接异常");
            }
            //获取会计分录数据库连接
            try {
                journalConnection = finAccountEngineJdbcService.getJournalConnection(finAccountEngineConfigDO.getFinAccountEngineDO());
            } catch (Exception e) {
                log.error("获取会计分录数据库连接异常,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), e);
                throw new BusinessException("获取会计分录数据库连接异常");
            }
            //查询数据begin
            List<Map> dataList = null;
            try {
                dataList = finAccountEngineJdbcService.queryEventTableData(eventTableConnection, finAccountEngineConfigDO, manualFlag, finAccountEngineConfigDO.getDocIds());
            } catch (Exception e) {
                log.error("查询事件表单数据异常,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), e);
                throw new BusinessException("查询事件表单数据异常");
            }
            if (CollectionUtils.isEmpty(dataList)) {
                log.error("未查询到匹配的单据,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId());
                String checkDocIdResult = manualProposedParam.getDocIds().stream().map(Objects::toString).collect(Collectors.joining("、", "[", "]"));
                String checkDocNumResult = manualProposedParam.getDocNums().stream().map(Objects::toString).collect(Collectors.joining("、", "[", "]"));
                throw new BusinessException("未查询到匹配的单据,单据ID:"+checkDocIdResult+";单据编码:"+checkDocNumResult);
                //throw new BusinessException("未查询到匹配的单据,单据编码:"+checkDocNumResult);
            }
            //是手动拟定需要删除原来已经生成的会计凭证
            if (manualFlag) {
                finAccountEngineJdbcService.deleteExistJournal(eventTableConnection, journalConnection, finAccountEngineConfigDO, finAccountEngineConfigDO.getDocNums());
            }
            //生成会计分录数据
            generateFinJournalMap(eventTableConnection, finAccountEngineConfigDO, dataList, saveMapList, proposingDocIds, successProposingDocIds);
            //分录入库
            saveFinJournal(finAccountEngineDO, finEventTableDO, journalConnection, saveMapList, proposingDocIds, successProposingDocIds, manualFlag);
            //批量更新拟定状态-拟定成功
            try {
                finAccountEngineJdbcService.batchUpdateProposedStatus(finAccountEngineConfigDO.getFinEventTableDO(), eventTableConnection, successProposingDocIds, UdcEnum.DOC_PROPOSED_STATUS_PROPOSED_SUCCESS.getValueCode());
            } catch (SQLException e) {
                log.error("批量更新单据拟定成功异常,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), e);
                throw new BusinessException("批量更新单据拟定成功异常");
            }
            //批量更新拟定状态-拟定失败
            proposingDocIds.removeAll(successProposingDocIds);
            try {
                finAccountEngineJdbcService.batchUpdateProposedStatus(finAccountEngineConfigDO.getFinEventTableDO(), eventTableConnection, proposingDocIds, UdcEnum.DOC_PROPOSED_STATUS_PROPOSED_FAIL.getValueCode());
            } catch (SQLException e) {
                log.error("批量更新单据拟定失败异常,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), e);
                throw new BusinessException("批量更新单据拟定失败异常");
            }
        } catch (Exception e) {
            saveFinJournalLog(null, String.format("%s,事件表单【%s】",
                    e.getMessage(), finEventTableDO.getEventTable()));
            //更新单据为拟定失败
            try {
                finAccountEngineJdbcService.batchUpdateProposedStatus(finAccountEngineConfigDO.getFinEventTableDO(), eventTableConnection, proposingDocIds, UdcEnum.DOC_PROPOSED_STATUS_PROPOSED_FAIL.getValueCode());
            } catch (SQLException sqle) {
                log.error("批量更新单据拟定失败异常,事件表单:{},会计引擎ID：{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), sqle);
                throw new BusinessException("批量更新单据拟定失败异常");
            }
            throw e;
        } finally {
            //释放事件表单数据库连接
            finAccountEngineJdbcService.closeConnection(eventTableConnection);
            //释放会计分录数据库连接
            finAccountEngineJdbcService.closeConnection(journalConnection);
        }
    }

    private void generateFinJournalMap(Connection eventTableConnection, FinAccountEngineConfigDO finAccountEngineConfigDO, List<Map> dataList, List<Map> saveMapList, Set<Long> proposingDocIds, Set<Long> successProposingDocIds) {
        //查询会计引擎相关数据
        boolean manualFlag = finAccountEngineConfigDO.getManualFlag() == null ? false : finAccountEngineConfigDO.getManualFlag();
        FinEventTableDO finEventTableDO = finAccountEngineConfigDO.getFinEventTableDO();
        FinAccountEngineDO finAccountEngineDO = finAccountEngineConfigDO.getFinAccountEngineDO();
        String docNumKey = finAccountEngineConfigDO.getFinAccountEngineDO().getColumnDocNum();
        Set<Long> updatedDocIds = new HashSet<>();
        //根据整单分组
        Map<Object, List<Map>> dataGroup = dataList.stream().collect(Collectors.groupingBy(v -> {
            return v.get("main_id");
        }));
        Set<Object> dataKeys = dataGroup.keySet();
        String docNum = null;
        JournalLogInfo journalLogInfo = null;
        for (Object key : dataKeys) {
            journalLogInfo = new JournalLogInfo(finEventTableDO.getEventTable(), finAccountEngineDO.getId());
            try {
                int successDetailDataSize = 0;
                List<Map> perDocDataMap = new ArrayList<>();
                //初始化日志
                journalLogInfo.setManualFlag(manualFlag);
                List<Map> resultList = dataGroup.get(key);

                docNum =  resultList.get(0).get(docNumKey).toString();
                if (docNum == null) {
                    journalLogInfo.setErrMsg("单据编码为空");
                    saveFinJournalLogNew(journalLogInfo);
                    continue;
                }
                journalLogInfo.setDocNum(docNum);
                //判断单据公司编码是否存在
                //String ouCode = (String) resultList.get(0).get(finAccountEngineConfigDO.getFinAccountEngineDO().getColumnCompany());
                String ouCode = String.valueOf(resultList.get(0).get(finAccountEngineConfigDO.getFinAccountEngineDO().getColumnCompany()));
                if (ouCode == null) {
                    journalLogInfo.setErrMsg("公司编码为空");
                    saveFinJournalLogNew(journalLogInfo);
                    continue;
                }
                journalLogInfo.setOuCode(ouCode);
                //收集拟定中的单据
                long docId = (long) key;
                journalLogInfo.setDocId(docId);
                proposingDocIds.add(docId);
                //修改业务单据主表状态为拟定中
                try {
                    if (!updatedDocIds.contains(docId)) {
                        finAccountEngineJdbcService.updateProposedStatus(finEventTableDO, eventTableConnection, docId, UdcEnum.DOC_PROPOSED_STATUS_PROPOSING.getValueCode());
                        updatedDocIds.add(docId);
                    }
                } catch (Exception e) {
                    journalLogInfo.setErrMsg(e.getMessage());
                    saveFinJournalLogNew(journalLogInfo);
                    continue;
                }
                for (Map result : resultList) {
                    //获取单据明细ID
                    if(!finAccountEngineConfigDO.isMasTableOnly()){
                        journalLogInfo.setDetailDocId((long) result.get("detail_id"));
                    }
                    boolean hitFinAccountEngineDetailFlag=false;
                    //会计引擎详情
                    for (FinAccountEngineDetailsDO finAccountEngineDetailsDO : finAccountEngineConfigDO.getFinAccountEngineDetailsDOList()) {
                        journalLogInfo.setAccountEngine(finAccountEngineDetailsDO.getAccountEngine());
                        //会计引擎详情条件
                        List<FinAccEngDetConditionDO> finAccEngDetConditionDOList = finAccountEngineConfigDO.getFinAccEngDetConditionDOList().stream()
                                .filter(finAccEngDetDataDO -> finAccountEngineDetailsDO.getId().equals(finAccEngDetDataDO.getMasId()))
                                .collect(Collectors.toList());
                        //校验数据是否满足会计引擎
                        if (!checkDataMeetCondition(result, finAccEngDetConditionDOList)) {
                            log.info("单据不满足会计明细条件，事件表单：{}，单据编码：{}，单据明细ID：{}，会计引擎明细：{}",
                                    finEventTableDO.getEventTable(), docNum, journalLogInfo.getDetailDocId(), finAccountEngineDetailsDO.getAccountEngine());
                            continue;
                        }
                        hitFinAccountEngineDetailFlag=true;
                        //数据行详情
                        List<FinAccEngDetDataDO> finAccEngDetDataDOListItem = finAccountEngineConfigDO.getFinAccEngDetDataDOList()
                                .stream()
                                .filter(finAccEngDetDataDO -> finAccountEngineDetailsDO.getId().equals(finAccEngDetDataDO.getMasId()))
                                .collect(Collectors.toList());
                        if (CollectionUtils.isEmpty(finAccEngDetDataDOListItem)) {
                            journalLogInfo.setErrMsg("会计引擎数据行未配置");
                            saveFinJournalLogNew(journalLogInfo);
                            break;
                        }
                        List<Map> detDataList = new ArrayList<>();
                        boolean needCycleAccountEngineDetails = true;
                        boolean setOfBookOuFoundYet = false;
                        for (FinAccEngDetDataDO finAccEngDetDataDO : finAccEngDetDataDOListItem) {
                            journalLogInfo.setDetDataName(finAccEngDetDataDO.getName());
                            //判断账套是否命中单据公司
                            if (!checkFinSetOfBookOu(finAccountEngineConfigDO, ouCode, docNum, finAccountEngineConfigDO.getFinAccountEngineDO(), finAccountEngineDetailsDO, finAccEngDetDataDO)) {
                                continue;
                            }
                            setOfBookOuFoundYet = true;
                            //数据映射
                            List<FinAccEngDetDataLineDO> finAccEngDetDataLineDOListItem = finAccountEngineConfigDO.getFinAccEngDetDataLineDOList().stream()
                                    .filter(finAccEngDetDataLineDO -> finAccEngDetDataDO.getId().equals(finAccEngDetDataLineDO.getMasId()))
                                    .collect(Collectors.toList());
                            if (CollectionUtils.isEmpty(finAccEngDetDataLineDOListItem)) {
                                journalLogInfo.setErrMsg("查不到数据映射");
                                saveFinJournalLogNew(journalLogInfo);
                                needCycleAccountEngineDetails = false;
                                break;
                            }
                            Map saveMap = new HashMap();
                            saveMap.put("detDataId", finAccEngDetDataDO.getId());
                            saveMap.put("main_id", result.get("main_id"));
                            for (FinAccEngDetDataLineDO finAccEngDetDataLineDO : finAccEngDetDataLineDOListItem) {
                                journalLogInfo.setColumnName(finAccEngDetDataLineDO.getColumnName());
                                if (!checkConstant(finAccEngDetDataLineDO, docNum, finAccountEngineConfigDO.getFinAccountEngineDO(), finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap, journalLogInfo)) {
                                    needCycleAccountEngineDetails = false;
                                    break;
                                }
                                if (!checkColumn(finAccEngDetDataLineDO, result, docNum, finAccountEngineConfigDO.getFinAccountEngineDO(), finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap, journalLogInfo)) {
                                    needCycleAccountEngineDetails = false;
                                    break;
                                }
                                if (!checkFastCode(finAccEngDetDataLineDO, finAccountEngineConfigDO, result, docNum, finAccountEngineConfigDO.getFinAccountEngineDO(), finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap, journalLogInfo)) {
                                    needCycleAccountEngineDetails = false;
                                    break;
                                }
                                if (!checkFlexible(finAccEngDetDataLineDO, finAccountEngineConfigDO, docNum, finAccountEngineConfigDO.getFinAccountEngineDO(), finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap, journalLogInfo)) {
                                    needCycleAccountEngineDetails = false;
                                    break;
                                }
                                if (!checkAccountPeriod(finAccEngDetDataLineDO, result, finAccountEngineConfigDO, ouCode, docNum, finAccountEngineConfigDO.getFinAccountEngineDO(), finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap, journalLogInfo)) {
                                    needCycleAccountEngineDetails = false;
                                    break;
                                }
                            }
                            detDataList.add(saveMap);
                        }
                        //数据行没有找到匹配的账套记录错误日志
                        if (!setOfBookOuFoundYet) {
                            saveFinJournalLog(docNum,
                                    String.format("未查询到匹配的公司账套，事件表单【%s】,会计引擎明细【%s】,单据公司编码【%s】",
                                            finAccountEngineDO.getEventTable(),
                                            finAccountEngineDetailsDO.getAccountEngine(),
                                            ouCode));
                            needCycleAccountEngineDetails = false;
                        }
                        if (!needCycleAccountEngineDetails) {
                            break;
                        }
                        //检验借方、贷方金额是否相等
                        if (!CollectionUtils.isEmpty(detDataList)) {
                            if (!checkDebtAndCreditSumEqual(detDataList, journalLogInfo)) {
                                break;
                            }
                            perDocDataMap.addAll(detDataList);
                            successDetailDataSize++;
                        }
                        break;
                    }
                    if(!hitFinAccountEngineDetailFlag){
                        journalLogInfo.setErrMsg("单据不满足会计明细条件");
                        journalLogInfo.setAccountEngine(null);
                        journalLogInfo.setDetDataName(null);
                        journalLogInfo.setColumnName(null);
                        saveFinJournalLogNew(journalLogInfo);
                    }
                }
                if (successDetailDataSize == resultList.size()) {
                    saveMapList.addAll(perDocDataMap);
                }
            } catch (Exception e) {
                log.error("生成会计凭证异常执行异常", e);
                journalLogInfo.setErrMsg("生成会计凭证异常," + e.getMessage());
                saveFinJournalLogNew(journalLogInfo);
            }

        }
    }

    private void saveFinJournalLog(String docNum, String log) {
        FinJournalLogDO finJournalLogDo = new FinJournalLogDO();
        finJournalLogDo.setDocNum(docNum);
        finJournalLogDo.setLog(log);
        finJournalLogRepo.save(finJournalLogDo);
    }

    private void saveFinJournalLogNew(JournalLogInfo journalLogInfo) {
        String docNum = journalLogInfo.getDocNum();
        String proposedType = "";
        if (journalLogInfo.getManualFlag() != null) {
            if (journalLogInfo.getManualFlag()) {
                proposedType = "手动拟定";
            } else {
                proposedType = "自动拟定";
            }
        }
        String formatMsg = String.format("%s,%s,事件表单【%s】,单据明细ID【%s】,会计引擎明细【%s】,数据行【%s】,列名称【%s】",
                proposedType,
                journalLogInfo.getErrMsg(),
                journalLogInfo.getEventTable(),
                journalLogInfo.getDetailDocId(),
                journalLogInfo.getAccountEngine(),
                journalLogInfo.getDetDataName(),
                journalLogInfo.getColumnName());
        FinJournalLogDO finJournalLogDo = new FinJournalLogDO();
        finJournalLogDo.setDocNum(docNum);
        finJournalLogDo.setLog(formatMsg);
        finJournalLogRepo.save(finJournalLogDo);
        log.error("会计引擎生成会计凭证异常日志：{}", JSON.toJSONString(finJournalLogDo));
    }

    private boolean checkDataMeetCondition(Map data, List<FinAccEngDetConditionDO> finAccEngDetConditionDOList) {
        if (CollectionUtils.isEmpty(finAccEngDetConditionDOList)) {
            return true;
        }
  /*      finAccEngDetConditionDOList.stream().map(item ->
                        buildWhere(item.getConditionType(),
                                item.getTableName(),
                                item.getColumnName(),
                                item.getValueFrom(),
                                item.getValueTo()))
                .collect(Collectors.toList());*/
        for (FinAccEngDetConditionDO finAccEngDetConditionDO : finAccEngDetConditionDOList) {
            //Map<String, String> eventTableConditionUdc = udcProvider.getValueMapByUdcCode(UdcEnum.EVENT_TABLE_CONDITION_EQUAL.getModel(), UdcEnum.EVENT_TABLE_CONDITION_EQUAL.getCode());
            String conditionType = finAccEngDetConditionDO.getConditionType();
            String tableName = finAccEngDetConditionDO.getTableName();
            String columnName = finAccEngDetConditionDO.getColumnName();
            String valueFrom = finAccEngDetConditionDO.getValueFrom();
            String valueTo = finAccEngDetConditionDO.getValueTo();
            Object value = data.get(columnName);
            boolean testResult = true;
            if (FinAccountEngineConditionEnum.EQUAL.name().equals(conditionType)) {
                if (StringUtils.isBlank(valueFrom)) {
                    continue;
                }
                if (value == null) {
                    return false;
                }
                if (value instanceof Number) {
                    testResult = DataCompareUtil.numberEqual((Number) value, valueFrom);
                } else if (value instanceof Boolean) {
                    testResult = DataCompareUtil.booleanEqual((Boolean) value, valueFrom);
                } else if (value instanceof String) {
                    testResult = DataCompareUtil.stringEqual((String) value, valueFrom);
                } else if (value instanceof LocalDateTime) {
                    testResult = DataCompareUtil.dateTimeEqual((LocalDateTime) value, valueFrom);
                }
            } else if (FinAccountEngineConditionEnum.GREATER.name().equals(conditionType)) {
                if (StringUtils.isBlank(valueFrom)) {
                    continue;
                }
                if (value == null) {
                    return false;
                }
                if (value instanceof Number) {
                    testResult = DataCompareUtil.numberGreater((Number) value, valueFrom);
                } else if (value instanceof LocalDateTime) {
                    testResult = DataCompareUtil.dateTimeGreater((LocalDateTime) value, valueFrom);
                }
            } else if (FinAccountEngineConditionEnum.LESS.name().equals(conditionType)) {
                if (StringUtils.isBlank(valueFrom)) {
                    continue;
                }
                if (value == null) {
                    return false;
                }
                if (value instanceof Number) {
                    testResult = DataCompareUtil.numberLess((Number) value, valueFrom);
                } else if (value instanceof LocalDateTime) {
                    testResult = DataCompareUtil.dateTimeLess((LocalDateTime) value, valueFrom);
                }
            } else if (FinAccountEngineConditionEnum.GREATER_EQUAL.name().equals(conditionType)) {
                if (StringUtils.isBlank(valueFrom)) {
                    continue;
                }
                if (value == null) {
                    return false;
                }
                if (value instanceof Number) {
                    testResult = DataCompareUtil.numberGreaterEqual((Number) value, valueFrom);
                } else if (value instanceof LocalDateTime) {
                    testResult = DataCompareUtil.dateTimeGreaterEqual((LocalDateTime) value, valueFrom);
                }

            } else if (FinAccountEngineConditionEnum.LESS_EQUAL.name().equals(conditionType)) {
                if (StringUtils.isBlank(valueFrom)) {
                    continue;
                }
                if (value == null) {
                    return false;
                }
                if (value instanceof Number) {
                    testResult = DataCompareUtil.numberLessEqual((Number) value, valueFrom);
                } else if (value instanceof LocalDateTime) {
                    testResult = DataCompareUtil.dateTimeLessEqual((LocalDateTime) value, valueFrom);
                }
            } else if (FinAccountEngineConditionEnum.IS_NULL.name().equals(conditionType)) {
                testResult = (value == null)||(value instanceof String && StringUtils.isBlank((String)value));
            } else if (FinAccountEngineConditionEnum.IS_NOT_NULL.name().equals(conditionType)) {
                testResult = (value != null && (value instanceof String?StringUtils.isNoneBlank((String)value):true));
            } else if (FinAccountEngineConditionEnum.IN.name().equals(conditionType)) {
                if (StringUtils.isBlank(valueFrom)) {
                    continue;
                }
                if (value == null) {
                    return false;
                }
                if (value instanceof Number) {
                    testResult = DataCompareUtil.numberIn((Number) value, valueFrom);
                } else if (value instanceof String) {
                    testResult = DataCompareUtil.StringIn((String) value, valueFrom);
                }
            } else if (FinAccountEngineConditionEnum.NOT_IN.name().equals(conditionType)) {
                if (StringUtils.isBlank(valueFrom)) {
                    continue;
                }
                if (value == null) {
                    return true;
                }
                if (value instanceof Number) {
                    testResult = DataCompareUtil.numberNotIn((Number) value, valueFrom);
                } else if (value instanceof String) {
                    testResult = DataCompareUtil.StringNotIn((String) value, valueFrom);
                }
            } else if (FinAccountEngineConditionEnum.BETWEEN_AND.name().equals(conditionType)) {
                if (StringUtils.isBlank(valueFrom) || StringUtils.isBlank(valueTo)) {
                    continue;
                }
                if (value == null) {
                    return false;
                }
                if (value instanceof Number) {
                    testResult = DataCompareUtil.numberBetweenAnd((Number) value, valueFrom, valueTo);
                } else if (value instanceof LocalDateTime) {
                    testResult = DataCompareUtil.dateTimeBetweenAnd((LocalDateTime) value, valueFrom, valueTo);
                }
            }
            if (!testResult) {
                return false;
            }
        }
        return true;
    }

    private boolean checkFinSetOfBookOu(FinAccountEngineConfigDO finAccountEngineConfigDO,
                                        String ouCode,
                                        String docNum,
                                        FinAccountEngineDO finAccountEngineDO,
                                        FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                        FinAccEngDetDataDO finAccEngDetDataDO) {
        String sobCode = finAccEngDetDataDO.getSobCode();
        FinSetOfBookDO finSetOfBookDOResult = finAccountEngineConfigDO.getFinSetOfBookDOList().stream()
                .filter(finSetOfBookDO -> finSetOfBookDO.getSobCode().equals(sobCode))
                .findFirst()
                .orElse(null);
        if (finSetOfBookDOResult == null) {
            log.info("查不到启用状态的账套，事件表单：{},会计引擎明细：{},数据行：{},单据公司编码：{}",
                    finAccountEngineDO.getEventTable(), finAccountEngineDetailsDO.getAccountEngine(), finAccEngDetDataDO.getName(),
                    ouCode);
            return false;
        }
        //帐套公司
        List<FinSetOfBookOuDO> finSetOfBookOuDOListItem = finAccountEngineConfigDO.getFinSetOfBookOuDOList().stream()
                .filter(finSetOfBookOuDO -> finSetOfBookDOResult.getId().equals(finSetOfBookOuDO.getMasId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(finSetOfBookOuDOListItem)) {
            log.info("查不到账套公司，事件表单：{},会计引擎明细：{},数据行：{},单据公司编码：{}",
                    finAccountEngineDO.getEventTable(), finAccountEngineDetailsDO.getAccountEngine(), finAccEngDetDataDO.getName(),
                    ouCode);
            return false;
        }
        FinSetOfBookOuDO finSetOfBookOuDOResult = finSetOfBookOuDOListItem.stream()
                .filter(finSetOfBookOuDO -> ouCode.equals(finSetOfBookOuDO.getOuCode()) || Objects.equals(ouCode,String.valueOf(finSetOfBookOuDO.getOuId())))
                .findFirst()
                .orElse(null);
        if (finSetOfBookOuDOResult == null) {
            log.info("未查询到匹配的公司账套，事件表单：{},会计引擎明细：{},数据行：{},单据公司编码：{}",
                    finAccountEngineDO.getEventTable(), finAccountEngineDetailsDO.getAccountEngine(), finAccEngDetDataDO.getName(),
                    ouCode);
            return false;
        }
        return true;
    }

    private FinAccountEngineConfigDO queryFinAccountEngineConfigData(FinAccountEngineDO finAccountEngineDO, FinEventTableDO finEventTableDO) {
        //事件表单明细
        List<FinEventTableLineDO> finEventTableLineDOList = finEventTableLineRepo.findAllByMasId(finEventTableDO.getId());
        Assert.notEmpty(finEventTableLineDOList, "查不到事件表单明细");

        //事件表单条件
        List<FinEventTableConditionDO> conditionDOListAll = finEventTableConditionRepo.findAllByMasId(finEventTableDO.getId());

        //会计引擎明细
        List<FinAccountEngineDetailsDO> finAccountEngineDetailsDOList = finAccountEngineDetailsRepo
                .findAllByMasIdAndActiveFlag(finAccountEngineDO.getId(), Boolean.TRUE)
                .stream()
                .sorted(Comparator.comparing(FinAccountEngineDetailsDO::getSerialNum).reversed())
                .collect(Collectors.toList());
        Assert.notEmpty(finAccountEngineDetailsDOList, "查不到启用的会计引擎明细");

        List<Long> finAccountEngineDetailsDOIds = finAccountEngineDetailsDOList.stream().map(FinAccountEngineDetailsDO::getId).collect(Collectors.toList());

        //会计引擎条件
        List<FinAccEngDetConditionDO> finAccEngDetConditionDOList = finAccEngDetConditionRepo.findAllByMasIdIn(finAccountEngineDetailsDOIds);

        //数据行
        List<FinAccEngDetDataDO> finAccEngDetDataDOList = finAccEngDetDataRepo.findAllByMasIdIn(finAccountEngineDetailsDOIds).stream()
                .sorted(Comparator.comparing(FinAccEngDetDataDO::getSerialNum).reversed())
                .collect(Collectors.toList());
        Assert.notEmpty(finAccEngDetDataDOList, "查不到数据行详情");

        //数据映射
        List<Long> finAccEngDetDataDOIds = finAccEngDetDataDOList.stream().map(FinAccEngDetDataDO::getId).collect(Collectors.toList());
        List<FinAccEngDetDataLineDO> finAccEngDetDataLineDOList = finAccEngDetDataLineRepo.findAllByMasIdIn(finAccEngDetDataDOIds);
        Assert.notEmpty(finAccEngDetDataLineDOList, "查不到数据映射");

        //快码
        List<String> fastCodes = finAccEngDetDataLineDOList.stream().filter(item -> StringUtils.isNotEmpty(item.getFastCode())).map(FinAccEngDetDataLineDO::getFastCode).collect(Collectors.toList());
        List<FinFastCodeDO> finFastCodeDOList = finFastCodeRepoProc.findAllByFastCodeInAndStatus(fastCodes, UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        //快码明细
        List<Long> finFastCodeDOIds = finFastCodeDOList.stream().map(FinFastCodeDO::getId).collect(Collectors.toList());
        List<FinFastCodeLineDO> finFastCodeLineDOList = finFastCodeLineRepoProc.findByMasIdIn(finFastCodeDOIds);

        //帐套
        List<String> sobCodes = finAccEngDetDataDOList.stream().map(FinAccEngDetDataDO::getSobCode).collect(Collectors.toList());
        List<FinSetOfBookDO> finSetOfBookDOList = finSetOfBookRepo.findAllBySobCodeInAndStatus(sobCodes, UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        Assert.notEmpty(finSetOfBookDOList, "查不到启用状态的帐套");

        //帐套公司
        List<Long> finSetOfBookDOIds = finSetOfBookDOList.stream().map(FinSetOfBookDO::getId).collect(Collectors.toList());
        List<FinSetOfBookOuDO> finSetOfBookOuDOList = finSetOfBookOuRepo.findAllByMasIdIn(finSetOfBookDOIds);
        Assert.notEmpty(finSetOfBookOuDOList, "查不到帐套帐套公司");

        //帐套明细
        List<FinSetOfBookLineDO> finSetOfBookLineDOList = finSetOfBookLineRepo.findAllByMasIdIn(finSetOfBookDOIds);
        Assert.notEmpty(finSetOfBookLineDOList, "查不到帐套明细");

        //会计期间
        List<FinSobAccountPeriodDO> finSobAccountPeriodDOList = finSobAccountPeriodRepo.findAllByStatus(UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_OPEN.getValueCode());

        List<String> flexibleCodes = finSetOfBookLineDOList.stream()
                .filter(finSetOfBookLineDO -> StringUtils.isNotEmpty(finSetOfBookLineDO.getFlexibleCode()))
                .map(FinSetOfBookLineDO::getFlexibleCode)
                .collect(Collectors.toList());

        //集值
        List<FinFlexibleDO> finFlexibleDOList = finFlexibleRepoProc.findAllByFlexibleCodeInAndStatus(flexibleCodes, UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        //集值明细
        List<Long> finSetOfBookLineDOIds = finFlexibleDOList.stream()
                .map(FinFlexibleDO::getId)
                .collect(Collectors.toList());

        List<FinFlexibleValueDO> finFlexibleValueDOList = finFlexibleValueRepoProc.findByMasIdInAndValidTime(finSetOfBookLineDOIds, LocalDateTime.now());


        FinAccountEngineConfigDO finAccountEngineConfigDO = new FinAccountEngineConfigDO();

        finAccountEngineConfigDO.setFinAccountEngineDO(finAccountEngineDO);
        finAccountEngineConfigDO.setFinEventTableDO(finEventTableDO);
        finAccountEngineConfigDO.setFinEventTableLineDOList(finEventTableLineDOList);
        finAccountEngineConfigDO.setFinEventTableConditionDOList(conditionDOListAll.stream().filter(v -> FinConstant.EVENT_TABLE_CONDITION_CATEGORY_DATA_SELECT.equals(v.getCategory())).collect(Collectors.toList()));
        finAccountEngineConfigDO.setFinEventTableProposedConditionList(conditionDOListAll.stream().filter(v -> FinConstant.EVENT_TABLE_CONDITION_CATEGORY_PROPOSED.equals(v.getCategory())).collect(Collectors.toList()));
        finAccountEngineConfigDO.setFinAccountEngineDetailsDOList(finAccountEngineDetailsDOList);
        finAccountEngineConfigDO.setFinAccEngDetConditionDOList(finAccEngDetConditionDOList);
        finAccountEngineConfigDO.setFinAccEngDetDataDOList(finAccEngDetDataDOList);
        finAccountEngineConfigDO.setFinAccEngDetDataLineDOList(finAccEngDetDataLineDOList);
        finAccountEngineConfigDO.setFinFastCodeDOList(finFastCodeDOList);
        finAccountEngineConfigDO.setFinFastCodeLineDOList(finFastCodeLineDOList);
        finAccountEngineConfigDO.setFinSetOfBookDOList(finSetOfBookDOList);
        finAccountEngineConfigDO.setFinSetOfBookOuDOList(finSetOfBookOuDOList);
        finAccountEngineConfigDO.setFinSetOfBookLineDOList(finSetOfBookLineDOList);
        finAccountEngineConfigDO.setFinSobAccountPeriodDOList(finSobAccountPeriodDOList);
        finAccountEngineConfigDO.setFinFlexibleDOList(finFlexibleDOList);
        finAccountEngineConfigDO.setFinFlexibleValueDOList(finFlexibleValueDOList);
        if(StringUtils.isNotBlank(finEventTableDO.getTableName())){
            finAccountEngineConfigDO.setMasTableOnly(false);
        }else{
            finAccountEngineConfigDO.setMasTableOnly(true);
        }
        return finAccountEngineConfigDO;
    }

    private boolean checkConstant(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                  String docNum,
                                  FinAccountEngineDO finAccountEngineDO,
                                  FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                  FinAccEngDetDataDO finAccEngDetDataDO,
                                  Map saveMap, JournalLogInfo journalLogInfo) {
        if (UdcEnum.SOURCE_TYPE_CONSTANT.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {
            if (StringUtils.isEmpty(finAccEngDetDataLineDO.getConstant())) {
                journalLogInfo.setErrMsg("常数列值为空");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }

            put(saveMap, finAccEngDetDataLineDO.getColumnName(), finAccEngDetDataLineDO.getConstant(), finAccEngDetDataLineDO);
        }
        return true;
    }

    private boolean checkColumn(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                Map result,
                                String docNum,
                                FinAccountEngineDO finAccountEngineDO,
                                FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                FinAccEngDetDataDO finAccEngDetDataDO,
                                Map saveMap, JournalLogInfo journalLogInfo) {
        if (UdcEnum.SOURCE_TYPE_COLUMN.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {
            if (!result.containsKey(finAccEngDetDataLineDO.getColumnSource())) {
                journalLogInfo.setErrMsg("表单数据未包含映射列");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            put(saveMap, finAccEngDetDataLineDO.getColumnName(), result.get(finAccEngDetDataLineDO.getColumnSource()) == null ? null : result.get(finAccEngDetDataLineDO.getColumnSource()), finAccEngDetDataLineDO);
        }
        return true;
    }

    private boolean checkFastCode(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                  FinAccountEngineConfigDO finAccountEngineConfigDO,
                                  Map result,
                                  String docNum,
                                  FinAccountEngineDO finAccountEngineDO,
                                  FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                  FinAccEngDetDataDO finAccEngDetDataDO,
                                  Map saveMap, JournalLogInfo journalLogInfo) {
        if (UdcEnum.SOURCE_TYPE_FAST_CODE.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {

            FinFastCodeDO finFastCodeDOResult = finAccountEngineConfigDO.getFinFastCodeDOList().stream()
                    .filter(finFastCodeDO -> finFastCodeDO.getFastCode().equals(finAccEngDetDataLineDO.getFastCode()))
                    .findFirst()
                    .orElse(null);

            if (finFastCodeDOResult == null) {
                journalLogInfo.setErrMsg("查不到启用状态的快码");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            List<FinFastCodeLineDO> finFastCodeLineDOListItem = finAccountEngineConfigDO.getFinFastCodeLineDOList().stream()
                    .filter(finFastCodeLineDO -> finFastCodeDOResult.getId().equals(finFastCodeLineDO.getMasId()))
                    .collect(Collectors.toList());
            if (CollectionUtils.isEmpty(finFastCodeLineDOListItem)) {
                journalLogInfo.setErrMsg("未查到快码明细");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }

            FinFastCodeLineDO finFastCodeLineDOResult = finFastCodeLineDOListItem.stream().filter(finFastCodeLineDO -> {
                if (StringUtils.isNotEmpty(finFastCodeLineDO.getConditionOne())) {
                    if (result.get(finFastCodeLineDO.getConditionOne()) == null) {
                        return false;
                    }
                    if (!finFastCodeLineDO.getConditionOneValue().equals(result.get(finFastCodeLineDO.getConditionOne()) + "")) {
                        return false;
                    }
                }

                if (StringUtils.isNotEmpty(finFastCodeLineDO.getConditionTwo())) {
                    if (result.get(finFastCodeLineDO.getConditionOne()) == null) {
                        return false;
                    }
                    if (!finFastCodeLineDO.getConditionTwoValue().equals(result.get(finFastCodeLineDO.getConditionTwo()) + "")) {
                        return false;
                    }
                }
                if (StringUtils.isNotEmpty(finFastCodeLineDO.getConditionThree())) {
                    if (result.get(finFastCodeLineDO.getConditionOne()) == null) {
                        return false;
                    }
                    if (!finFastCodeLineDO.getConditionThreeValue().equals(result.get(finFastCodeLineDO.getConditionThree()) + "")) {
                        return false;
                    }
                }
                return true;
            }).findFirst().orElse(null);
            if (finFastCodeLineDOResult == null) {
                journalLogInfo.setErrMsg("快码条件不满足");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            if (StringUtils.isBlank(finAccEngDetDataLineDO.getOutputColumn())) {
                journalLogInfo.setErrMsg("输出字段未配置");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            String outputValue = null;
            if (FinConstant.OUT_PUT_ONE.equals(finAccEngDetDataLineDO.getOutputColumn())) {
                outputValue = "'" + finFastCodeLineDOResult.getOutputOne() + "'";
            }
            if (FinConstant.OUT_PUT_TWO.equals(finAccEngDetDataLineDO.getOutputColumn())) {
                outputValue = "'" + finFastCodeLineDOResult.getOutputTwo() + "'";
            }
            if (FinConstant.OUT_PUT_THREE.equals(finAccEngDetDataLineDO.getOutputColumn())) {
                outputValue = "'" + finFastCodeLineDOResult.getOutputThree() + "'";
            }
            if (outputValue == null) {
                journalLogInfo.setErrMsg("输出字段配置有误");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            saveMap.put(finAccEngDetDataLineDO.getColumnName(), outputValue);
        }
        return true;
    }

    private boolean checkFlexible(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                  FinAccountEngineConfigDO finAccountEngineConfigDO,
                                  String docNum,
                                  FinAccountEngineDO finAccountEngineDO,
                                  FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                  FinAccEngDetDataDO finAccEngDetDataDO,
                                  Map saveMap, JournalLogInfo journalLogInfo) {
        if (UdcEnum.SOURCE_TYPE_FLEXIBLE.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {

            FinFlexibleValueDO finFlexibleValueDOResult = finAccountEngineConfigDO.getFinFlexibleValueDOList().stream()
                    .filter(finFlexibleValueDO -> finFlexibleValueDO.getFlexibleValueCode().equals(finAccEngDetDataLineDO.getFlexibleCode()))
                    .findFirst()
                    .orElse(null);
            if (finFlexibleValueDOResult == null) {
                journalLogInfo.setErrMsg("查不到有效的值集明细");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            put(saveMap, finAccEngDetDataLineDO.getColumnName(), finFlexibleValueDOResult.getFlexibleValueName(), finAccEngDetDataLineDO);
        }
        return true;
    }

    private boolean checkAccountPeriod(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                       Map result,
                                       FinAccountEngineConfigDO finAccountEngineConfigDO,
                                       String ouCode,
                                       String docNum,
                                       FinAccountEngineDO finAccountEngineDO,
                                       FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                       FinAccEngDetDataDO finAccEngDetDataDO,
                                       Map saveMap, JournalLogInfo journalLogInfo) {
        if (UdcEnum.SOURCE_TYPE_ACCOUNT_PERIOD.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {
            if (!result.containsKey(finAccEngDetDataLineDO.getColumnSource())) {
                journalLogInfo.setErrMsg("会计期间字段列来源未配置");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            if (result.get(finAccEngDetDataLineDO.getColumnSource()) == null) {
                journalLogInfo.setErrMsg("会计期间字段列值为空");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            Object detDate = result.get(finAccEngDetDataLineDO.getColumnSource());
            LocalDateTime localDateTime;
            if(detDate instanceof Timestamp){
                 localDateTime = ((Timestamp) detDate).toLocalDateTime();
            }else {
                 localDateTime = (LocalDateTime) result.get(finAccEngDetDataLineDO.getColumnSource());
            }
            String sobCode = finAccEngDetDataDO.getSobCode();
            FinSetOfBookDO finSetOfBookDOResult = finAccountEngineConfigDO.getFinSetOfBookDOList().stream()
                    .filter(finSetOfBookDO -> finSetOfBookDO.getSobCode().equals(sobCode))
                    .findFirst()
                    .orElse(null);
            if (finSetOfBookDOResult == null) {
                journalLogInfo.setErrMsg("查不到启用状态的账套");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            FinSobAccountPeriodDO finSobAccountPeriodDOResult = finAccountEngineConfigDO.getFinSobAccountPeriodDOList().stream().filter(finSobAccountPeriodDO ->

                            finSetOfBookDOResult.getSobCode().equals(finSobAccountPeriodDO.getSobCode()) &&
                                    checkFinSobAccountPeriodOu(ouCode,finSobAccountPeriodDO) &&
                                    UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_OPEN.getValueCode().equals(finSobAccountPeriodDO.getStatus()) &&
                                    localDateTime.compareTo(finSobAccountPeriodDO.getActiveStartTime()) >= 0 &&
                                    localDateTime.compareTo(finSobAccountPeriodDO.getActiveEndTime()) <= 0)
                    .findFirst().orElse(null);
            if (finSobAccountPeriodDOResult == null) {
                journalLogInfo.setErrMsg("未查询到会计期间");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            put(saveMap, finAccEngDetDataLineDO.getColumnName(), finSobAccountPeriodDOResult.getPeriodStyle(), finAccEngDetDataLineDO);
        }
        return true;
    }

    private Boolean checkFinSobAccountPeriodOu(String ouCode,FinSobAccountPeriodDO finSobAccountPeriodDO){

        if (ouCode.equals(finSobAccountPeriodDO.getOuCode()) || Objects.equals(ouCode,String.valueOf(finSobAccountPeriodDO.getOuId()))){
            return true;
        }

        return false;
    }

    private boolean checkDebtAndCreditSumEqual(List<Map> dataList, JournalLogInfo journalLogInfo) {
        if (CollectionUtils.isEmpty(dataList)) {
            return true;
        }
        //借方金额
        BigDecimal debit_amt = BigDecimal.ZERO;
        //贷方金额
        BigDecimal credit_amt = BigDecimal.ZERO;
        //本币借方金额
        BigDecimal debit_cur_amt = BigDecimal.ZERO;
        //本币贷方金额
        BigDecimal credit_cur_amt = BigDecimal.ZERO;
        for (Map data : dataList) {
            if (data.get("debit_amt") == null || data.get("credit_amt") == null
                    || data.get("debit_cur_amt") == null || data.get("credit_cur_amt") == null) {
                journalLogInfo.setErrMsg("校验借方、贷方金额是否相等失败，存在值为空的金额字段");
                saveFinJournalLogNew(journalLogInfo);
                return false;
            }
            debit_amt = debit_amt.add((BigDecimal) data.get("debit_amt"));
            credit_amt = credit_amt.add((BigDecimal) data.get("credit_amt"));
            debit_cur_amt = debit_cur_amt.add((BigDecimal) data.get("debit_cur_amt"));
            credit_cur_amt = credit_cur_amt.add((BigDecimal) data.get("credit_cur_amt"));
        }
        if (debit_amt.compareTo(credit_amt) != 0 || debit_cur_amt.compareTo(credit_cur_amt) != 0) {
            journalLogInfo.setErrMsg("校验借方、贷方金额是否相等失败，借方和贷方金额不相等");
            saveFinJournalLogNew(journalLogInfo);
            return false;
        }
        return true;
    }

    private void saveFinJournal(FinAccountEngineDO finAccountEngineDO, FinEventTableDO finEventTableDO, Connection connection, List<Map> saveMapList, Set<Long> proposingDocIds, Set<Long> successProposingDocIds, boolean manualFlag) {
        if (CollectionUtils.isEmpty(saveMapList)) {
            return;
        }
        //会计引擎
        Assert.notNull(finAccountEngineDO, "查不到会计引擎");
        try {
            //设置下租户ID字段
            long tenantId = -1L;
            if (tenantClientProvider.getSessionTenant() != null) {
                tenantId = tenantClientProvider.getSessionTenant().getId();
            }
            final long finalTenantId = tenantId;
            String creator = "系统自动";
            if (manualFlag) {
                GeneralUserDetails principal = (GeneralUserDetails) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
                if (principal != null) {
                    SysUserDTO sysUserDTO = principal.getUser();
                    if (sysUserDTO != null) {
                        creator = sysUserDTO.getLastName();
                    }
                }
            }
            final String finalCreator = creator;
            DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            if (CollectionUtils.isNotEmpty(saveMapList)) {
                saveMapList.stream().forEach(v -> {
                    v.put("tenant_id", finalTenantId);
                });
                saveMapList.stream().forEach(v -> {
                    v.put("create_user_id", -1);
                });
                saveMapList.stream().forEach(v -> {
                    v.put("create_time", LocalDateTime.now());
                });
                saveMapList.stream().forEach(v -> {
                    v.put("creator",finalCreator);
                });
            }
            Map<Object, List<Map>> detDataIdMap = saveMapList.stream().collect(Collectors.groupingBy(v -> {
                return v.get("detDataId");
            }));
            for (Object key : detDataIdMap.keySet()) {
                List<Map> maps = detDataIdMap.get(key);
                maps.stream().forEach(v -> {
                    v.remove("detDataId");
                    successProposingDocIds.add((long) v.get("main_id"));
                    v.remove("main_id");
                });
                finAccountEngineJdbcService.finJournalBatchSave(finAccountEngineDO, connection, maps);
            }

        } catch (Exception e) {
            log.error("保存会计凭证失败,事件表单:{},会计引擎ID:{}", finEventTableDO.getEventTable(), finAccountEngineDO.getId(), e);
            throw new BusinessException("保存会计凭证失败，" + e.getMessage());
        }
    }

    private void put(Map map, String key, Object value, FinAccEngDetDataLineDO finAccEngDetDataLineDO) {
        if (StringUtils.isBlank(finAccEngDetDataLineDO.getColumnType())) {
            map.put(key, value+"");
        } else if (finAccEngDetDataLineDO.getColumnType().trim().equals("varchar")) {
            map.put(key,value+"");
        } else if (finAccEngDetDataLineDO.getColumnType().trim().equals("decimal")) {
            if (value instanceof BigDecimal) {
                map.put(key, value);
            }else if (value instanceof Float){
                if (Objects.isNull(value)){
                    map.put(key, BigDecimal.ZERO);
                }else {
                    map.put(key, BigDecimal.valueOf(Float.valueOf(value.toString())));
                }
            } else {
                if (Objects.isNull(value)){
                    map.put(key, BigDecimal.ZERO);
                }else {
                    map.put(key, BigDecimal.valueOf(Double.valueOf((String) value)));
                }
            }
        } else if (finAccEngDetDataLineDO.getColumnType().trim().equals("datetime")) {
            map.put(key, value);
        } else if (finAccEngDetDataLineDO.getColumnType().trim().equals("bigint")) {
            map.put(key, value);
        } else if (finAccEngDetDataLineDO.getColumnType().trim().equals("int")) {
            map.put(key, value);
        }
    }

    @Getter
    @Setter
    static class JournalLogInfo {
        //单据ID
        private Long docId;
        //单据ID
        private Long detailDocId;
        //单据编码
        private String docNum;
        //单据公司编码
        private String ouCode;
        //事件表单
        private String eventTable;
        //会计引擎ID
        private Long accountEngineId;
        //会计引擎明细名称
        private String accountEngine;
        //会计引擎数据行名称
        private String detDataName;
        //数据映射列名称
        private String columnName;
        //异常信息
        private String errMsg;
        //是否手动拟定
        private Boolean manualFlag;

        public JournalLogInfo(String eventTable, Long accountEngineId) {
            this.eventTable = eventTable;
            this.accountEngineId = accountEngineId;
        }

        public JournalLogInfo(String docNum, String eventTable, String accountEngine) {
            this.docNum = docNum;
            this.eventTable = eventTable;
            this.accountEngine = accountEngine;
        }
    }

}
