package com.elitesland.fin.application.service.flowrepair;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.application.facade.dto.flowrepair.AccountFlowRepairHandle20DTO;
import com.elitesland.fin.application.facade.dto.flowrepair.AccountFlowRepairHandleDTO;
import com.elitesland.fin.application.facade.dto.flowrepair.CreditAccountFlowRepairHandle20DTO;
import com.elitesland.fin.application.facade.dto.flowrepair.CreditAccountFlowRepairHandleDTO;
import com.elitesland.fin.application.facade.param.account.AccountSnapshotParam;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountSnapshotParam;
import com.elitesland.fin.application.facade.param.creditaccountflow.CreditAccountFlowParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowPageParam;
import com.elitesland.fin.application.facade.param.flowrepair.AccountFlowRepairParam;
import com.elitesland.fin.application.facade.param.flowrepair.CreditAccountFlowRepairPageParam;
import com.elitesland.fin.application.facade.param.flowrepair.CreditAccountFlowRepairParam;
import com.elitesland.fin.application.facade.param.flowrepair.CreditAccountFlowRepairRedoParam;
import com.elitesland.fin.application.facade.vo.creditaccountflow.CreditAccountFlowVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.facade.vo.flowrepair.AccountFlowRepairVO;
import com.elitesland.fin.application.facade.vo.flowrepair.CreditAccountFlowRepairVO;
import com.elitesland.fin.application.service.account.AccountService;
import com.elitesland.fin.application.service.creditaccount.CreditAccountService;
import com.elitesland.fin.application.service.creditaccountflow.CreditAccountFlowService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinRedisConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.account.AccountSnapshotDO;
import com.elitesland.fin.entity.creditaccount.CreditAccountDO;
import com.elitesland.fin.entity.creditaccount.CreditAccountSnapshotDO;
import com.elitesland.fin.entity.creditaccountflow.CreditAccountFlowDO;
import com.elitesland.fin.entity.flowrepair.AccountFlowRepairDO;
import com.elitesland.fin.entity.flowrepair.CreditAccountFlowRepairDO;
import com.elitesland.fin.repo.account.AccountSnapshotRepo;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepoProc;
import com.elitesland.fin.repo.creditaccount.CreditAccountSnapshotRepo;
import com.elitesland.fin.repo.creditaccountflow.CreditAccountFlowRepo;
import com.elitesland.fin.repo.creditaccountflow.CreditAccountFlowRepoProc;
import com.elitesland.fin.repo.flowrepair.CreditAccountFlowRepairRepo;
import com.elitesland.fin.repo.flowrepair.CreditAccountFlowRepairRepoProc;
import groovy.lang.Lazy;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RBucket;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import javax.persistence.EntityManager;
import javax.persistence.ParameterMode;
import javax.persistence.PersistenceContext;
import javax.persistence.StoredProcedureQuery;
import java.math.BigDecimal;
import java.sql.*;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;
import java.util.Date;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2023/10/25
 * @Version 1.0
 * @Content:
 */
@Service
@Slf4j
@AllArgsConstructor
public class CreditAccountFlowRepairServiceImpl implements CreditAccountFlowRepairService{

    @Autowired
    private CreditAccountFlowRepairRepo creditAccountFlowRepairRepo;
    @Autowired
    private CreditAccountFlowRepairRepoProc creditAccountFlowRepairRepoProc;
    @Autowired
    private CreditAccountFlowService creditAccountFlowService;
    @Autowired
    private CreditAccountFlowRepoProc creditAccountFlowRepoProc;
    @Autowired
    private CreditAccountFlowRepo creditAccountFlowRepo;
    @Autowired
    private CreditAccountRepoProc creditAccountRepoProc;
    @Autowired
    private CreditAccountSnapshotRepo creditAccountSnapshotRepo;
    @Autowired
    private CreditAccountService creditAccountService;
    private final RedissonClient redissonClient;
    @PersistenceContext
    private EntityManager entityManager;
    private final NamedParameterJdbcTemplate namedTemplate;
    private final TransactionTemplate transactionTemplate;

    /**
     * 信用账户流水修复重算分页查询
     *
     * @param pageParam 入参
     * @return 信用账户流水修复重算信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<CreditAccountFlowRepairVO> page(CreditAccountFlowRepairPageParam pageParam){

        PagingVO<CreditAccountFlowRepairVO> pagingVO = creditAccountFlowRepairRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<CreditAccountFlowRepairVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        return PagingVO.<CreditAccountFlowRepairVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }


    /**
     * 信用账户流水修复重算保存
     *
     * @param param 入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void save(CreditAccountFlowRepairParam param){

        if (Objects.equals(param.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode())){
            saveFlowRepair10(param);
        }else if (Objects.equals(param.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_20.getValueCode())){
            if (Objects.isNull(param.getId())){
                saveFlowRepair20(param);
            }else {
                saveFlowRepair10(param);
            }
        }else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未匹配到重算原因!");

        }

    }

    private void saveFlowRepair10(CreditAccountFlowRepairParam param){
        if (Objects.equals(param.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode())) {
            if (StringUtils.isBlank(param.getFlowNo())){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "流水号不能为空!");
            }
            if (Objects.isNull(param.getActualAmount())){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "实际金额不能为空!");
            }
        }

        CreditAccountFlowVO flowVO = selectFlowByFlowNo(param.getFlowNo());

        if (Objects.isNull(param.getId())){
            CreditAccountFlowRepairDO flowRepairDO = new CreditAccountFlowRepairDO();
            assembleFlowRepair(flowRepairDO,param,flowVO);
            creditAccountFlowRepairRepo.save(flowRepairDO);
        }else {
            Optional<CreditAccountFlowRepairDO> flowRepairOptional = creditAccountFlowRepairRepo.findById(param.getId());
            if (flowRepairOptional.isEmpty()){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到此条流水修复数据!");
            }
            CreditAccountFlowRepairDO flowRepairDO = flowRepairOptional.get();
            if (!Objects.equals(flowRepairDO.getRepairStatus(), UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode())){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "草稿状态才可修改!");
            }
            assembleFlowRepair(flowRepairDO,param,flowVO);
            creditAccountFlowRepairRepo.save(flowRepairDO);
        }
    }

    private void assembleFlowRepair(CreditAccountFlowRepairDO flowRepairDO,
                                    CreditAccountFlowRepairParam param,CreditAccountFlowVO flowVO){
        if (StringUtils.isNotBlank(param.getFlowNo())) {
            flowRepairDO.setFlowNo(param.getFlowNo());
        }
        if (Objects.nonNull(param.getActualAmount())) {
            flowRepairDO.setActualAmount(param.getActualAmount());
        }
        flowRepairDO.setAmount(flowVO.getAmount());
        flowRepairDO.setOpenAccountEntityCode(flowVO.getObjectCode());
        flowRepairDO.setOpenAccountEntityName(flowVO.getObjectName());
        flowRepairDO.setAccountType(flowVO.getCreditAccountType());
        flowRepairDO.setAccountCode(flowVO.getCreditAccountCode());
        flowRepairDO.setAccountName(flowVO.getCreditAccountName());
        flowRepairDO.setSourceNo(flowVO.getSourceNo());
        flowRepairDO.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());

        if (StringUtils.isNotBlank(param.getRecalculationReason())) {
            flowRepairDO.setRecalculationReason(param.getRecalculationReason());
        }
        if (Objects.nonNull(param.getRecalculationDate())){
            flowRepairDO.setRecalculationDate(param.getRecalculationDate());
        }else {
            if (Objects.equals(param.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode())) {
                if (Objects.isNull(flowRepairDO.getRecalculationDate())){
                    flowRepairDO.setRecalculationDate(flowVO.getAuditDate());

                }
            }
        }

        SysUserDTO sysUserDTO = getUser();
        if (StringUtils.isNotBlank(param.getRecalculationUser())) {
            flowRepairDO.setRecalculationUser(param.getRecalculationUser());
        }else {
            flowRepairDO.setRecalculationUser(sysUserDTO.getPrettyName());
        }
        if (Objects.nonNull(param.getRecalculationUserId())){
            flowRepairDO.setRecalculationUserId(param.getRecalculationUserId());
        }else {
            flowRepairDO.setRecalculationUserId(sysUserDTO.getId());
        }
    }

    private SysUserDTO getUser() {
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (userDetails == null) {
            throw new BusinessException("当前登陆人信息获取为空!");
        }
        if (Objects.isNull(userDetails.getUser())){
            throw new BusinessException("当前登陆人信息为空!");
        }
        return userDetails.getUser();
    }

    private CreditAccountFlowVO selectFlowByFlowNo(String flowNo){
        if (StringUtils.isBlank(flowNo)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "流水号为空!");
        }
        CreditAccountFlowParam flowQueryParam = new CreditAccountFlowParam();
        flowQueryParam.setFlowNo(flowNo);
        List<CreditAccountFlowVO> flowVOList = creditAccountFlowService.selectListByQueryParam(flowQueryParam);
        if (CollectionUtils.isEmpty(flowVOList)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到此流水号("+flowNo+")的流水信息!");
        }
        if (CollectionUtils.isNotEmpty(flowVOList) && flowVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "此流水号("+flowNo+")的存在多条流水信息!");
        }

        return flowVOList.get(0);
    }

    /**
     * 根据信用账户流水修复重算ID批量删除
     *
     * @param ids ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteBatch(List<Long> ids) {
        CreditAccountFlowRepairPageParam flowRepairPageParam = new CreditAccountFlowRepairPageParam();
        flowRepairPageParam.setIds(ids);
        List<CreditAccountFlowRepairVO> respVOList = creditAccountFlowRepairRepoProc.selectListByParam(flowRepairPageParam);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        //针对‘待发布’状态的任务（主表）
        List<CreditAccountFlowRepairVO> voList = respVOList.stream().filter(respVO -> !Objects.equals(respVO.getRepairStatus(), UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(voList)) {
            String checkResult = voList.stream().map(vo ->
                    "流水号:" + vo.getFlowNo() + ""
            ).collect(Collectors.joining(";", "只有草稿可删除,[", "], 请检查"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        //逻辑删除
        creditAccountFlowRepairRepoProc.updateDeleteFlagBatch(1, ids);

    }


    /**
     * 信用账户流水修复重算
     *
     * @param repairRedoParam 重算入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void repairRedo(CreditAccountFlowRepairRedoParam repairRedoParam){
        //选中或者全部草稿状态的才可重算

        //选中或者不选(默认全部)的草稿状态的才可重算
        CreditAccountFlowRepairPageParam flowRepairPageParam = new CreditAccountFlowRepairPageParam();
        flowRepairPageParam.setIds(repairRedoParam.getIds());
        flowRepairPageParam.setFlowNoList(repairRedoParam.getFlowNoList());
        flowRepairPageParam.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());
        List<CreditAccountFlowRepairVO> repairFlowVOList = creditAccountFlowRepairRepoProc.selectListByParam(flowRepairPageParam);
        if (CollectionUtils.isEmpty(repairFlowVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "无符合重算的信用账户流水数据!");
        }

        val flowNoCheckList = repairFlowVOList.stream().filter(repairParam -> !StringUtils.isBlank(repairParam.getFlowNo()))
                // 获得元素出现频率的 Map，键为元素，值为元素出现的次数
                .map(CreditAccountFlowRepairVO::getFlowNo).collect(Collectors.toMap(e -> e, e -> 1, (a, b) -> a + b))
                // 所有 entry 对应的 Stream
                .entrySet().stream()
                // 过滤出元素出现次数大于 1 的 entry
                .filter(entry -> entry.getValue() > 1)
                // 获得 entry 的键（重复元素）对应的 Stream
                .map(entry -> entry.getKey())
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(flowNoCheckList)) {
            String checkResult = flowNoCheckList.stream().map(Objects::toString).collect(Collectors.joining("、", "信用账户流水号[", "], 存在重复数据,请检查!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        List<String> repairFlowNoList = repairFlowVOList.stream().map(CreditAccountFlowRepairVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(repairFlowNoList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重算的信用账户流水数据的流水号为空!");
        }
        /*CreditAccountFlowParam flowQueryParam = new CreditAccountFlowParam();
        flowQueryParam.setFlowNoList(repairFlowNoList);
        log.info("修复重算信用账户流水查询数据,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowQueryParam));
        List<CreditAccountFlowVO> flowVOList = creditAccountFlowService.selectListByQueryParam(flowQueryParam);
        log.info("修复重算信用账户流水查询数据,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));

        if (CollectionUtils.isEmpty(flowVOList)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到信用账户流水信息!");
        }*/

        List<String> repairFlowNoCacheList = new ArrayList<>();
        repairFlowNoList.forEach(flowNo -> {
            if (StringUtils.isNotBlank(getFlowNoCache(flowNo))){
                repairFlowNoCacheList.add(flowNo);
            }else {
                saveFlowNoCache(flowNo);
            }
        });
        if (CollectionUtils.isNotEmpty(repairFlowNoCacheList)) {
            String checkResult = repairFlowNoCacheList.stream().map(Objects::toString).collect(Collectors.joining("、", "账户流水号[", "], 修复重算中,请稍后再次重算!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        if (CollectionUtils.isNotEmpty(repairFlowVOList)){
            //区分10和20
            List<CreditAccountFlowRepairVO> repairFlow10List = repairFlowVOList.stream().filter(accountFlowRepairVO -> Objects.equals(accountFlowRepairVO.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode())).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(repairFlow10List)){
                List<String> repairFlowNo10List = repairFlow10List.stream().map(CreditAccountFlowRepairVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                processingCalculation(repairFlow10List,repairFlowNo10List);
            }

            List<CreditAccountFlowRepairVO> repairFlow20List = repairFlowVOList.stream().filter(accountFlowRepairVO -> Objects.equals(accountFlowRepairVO.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_20.getValueCode())).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(repairFlow20List)){
                List<String> repairFlowNo20List = repairFlow20List.stream().map(CreditAccountFlowRepairVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                processingCalculation20(repairFlow20List,repairFlowNo20List);
            }

            //processingCalculation(repairFlowVOList,repairFlowNoList);
        }

    }

    @Override
    public void removeFlowNoCacheBatch(List<String> flowNoList) {
        flowNoList.forEach(flowNo -> {
            removeFlowNoCache(flowNo);
        });
    }

    @Override
    public void removeFlowNoCache(String flowNo) {
        RBucket<String> bucket = redissonClient.getBucket(FinRedisConstant.ACCOUNT_FLOW_REPAIR_KEY+flowNo);
        if (bucket.isExists()) {
            bucket.delete();
        }
    }

    @Override
    public String getFlowNoCache(String flowNo) {
        RBucket<String> bucket = redissonClient.getBucket(FinRedisConstant.ACCOUNT_FLOW_REPAIR_KEY+flowNo);
        if (bucket.isExists()) {
            return bucket.get();
        }
        return null;
    }

    @Override
    public void saveFlowNoCache(String flowNo){
        RBucket<String> bucket = redissonClient.getBucket(FinRedisConstant.ACCOUNT_FLOW_REPAIR_KEY+flowNo);
        bucket.set(flowNo);
    }

    private void processingCalculation(List<CreditAccountFlowRepairVO> repairFlowVOList,List<String> repairFlowNoList){

        List<String> accCodeList = new ArrayList<>();

        repairFlowVOList.forEach(flowRepairVO -> {
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            String processingFlowNo = transactionTemplate.execute(transactionStatus -> {
                String repairFlowNo = flowRepairVO.getFlowNo();
                try {
                    CreditAccountFlowParam flowQueryParam = new CreditAccountFlowParam();
                    flowQueryParam.setFlowNo(flowRepairVO.getFlowNo());
                    log.info("修复重算信用账户流水查询数据,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowQueryParam));
                    List<CreditAccountFlowVO> flowVOList = creditAccountFlowService.selectListByQueryParam(flowQueryParam);
                    log.info("修复重算信用账户流水查询数据,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));
                    if (CollectionUtils.isNotEmpty(flowVOList) && flowVOList.size() == 1 ){
                        if (Objects.nonNull(flowVOList.get(0))) {
                            CreditAccountFlowVO flowVO = flowVOList.get(0);

                            if (Objects.equals(flowVO.getCreditAccountType(),UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getValueCode()) &&
                                    FinConstant.CREDIT_ACCOUNT_FLOW_REPAIR_TRANSACTION_TYPE.contains(flowVO.getTransactionType())){
                                repairFlow(flowRepairVO,flowVO,accCodeList);
                            }else {
                                log.info("未修复信用账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
                            }

                            if (accCodeList.contains(flowVO.getCreditAccountCode())){
                                //更新完成时同步更新信用账户列表的金额，用最新一条流水记录的更新同账户编码的账户额度、账户使用额度、账户占用额度、账户可用额度。
                                updateAccountAmt(Collections.singletonList(flowVO.getCreditAccountCode()),repairFlowNoList);
                                //更新信用账户快照表的对应数据
                                //原先流水修复程序的重算日期 取流水的审核日期
                                if (Objects.isNull(flowRepairVO.getRecalculationDate())){
                                    flowRepairVO.setRecalculationDate(flowVO.getAuditDate());
                                }
                                updateCreditAccountSnapshot(Collections.singletonList(flowRepairVO),repairFlowNoList);

                            }

                        } else {
                            log.info("未查询到修复信用账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowRepairVO));
                        }
                    }else {
                        log.info("修复信用账户流水数据为空或者有多条,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));
                    }

                    //释放校验缓存
                    removeFlowNoCache(flowRepairVO.getFlowNo());
                    return flowRepairVO.getFlowNo();
                } catch (Exception e) {

                    log.error("流水号("+repairFlowNo+")信用账户流水修复重算失败!:{}", e.getMessage());
                    //回滚
                    transactionStatus.setRollbackOnly();
                    repairFlowNoList.forEach(flowNo -> {
                        removeFlowNoCache(flowNo);
                    });
                    //抛出异常，不往下执行
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "流水号("+repairFlowNo+")信用账户流水修复重算失败!");
                }
            });



        });

        /*//更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户额度、账户使用额度、账户占用额度、账户可用额度。
        if (CollectionUtils.isNotEmpty(accCodeList)) {
            List<String> accCodes = accCodeList.stream().filter(Objects::nonNull).distinct().collect(Collectors.toList());
            updateAccountAmt(accCodes,repairFlowNoList);
        }*/

        repairFlowNoList.forEach(flowNo -> {
            removeFlowNoCache(flowNo);
        });
    }


    @Transactional(rollbackFor = Exception.class)
    public void repairFlow(CreditAccountFlowRepairVO flowRepairVO,CreditAccountFlowVO flowVO,List<String> accCodeList) {
        repairCurrentFlow(flowRepairVO,flowVO,accCodeList);


    }

    /**
     * 根据账户编码更新账户金额：更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户额度、账户使用额度、账户占用额度、账户可用额度。
     *
     * @param accCodeList 账户编码
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateAccountAmt(List<String> accCodeList,List<String> repairFlowNoList) {
        accCodeList.forEach(accCode -> {
            //根据账户编码查询,按照创建时间进行倒叙查询并取第一条数据
            CreditAccountFlowParam flowQueryParam = new CreditAccountFlowParam();
            flowQueryParam.setCreditAccountCode(accCode);
            CreditAccountFlowVO flowVO = creditAccountFlowRepoProc.selectPreviousRepairAmtByParam(flowQueryParam);
            //根据编码更新账户金额
            CreditAccountDO accountDO = creditAccountRepoProc.findByCreditAccountCode(accCode);

            if (Objects.nonNull(flowVO) && Objects.nonNull(accountDO)){
                BigDecimal creditAccountLimit = flowVO.getCreditAccountLimit();
                BigDecimal creditAccountUsedLimit = flowVO.getCreditAccountUsedLimit();
                BigDecimal creditAccountOccupancyLimit = flowVO.getCreditAccountOccupancyLimit();

                BigDecimal creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

                accountDO.setCreditAccountLimit(creditAccountLimit);
                accountDO.setCreditAccountUsedLimit(creditAccountUsedLimit);
                accountDO.setCreditAccountOccupancyLimit(creditAccountOccupancyLimit);
                accountDO.setCreditAccountAvailableLimit(creditAccountAvailableLimit);
                creditAccountRepoProc.save(accountDO);
            }else if (Objects.isNull(accountDO)){
                repairFlowNoList.forEach(flowNo -> {
                    removeFlowNoCache(flowNo);
                });

                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到信用账户("+accCode+")信息!");

            }


        });
    }

    /**
     * 修复重算当前此条信用账户流水表的信息和修复信用流水记录表的信息
     *
     * @param flowRepairVO   修复信用流水表的此条数据
     * @param flowVO   要修复的此条信用账户流水数据
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairCurrentFlow(CreditAccountFlowRepairVO flowRepairVO,CreditAccountFlowVO flowVO,List<String> accCodeList) {
        CreditAccountFlowRepairHandleDTO flowRepairHandleDTO = assembleRepairHandle(flowRepairVO,flowVO);
        if (Objects.nonNull(flowRepairHandleDTO)){
            repairCalculationFlow(flowRepairHandleDTO,accCodeList);
            creditAccountFlowRepairRepoProc.updateAmountAndRepairStatusById(
                    flowRepairHandleDTO.getAmount(),UdcEnum.FLOW_REPAIR_STATUS_FIXED.getValueCode(),flowRepairHandleDTO.getFlowRepairId());
            repairAfterFlow(flowRepairVO,flowVO,flowRepairHandleDTO,accCodeList);
        }
    }

    /**
     * 修复重算当前此条之后的信用账户流水表的信息
     *
     * @param flowRepairVO   修复信用流水表的此条数据
     * @param flowVO   要修复的此条信用账户流水数据
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairAfterFlow(CreditAccountFlowRepairVO flowRepairVO,CreditAccountFlowVO flowVO,CreditAccountFlowRepairHandleDTO flowRepairCurrentHandleDTO,List<String> accCodeList) {
        List<CreditAccountFlowRepairHandleDTO> flowRepairHandleList = selectRepairAfterFlow(flowRepairVO,flowVO,flowRepairCurrentHandleDTO);
        if (CollectionUtils.isNotEmpty(flowRepairHandleList)){
            flowRepairHandleList.forEach(flowRepairHandleDTO -> {
                repairCalculationFlowAfter(flowRepairHandleDTO,accCodeList);
            });
        }
    }

    /**
     * 查询某条信用账户流水以及之后账户流水的数据-相同账户编码的
     *
     * @param flowRepairVO   修复信用流水表的此条数据
     * @param flowVO   要修复的此条信用账户流水数据
     * @return
     */
    public List<CreditAccountFlowRepairHandleDTO> selectRepairAfterFlow(CreditAccountFlowRepairVO flowRepairVO, CreditAccountFlowVO flowVO, CreditAccountFlowRepairHandleDTO flowRepairCurrentHandleDTO) {
        int size = 500;
        int page = 1;

        List<CreditAccountFlowRepairHandleDTO>  flowRepairHandleList= new ArrayList<>();

        //查询要修复的该条账户流水及之后的相同账户编码的全部流水的
        CreditAccountFlowParam flowPageParam = new CreditAccountFlowParam();
        flowPageParam.setCreditAccountCode(flowVO.getCreditAccountCode());
        flowPageParam.setRepairTime(flowVO.getAuditDate());
        //之后的账户流水不做全部查出，要分批查出循环处理
        while (true) {
            flowPageParam.setCurrent(page++);
            flowPageParam.setSize(size);
            log.info("查询修复信用账户流水及之后流水,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowPageParam));
            List<CreditAccountFlowVO> flowAfterList = creditAccountFlowService.selectRepairAfterPage(flowPageParam);
            log.info("查询修复信用账户流水及之后流水,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowAfterList));
            if (CollectionUtils.isEmpty(flowAfterList)) {
                break;
            }

            flowAfterList.forEach(flowAfterVO -> {
                CreditAccountFlowRepairHandleDTO flowRepairHandleDTO = assembleRepairAfterHandle(flowRepairVO,flowAfterVO,flowRepairCurrentHandleDTO);
                if (Objects.nonNull(flowRepairHandleDTO)){
                    flowRepairHandleList.add(flowRepairHandleDTO);
                }else {
                    log.info("按照交易类型未匹配到信用账户流水,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowAfterList));
                }
            });

            if (flowAfterList.size() < size) {
                // 少于要查询的数量，则说明已无数据
                break;
            }
        }

        return flowRepairHandleList;
    }

    private CreditAccountFlowRepairHandleDTO assembleRepairHandle(CreditAccountFlowRepairVO flowRepairVO,CreditAccountFlowVO flowVO){
        if (Objects.isNull(flowRepairVO.getActualAmount()) || Objects.isNull(flowVO.getCreditAccountLimit()) ||
                Objects.isNull(flowVO.getCreditAccountOccupancyLimit()) ||
                Objects.isNull(flowVO.getCreditAccountUsedLimit()) || Objects.isNull(flowVO.getAmount())){
            log.info("未修复信用账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
            return null;
        }
        CreditAccountFlowRepairHandleDTO flowRepairHandleDTO = new CreditAccountFlowRepairHandleDTO();
        flowRepairHandleDTO.setFlowRepairId(flowRepairVO.getId());
        flowRepairHandleDTO.setFlowRepairFlowNo(flowRepairVO.getFlowNo());
        flowRepairHandleDTO.setFlowId(flowVO.getId());
        flowRepairHandleDTO.setFlowNo(flowVO.getFlowNo());
        flowRepairHandleDTO.setActualAmount(flowRepairVO.getActualAmount());
        flowRepairHandleDTO.setAmount(flowVO.getAmount());
        //变动金额=实际交易金额-原先错误流水交易金额
        BigDecimal variableAmount = flowRepairVO.getActualAmount().subtract(flowVO.getAmount());
        flowRepairHandleDTO.setVariableAmount(variableAmount);
        flowRepairHandleDTO.setAccountType(flowVO.getCreditAccountType());
        flowRepairHandleDTO.setAccountCode(flowVO.getCreditAccountCode());
        flowRepairHandleDTO.setAccountName(flowVO.getCreditAccountName());
        flowRepairHandleDTO.setTransactionType(flowVO.getTransactionType());

        flowRepairHandleDTO.setCreditAccountLimit(flowVO.getCreditAccountLimit());
        flowRepairHandleDTO.setCreditAccountOccupancyLimit(flowVO.getCreditAccountOccupancyLimit());
        flowRepairHandleDTO.setCreditAccountUsedLimit(flowVO.getCreditAccountUsedLimit());
        flowRepairHandleDTO.setCreditAccountAvailableLimit(flowVO.getCreditAccountAvailableLimit());

        return flowRepairHandleDTO;
    }

    private CreditAccountFlowRepairHandleDTO assembleRepairAfterHandle(CreditAccountFlowRepairVO flowRepairVO,CreditAccountFlowVO flowVO, CreditAccountFlowRepairHandleDTO flowRepairCurrentHandleDTO){
        if (Objects.isNull(flowRepairVO.getActualAmount()) || Objects.isNull(flowVO.getCreditAccountLimit()) ||
                Objects.isNull(flowVO.getCreditAccountOccupancyLimit()) ||
                Objects.isNull(flowVO.getCreditAccountUsedLimit()) || Objects.isNull(flowVO.getAmount())){
            log.info("未修复信用账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
            return null;
        }
        CreditAccountFlowRepairHandleDTO flowRepairHandleDTO = new CreditAccountFlowRepairHandleDTO();
        flowRepairHandleDTO.setFlowRepairId(flowRepairVO.getId());
        flowRepairHandleDTO.setFlowRepairFlowNo(flowRepairVO.getFlowNo());
        flowRepairHandleDTO.setFlowId(flowVO.getId());
        flowRepairHandleDTO.setFlowNo(flowVO.getFlowNo());
        flowRepairHandleDTO.setActualAmount(flowRepairVO.getActualAmount());
        flowRepairHandleDTO.setAmount(flowVO.getAmount());
        //变动金额=实际交易金额-原先错误流水交易金额
        //BigDecimal variableAmount = flowRepairVO.getActualAmount().subtract(flowVO.getAmount());
        BigDecimal variableAmount = flowRepairCurrentHandleDTO.getVariableAmount();
        flowRepairHandleDTO.setVariableAmount(variableAmount);
        flowRepairHandleDTO.setAccountType(flowVO.getCreditAccountType());
        flowRepairHandleDTO.setAccountCode(flowVO.getCreditAccountCode());
        flowRepairHandleDTO.setAccountName(flowVO.getCreditAccountName());
        flowRepairHandleDTO.setTransactionType(flowRepairCurrentHandleDTO.getTransactionType());

        flowRepairHandleDTO.setCreditAccountLimit(flowVO.getCreditAccountLimit());
        flowRepairHandleDTO.setCreditAccountOccupancyLimit(flowVO.getCreditAccountOccupancyLimit());
        flowRepairHandleDTO.setCreditAccountUsedLimit(flowVO.getCreditAccountUsedLimit());
        flowRepairHandleDTO.setCreditAccountAvailableLimit(flowVO.getCreditAccountAvailableLimit());

        return flowRepairHandleDTO;
    }

    /**
     * 修复的计算处理逻辑
     *
     * @param flowRepairHandleDTO 信用账户流水修复重算处理计算DTO
     *
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairCalculationFlow(CreditAccountFlowRepairHandleDTO flowRepairHandleDTO,List<String> accCodeList) {
        //账户额度
        BigDecimal creditAccountLimit = flowRepairHandleDTO.getCreditAccountLimit();
        //账户占用额度
        BigDecimal creditAccountOccupancyLimit = flowRepairHandleDTO.getCreditAccountOccupancyLimit();
        //账户使用额度
        BigDecimal creditAccountUsedLimit = flowRepairHandleDTO.getCreditAccountUsedLimit();
        //账户可用额度 = 账户额度-账户使用额度-账户占用额度
        BigDecimal creditAccountAvailableLimit = BigDecimal.ZERO;
        //变动金额
        BigDecimal variableAmount = flowRepairHandleDTO.getVariableAmount();
        //实际金额
        BigDecimal actualAmount = flowRepairHandleDTO.getActualAmount();

        if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_R.getValueCode())) {
            //交易类型=释放占用时，该条流水及之后的相同账户编码的全部流水的账户占用额度改成原账户占用额度 - 变动金额，
            creditAccountOccupancyLimit = creditAccountOccupancyLimit.subtract(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateOccupancyAndAvailableAndAmountById(
                    creditAccountOccupancyLimit,creditAccountAvailableLimit,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());

        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_T.getValueCode())) {
            //交易类型=账户占用时，该条流水及之后的相同账户编码的全部流水的账户占用额度改成原账户占用额度 + 变动金额，
            creditAccountOccupancyLimit = creditAccountOccupancyLimit.add(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateOccupancyAndAvailableAndAmountById(
                    creditAccountOccupancyLimit,creditAccountAvailableLimit,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_O.getValueCode())) {
            //交易类型=账户扣减时，该条流水之后的相同账户编码的全部流水的账户额度改成原账户额度 - 变更金额，
            creditAccountLimit = creditAccountLimit.subtract(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateAccountAndAvailableAndAmountById(
                    creditAccountLimit,creditAccountAvailableLimit,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_I.getValueCode())) {
            //交易类型=账户增加时，该条流水及之后的相同账户编码的全部流水的账户额度改成原账户额度 + 变更金额，
            creditAccountLimit = creditAccountLimit.add(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateAccountAndAvailableAndAmountById(
                    creditAccountLimit,creditAccountAvailableLimit,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_U.getValueCode())) {
            //交易类型=使用增加时，该条流水及之后的相同账户编码的全部流水的账户使用额度改成原账户使用额度 + 变更金额
            creditAccountUsedLimit = creditAccountUsedLimit.add(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateUsedAndAvailableAndAmountById(
                    creditAccountUsedLimit,creditAccountAvailableLimit,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_D.getValueCode())) {
            //交易类型=使用减少时，该条流水及之后的相同账户编码的全部流水的账户使用额度改成原账户使用额度 - 变更金额，
            creditAccountUsedLimit = creditAccountUsedLimit.subtract(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateUsedAndAvailableAndAmountById(
                    creditAccountUsedLimit,creditAccountAvailableLimit,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        }


    }

    /**
     * 修复的计算处理逻辑
     *
     * @param flowRepairHandleDTO 信用账户流水修复重算处理计算DTO
     *
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairCalculationFlowAfter(CreditAccountFlowRepairHandleDTO flowRepairHandleDTO,List<String> accCodeList) {
        //账户额度
        BigDecimal creditAccountLimit = flowRepairHandleDTO.getCreditAccountLimit();
        //账户占用额度
        BigDecimal creditAccountOccupancyLimit = flowRepairHandleDTO.getCreditAccountOccupancyLimit();
        //账户使用额度
        BigDecimal creditAccountUsedLimit = flowRepairHandleDTO.getCreditAccountUsedLimit();
        //账户可用额度 = 账户额度-账户使用额度-账户占用额度
        BigDecimal creditAccountAvailableLimit = BigDecimal.ZERO;
        //变动金额
        BigDecimal variableAmount = flowRepairHandleDTO.getVariableAmount();
        //实际金额
       // BigDecimal actualAmount = flowRepairHandleDTO.getActualAmount();

        if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_R.getValueCode())) {
            //交易类型=释放占用时，该条流水及之后的相同账户编码的全部流水的账户占用额度改成原账户占用额度 - 变动金额，
            creditAccountOccupancyLimit = creditAccountOccupancyLimit.subtract(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateOccupancyAndAvailableById(
                    creditAccountOccupancyLimit,creditAccountAvailableLimit,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());

        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_T.getValueCode())) {
            //交易类型=账户占用时，该条流水及之后的相同账户编码的全部流水的账户占用额度改成原账户占用额度 + 变动金额，
            creditAccountOccupancyLimit = creditAccountOccupancyLimit.add(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateOccupancyAndAvailableById(
                    creditAccountOccupancyLimit,creditAccountAvailableLimit,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_O.getValueCode())) {
            //交易类型=账户扣减时，该条流水之后的相同账户编码的全部流水的账户额度改成原账户额度 - 变更金额，
            creditAccountLimit = creditAccountLimit.subtract(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateAccountAndAvailableById(
                    creditAccountLimit,creditAccountAvailableLimit,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_I.getValueCode())) {
            //交易类型=账户增加时，该条流水及之后的相同账户编码的全部流水的账户额度改成原账户额度 + 变更金额，
            creditAccountLimit = creditAccountLimit.add(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateAccountAndAvailableById(
                    creditAccountLimit,creditAccountAvailableLimit,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_U.getValueCode())) {
            //交易类型=使用增加时，该条流水及之后的相同账户编码的全部流水的账户使用额度改成原账户使用额度 + 变更金额
            creditAccountUsedLimit = creditAccountUsedLimit.add(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateUsedAndAvailableById(
                    creditAccountUsedLimit,creditAccountAvailableLimit,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_D.getValueCode())) {
            //交易类型=使用减少时，该条流水及之后的相同账户编码的全部流水的账户使用额度改成原账户使用额度 - 变更金额，
            creditAccountUsedLimit = creditAccountUsedLimit.subtract(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

            creditAccountFlowRepoProc.updateUsedAndAvailableById(
                    creditAccountUsedLimit,creditAccountAvailableLimit,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        }


    }



    /*****************************************/
    private void saveFlowRepair20(CreditAccountFlowRepairParam param){
        String accountCode = param.getAccountCode();
        LocalDateTime recalculationDate = param.getRecalculationDate();

        List<CreditAccountFlowRepairDO> repairDoAllList = new ArrayList<>();

        //1.信用账户
        List<CreditAccountFlowRepairHandle20DTO> xyRepairHandleAllList = new ArrayList<>();
        //1.1 查询信用账户 交易类型为账户增加\账户扣减
        List<CreditAccountFlowRepairHandle20DTO> xyRepairHandle21List = xyAppendSql21(accountCode,recalculationDate);
        if (CollectionUtil.isNotEmpty(xyRepairHandle21List)){
            xyRepairHandleAllList.addAll(xyRepairHandle21List);
        }
        //1.2 查询信用账户 交易类型为账户占用\占用释放
        List<CreditAccountFlowRepairHandle20DTO> xyRepairHandle22List = xyAppendSql22(accountCode,recalculationDate);
        if (CollectionUtil.isNotEmpty(xyRepairHandle22List)){
            xyRepairHandleAllList.addAll(xyRepairHandle22List);
        }
        //1.2 查询信用账户 交易类型为使用增加\使用减少
        List<CreditAccountFlowRepairHandle20DTO> xyRepairHandle23List = xyAppendSql23(accountCode,recalculationDate);
        if (CollectionUtil.isNotEmpty(xyRepairHandle23List)){
            xyRepairHandleAllList.addAll(xyRepairHandle23List);
        }
        //1.4组装保存信用账户
        assembleHandleFlowRepair20(param,xyRepairHandleAllList,repairDoAllList);



        if (CollectionUtil.isNotEmpty(repairDoAllList)){
            creditAccountFlowRepairRepo.saveAll(repairDoAllList);
        }else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未匹配到符合重算的数据!");
        }
    }

    private void assembleHandleFlowRepair20(CreditAccountFlowRepairParam param,List<CreditAccountFlowRepairHandle20DTO> repairHandleAllList,List<CreditAccountFlowRepairDO> repairDoAllList){
        if (CollectionUtil.isNotEmpty(repairHandleAllList)){
            Map<String, List<CreditAccountFlowRepairHandle20DTO>> xyRepairHandleAllMap = repairHandleAllList.stream().collect(Collectors.groupingBy(i -> i.getCreditAccountCode()));
            xyRepairHandleAllMap.forEach((key, value) -> {

                CreditAccountFlowRepairHandle20DTO repairHandle20DTO = value.stream().min((u1, u2) -> u1.getAuditDate().compareTo(u2.getAuditDate())).orElse(null);
                if (Objects.isNull(repairHandle20DTO)){
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "此信用账户("+key+")未匹配到最早审核时间的数据!");
                }

                CreditAccountFlowRepairDO flowRepairDO = new CreditAccountFlowRepairDO();
                assembleFlowRepair20(flowRepairDO,param,repairHandle20DTO);
                repairDoAllList.add(flowRepairDO);
            });

        }
    }

    private void assembleFlowRepair20(CreditAccountFlowRepairDO flowRepairDO,
                                      CreditAccountFlowRepairParam param,
                                      CreditAccountFlowRepairHandle20DTO repairHandle20DTO){

        flowRepairDO.setFlowNo(repairHandle20DTO.getFlowNo());
        flowRepairDO.setActualAmount(repairHandle20DTO.getAmount());
        flowRepairDO.setAmount(repairHandle20DTO.getAmount());
        flowRepairDO.setOpenAccountEntityCode(repairHandle20DTO.getObjectCode());
        flowRepairDO.setOpenAccountEntityName(repairHandle20DTO.getObjectName());
        flowRepairDO.setAccountType(repairHandle20DTO.getCreditAccountType());
        flowRepairDO.setAccountCode(repairHandle20DTO.getCreditAccountCode());
        flowRepairDO.setAccountName(repairHandle20DTO.getCreditAccountName());
        flowRepairDO.setSourceNo(repairHandle20DTO.getSourceNo());
        flowRepairDO.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());

        if (StringUtils.isNotBlank(param.getRecalculationReason())) {
            flowRepairDO.setRecalculationReason(param.getRecalculationReason());
        }
        if (Objects.nonNull(param.getRecalculationDate())){
            flowRepairDO.setRecalculationDate(param.getRecalculationDate());
        }else {
            flowRepairDO.setRecalculationDate(repairHandle20DTO.getPrevAuditDate());
        }

        SysUserDTO sysUserDTO = getUser();
        if (StringUtils.isNotBlank(param.getRecalculationUser())) {
            flowRepairDO.setRecalculationUser(param.getRecalculationUser());
        }else {
            flowRepairDO.setRecalculationUser(sysUserDTO.getPrettyName());
        }
        if (Objects.nonNull(param.getRecalculationUserId())){
            flowRepairDO.setRecalculationUserId(param.getRecalculationUserId());
        }else {
            flowRepairDO.setRecalculationUserId(sysUserDTO.getId());
        }
    }

    /**
     * 查询信用账户账户增加\账户扣减
     *
     * @param accountCode 信用账户编码
     * @param recalculationDate 重算日期
     * @return
     */
    private List<CreditAccountFlowRepairHandle20DTO> xyAppendSql21(String accountCode, LocalDateTime recalculationDate) {
        StringBuilder sqlWhereBuffer = new StringBuilder();
        Map<String, Object> params = new HashMap<>();
        List<Object> args = new ArrayList<>();
        appendWhere(accountCode,recalculationDate,sqlWhereBuffer,params,args);

        StringBuilder sqlBuffer = new StringBuilder();
        sqlBuffer.append("select b.* from \n" +
                "        (select \n" +
                "        credit_account_code,credit_account_name,credit_account_type,transaction_type,object_code,object_name, \n" +
                "        flow_no,amount,credit_account_limit,credit_account_used_limit,credit_account_occupancy_limit,credit_account_available_limit,audit_date,source_no, \n" +
                "        prev_credit_account_limit, \n" +
                "        prev_flow_no, \n" +
                "        prev_audit_date, \n" +
                "        if(prev_flow_no is null,amount,if(prev_credit_account_limit is null,0,prev_credit_account_limit - credit_account_limit)) difference_value \n" +
                "        from \n" +
                "                (select A.*, \n" +
                "                        lag(credit_account_limit, 1,0) over(partition by credit_account_code order by audit_date) prev_credit_account_limit, \n" +
                "                        lag(flow_no, 1) over(partition by account_code order by audit_date) prev_flow_no, \n" +
                "                        lag(audit_date, 1) over(partition by account_code order by audit_date) prev_audit_date \n" +
                "                from \n" +
                "                        credit_account_flow as A \n" +
                "                         where \n" +
                "                                A.delete_flag = 0 \n" +
                "                                and A.transaction_type in ('I','O') \n" +
                "                               and A.credit_account_type in ('CREDIT') \n" +
                "                               and A.audit_date is not null \n" +
                                                sqlWhereBuffer +
                "                ) T \n" +
                "        ) b \n" +
                "where \n" +
                "        abs(b.difference_value) != abs(b.amount) \n" +
                "       order by b.audit_date");


        List<CreditAccountFlowRepairHandle20DTO> creditAccountFlowRepairHandle20DTOList = namedTemplate.query(sqlBuffer.toString(),params, new BeanPropertyRowMapper<>(CreditAccountFlowRepairHandle20DTO.class));

        return creditAccountFlowRepairHandle20DTOList;
    }

    /**
     * 查询信用账户账户占用\占用释放
     *
     * @param accountCode 信用账户编码
     * @param recalculationDate 重算日期
     * @return
     */
    private List<CreditAccountFlowRepairHandle20DTO> xyAppendSql22(String accountCode,LocalDateTime recalculationDate) {
        StringBuilder sqlWhereBuffer = new StringBuilder();
        Map<String, Object> params = new HashMap<>();
        List<Object> args = new ArrayList<>();
        appendWhere(accountCode,recalculationDate,sqlWhereBuffer,params,args);

        StringBuilder sqlBuffer = new StringBuilder();
        sqlBuffer.append("select b.* from \n" +
                "        (select \n" +
                "        credit_account_code,credit_account_name,credit_account_type,transaction_type,object_code,object_name, \n" +
                "        flow_no,amount,credit_account_limit,credit_account_used_limit,credit_account_occupancy_limit,credit_account_available_limit,audit_date,source_no, \n" +
                "        prev_credit_account_occupancy_limit, \n" +
                "        prev_flow_no, \n" +
                "        prev_audit_date, \n" +
                "        if(prev_flow_no is null,amount,if(prev_credit_account_occupancy_limit is null,0,prev_credit_account_occupancy_limit - credit_account_occupancy_limit)) difference_value \n" +
                "        from \n" +
                "                (select A.*, \n" +
                "                        lag(credit_account_occupancy_limit, 1,0) over(partition by credit_account_code order by audit_date) prev_credit_account_occupancy_limit, \n" +
                "                        lag(flow_no, 1) over(partition by account_code order by audit_date) prev_flow_no, \n" +
                "                        lag(audit_date, 1) over(partition by account_code order by audit_date) prev_audit_date \n" +
                "                from \n" +
                "                        credit_account_flow as A \n" +
                "                         where \n" +
                "                                A.delete_flag = 0 \n" +
                "                                and A.transaction_type in ('T','R') \n" +
                "                               and A.credit_account_type in ('CREDIT') \n" +
                "                               and A.audit_date is not null \n" +
                                                sqlWhereBuffer +
                "                ) T \n" +
                "        ) b \n" +
                "where \n" +
                "        abs(b.difference_value) != abs(b.amount) \n" +
                "       order by b.audit_date");


        List<CreditAccountFlowRepairHandle20DTO> creditAccountFlowRepairHandle20DTOList = namedTemplate.query(sqlBuffer.toString(),params, new BeanPropertyRowMapper<>(CreditAccountFlowRepairHandle20DTO.class));

        return creditAccountFlowRepairHandle20DTOList;
    }


    /**
     * 查询信用账户使用增加\使用减少
     *
     * @param accountCode 信用账户编码
     * @param recalculationDate 重算日期
     * @return
     */
    private List<CreditAccountFlowRepairHandle20DTO> xyAppendSql23(String accountCode,LocalDateTime recalculationDate) {
        StringBuilder sqlWhereBuffer = new StringBuilder();
        Map<String, Object> params = new HashMap<>();
        List<Object> args = new ArrayList<>();
        appendWhere(accountCode,recalculationDate,sqlWhereBuffer,params,args);

        StringBuilder sqlBuffer = new StringBuilder();
        sqlBuffer.append("select b.* from \n" +
                "        (select \n" +
                "        credit_account_code,credit_account_name,credit_account_type,transaction_type,object_code,object_name, \n" +
                "        flow_no,amount,credit_account_limit,credit_account_used_limit,credit_account_occupancy_limit,credit_account_available_limit,audit_date,source_no, \n" +
                "        prev_credit_account_used_limit, \n" +
                "        prev_flow_no, \n" +
                "        prev_audit_date, \n" +
                "        if(prev_flow_no is null,amount,if(prev_credit_account_used_limit is null,0,prev_credit_account_used_limit - credit_account_used_limit)) difference_value \n" +
                "        from \n" +
                "                (select A.*, \n" +
                "                        lag(credit_account_used_limit, 1,0) over(partition by credit_account_code order by audit_date) prev_credit_account_used_limit, \n" +
                "                        lag(flow_no, 1) over(partition by account_code order by audit_date) prev_flow_no, \n" +
                "                        lag(audit_date, 1) over(partition by account_code order by audit_date) prev_audit_date \n" +
                "                from \n" +
                "                        credit_account_flow as A \n" +
                "                         where \n" +
                "                                A.delete_flag = 0 \n" +
                "                                and A.transaction_type in ('U','D') \n" +
                "                               and A.credit_account_type in ('CREDIT') \n" +
                "                               and A.audit_date is not null \n" +
                                                sqlWhereBuffer +
                "                ) T \n" +
                "        ) b \n" +
                "where \n" +
                "        abs(b.difference_value) != abs(b.amount) \n" +
                "       order by b.audit_date ");


        List<CreditAccountFlowRepairHandle20DTO> creditAccountFlowRepairHandle20DTOList = namedTemplate.query(sqlBuffer.toString(),params, new BeanPropertyRowMapper<>(CreditAccountFlowRepairHandle20DTO.class));

        return creditAccountFlowRepairHandle20DTOList;
    }



    private void appendWhere(String creditAccountCode, LocalDateTime recalculationDate, StringBuilder sqlWhereBuffer, Map<String, Object> params, List<Object> args) {
        if (StringUtils.isNotBlank(creditAccountCode)) {
            sqlWhereBuffer.append(" and A.credit_account_code = :creditAccountCode ");
            params.put("creditAccountCode", creditAccountCode);
        }
        if (Objects.nonNull(recalculationDate)) {
            sqlWhereBuffer.append(" and A.audit_date >= :auditDate ");
            params.put("auditDate", recalculationDate);

        }

       /* if (StringUtils.isNotBlank(creditAccountCode)) {
            sqlWhereBuffer.append(" and A.credit_account_code = ? ");
            args.add(creditAccountCode);
        }
        if (Objects.nonNull(recalculationDate)) {
            sqlWhereBuffer.append(" and A.audit_date >= ? ");
            args.add(recalculationDate);
        }*/

    }


    private void processingCalculation20(List<CreditAccountFlowRepairVO> repairFlowVOList,List<String> repairFlowNoList){

        List<String> accountCodeAllList = repairFlowVOList.stream().map(CreditAccountFlowRepairVO::getAccountCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<CreditAccountFlowDO> creditAccountFlowAllList = creditAccountFlowRepo.selectByCreditAccountCodeIn(false,accountCodeAllList);
        Map<String, CreditAccountFlowDO> creditAccountFlowAllMap = CollectionUtil.isEmpty(creditAccountFlowAllList) ? new HashMap<>() :
                creditAccountFlowAllList.stream().collect(Collectors.toMap(CreditAccountFlowDO::getCreditAccountCode, t -> t, (t1, t2) -> t1));

        for (CreditAccountFlowRepairVO creditAccountFlowRepairVO : repairFlowVOList) {
            String accountCode = creditAccountFlowRepairVO.getAccountCode();
            LocalDateTime recalculationDate = creditAccountFlowRepairVO.getRecalculationDate();
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            String successCode = transactionTemplate.execute(transactionStatus -> {

                try {
                    //信用账户的期初重算需要单独再写一下 第一条自身的不通类型的值变化是否一致，并且重算日期小于第一条时才走期初的
                    CreditAccountFlowDO creditAccountFlowDO = creditAccountFlowAllMap.get(accountCode);
                    if (Objects.isNull(creditAccountFlowDO)){
                        log.info("信用账户("+accountCode+")未查询到最早一条的审核流水!");
                        //释放校验缓存
                        removeFlowNoCache(creditAccountFlowRepairVO.getFlowNo());
                        return null;
                    }
                    if (Objects.isNull(recalculationDate)){
                        log.info("信用账户("+accountCode+")的重算日期为空!");
                        //释放校验缓存
                        removeFlowNoCache(creditAccountFlowRepairVO.getFlowNo());
                        return null;
                    }
                    String successAccountCode = null;

                    if (creditAccountFlowDO.getAuditDate().isAfter(recalculationDate)){
                        //期初重算 存储过程
                        successAccountCode = executeCreditAccountFlowProcedureInitial(accountCode,repairFlowNoList);

                    }else {
                        //重算 存储过程
                        successAccountCode = executeCreditAccountFlowProcedure(accountCode,recalculationDate,repairFlowNoList);

                    }

                    if (Objects.nonNull(successAccountCode)){
                        creditAccountFlowRepairRepoProc.updateRepairStatusById(UdcEnum.FLOW_REPAIR_STATUS_FIXED.getValueCode(),creditAccountFlowRepairVO.getId());
                        //更新完成时同步更新信用账户列表的金额，用最新一条流水记录的更新同账户编码的账户额度、账户使用额度、账户占用额度、账户可用额度。
                        updateAccountAmt(Collections.singletonList(accountCode),repairFlowNoList);

                        //更新信用账户快照表的对应数据
                        updateCreditAccountSnapshot(Collections.singletonList(creditAccountFlowRepairVO),repairFlowNoList);

                    }else {
                       /* repairFlowNoList.forEach(flowNo -> {
                            removeFlowNoCache(flowNo);
                        });*/
                    }

                    //释放校验缓存
                    removeFlowNoCache(creditAccountFlowRepairVO.getFlowNo());
                    return successAccountCode;
                } catch (Exception e) {

                    log.error("信用账户("+accountCode+")流水修复重算失败!:{}", e.getMessage());
                    //回滚
                    transactionStatus.setRollbackOnly();
                    repairFlowNoList.forEach(flowNo -> {
                        removeFlowNoCache(flowNo);
                    });
                    //抛出异常，不往下执行
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "信用账户("+accountCode+")流水修复重算失败!");
                }
            });

        }

        repairFlowNoList.forEach(flowNo -> {
            removeFlowNoCache(flowNo);
        });
    }

    private void updateCreditAccountSnapshot(List<CreditAccountFlowRepairVO> repairFlowVOList,List<String> repairFlowNoList){
        repairFlowVOList.forEach(accountFlowRepairVO -> {
            //更新信用账户快照表的对应数据
            //双层循环，快照表外层，流水表内层，当快照时间小于当前轮次的审核时间时返回上一个坐标i-1,若相等则返回本次坐标i
            CreditAccountFlowParam creditAccountFlowParam = new CreditAccountFlowParam();
            creditAccountFlowParam.setCreditAccountCode(accountFlowRepairVO.getAccountCode());
            creditAccountFlowParam.setRepairTime(accountFlowRepairVO.getRecalculationDate());
            List<CreditAccountFlowVO> creditAccountFlowVOList = creditAccountFlowService.selectRepairAfter(creditAccountFlowParam);
            Map<String, List<CreditAccountFlowVO>> creditAccountFlowAllMap = CollectionUtil.isEmpty(creditAccountFlowVOList) ? new HashMap<>() :
                    creditAccountFlowVOList.stream().collect(Collectors.groupingBy(i -> i.getCreditAccountCode()));


            CreditAccountSnapshotParam creditAccountSnapshotParam = new CreditAccountSnapshotParam();
            creditAccountSnapshotParam.setCreditAccountCode(accountFlowRepairVO.getAccountCode());
            creditAccountSnapshotParam.setRepairTime(accountFlowRepairVO.getRecalculationDate());
            List<CreditAccountSnapshotDO>  creditAccountSnapshotDOList = creditAccountService.selectCreditAccountSnapshotByParam(creditAccountSnapshotParam);
            Map<String, List<CreditAccountSnapshotDO>> creditAccountSnapshotAllMap = CollectionUtil.isEmpty(creditAccountSnapshotDOList) ? new HashMap<>() :
                    creditAccountSnapshotDOList.stream().collect(Collectors.groupingBy(i -> i.getCreditAccountCode()));


            List<CreditAccountSnapshotDO>  creditAccountSnapshotUpdateList = new ArrayList<>();
            //accountSnapshotAllMap.forEach((key, value) -> {
            for (Map.Entry<String, List<CreditAccountSnapshotDO>> entry : creditAccountSnapshotAllMap.entrySet()) {
                String creditAccountCode = entry.getKey();
                // 根据快照时间排序
                Collections.sort(entry.getValue(), Comparator.comparing(CreditAccountSnapshotDO::getSnapshotTime));

                List<CreditAccountFlowVO> creditAccountFlowList = creditAccountFlowAllMap.get(creditAccountCode);
                // 根据审核时间排序
                Collections.sort(creditAccountFlowList, Comparator.comparing(CreditAccountFlowVO::getAuditDate));

                //Integer lastSnapshotCoordinate = 0;
                //Integer snapshotSize = entry.getValue().size();
                //Integer flowSize = creditAccountFlowList.size();
                for (CreditAccountSnapshotDO creditAccountSnapshotDO : entry.getValue()) {

                    //lastSnapshotCoordinate = lastSnapshotCoordinate + 1;
                    //流水坐标
                    //Integer flowCoordinate = null;
                    //Integer lastFlowCoordinate = 0;
                    List<CreditAccountFlowVO> creditAccountFlowFiltrationList = new ArrayList<>();
                    for (int i = 0; i < creditAccountFlowList.size(); i++) {
                        CreditAccountFlowVO creditAccountFlowVO = creditAccountFlowList.get(i);

                        //lastFlowCoordinate = lastFlowCoordinate + 1;
                        //Integer j = i;
                        if (creditAccountFlowVO.getAuditDate().isEqual(creditAccountSnapshotDO.getSnapshotTime())){
                            //flowCoordinate = j;
                            creditAccountFlowFiltrationList.add(creditAccountFlowVO);
                            //break;
                        } else if (creditAccountFlowVO.getAuditDate().isBefore(creditAccountSnapshotDO.getSnapshotTime())) {
                            //flowCoordinate = (j == 0) ? j : j-1;
                            creditAccountFlowFiltrationList.add(creditAccountFlowVO);
                            //break;
                        }else {
                            break;
                        }
                        /*else if (lastSnapshotCoordinate.equals(snapshotSize) && lastFlowCoordinate.equals(flowSize)){
                            creditAccountFlowFiltrationList.add(creditAccountFlowVO);
                        }*/
                    }

                    CreditAccountFlowVO maxDateAccountFlow = creditAccountFlowFiltrationList.stream().filter(Objects::nonNull)
                            .max(Comparator.comparing(CreditAccountFlowVO::getAuditDate)).orElse(null);

                    if (Objects.nonNull(maxDateAccountFlow)){
                        //CreditAccountFlowVO creditAccountFlowVO = creditAccountFlowList.get(flowCoordinate);
                        creditAccountSnapshotDO.setCreditAccountLimit(maxDateAccountFlow.getCreditAccountLimit());
                        creditAccountSnapshotDO.setCreditAccountAvailableLimit(maxDateAccountFlow.getCreditAccountAvailableLimit());
                        creditAccountSnapshotDO.setCreditAccountOccupancyLimit(maxDateAccountFlow.getCreditAccountOccupancyLimit());
                        creditAccountSnapshotDO.setCreditAccountUsedLimit(maxDateAccountFlow.getCreditAccountUsedLimit());

                        //creditAccountSnapshotDO.setSnapshotTime(LocalDateTime.now());
                        creditAccountSnapshotUpdateList.add(creditAccountSnapshotDO);
                    }
                }

                //处理最后一条快照对应最后一条流水
                CreditAccountSnapshotDO lastSnapshot = entry.getValue().get(entry.getValue().size() - 1);
                CreditAccountFlowVO lastFlow = creditAccountFlowList.get(creditAccountFlowList.size() - 1);

               /*Optional<CreditAccountSnapshotDO> lastSnapshotOptional = creditAccountSnapshotUpdateList.stream().filter(creditAccountSnapshotDO -> Objects.equals(creditAccountSnapshotDO.getId(),lastSnapshot.getId())).findFirst();
               if (lastSnapshotOptional.isPresent() || lastSnapshot.getSnapshotTime().isAfter(lastFlow.getAuditDate())){

               }*/

                creditAccountSnapshotUpdateList.forEach(creditAccountSnapshotDO -> {
                  if (Objects.equals(creditAccountSnapshotDO.getId(),lastSnapshot.getId())) {
                      creditAccountSnapshotDO.setCreditAccountLimit(lastFlow.getCreditAccountLimit());
                      creditAccountSnapshotDO.setCreditAccountAvailableLimit(lastFlow.getCreditAccountAvailableLimit());
                      creditAccountSnapshotDO.setCreditAccountOccupancyLimit(lastFlow.getCreditAccountOccupancyLimit());
                      creditAccountSnapshotDO.setCreditAccountUsedLimit(lastFlow.getCreditAccountUsedLimit());
                  }
                });
            }

            if (CollectionUtil.isNotEmpty(creditAccountSnapshotUpdateList)){
                creditAccountSnapshotRepo.saveAll(creditAccountSnapshotUpdateList);

            }
        });

    }


    public String executeCreditAccountFlowProcedure(String creditAccountCode,LocalDateTime retryTime,List<String> repairFlowNoList) {

        if (StringUtils.isBlank(creditAccountCode)){
            log.info("信用账户编码为空无法重算");
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "信用账户编码为空无法重算!");
        }
        if (Objects.isNull(retryTime)){
            log.info("重算时间为空无法重算");
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重算时间为空无法重算!");
        }

        StoredProcedureQuery storedProcedureQuery = entityManager.createStoredProcedureQuery("credit_account_flow_repair");
        storedProcedureQuery.registerStoredProcedureParameter("credit_account_code_i", String.class, ParameterMode.IN);
        storedProcedureQuery.registerStoredProcedureParameter("audit_date_i", LocalDateTime.class, ParameterMode.IN);
        storedProcedureQuery.setParameter("credit_account_code_i", creditAccountCode);
        storedProcedureQuery.setParameter("audit_date_i", retryTime);
        storedProcedureQuery.execute();


        List  resultList = storedProcedureQuery.getResultList();
        log.info("处理信用账户流水存储过程的结果:"+resultList.toString());
        if (CollectionUtil.isEmpty(resultList)){
            return null;
        }

        return creditAccountCode;
    }

    public String executeCreditAccountFlowProcedureInitial(String creditAccountCode,List<String> repairFlowNoList) {

        if (StringUtils.isBlank(creditAccountCode)){
            log.info("信用账户编码为空无法期初重算");
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "信用账户编码为空无法期初重算!");
        }


        StoredProcedureQuery storedProcedureQuery = entityManager.createStoredProcedureQuery("credit_account_flow_repair_from_initial");
        storedProcedureQuery.registerStoredProcedureParameter("credit_account_code_i", String.class, ParameterMode.IN);
        storedProcedureQuery.setParameter("credit_account_code_i", creditAccountCode);
        storedProcedureQuery.execute();


        List  resultList = storedProcedureQuery.getResultList();
        log.info("处理信用账户流水期初存储过程的结果:"+resultList.toString());
        if (CollectionUtil.isEmpty(resultList)){
            return null;
        }

        return creditAccountCode;
    }



    /*public Boolean repairCalculationInit(CreditAccountFlowDO creditAccountFlowDO) {
        //账户额度
        BigDecimal creditAccountLimit = creditAccountFlowDO.getCreditAccountLimit();
        //账户占用额度
        BigDecimal creditAccountOccupancyLimit = creditAccountFlowDO.getCreditAccountOccupancyLimit();
        //账户使用额度
        BigDecimal creditAccountUsedLimit = creditAccountFlowDO.getCreditAccountUsedLimit();
        //账户可用额度 = 账户额度-账户使用额度-账户占用额度
        BigDecimal creditAccountAvailableLimit = creditAccountFlowDO.getCreditAccountAvailableLimit();
        //发生金额
        BigDecimal amount = creditAccountFlowDO.getAmount();


        if (Objects.equals(creditAccountFlowDO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_R.getValueCode())) {
            //交易类型=释放占用时，该条流水及之后的相同账户编码的全部流水的账户占用额度改成原账户占用额度 - 变动金额，
            BigDecimal creditAccountOccupancyAmt = creditAccountOccupancyLimit.subtract(amount);
            BigDecimal creditAccountAvailableAmt = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);


        } else if (Objects.equals(creditAccountFlowDO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_T.getValueCode())) {
            //交易类型=账户占用时，该条流水及之后的相同账户编码的全部流水的账户占用额度改成原账户占用额度 + 变动金额，
            creditAccountOccupancyLimit = creditAccountOccupancyLimit.add(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);


        } else if (Objects.equals(creditAccountFlowDO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_O.getValueCode())) {
            //交易类型=账户扣减时，该条流水之后的相同账户编码的全部流水的账户额度改成原账户额度 - 变更金额，
            creditAccountLimit = creditAccountLimit.subtract(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);


        } else if (Objects.equals(creditAccountFlowDO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_I.getValueCode())) {
            //交易类型=账户增加时，该条流水及之后的相同账户编码的全部流水的账户额度改成原账户额度 + 变更金额，
            creditAccountLimit = creditAccountLimit.add(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);




        } else if (Objects.equals(creditAccountFlowDO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_U.getValueCode())) {
            //交易类型=使用增加时，该条流水及之后的相同账户编码的全部流水的账户使用额度改成原账户使用额度 + 变更金额
            creditAccountUsedLimit = creditAccountUsedLimit.add(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);


        } else if (Objects.equals(creditAccountFlowDO.getTransactionType(), UdcEnum.CREDIT_IO_TYPE_D.getValueCode())) {
            //交易类型=使用减少时，该条流水及之后的相同账户编码的全部流水的账户使用额度改成原账户使用额度 - 变更金额，
            creditAccountUsedLimit = creditAccountUsedLimit.subtract(variableAmount);
            creditAccountAvailableLimit = creditAccountLimit.subtract(creditAccountUsedLimit).subtract(creditAccountOccupancyLimit);

        }

    }*/

}
