package com.elitesland.fin.repo.account;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.facade.dto.account.AccountDTO;
import com.elitesland.fin.application.facade.param.account.AccountPageParam;
import com.elitesland.fin.application.facade.param.account.AccountParam;
import com.elitesland.fin.application.facade.vo.account.AccountVO;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.account.AccountDO;
import com.elitesland.fin.entity.account.QAccountDO;
import com.elitesland.fin.utils.StringUtil;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.repo
 * @date 2023/2/16 15:00
 */
@Component
@RequiredArgsConstructor
public class AccountRepoProc {

    private final JPAQueryFactory jpaQueryFactory;

    private final QAccountDO qAccountDO = QAccountDO.accountDO;

    private final AccountRepo accountRepo;

    public PagingVO<AccountVO> page(AccountPageParam accountPageParam){
        JPAQuery<AccountVO> query = buildJPAQuery(accountPageParam);

        accountPageParam.setPaging(query);
        accountPageParam.fillOrders(query, qAccountDO);
        List<AccountVO> fetch = query.fetch();

        return PagingVO.<AccountVO>builder()
                .total(query.fetchCount())
                .records(fetch)
                .build();
    }

    public List<AccountVO> queryList(AccountPageParam accountPageParam){
        JPAQuery<AccountVO> query = buildJPAQuery(accountPageParam);
        return query.fetch();
    }



    public JPAQuery<AccountVO> buildJPAQuery(AccountPageParam accountPageParam) {

        List<Predicate> predicates = new ArrayList<>();

        if (StringUtils.isNotBlank(accountPageParam.getAccountHolderName())) {
            predicates.add(qAccountDO.accountHolderName.like('%' + accountPageParam.getAccountHolderName() + '%'));
        }
        if(CollectionUtils.isNotEmpty(accountPageParam.getAccountHolderNameList())){
            predicates.add(qAccountDO.accountHolderName.in(accountPageParam.getAccountHolderNameList()));
        }
        if (StringUtils.isNotBlank(accountPageParam.getAccountHolderCode())) {
            predicates.add(qAccountDO.accountHolderCode.eq(accountPageParam.getAccountHolderCode()));
        }
        if(CollectionUtils.isNotEmpty(accountPageParam.getAccountHolderCodeList())){
            predicates.add(qAccountDO.accountHolderCode.in(accountPageParam.getAccountHolderCodeList()));
        }
        if (StringUtils.isNotBlank(accountPageParam.getAccountName())) {
            predicates.add(qAccountDO.accountName.like('%' + accountPageParam.getAccountName() + '%'));
        }
        if(CollectionUtils.isNotEmpty(accountPageParam.getAccountNameList())){
            predicates.add(qAccountDO.accountName.in(accountPageParam.getAccountNameList()));
        }
        if (StringUtils.isNotBlank(accountPageParam.getAccountType())) {
            predicates.add(qAccountDO.accountType.eq(accountPageParam.getAccountType()));
        }
        if(CollectionUtils.isNotEmpty(accountPageParam.getAccountTypeList())){
            predicates.add(qAccountDO.accountType.in(accountPageParam.getAccountTypeList()));
        }
        if (StringUtils.isNotBlank(accountPageParam.getState())) {
            predicates.add(qAccountDO.state.eq(accountPageParam.getState()));
        }

        if (StringUtils.isNotBlank(accountPageParam.getAccountCode())) {
            predicates.add(qAccountDO.accountCode.like(StringUtil.buildLikeString(accountPageParam.getAccountCode())));
        }
        if(CollectionUtils.isNotEmpty(accountPageParam.getAccountCodeList())){
            predicates.add(qAccountDO.accountCode.in(accountPageParam.getAccountCodeList()));
        }
        if (StringUtils.isNotBlank(accountPageParam.getSecOuCode())) {
            predicates.add(qAccountDO.secOuCode.eq(accountPageParam.getSecOuCode()));
        }

        if (StringUtils.isNotBlank(accountPageParam.getAccountHolderNameCode())) {
            predicates.add(qAccountDO.accountHolderName.like(StringUtil.buildLikeString(accountPageParam.getAccountHolderNameCode())).
                    or(qAccountDO.accountHolderCode.like(StringUtil.buildLikeString(accountPageParam.getAccountHolderNameCode()))));
        }

        if (CollectionUtil.isNotEmpty(accountPageParam.getIds())) {
            predicates.add(qAccountDO.id.in(accountPageParam.getIds()));
        }

        JPAQuery<AccountVO> query = select(AccountVO.class).where(ExpressionUtils.allOf(predicates));
        return query;
    }

    public List<AccountVO> queryByAccounts(List<String> accounts, String type, String state) {
        List<Predicate> predicates = new ArrayList<>();
        if (!StringUtils.isEmpty(type)) {
            predicates.add(qAccountDO.accountType.eq(type));
        }
        if (!CollectionUtils.isEmpty(accounts)) {
            predicates.add(qAccountDO.accountCode.in(accounts));
        }
        if (!StringUtils.isEmpty(state)) {
            predicates.add(qAccountDO.state.eq(state));
        }
        return select(AccountVO.class)
                .where(ExpressionUtils.allOf(predicates))
                .fetch();
    }

    public Boolean updateAmtByCode(AccountParam accountParam) {
        Long res = jpaQueryFactory.update(qAccountDO)
                .set(qAccountDO.accountAmount, accountParam.getAccAmt())
                .set(qAccountDO.accountOccupancyAmount, accountParam.getAccOccAmt())
                .where(qAccountDO.accountCode.eq(accountParam.getAccCode()))
                .execute();
        return res > 0;
    }

    public <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                qAccountDO.id,
                qAccountDO.accountName,
                qAccountDO.accountCode,
                qAccountDO.accountType,
                qAccountDO.state,
                qAccountDO.createTime,
                qAccountDO.creator,
                qAccountDO.accountOccupancyAmount,
                qAccountDO.accountAmount.subtract(qAccountDO.accountOccupancyAmount).as("accAvaAmt"),
                qAccountDO.accountAmount,
                qAccountDO.accountAvailableAmount,
                qAccountDO.secOuCode,
                qAccountDO.secOuName,
                qAccountDO.secFranchiseeCode,
                qAccountDO.accountHolderType,
                qAccountDO.accountHolderName,
                qAccountDO.accountHolderCode,
                qAccountDO.remark
                )
        ).from(qAccountDO);
    }

    public Long updateState(AccountParam accountParam) {
        long res = jpaQueryFactory.update(qAccountDO)
                .set(qAccountDO.state, accountParam.getState())
                .where(qAccountDO.id.in(accountParam.getIds()))
                .execute();
        return res;
    }

    public AccountVO get(Long id) {
        AccountVO accountVO = select(AccountVO.class)
                .where(qAccountDO.id.eq(id))
                .fetchOne();
        return accountVO;
    }

    public AccountVO getByCode(String accCode) {
        AccountVO res = select(AccountVO.class)
                .where(qAccountDO.accountCode.eq(accCode))
                .fetchOne();
        return res;
    }

    public Boolean updateOccAmtByCode(AccountParam accountParam) {
        long res = jpaQueryFactory.update(qAccountDO)
                .set(qAccountDO.accountOccupancyAmount, accountParam.getAccOccAmt())
                .where(qAccountDO.accountCode.eq(accountParam.getAccCode()))
                .execute();
        return res > 0;

    }

    public Boolean updateAccAmtByCode(AccountParam accountParam) {
        long res = jpaQueryFactory.update(qAccountDO)
                .set(qAccountDO.accountAmount, accountParam.getAccAmt())
                .where(qAccountDO.accountCode.eq(accountParam.getAccCode()))
                .execute();
        return res > 0;

    }


    public void updateAccountAmountByAccountCode(AccountParam accountParam) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qAccountDO);
        if (accountParam.getAccountAmount() != null) {
            jpaUpdateClause.set(qAccountDO.accountAmount, accountParam.getAccountAmount());
        }
        if (accountParam.getAccountOccupancyAmount() != null) {
            jpaUpdateClause.set(qAccountDO.accountOccupancyAmount, accountParam.getAccountOccupancyAmount());
        }

        if (accountParam.getAccountAvailableAmount() != null) {
            jpaUpdateClause.set(qAccountDO.accountAvailableAmount, accountParam.getAccountAvailableAmount());
        }
        jpaUpdateClause.where(qAccountDO.accountCode.eq(accountParam.getAccountCode())).execute();
    }

    public List<AccountDTO> getAccountByAccountParam(AccountParam param) {
        JPAQuery<AccountDTO> accountVOJPAQuery = select(AccountDTO.class);
        if (CollectionUtils.isNotEmpty(param.getAccountCodes())) {
            accountVOJPAQuery.where(qAccountDO.accountCode.in(param.getAccountCodes()));
        }

        if (CollectionUtils.isNotEmpty(param.getAccountHolderNames())) {
            accountVOJPAQuery.where(qAccountDO.accountHolderName.in(param.getAccountHolderNames()));
        }

        if (StringUtils.isNotEmpty(param.getAccountCode())) {
            accountVOJPAQuery.where(qAccountDO.accountCode.eq(param.getAccountCode()));
        }

        if (StringUtils.isNotEmpty(param.getAccountName())) {
            accountVOJPAQuery.where(qAccountDO.accountName.eq(param.getAccountName()));
        }

        if (StringUtils.isNotEmpty(param.getAccountHolderName())) {
            accountVOJPAQuery.where(qAccountDO.accountHolderName.eq(param.getAccountHolderName()));
        }

        if (StringUtils.isNotEmpty(param.getAccountHolderCode())) {
            accountVOJPAQuery.where(qAccountDO.accountHolderCode.eq(param.getAccountHolderCode()));
        }

        if (StringUtils.isNotEmpty(param.getSecOuCode())) {
            accountVOJPAQuery.where(qAccountDO.secOuCode.eq(param.getSecOuCode()));
        }

        if (StringUtils.isNotEmpty(param.getAccountType())) {
            accountVOJPAQuery.where(qAccountDO.accountType.eq(param.getAccountType()));
        }

        if (StringUtils.isNotEmpty(param.getState())) {
            accountVOJPAQuery.where(qAccountDO.state.eq(param.getState()));
        }

        return accountVOJPAQuery.fetch();
    }

    public AccountDO findByAccountCode(String accountCode) {
        List<AccountDO> accountDOList = accountRepo.findAllByAccountCode(accountCode);
        Assert.notEmpty(accountDOList, "账户不存在");
        Assert.isTrue(accountDOList.size() == 1, "查询到多条账户");
        AccountDO accountDO = accountDOList.get(0);
        Assert.equals(accountDO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");
        return accountDOList.get(0);
    }
    public AccountVO getVOByCode(String accCode) {
        List<AccountVO>  accountVoList= select(AccountVO.class)
                .where(qAccountDO.accountCode.eq(accCode))
                .fetch();
        Assert.notEmpty(accountVoList, "账户不存在");
        Assert.isTrue(accountVoList.size() == 1, "查询到多条账户");
        AccountVO accountVO = accountVoList.get(0);
        Assert.equals(accountVO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");
        return accountVO;
    }
}
