package com.elitesland.fin.application.service.creditaccount;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.fin.application.convert.adjusttoorder.AdjustToOrderConvert;
import com.elitesland.fin.application.convert.creditaccount.CreditAutoRepaymentConditionConvert;
import com.elitesland.fin.application.facade.param.account.AccountPageParam;
import com.elitesland.fin.application.facade.param.adjusttoorder.AdjustToOrderParam;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountPageParam;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAutoRepaymentJobParam;
import com.elitesland.fin.application.facade.vo.account.AccountVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountPageVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditSettingDetailVO;
import com.elitesland.fin.application.service.adjusttoorder.AdjustToOrderService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.SysNumEnum;
import com.elitesland.fin.common.SysNumberGenerator;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.account.AccountDO;
import com.elitesland.fin.entity.adjusttoorder.AdjustToOrderDO;
import com.elitesland.fin.entity.creditaccount.CreditAccountDO;
import com.elitesland.fin.entity.creditaccount.CreditAutoRepaymentConditionDO;
import com.elitesland.fin.repo.account.AccountRepoProc;
import com.elitesland.fin.repo.adjusttoorder.AdjustToOrderRepo;
import com.elitesland.fin.repo.adjusttoorder.AdjustToOrderRepoProc;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepoProc;
import com.elitesland.fin.repo.creditaccount.CreditAutoRepaymentConditionRepo;
import com.elitesland.fin.repo.creditaccount.CreditSettingRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

/**
 * <p>
 * 功能说明:信用账户自动还款
 * </p>
 *
 * @Author Darren
 * @Date 2024/01/09
 * @Version 1.0
 * @Content:
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class CreditAccountAutoRepaymentServiceImpl implements CreditAccountAutoRepaymentService{

    private final CreditSettingRepoProc creditSettingRepoProc;
    private final AccountRepoProc accountRepoProc;
    private final CreditAccountRepoProc creditAccountRepoProc;
    private final CreditAutoRepaymentConditionRepo autoRepaymentConditionRepo;
    public static final String CREDIT_ACCOUNT_TABLE_NAME ="credit_account";
    public static final String ACCOUNT_TABLE_NAME = "account";
    public static final String DEFAULT_USER_NAME = "系统自动";
    public static final Long DEFAULT_USER_ID = 0L;
    public static final String DEFAULT_ADJUST_TO_ORDER_REMARK = "信用账户自动还款";

    private final SysNumberGenerator sysNumberGenerator;
    private final AdjustToOrderRepo adjustToOrderRepo;
    private final AdjustToOrderRepoProc adjustToOrderRepoProc;
    private final AdjustToOrderService adjustToOrderService;
    private final UdcProvider udcProvider;
    // 注入EntityManager
    @PersistenceContext
    private EntityManager entityManager;
    private final TransactionTemplate transactionTemplate;


    @Override
    public void creditAccountAutoRepayment(String param){

        CreditAutoRepaymentJobParam jobParam = new CreditAutoRepaymentJobParam();
        if (StringUtils.isNotBlank(param)) {
            JSONObject jsonObject = JSONObject.parseObject(param);
            CreditAutoRepaymentJobParam autoRepaymentJobParam = JSONObject.toJavaObject(jsonObject, CreditAutoRepaymentJobParam.class);
            if (Objects.nonNull(autoRepaymentJobParam)) {
                CreditAutoRepaymentConditionConvert.INSTANCE.oldToNewDO(autoRepaymentJobParam,jobParam);
            }
        }

        this.autoRepayment(jobParam);

    }

    @Override
    public void autoRepayment(CreditAutoRepaymentJobParam jobParam) {
        log.info("登录人信息：{}", SecurityContextUtil.currentUser());
        log.info("信用账户自动还款定时任务参数:{}", JSON.toJSONString(jobParam));

        //查询信用设置
        CreditSettingDetailVO creditSettingDetailVO = creditSettingRepoProc.findAll();
        if (Objects.isNull(creditSettingDetailVO)){
            log.info("查询信用设置为空");
            return;
        }
        //先断信用设置项是否设置自动还款，设置【否】时无需触发自动还款；
        if (Objects.nonNull(creditSettingDetailVO.getAutoRepaymentFlag()) && creditSettingDetailVO.getAutoRepaymentFlag()){
            //设置【是】时 先判断信用账户是否满足设置项的条件，满足的通过信用账户对应的公司和客户去找到该储值账户是否满足设置的【编辑条件】，
            // 如果满足触发信用账户的自动还款。不满足不触发。（条件默认为储值账户可用金额>0,信用账户使用额度>0，账户是启用状态）

            //满足自动还款条件的信用账户
            List<CreditAccountDO> creditAccountConditionList = new ArrayList<>();
            //满足自动还款条件的账户
            List<AccountDO> accountConditionList = new ArrayList<>();

            //查询信用设置的自动还款条件
            List<CreditAutoRepaymentConditionDO> autoRepaymentConditionDOList = autoRepaymentConditionRepo.findAllByMasId(creditSettingDetailVO.getId());
            //按照表名分组
            Map<String, List<CreditAutoRepaymentConditionDO>> autoRepaymentConditionMap = autoRepaymentConditionDOList.stream().filter(i -> StringUtils.isNotBlank(i.getTableName()))
                    .collect(Collectors.groupingBy(i -> i.getTableName()));

            autoRepaymentConditionMap.forEach((key, values) -> {
                String buildSql = buildSql(key,values);
                if (Objects.equals(key,CREDIT_ACCOUNT_TABLE_NAME)){
                    List<CreditAccountDO> creditAccountDOList = selectCreditAccountData(buildSql);
                    log.info("查询满足自动还款条件的信用账户:{}", JSON.toJSONString(creditAccountDOList));
                    if (CollectionUtil.isNotEmpty(creditAccountDOList)){
                        creditAccountConditionList.addAll(creditAccountDOList);
                    }
                }else if (Objects.equals(key,ACCOUNT_TABLE_NAME)){
                    List<AccountDO> accountDOList = selectAccountData(buildSql);
                    log.info("查询满足自动还款条件的账户:{}", JSON.toJSONString(accountDOList));
                    if (CollectionUtil.isNotEmpty(accountDOList)){
                        accountConditionList.addAll(accountDOList);

                    }
                }

            });

            //判断信用账户是否满足设置项的条件
            if (CollectionUtil.isEmpty(creditAccountConditionList) || creditAccountConditionList.size() < 1){
                log.info("无满足自动还款条件的信用账户");
                return;
            }
            List<Long> creditAccountConditionIds = creditAccountConditionList.stream().map(CreditAccountDO::getId).collect(Collectors.toList());
            //判断账户是否满足设置项的条件
            if (CollectionUtil.isEmpty(accountConditionList) || accountConditionList.size() < 1){
                log.info("无满足自动还款条件的账户");
                return;
            }
            List<Long> accountConditionIds = accountConditionList.stream().map(AccountDO::getId).collect(Collectors.toList());

            //查询信用账户列表-启用
            CreditAccountPageParam creditAccountPageParam = new CreditAccountPageParam();
            creditAccountPageParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
            List<CreditAccountPageVO> creditAccountVOList = creditAccountRepoProc.queryList(creditAccountPageParam);
            log.info("查询信用账户集合:{}", JSON.toJSONString(creditAccountVOList));

            if (CollectionUtil.isEmpty(creditAccountVOList)){
                log.info("查询信用账户集合为空");
                return;
            }

            //过滤出符合自动还款设置条件的信用账户
            List<CreditAccountPageVO> creditAccountVoFilters = creditAccountVOList.stream().filter(creditAccountPageVO -> creditAccountConditionIds.contains(creditAccountPageVO.getId())).collect(Collectors.toList());
            if (CollectionUtil.isEmpty(creditAccountVoFilters) || creditAccountVoFilters.size() < 1){
                log.info("无符合自动还款设置条件的信用账户");
                return;
            }

            Map<String, Map<String, String>> udcCode = udcProvider.getValueMapByUdcCode(UdcEnum.ACCOUNT_TYPE_STORE.getModel(), Set.of(UdcEnum.ACCOUNT_TYPE_STORE.getCode(), UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getCode()));
            Map<String, String> accountTypeMap = udcCode.get(UdcEnum.ACCOUNT_TYPE_STORE.getCode());
            Map<String, String> creditAccountTypeMap = udcCode.get(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getCode());

            //循环处理
            for (CreditAccountPageVO creditAccountPageVO :creditAccountVoFilters) {
                String ouCode = creditAccountPageVO.getOuCode();
                String objectCode = creditAccountPageVO.getObjectCode();
                //查询账户列表
                AccountPageParam accountPageParam = new AccountPageParam();
                accountPageParam.setSecOuCode(ouCode);
                accountPageParam.setAccountHolderCode(objectCode);
                accountPageParam.setState(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
                accountPageParam.setAccountType(UdcEnum.ACCOUNT_TYPE_STORE.getValueCode());
                List<AccountVO> accountVOList = accountRepoProc.queryList(accountPageParam);
                log.info("查询储值账户集合:{}", JSON.toJSONString(accountVOList));
                if (CollectionUtil.isEmpty(accountVOList)){
                    log.info("查询储值账户集合为空");
                    continue;
                }
                if (CollectionUtil.isNotEmpty(accountVOList) && accountVOList.size() > 1){
                    //throw new BusinessException("根据公司("+ouCode+")" + "+客户("+objectCode+")维度查出多个储值账户数据");
                    log.info("根据公司("+ouCode+")" + "+客户("+objectCode+")维度查出多个储值账户数据");
                    continue;
                }
                AccountVO accountVO = accountVOList.get(0);
                //过滤出符合自动还款设置条件的储值账户
                /*List<AccountVO> accountVoFilters = accountVOList.stream().filter(accountVO -> accountConditionIds.contains(accountVO.getId())).collect(Collectors.toList());
                if (CollectionUtil.isEmpty(accountVoFilters) || accountVoFilters.size() < 1){
                    log.info("无符合自动还款设置条件的储值账户,公司:{}, 客户:{}",ouCode,objectCode);
                    break;
                }*/
                if (!accountConditionIds.contains(accountVO.getId())){
                    log.info("无符合自动还款设置条件的储值账户,公司:{}, 客户:{}",ouCode,objectCode);
                    continue;
                }

                //返回的调整金额为空则表示无需还款
                BigDecimal adjustmentAmt = computeAdjustmentAmt(creditAccountPageVO,accountVO);
                if (Objects.isNull(adjustmentAmt)){
                    log.info("调整金额为空则无需还款");
                    continue;
                }

                transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
                Integer successCode = transactionTemplate.execute(transactionStatus -> {
                    try {
                        //满足触发信用账户的自动还款
                        //自动生成审批通过的储值账户调剂到信用账户的调剂单，按照调剂单的账户规则生成对应的账户流水。
                        //调剂单生成时默认审核通过，自动生成审批中和审批通过时调用的账户流水规则。
                        generateAdjustToOrder(creditAccountPageVO,accountVO,adjustmentAmt,accountTypeMap,creditAccountTypeMap,jobParam);
                        return 1;
                    } catch (Exception e) {
                        log.error("公司("+ouCode+")与对象编码("+objectCode+")信用账户自动还款失败!:{}",  e.getMessage());
                        //回滚
                        transactionStatus.setRollbackOnly();
                        //抛出异常，不往下执行
                        //throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "公司("+ouCode+")与对象编码("+objectCode+")信用账户自动还款失败!");
                        return 0;
                    }


                });
            }


        }else {
            log.info("信用设置的是否设置自动还款为【否】时无需触发");
            return;
        }


    }

    /**
     *
     *
     *先判断对应的信用账户是否启用，未启用不用做后续判断；如果信用账户启用，做下述判断
     *   （1）如果信用账户的的账户使用金额<=0，则无需还款；
     *   （2）如果信用账户的的账户使用金额是否>0，且该公司+客户对应的储值账户可用金额<=0,则无需还款；
     *   （3）如果信用账户的的账户使用金额是否>0，且该公司+客户对应的储值账户可用金额>0,则需要还款。
     *   信用账户使用金额>=储值账户可用金额，调整金额=储值账户可用金额；信用账户使用金额<储值账户可用金额，调整金额=信用账户使用金额。
     *
     * @param creditAccountPageVO 信用账户
     * @param accountVO 储值账户
     *
     * @return
     */
    private BigDecimal computeAdjustmentAmt(CreditAccountPageVO creditAccountPageVO,AccountVO accountVO){

        BigDecimal creditAccountUsedLimit = creditAccountPageVO.getCreditAccountUsedLimit();
        BigDecimal accountAvailableAmount = accountVO.getAccountAvailableAmount();

        if ((creditAccountUsedLimit.compareTo(BigDecimal.ZERO) == 1) && (accountAvailableAmount.compareTo(BigDecimal.ZERO) == 1)){
            BigDecimal adjustmentAmt = BigDecimal.ZERO;
            if(creditAccountUsedLimit.compareTo(accountAvailableAmount) > -1){
                adjustmentAmt = accountAvailableAmount;
                return adjustmentAmt;
            }else if(creditAccountUsedLimit.compareTo(accountAvailableAmount) == -1){
                adjustmentAmt = creditAccountUsedLimit;
                return adjustmentAmt;
            }
        }

        return null;
    }

    private String buildSql(String tableName,List<CreditAutoRepaymentConditionDO> autoRepaymentConditionDOList){
        StringBuilder stringBuilder  = new StringBuilder();
        stringBuilder.append(FinConstant.SELECT);
        stringBuilder.append(FinConstant.BLANK);
        stringBuilder.append(FinConstant.ASTERISK);
        stringBuilder.append(FinConstant.BLANK);
        stringBuilder.append(FinConstant.FROM);
        stringBuilder.append(FinConstant.BLANK);
        stringBuilder.append(tableName);
        stringBuilder.append(FinConstant.BLANK);
        stringBuilder.append(FinConstant.WHERE);
        stringBuilder.append(FinConstant.BLANK);
        stringBuilder.append(FinConstant.DELETE_FLAG_WHERE);
        stringBuilder.append(FinConstant.BLANK);

        String buildWhere = buildWhere(autoRepaymentConditionDOList);
        if (StringUtils.isNotBlank(buildWhere)){
            stringBuilder.append(FinConstant.AND);
            stringBuilder.append(FinConstant.BLANK);
            stringBuilder.append(buildWhere);
        }

        return stringBuilder.toString();
    }

    private String buildWhere(List<CreditAutoRepaymentConditionDO> autoRepaymentConditionDOList) {

        if (CollectionUtils.isEmpty(autoRepaymentConditionDOList)) {
            return null;
        }

        List<String> wheres = autoRepaymentConditionDOList.stream().map(item ->
                buildWhere(item.getConditionType(),
                        item.getTableName(),
                        item.getColumnName(),
                        item.getValueFrom(),
                        item.getValueTo()))
                .collect(Collectors.toList());


        return StringUtils.join(wheres, FinConstant.BLANK.concat(FinConstant.AND).concat(FinConstant.BLANK));
    }

    private String buildWhere(String conditionType, String tableName, String columnName, String valueFrom, String valueTo) {
        Map<String, String> eventTableConditionUdc = udcProvider.getValueMapByUdcCode(UdcEnum.EVENT_TABLE_CONDITION_EQUAL.getModel(), UdcEnum.EVENT_TABLE_CONDITION_EQUAL.getCode());
        conditionType = eventTableConditionUdc.get(conditionType);

        String concat = FinConstant.BLANK.concat(columnName).concat(FinConstant.BLANK);

        if (UdcEnum.EVENT_TABLE_CONDITION_BETWEEN_AND.getValueCode().equals(conditionType)) {
            return concat
                    .concat(FinConstant.BETWEEN)
                    .concat(FinConstant.BLANK)
                    .concat(valueFrom)
                    .concat(FinConstant.BLANK)
                    .concat(FinConstant.AND)
                    .concat(FinConstant.BLANK)
                    .concat(valueTo)
                    .concat(FinConstant.BLANK);
        }

        return concat
                .concat(conditionType)
                .concat(FinConstant.BLANK)
                .concat(valueFrom)
                .concat(FinConstant.BLANK);
    }


    /**
     * 查询满足自动还款条件的账户
     * 使用纯SQL查询数据
     *
     * @param sql
     * @return
     */
    private List<AccountDO> selectAccountData(String sql) {
        Query query = entityManager.createNativeQuery(sql, AccountDO.class);
        // 获取结果集
        List<AccountDO> resultList = query.getResultList();
        return resultList;
    }

    /**
     * 查询满足自动还款条件的信用账户
     * 使用纯SQL查询数据
     *
     * @param sql
     * @return
     */
    private List<CreditAccountDO> selectCreditAccountData(String sql) {
        Query query = entityManager.createNativeQuery(sql, CreditAccountDO.class);
        // 获取结果集
        List<CreditAccountDO> resultList = query.getResultList();
        return resultList;
    }

    /**
     *
     * 生成调剂单和流水
     *
     * @param creditAccountVO 信用账户
     * @param accountVO 账户
     * @param adjustmentAmt 调整金额
     * @param accountTypeMap 账户类型
     * @param creditAccountTypeMap 信用账户类型
     *
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public void generateAdjustToOrder(CreditAccountPageVO creditAccountVO,AccountVO accountVO,BigDecimal adjustmentAmt,
                                       Map<String, String> accountTypeMap,Map<String, String> creditAccountTypeMap,
                                      CreditAutoRepaymentJobParam jobParam){
        log.info("生成调剂单和流水入参,信用账户:{},账户:{},调整金额:{}", JSON.toJSONString(creditAccountVO),JSON.toJSONString(accountVO),JSON.toJSONString(adjustmentAmt));

        //组装生成调剂单
        AdjustToOrderParam adjustOrderParam = new AdjustToOrderParam();
        adjustOrderParam.setAccountNameFrom(accountVO.getAccountName());
        adjustOrderParam.setAccountCodeFrom(accountVO.getAccountCode());
        adjustOrderParam.setAccountTypeFrom(UdcEnum.ACCOUNT_TYPE_STORE.getValueCode());
        adjustOrderParam.setAccountNameTo(creditAccountVO.getCreditAccountName());
        adjustOrderParam.setAccountCodeTo(creditAccountVO.getCreditAccountCode());
        adjustOrderParam.setAccountTypeTo(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getValueCode());
        adjustOrderParam.setAdjustReason(UdcEnum.FIN_ADJUST_REASON_5.getValueCode());
        adjustOrderParam.setAdjustAmount(adjustmentAmt);
        adjustToOrderService.checkSave(adjustOrderParam);

        AdjustToOrderDO adjustToOrderDO = AdjustToOrderConvert.INSTANCE.paramToDO(adjustOrderParam);
        String docNo = sysNumberGenerator.generate(SysNumEnum.FIN_ADJTO_ORDER.getCode());
        adjustToOrderDO.setDocNo(docNo);
        adjustToOrderDO.setDocState(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());

        String userName = StringUtils.isNotBlank(jobParam.getUserName()) ? jobParam.getUserName() : DEFAULT_USER_NAME;
        Long userId = Objects.nonNull(jobParam.getUserId()) ? jobParam.getUserId() : DEFAULT_USER_ID;

        adjustToOrderDO.setAuditTime(LocalDateTime.now());
        adjustToOrderDO.setAuditUser(userName);
        adjustToOrderDO.setAuditUserId(userId);
        adjustToOrderDO.setRemark(DEFAULT_ADJUST_TO_ORDER_REMARK);

        adjustToOrderDO.setCreateTime(LocalDateTime.now());
        adjustToOrderDO.setCreator(userName);
        adjustToOrderDO.setCreateUserId(userId);
        adjustToOrderDO.setModifyUserId(userId);
        adjustToOrderDO.setUpdater(userName);
        adjustToOrderDO.setModifyTime(LocalDateTime.now());

        adjustToOrderRepo.save(adjustToOrderDO);
        //Long adjustToOrderId = adjustToOrderDO.getId();


        //生成审批中时调用的账户流水规则
        //查询从账户
        AccountDO fromAccountDO = accountRepoProc.findByAccountCode(adjustToOrderDO.getAccountCodeFrom());
        // 校验金额
        if (NumberUtil.isGreater(adjustToOrderDO.getAdjustAmount(), fromAccountDO.getAccountAvailableAmount())) {
            throw new BusinessException(String.format("从账户("+adjustToOrderDO.getAccountCodeFrom()+")调出金额%s大于账户可用额度%s",
                    adjustToOrderDO.getAdjustAmount().setScale(2, RoundingMode.HALF_UP),
                    fromAccountDO.getAccountAvailableAmount().setScale(2, RoundingMode.HALF_UP)));
        }
        //调剂单调出 占用流水
        AdjustToOrderDO adjustToOrderDoingDO = new AdjustToOrderDO();
        AdjustToOrderConvert.INSTANCE.oldToNewDO(adjustToOrderDO, adjustToOrderDoingDO);
        adjustToOrderDoingDO.setDocState(UdcEnum.APPLY_STATUS_DOING.getValueCode());

        adjustToOrderService.generateAccountFlow(adjustToOrderDoingDO,adjustToOrderDoingDO.getAccountCodeFrom(),UdcEnum.ADJUST_TYPE_2.getValueCode());

        //生成审批通过时调用的账户流水规则
        //生成账户流水 调剂单调出
        AdjustToOrderDO adjustToOrderCompleteDO = new AdjustToOrderDO();
        AdjustToOrderConvert.INSTANCE.oldToNewDO(adjustToOrderDO, adjustToOrderCompleteDO);
        adjustToOrderCompleteDO.setDocState(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());

        adjustToOrderService.generateAccountFlow(adjustToOrderCompleteDO,adjustToOrderCompleteDO.getAccountCodeFrom(),UdcEnum.ADJUST_TYPE_2.getValueCode());
        // 调价单调入
        if (accountTypeMap.containsKey(adjustToOrderCompleteDO.getAccountTypeTo())) {
            // 生成账户流水
            adjustToOrderService.generateAccountFlow(adjustToOrderCompleteDO,adjustToOrderCompleteDO.getAccountCodeTo(),UdcEnum.ADJUST_TYPE_1.getValueCode());
        } else if (creditAccountTypeMap.containsKey(adjustToOrderCompleteDO.getAccountTypeTo())) {
            // 生成信用账户流水
            adjustToOrderService.generateCreditAccountFlow(adjustToOrderCompleteDO);
        } else {
            throw new BusinessException(String.format("至账户类型%s没有维护", adjustToOrderCompleteDO.getAccountTypeTo()));
        }

    }



}
