package com.elitesland.fin.application.service.flow;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.core.base.SeqNumProvider;
import com.elitescloud.cloudt.authorization.core.SecurityContextUtil;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.common.util.RedLockUtils;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.Application;
import com.elitesland.fin.application.convert.account.AccountConvert;
import com.elitesland.fin.application.convert.flow.AccountFlowConvert;
import com.elitesland.fin.application.facade.dto.account.AccountIocDTO;
import com.elitesland.fin.application.facade.param.account.AccountIocParam;
import com.elitesland.fin.application.facade.param.account.AccountRuleConfigQueryParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowDealerParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowPageParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowQueryParam;
import com.elitesland.fin.application.facade.vo.account.AccountRuleConfigPageVO;
import com.elitesland.fin.application.facade.vo.account.AccountVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowDealerVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.service.account.AccountIocService;
import com.elitesland.fin.application.service.account.AccountRuleConfigService;
import com.elitesland.fin.application.service.account.AccountService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.SysNumberGenerator;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.account.AccountDO;
import com.elitesland.fin.entity.account.AccountRuleConfigDO;
import com.elitesland.fin.entity.flow.AccountFlowDO;
import com.elitesland.fin.repo.account.AccountRepo;
import com.elitesland.fin.repo.account.AccountRepoProc;
import com.elitesland.fin.repo.account.AccountRuleConfigRepo;
import com.elitesland.fin.repo.flow.AccountFlowRepo;
import com.elitesland.fin.repo.flow.AccountFlowRepoProc;
import com.elitesland.fin.rocketmq.channel.FinSalMqProcessor;
import com.elitesland.fin.rpc.order.RmiOrderRpcService;
import com.elitesland.fin.rpc.ystsupp.RmiOrgOuRpcServiceService;
import com.elitesland.support.provider.org.dto.OrgOuRpcDTO;
import com.elitesland.support.provider.org.param.OrgOuRpcDtoParam;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.redisson.RedissonRedLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.integration.support.MessageBuilder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;


/**
 * <p>
 * 功能说明:账户流水
 * </p>
 *
 * @Author Darren
 * @Date 2023/02/25
 * @Version 1.0
 * @Content:
 */
@Service
@Slf4j
public class AccountFlowServiceImpl implements AccountFlowService {
    @Autowired
    private AccountFlowRepo accountFlowRepo;
    @Autowired
    private AccountFlowRepoProc accountFlowRepoProc;

    @Autowired
    private SysNumberGenerator sysNumberGenerator;

    @Autowired
    private AccountFlowZcService accountFlowZcService;
    @Autowired
    private AccountFlowPjService accountFlowPjService;
    @Autowired
    private AccountFlowFjbService accountFlowFjbService;
    @Autowired
    private AccountFlowTcService accountFlowTcService;
    @Autowired
    private AccountFlowCommonService accountFlowCommonService;

    @Autowired
    private AccountRepoProc accountRepoProc;

    @Autowired
    private AccountRepo accountRepo;

    @Autowired
    private RedissonClient redissonClient;


    @Autowired
    private RmiOrgOuRpcServiceService rmiOrgOuRpcServiceService;

    @Autowired
    private AccountRuleConfigRepo accountRuleConfigRepo;

    @Autowired
    private RmiOrderRpcService rmiOrderRpcService;

    @Autowired
    private AccountFlowService accountFlowService;

    @Autowired
    private AccountIocService accountIocService;

    @Autowired
    private SeqNumProvider seqNumProvider;

    @Autowired
    private AccountRuleConfigService accountRuleConfigService;

    @Autowired
    private AccountService accountService;

    @Autowired
    private FinSalMqProcessor finSalMqProcessor;
    @Autowired
    private  RedLockUtils redLockUtils;
    private static final BigDecimal ONE_HUNDRED_BIGDECIMAL = new BigDecimal("100");


    /**
     * 账户流水分页查询：中台端
     *
     * @param pageParam 入参
     * @return 账户流水信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountFlowVO> page(AccountFlowPageParam pageParam) {

        PagingVO<AccountFlowVO> pagingVO = accountFlowRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AccountFlowVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        //根据公司编码、加盟商编码、规则编码获取名称
        List<AccountFlowVO> accountFlowVOList = pagingVO.getRecords();

        OrgOuRpcDtoParam orgOuRpcDtoParam = new OrgOuRpcDtoParam();
        orgOuRpcDtoParam.setOuCodes(accountFlowVOList.stream().map(AccountFlowVO::getCompanyCode).collect(Collectors.toList()));
        List<OrgOuRpcDTO> orgOuRpcDTOList = rmiOrgOuRpcServiceService.findOuDtoByParam(orgOuRpcDtoParam);

        List<AccountRuleConfigDO> accountRuleConfigDOList = accountRuleConfigRepo.findAll();

        accountFlowVOList.stream().forEach(item -> {

            OrgOuRpcDTO orgOuRpcDTOResult = orgOuRpcDTOList.stream()
                    .filter(orgOuRpcDTO -> StringUtils.equals(orgOuRpcDTO.getOuCode(), item.getCompanyCode()))
                    .findFirst()
                    .orElse(null);
            if (orgOuRpcDTOResult != null) {
                item.setCompanyCodeName(orgOuRpcDTOResult.getOuName());
            }

            AccountRuleConfigDO accountRuleConfigDOResult = accountRuleConfigDOList.stream().filter(accountRuleConfigDO -> StringUtils.equals(item.getRuleCode(), accountRuleConfigDO.getRuleCode()))
                    .findFirst()
                    .orElse(null);
            if (accountRuleConfigDOResult != null) {
                item.setRuleName(accountRuleConfigDOResult.getRuleName());
            }
        });

        return PagingVO.<AccountFlowVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }

    /**
     * 根据数据来源+来源单号查询流水的信息
     * <p>
     * 使用场景1：校验是否有重复数据已经存在
     *
     * @param dataSource 数据来源
     * @param sourceNo   来源单号
     * @return 账户流水信息
     */
    @Override
    @SysCodeProc
    public List<AccountFlowVO> selectBySourceAndNo(String dataSource, String sourceNo) {
        if (StringUtils.isEmpty(dataSource)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据来源为空!");
        }
        if (StringUtils.isEmpty(sourceNo)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "来源单号为空!");
        }
        AccountFlowPageParam accountFlowPageParam = new AccountFlowPageParam();
        accountFlowPageParam.setDataSource(dataSource);
        accountFlowPageParam.setSourceNoEq(sourceNo);
        List<AccountFlowVO> accountFlowVOList = accountFlowRepoProc.selectListByParam(accountFlowPageParam);
        if (CollectionUtils.isEmpty(accountFlowVOList)) {
            return Collections.emptyList();
        }
        return accountFlowVOList;
    }

    /**
     * 账户流水分页查询：经销商端
     *
     * @param pageParam 入参
     * @return 账户流水信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountFlowVO> dealerPage(AccountFlowPageParam pageParam) {
        //经销商下全部整车储值账户/配件储值账户/附件包储值账户/整车返利账户/配件返利账户/统筹账户
        // 当月实际增加金额之和     当月实际减少金额之和
        //经销商账户金额是 +交易金额  的    就是入账金额    经销商账户金额是-交易金额  的    就是使用金额
        //可以这么理解   不过经销商账户金额是 +-交易金额  的  也是根据交易类型来做的
        //本质是根据交易类型来判断的
        //占用和释放的这种  不需要计算
        //入账和使用  是实际的扣减和增加

        //校验必填项02:年月日期、账户编码
        checkMandatoryField02(pageParam);
        if (StringUtils.isNotBlank(pageParam.getYearMonthStr())) {
            //把年月格式yyyy-MM字符串转换拼接成年月日格式的日期
            LocalDate transactionDate = accountFlowCommonService.transitionYearMonthStr(pageParam.getYearMonthStr());
            //根据年月日格式的日期拼装入参里的交易起始时间、交易截止时间
            accountFlowCommonService.splicTransactionTimeSection(transactionDate, pageParam);
        }
        //PC端只展示实际扣减和增加的流水
        pageParam.setAccIoTypeList(Lists.newArrayList(FinConstant.ACC_IO_TYPE_ADD, FinConstant.ACC_IO_TYPE_SUB));
        PagingVO<AccountFlowVO> pagingVO = accountFlowRepoProc.dealerPage(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AccountFlowVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        return PagingVO.<AccountFlowVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }

    /**
     * 经销商端查询相关统计金额：入账金额、使用金额
     *
     * @param pageParam 入参
     * @return 相关统计金额出参对象
     */
    @Override
    public AccountFlowDealerVO dealerSelectAmt(AccountFlowPageParam pageParam) {
        //经销商下全部整车储值账户/配件储值账户/附件包储值账户/整车返利账户/配件返利账户/统筹账户
        // 当月实际增加金额之和     当月实际减少金额之和
        //经销商账户金额是 +交易金额  的    就是入账金额    经销商账户金额是-交易金额  的    就是使用金额
        //可以这么理解   不过经销商账户金额是 +-交易金额  的  也是根据交易类型来做的
        //本质是根据交易类型来判断的
        //占用和释放的这种  不需要计算
        //入账和使用  是实际的扣减和增加
        //那么标产里就是对应账户金额+-的发生金额，通过交易类型来分区发生金额是扣减还是增加，然后求的发生金额之和

        //校验必填项02:年月日期、账户编码
        checkMandatoryField02(pageParam);
        //把年月格式yyyy-MM字符串转换拼接成年月日格式的日期
        LocalDate transactionDate = accountFlowCommonService.transitionYearMonthStr(pageParam.getYearMonthStr());
        //根据年月日格式的日期拼装入参里的交易起始时间、交易截止时间
        accountFlowCommonService.splicTransactionTimeSection(transactionDate, pageParam);

        AccountFlowDealerVO accountFlowDealerVO = new AccountFlowDealerVO();
        accountFlowDealerVO.setUseAmt(BigDecimal.ZERO);
        accountFlowDealerVO.setRecordAmt(BigDecimal.ZERO);
        //等产品确定根据新的维度进行判断扣减或增加统计
        //获取发生金额为增加的交易类型编码的集合，并查询数据对发生金额进行相加求和
        //pageParam.setTransactionTypeList(FinConstant.ADD_TRANSACTION_TYPE);
        pageParam.setAccIoType(FinConstant.ACC_IO_TYPE_ADD);
        List<AccountFlowVO> addAmtList = accountFlowRepoProc.dealerSelectAmt(pageParam);
        if (!CollectionUtils.isEmpty(addAmtList)) {
            //入账金额
            BigDecimal recordAmt = addAmtList.stream().map(AccountFlowVO::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            accountFlowDealerVO.setRecordAmt(recordAmt);
        }
        //获取发生金额为扣减的交易类型编码的集合，并查询数据对发生金额进行相加求和
        //pageParam.setTransactionTypeList(FinConstant.SUB_TRANSACTION_TYPE);
        pageParam.setAccIoType(FinConstant.ACC_IO_TYPE_SUB);
        List<AccountFlowVO> subAmtList = accountFlowRepoProc.dealerSelectAmt(pageParam);
        if (!CollectionUtils.isEmpty(subAmtList)) {
            //使用金额
            BigDecimal useAmt = subAmtList.stream().map(AccountFlowVO::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            accountFlowDealerVO.setUseAmt(useAmt);
        }

        return accountFlowDealerVO;
    }


    /**
     * 账户流水保存
     *
     * @param paramList 入参
     * @return 账户流水返回对象
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public AccountFlowVO save(List<AccountFlowParam> paramList) {
        log.info("获取流水入参{}", JSON.toJSONString(paramList));
        //校验必填项
        this.checkMandatoryFieldList(paramList);
        //以防万一，正常不需要这种校验处理
        this.checkRepeatability(paramList);
        //默认赋值，放到校验逻辑之后,目前下放到每个业务里了
        //this.defaultAssignment(paramList);

        this.distinguishAccType(paramList);

        //TODO 占用前都会有这个   校验的接口逻辑
        // 则朋 就是这块不是先释放  再占用吗     就是释放后   还有一个校验的接口检查完了之后看看是否满足  才能占用
        // 校验逻辑：整车下单、整车订单修改需要进行校验
        // 订单的时候需要返回结果和值给他们
        // 金额扣减为负的情况，在校验接口里处理
        // 流水第二条是占用的情况，需要再次反向调用校验接口，只有外部接口里需要


        return new AccountFlowVO();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public AccountFlowDO saveAccountFlow(AccountFlowParam accountFlowParam) {

        checkMandatoryField(accountFlowParam);

        return accountFlowRepo.save(AccountFlowConvert.INSTANCE.paramToDo(accountFlowParam));

    }

    /**
     * 按照账户类型区分
     *
     * @param paramList 入参
     */
    private void distinguishAccType(List<AccountFlowParam> paramList) {

        //避免批量穿过来会交叉多种账户类型模块的数据，故每个大模块账户类型都做了过滤区分
        List<AccountFlowParam> accountFlowZcList = paramList.stream().filter(param -> accountFlowCommonService.judgeZcAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowZcList)) {
            accountFlowZcService.saveZc(accountFlowZcList);
        }
        List<AccountFlowParam> accountFlowPjList = paramList.stream().filter(param -> accountFlowCommonService.judgePjAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowPjList)) {
            accountFlowPjService.savePj(accountFlowPjList);
        }
        List<AccountFlowParam> accountFlowFjbList = paramList.stream().filter(param -> accountFlowCommonService.judgeFjbAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowFjbList)) {
            accountFlowFjbService.saveFjb(accountFlowFjbList);
        }
        List<AccountFlowParam> accountFlowTcList = paramList.stream().filter(param -> accountFlowCommonService.judgeTcAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowTcList)) {
            accountFlowTcService.saveTc(accountFlowTcList);
        }


        //1.按照账户类型拆分成4个大模块
        //2.在每个模块里根据数据来源区分不同的方法，方便后续变动或扩展
        //3.把设计到账户金额、账户占用金额(子账户金额、子账户占用金额、经销商账户金额、经销商占用金额)
        // 查询上一条记录的能共用的抽离出单独的方法块，即使立马项目里无法引用，但是代码可以复用

    }


    /**
     * 默认赋值
     *
     * @param paramList 入参
     */
    /*private void defaultAssignment(List<AccountFlowParam> paramList) {
        paramList.forEach(accountFlowParam -> {
            if (StringUtils.isEmpty(accountFlowParam.getSourcePlatform())) {
                accountFlowParam.setSourcePlatform(UdcEnum.FIN_SOURCE_PLATFORM_TYPE_OTHER.getValueCode());
            }
            String flowNo = sysNumberGenerator.generate(SysNumEnum.FIN_FLOW_NO.getCode());
            accountFlowParam.setFlowNo(flowNo);
        });
    }*/

    /**
     * 校验必填项
     *
     * @param paramList 入参
     */
    @Override
    public void checkMandatoryFieldList(List<AccountFlowParam> paramList) {

        Assert.notEmpty(paramList, "参数不能为空");
        paramList.stream().forEach(accountFlowParam -> checkMandatoryField(accountFlowParam));
    }

    /**
     * 校验必填项
     *
     * @param param 入参
     */
    @Override
    public void checkMandatoryField(AccountFlowParam param) {
        if (StringUtils.isBlank(param.getAccountType())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "账户类型为空!");
        }
        if (StringUtils.isBlank(param.getSourceDoc())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "来源单据为空!");
        }
        if (StringUtils.isBlank(param.getSourceNo())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "来源单号为空!");
        }
        if (StringUtils.isBlank(param.getTransactionType())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "交易类型为空!");
        }
        if (Objects.isNull(param.getAmount())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "发生金额为空!");
        }
        if (Objects.isNull(param.getTransactionTime())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "交易日期为空!");
        }
        if (StringUtils.isBlank(param.getAccountCode())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "账户编码为空!");
        }
        if (StringUtils.isBlank(param.getAccountName())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "账户名称为空!");
        }

    }

    /**
     * 校验是否有重复数据或脏数据、数据结构是否正确：
     * 必须每次只有一种数据来源，要不数据融合在一起无法判断谁先谁后，会涉及到上一条数据问题。
     * 同种数据来源，也应该是单次的（一个单号），就是不是批量的，会涉及到上一条数据问题。
     * 可以先判断数据来源是否是多种，再判断账户类型+数据来源+交易类型是否是唯一
     *
     * @param paramList 入参
     */
    @Override
    public void checkRepeatability(List<AccountFlowParam> paramList) {
        val dataSourceList = paramList.stream().map(AccountFlowParam::getDataSource).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(dataSourceList) && dataSourceList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "存在多种数据来源!");
        }
        Map<String, List<AccountFlowParam>> accountFlowListMap = paramList.stream().collect(Collectors.groupingBy(
                i -> i.getAccType() + FinConstant.LINE_SPLIT + i.getDataSource() + FinConstant.LINE_SPLIT + i.getTransactionType()));
        for (String accountFlowKey : accountFlowListMap.keySet()) {
            val accountFlowList = accountFlowListMap.get(accountFlowKey);

            String[] keyParams = accountFlowKey.split(FinConstant.LINE_SPLIT);
            String dataSource = keyParams[FinConstant.ARRAY_INDEX_1];
            //数据来源是调剂单审核通过时，交易类型可能是相同的，所以使用拆分类型来区分
            if (Objects.equals(dataSource, UdcEnum.FIN_DATA_SOURCE_TYPE_TJD03.getValueCode())) {
                List<String> splitTypeList = accountFlowList.stream().map(AccountFlowParam::getSplitType).distinct().collect(Collectors.toList());
                if (!Objects.equals(accountFlowList.size(), splitTypeList.size())) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "调剂单数据不唯一，无法处理!");
                }
            } else {
                if (!CollectionUtils.isEmpty(accountFlowList) && accountFlowList.size() > 1) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据不唯一，无法处理!");
                }
            }

        }
    }


    /**
     * 校验必填项02:年月日期、账户编码、账户类型
     *
     * @param pageParam 查询入参
     */
    @Override
    public void checkMandatoryField02(AccountFlowPageParam pageParam) {
        if (StringUtils.isBlank(pageParam.getYearMonthStr())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "年月日期为空!");
        }
        if (StringUtils.isBlank(pageParam.getAccCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户编码为空!");
        }
        if (StringUtils.isBlank(pageParam.getAccType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户类型为空!");
        }
    }


    /**
     * 经销商端查询相关统计金额：储值账户金额、返利账户金额
     *
     * @param param 入参
     * @return 相关统计金额出参对象
     */
    @Override
    public AccountFlowDealerVO dealerSelectAccAmt(AccountFlowDealerParam param) {
        AccountFlowDealerVO accountFlowDealerVO = new AccountFlowDealerVO();
        accountFlowDealerVO.setCzAccAmt(BigDecimal.ZERO);
        accountFlowDealerVO.setFlAccAmt(BigDecimal.ZERO);

        //整车/配件储值账户金额
        Optional<AccountFlowVO> czOptional = accountFlowCommonService.selectByAccCode(param.getCzAccCode());
        czOptional.ifPresent(accountFlowVO -> {
            accountFlowDealerVO.setCzAccAmt(accountFlowVO.getAccAmt());
        });

        //整车/配件返利账户金额
        Optional<AccountFlowVO> flOptional = accountFlowCommonService.selectByAccCode(param.getFlAccCode());
        flOptional.ifPresent(accountFlowVO -> {
            accountFlowDealerVO.setFlAccAmt(accountFlowVO.getAccAmt());
        });

        return accountFlowDealerVO;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void approve(AccountFlowParam accountFlowParam) {
        checkApproveParam(accountFlowParam);
        approve(accountFlowParam, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode(), true);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void reject(AccountFlowParam accountFlowParam) {
        checkApproveParam(accountFlowParam);
        approve(accountFlowParam, UdcEnum.APPLY_STATUS_REJECTED.getValueCode(), false);
    }

    @Override
    @SysCodeProc
    public List<AccountFlowVO> queryByAccountFlowQueryParam(AccountFlowQueryParam accountFlowQueryParam) {
        List<AccountFlowVO> flowVOList = accountFlowRepoProc.queryByAccountFlowQueryParam(accountFlowQueryParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }

    @Override
    public void accountFlowCallBack(String doc, AccountFlowDO accountFlowDO, boolean isApprove) {

        if (UdcEnum.DOC_CLS_SVO.getValueCode().equals(doc) || UdcEnum.DOC_CLS_AO.getValueCode().equals(doc)) {
            return;
        }
//        OrderAccountFlowParam orderAccountFlowParam = new OrderAccountFlowParam();
//        //销售订单id
//        orderAccountFlowParam.setSoId(accountFlowDO.getSourceId());
//        //订单类别
//        orderAccountFlowParam.setDocCls(accountFlowDO.getSourceDoc());
//        //流水号
//        orderAccountFlowParam.setReceiptSerial(accountFlowDO.getFlowNo());
//        //交易类型
//        orderAccountFlowParam.setTransType(accountFlowDO.getTransactionType());
//        //发生金额
//        orderAccountFlowParam.setReceiptAmt(accountFlowDO.getAmount());
//        //交易日期
//        orderAccountFlowParam.setReceiptDate(accountFlowDO.getTransactionTime());
//        //账户编码
//        orderAccountFlowParam.setPayAccNo(accountFlowDO.getAccountCode());
//        //账户名称
//        orderAccountFlowParam.setPayAccName(accountFlowDO.getAccountName());
//        //账户名称
//        orderAccountFlowParam.setIsApprove(isApprove);
//
//        rmiOrderRpcService.orderAccountFlow(orderAccountFlowParam);
    }


    private void checkApproveParam(AccountFlowParam accountFlowParam) {
        Assert.notNull(accountFlowParam, "参数不能为空");
        Assert.notNull(accountFlowParam.getId(), "id不能为空");
        Assert.notEmpty(accountFlowParam.getAccountCode(), "账户编码不能为空");
        Assert.notEmpty(accountFlowParam.getTransactionType(), "交易类型不能为空");
        Assert.notNull(accountFlowParam.getAmount(), "发生金额不能为空");
    }

    private void approve(AccountFlowParam accountFlowParam, String state, boolean isApprove) {


        Optional<AccountFlowDO> accountFlowDOOptional = accountFlowRepo.findById(accountFlowParam.getId());
        AccountFlowDO accountFlowDO = accountFlowDOOptional.get();
        Assert.equals(UdcEnum.APPLY_STATUS_DOING.getValueCode(), accountFlowDO.getOrderState(), "只有审批中状态单据才可审批");

        //账户占用特殊处理
        if (UdcEnum.IO_TYPE_T.getValueCode().equals(accountFlowParam.getTransactionType())) {
            updateAccountAmountFor_IO_TYPE_T(accountFlowParam, state);
        }else{
            updateAccountAmountOther(accountFlowParam, state);
        }
        //更新流水单状态
        SysUserDTO user = getUser();

        accountFlowParam.setOrderState(state);
        accountFlowParam.setAuditDate(LocalDateTime.now());
        accountFlowParam.setAuditUserName(FinConstant.SYSTEM);

        if (user != null) {
            accountFlowParam.setAuditUserId(user.getId());
            accountFlowParam.setAuditUserName(user.getLastName());
        }

        accountFlowRepoProc.approveAccountFlow(accountFlowParam);
    }

    private void updateAccountAmount(AccountFlowParam accountFlowParam, String state) {
        if (UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(state)) {

            AccountDO accountDO = accountRepoProc.findByAccountCode(accountFlowParam.getAccountCode());

            BigDecimal accountAmount = accountDO.getAccountAmount();
            BigDecimal accountOccupancyAmount = accountDO.getAccountOccupancyAmount();

            Assert.equals(accountDO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");

            if (UdcEnum.IO_TYPE_I.getValueCode().equals(accountFlowParam.getTransactionType())) {
                accountAmount = accountAmount.add(accountFlowParam.getAmount());
            }

            if (UdcEnum.IO_TYPE_O.getValueCode().equals(accountFlowParam.getTransactionType())) {
                accountAmount = accountAmount.subtract(accountFlowParam.getAmount());
            }

            if (UdcEnum.IO_TYPE_T.getValueCode().equals(accountFlowParam.getTransactionType())) {
                accountOccupancyAmount = accountOccupancyAmount.add(accountFlowParam.getAmount());
            }

            if (UdcEnum.IO_TYPE_R.getValueCode().equals(accountFlowParam.getTransactionType())) {
                accountOccupancyAmount = accountOccupancyAmount.subtract(accountFlowParam.getAmount());
            }
            accountDO.setAccountAmount(accountAmount);
            accountDO.setAccountOccupancyAmount(accountOccupancyAmount);
            accountDO.setAccountAvailableAmount(accountAmount.subtract(accountOccupancyAmount));
            //更新审批中的流水的状态
            accountFlowParam.setAccountAmount(accountAmount);
            accountFlowParam.setAccountOccupancyAmount(accountOccupancyAmount);
            accountFlowParam.setAccountAvailableAmount(accountAmount.subtract(accountOccupancyAmount));
            accountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            accountFlowRepoProc.updateAccountFlowAmount(accountFlowParam);

        }
    }
    private void updateAccountAmountFor_IO_TYPE_T(AccountFlowParam accountFlowParam, String state) {
        if (UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(state)) {
            AccountVO accountVOOld = accountRepoProc.getVOByCode(accountFlowParam.getAccountCode());
            Assert.equals(accountVOOld.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");
            int res = accountRepo.updateAccountOccupancyAmountAndAccountAvailableAmountById(accountVOOld.getId(), accountFlowParam.getAmount());
            log.info("账户流水生成，更新记录数：{}",res);
            if(res<1){
                log.error("账户流水生成,更新账户金额失败,可用金额不足");
                throw new BusinessException("账户流水生成,更新账户金额失败,可用金额不足");
            }
            AccountVO accountVO = accountRepoProc.get(accountVOOld.getId());
            if(accountVO==null){
                log.error("账户流水生成,查询账户信息失败");
                throw new BusinessException("账户流水生成,查询账户信息失败");
            }
            //更新审批中的流水的状态
            accountFlowParam.setAccountAmount(accountVO.getAccountAmount());
            accountFlowParam.setAccountOccupancyAmount(accountVO.getAccountOccupancyAmount());
            accountFlowParam.setAccountAvailableAmount(accountVO.getAccountAvailableAmount());
            accountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            accountFlowRepoProc.updateAccountFlowAmount(accountFlowParam);

        }
    }

    private void updateAccountAmountOther(AccountFlowParam accountFlowParam, String state) {
        if (UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(state)) {

            AccountVO oldAccountVO = accountRepoProc.getVOByCode(accountFlowParam.getAccountCode());
            Assert.equals(oldAccountVO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");

            if (UdcEnum.IO_TYPE_I.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateAddAccountAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户增加流水生成，更新记录数：{}",res);
                if(res<1){
                    log.error("账户增加流水生成,更新账户金额失败!");
                    throw new BusinessException("账户增加流水生成,更新账户金额失败!");
                }
            }else if (UdcEnum.IO_TYPE_O.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateSubtractAccountAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户扣减流水生成，更新记录数：{}",res);
                if(res<1){
                    log.error("账户扣减流水生成,更新账户金额失败!");
                    throw new BusinessException("账户扣减流水生成,更新账户金额失败!");
                }
            }else if (UdcEnum.IO_TYPE_R.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateSubtractAccountOccupancyAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户占用释放流水生成，更新记录数：{}",res);
                if(res<1){
                    log.error("账户占用释放流水生成,更新账户金额失败!");
                    throw new BusinessException("账户占用释放流水生成,更新账户金额失败!");
                }
            }else {
                throw new BusinessException("账户未匹配到交易类型!");
            }

            AccountVO newAccountVO = accountRepoProc.get(oldAccountVO.getId());
            if(newAccountVO==null){
                log.error("账户流水生成,查询更新后账户信息失败");
                throw new BusinessException("账户流水生成,查询更新后账户信息失败");
            }

            //更新审批中的流水的状态
            accountFlowParam.setAccountAmount(newAccountVO.getAccountAmount());
            accountFlowParam.setAccountOccupancyAmount(newAccountVO.getAccountOccupancyAmount());
            accountFlowParam.setAccountAvailableAmount(newAccountVO.getAccountAvailableAmount());
            accountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            accountFlowRepoProc.updateAccountFlowAmount(accountFlowParam);

        }
    }

    private SysUserDTO getUser() {
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (userDetails == null) {
            return null;
        }
        return userDetails.getUser();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String generateAccountFlow(AccountFlowParam param) {
        log.info("生成账户流水，{}", JSONUtil.toJsonStr(param));
        //参数检查
        checkGenerateAccountFlowParam(param);

        AccountFlowParam accountFlowParam = AccountFlowConvert.INSTANCE.copyParam(param);

        //组装财务域udc
        buildAccountFlowParam(accountFlowParam);

        //获取账户信息
        AccountDO accountDO = getAccountMsg(accountFlowParam);

        //幂等
		// 账户流水校验重复：增加是否启用工作流判断，如果单据已有工作流，就允许重复；如果单据没有工作流，就不允许重复
		if(StrUtil.isEmpty(param.getWorkflowProcInstId())){
        	checkIdempotent(accountDO,accountFlowParam);
		}

        //获取账户规则
        List<AccountRuleConfigPageVO> accountRuleConfigPageVOList = getAccountRuleMsg(accountDO, accountFlowParam);

        //获取账户事务码
        List<AccountIocDTO> accountIocDTOList = getAccountIocMsg(accountRuleConfigPageVOList);

        //根据优先级排序
        accountRuleConfigPageVOList = accountRuleConfigPageVOList.stream()
                .sorted(Comparator.comparing(AccountRuleConfigPageVO::getPriorityNo))
                .collect(Collectors.toList());

        //校验账户余额
      /*  checkAccountAmount(accountRuleConfigPageVOList, accountDO, accountFlowParam, accountIocDTOList);*/

        //生成账户流水
      /*  List<AccountFlowParam> accountFlowParamList = generateAccountFlow(accountRuleConfigPageVOList,
                accountDO,
                accountIocDTOList,
                accountFlowParam);*/

        //流水审批
      /*  accountFlowReview(accountFlowParamList);*/
        //加锁begin
        RedissonRedLock redLock=null;
        try {
            String lockKey="ACCOUNT_FLOW_GENERATE:"+accountDO.getAccountCode();
            log.info("账户流水生成，锁:{}",lockKey);
            redLock = redLockUtils.getRedLock(lockKey);
            boolean lockFlag = redLock.tryLock(3, 300, TimeUnit.SECONDS);
            if (!lockFlag) {
                log.error("账户流水生成，获取锁失败");
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户流水生成并发过大,请稍后再试！");
            }
            log.info("账户流水生成，获取锁成功");
            //再次获取账户信息
            AccountVO accountVO = accountRepoProc.getVOByCode(accountDO.getAccountCode());
            AccountDO accountDO1=AccountConvert.INSTANCE.VO2DO(accountVO);
            //校验账户余额
            checkAccountAmount(accountRuleConfigPageVOList, accountDO1, accountFlowParam, accountIocDTOList);
            //生成流水单
            List<AccountFlowParam> accountFlowParamList = generateAccountFlow(accountRuleConfigPageVOList,
                    accountDO1,
                    accountIocDTOList,
                    accountFlowParam);

            //流水审批
            accountFlowReview(accountFlowParamList);
            return accountFlowParamList.get(0).getFlowNo();
        } catch (InterruptedException e) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户流水生成，获取redis锁异常,请稍后再试！");
        } finally {
            redLock.unlock();
            log.info("账户流水生成，释放锁成功");
        }
        //加锁end

    }

    private void accountFlowReview(List<AccountFlowParam> accountFlowParamList) {
        accountFlowParamList.stream().forEach(item -> {
            if (item.isAutomaticReview()) {
                accountFlowService.approve(item);
            }
        });
    }

    private void sendAccountFlowReviewMsg(List<AccountFlowParam> accountFlowParamList) {
        accountFlowParamList.stream().forEach(item -> {
            if (item.isAutomaticReview()) {
                //发送流水审批消息
                try {
                    boolean result = finSalMqProcessor.approveAccountFlowOutput().send(MessageBuilder.withPayload(item).build());
                    if (!result) {
                        log.error("流水自动审批消息发送失败");
                    }
                } catch (Exception e) {
                    log.error("流水自动审批消息发送失败，失败原因：" + e.getMessage(), e);
                }
            }
        });
    }

    private List<AccountFlowParam> generateAccountFlow(List<AccountRuleConfigPageVO> accountRuleConfigPageVOList,
                                                       AccountDO accountDO,
                                                       List<AccountIocDTO> accountIocDTOList,
                                                       AccountFlowParam accountFlowParam) {

        List<AccountRuleConfigPageVO> finalAccountRuleConfigPageVOList = accountRuleConfigPageVOList;
        List<AccountFlowParam> accountFlowParamList = new ArrayList<>();

        //生成流水单
        finalAccountRuleConfigPageVOList.stream().forEach(accountRuleConfigPageVO -> {

            AccountFlowParam param = buildAccountFlowParam(accountFlowParam,
                    accountDO,
                    accountIocDTOList,
                    accountRuleConfigPageVO);

            //创建账户流水
            AccountFlowDO accountFlowDO = accountFlowService.saveAccountFlow(param);

            param.setId(accountFlowDO.getId());
            param.setFlowNo(accountFlowDO.getFlowNo());

            accountFlowParamList.add(param);
        });

        return accountFlowParamList;
    }

    @Override
    public Boolean checkAccountBalance(AccountFlowParam param) {
        //参数检查
        checkGenerateAccountFlowParam(param);

        AccountFlowParam accountFlowParam = AccountFlowConvert.INSTANCE.copyParam(param);

        //组装财务域udc
        buildAccountFlowParam(accountFlowParam);

        //获取账户信息
        AccountDO accountDO = getAccountMsg(accountFlowParam);

        //幂等
        checkIdempotent(accountDO,accountFlowParam);

        //获取账户规则
        List<AccountRuleConfigPageVO> accountRuleConfigPageVOList = getAccountRuleMsg(accountDO, accountFlowParam);

        //获取账户事务码
        List<AccountIocDTO> accountIocDTOList = getAccountIocMsg(accountRuleConfigPageVOList);

        //根据优先级排序
        accountRuleConfigPageVOList = accountRuleConfigPageVOList.stream()
                .sorted(Comparator.comparing(AccountRuleConfigPageVO::getPriorityNo))
                .collect(Collectors.toList());

        //校验账户余额
        checkAccountAmount(accountRuleConfigPageVOList, accountDO, accountFlowParam, accountIocDTOList);

        return true;
    }


    private void buildAccountFlowParam(AccountFlowParam accountFlowParam) {
        accountFlowParam.setSourceDocType(accountFlowParam.getSourceDoc()
                .concat(FinConstant.UNDERLINE)
                .concat(accountFlowParam.getSourceDocType()));

        accountFlowParam.setSourceDocStatus(accountFlowParam.getSourceDoc()
                .concat(FinConstant.UNDERLINE)
                .concat(accountFlowParam.getSourceDocStatus()));
    }

    private void checkAccountAmount(List<AccountRuleConfigPageVO> accountRuleConfigPageVOList,
                                    AccountDO accountDO,
                                    AccountFlowParam accountFlowParam,
                                    List<AccountIocDTO> accountIocDTOList) {

        accountRuleConfigPageVOList.stream().forEach(item -> {
            //交易类型
            AccountIocDTO accountIocDTO = accountIocDTOList.stream().filter(accountIocRespVO -> accountIocRespVO.getIoCode()
                    .equals(item.getIoCode()))
                    .findFirst()
                    .orElse(null);

            Assert.notNull(accountIocDTO, "没有查询到启用的账户事务");

            if (UdcEnum.IO_TYPE_T.getValueCode().equals(accountIocDTO.getIoType())) {
                Assert.isTrue(accountDO.getAccountAvailableAmount().compareTo(accountFlowParam.getSourceDocAmount()
                        .multiply(item.getCalculatePercent())
                        .divide(ONE_HUNDRED_BIGDECIMAL)) >= 0, "账户可用余额不足");
            }
        });
    }

    private void checkIdempotent(AccountDO accountDO,AccountFlowParam accountFlowParam) {

        AccountFlowQueryParam accountFlowQueryParam = new AccountFlowQueryParam();
        accountFlowQueryParam.setSourceId(accountFlowParam.getSourceId());
        accountFlowQueryParam.setSourceDoc(accountFlowParam.getSourceDoc());
        accountFlowQueryParam.setSourceDocStatus(accountFlowParam.getSourceDocStatus());
        accountFlowQueryParam.setSourceNo(accountFlowParam.getSourceNo());
        accountFlowQueryParam.setAccountType(accountDO.getAccountType());
        if (ObjUtil.equals(accountFlowParam.getOverOneFlow(), Boolean.TRUE)) {
            accountFlowQueryParam.setAccountCode(accountFlowParam.getAccountCode());
        }

        Assert.isTrue(CollectionUtils.isEmpty(accountFlowService.queryByAccountFlowQueryParam(accountFlowQueryParam)), "订单已经生成过流水");
    }

    private List<AccountIocDTO> getAccountIocMsg(List<AccountRuleConfigPageVO> accountRuleConfigPageVOList) {
        Set<String> ioCodeSet = new HashSet<>();
        accountRuleConfigPageVOList.stream().forEach(item -> ioCodeSet.add(item.getIoCode()));

        AccountIocParam accountIocParam = new AccountIocParam();
        accountIocParam.setIoCodeSet(ioCodeSet);
        accountIocParam.setState(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<AccountIocDTO> accountIocDTOList = accountIocService.queryByAccountIocParam(accountIocParam);
        Assert.notEmpty(accountIocDTOList, "查询账户事务码失败");
        return accountIocDTOList;
    }

    public AccountFlowParam buildAccountFlowParam(AccountFlowParam accountFlowParam,
                                                  AccountDO accountDO,
                                                  List<AccountIocDTO> accountIocDTOList,
                                                  AccountRuleConfigPageVO accountRuleConfigPageVO) {

        AccountFlowParam result = new AccountFlowParam();
        //公司
        result.setCompanyCode(accountDO.getSecOuCode());
        //归属加盟商
        result.setSecFranchiseeCode(accountDO.getSecFranchiseeCode());
        //流水号
        result.setFlowNo(seqNumProvider.generateCode(Application.NAME, FinConstant.ACCOUNT_FLOW_NO, Collections.emptyList()));
        //交易类型
        AccountIocDTO accountIocDTO = accountIocDTOList.stream().filter(accountIocRespVO -> accountIocRespVO.getIoCode()
                .equals(accountRuleConfigPageVO.getIoCode()))
                .findFirst()
                .orElse(null);

        Assert.notNull(accountIocDTO, "没有查询到启用的账户事务");

        result.setTransactionType(accountIocDTO.getIoType());

        //发生金额
        result.setAmount(accountFlowParam.getSourceDocAmount()
                .multiply(accountRuleConfigPageVO.getCalculatePercent())
                .divide(ONE_HUNDRED_BIGDECIMAL));

        //交易日期
        result.setTransactionTime(LocalDateTime.now());
        //开户主体名称
        result.setAccountHolderName(accountDO.getAccountHolderName());
        //账户类型
        result.setAccountType(accountDO.getAccountType());
        //账户编码
        result.setAccountCode(accountDO.getAccountCode());
        //账户名称
        result.setAccountName(accountDO.getAccountName());
        //来源单据
        result.setSourceDoc(accountFlowParam.getSourceDoc());
        //来源单号
        result.setSourceNo(accountFlowParam.getSourceNo());
        //来源id
        result.setSourceId(accountFlowParam.getSourceId());

        //规则编码
        result.setRuleCode(accountRuleConfigPageVO.getRuleCode());
        //规则优先级
        result.setPriorityNo(accountRuleConfigPageVO.getPriorityNo());

        //单据状态
        result.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());

        //适用单据状态
        result.setSourceDocStatus(accountFlowParam.getSourceDocStatus());

        result.setAutomaticReview(accountRuleConfigPageVO.isAutomaticReview());

        result.setAccountAmount(accountDO.getAccountAmount());

        result.setAccountOccupancyAmount(accountDO.getAccountOccupancyAmount());

        result.setAccountAvailableAmount(accountDO.getAccountAvailableAmount());

        return result;
    }

    private List<AccountRuleConfigPageVO> getAccountRuleMsg(AccountDO accountDO, AccountFlowParam accountFlowParam) {
        AccountRuleConfigQueryParam accountRuleConfigQueryParam = new AccountRuleConfigQueryParam();

        //适用单据
        accountRuleConfigQueryParam.setOptDoc(accountFlowParam.getSourceDoc());
        //适用单据类型
        accountRuleConfigQueryParam.setOptDocType(accountFlowParam.getSourceDocType());
        //适用单据状态
        accountRuleConfigQueryParam.setOptDocStatus(accountFlowParam.getSourceDocStatus());
        //适用账户类型
        accountRuleConfigQueryParam.setOptAccountType(accountDO.getAccountType());
        //指定账户
        accountRuleConfigQueryParam.setSpecificAccount(accountDO.getAccountCode());
        accountRuleConfigQueryParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        PagingVO<AccountRuleConfigPageVO> accountRuleConfigPageVOPagingVO = accountRuleConfigService.searchPage(accountRuleConfigQueryParam);
        Assert.notNull(accountRuleConfigPageVOPagingVO, "查询订单能匹配到的规则失败");
        Assert.notEmpty(accountRuleConfigPageVOPagingVO.getRecords(), "没有匹配到账户规则配置");

        return accountRuleConfigPageVOPagingVO.getRecords();

    }

    private AccountDO getAccountMsg(AccountFlowParam accountFlowParam) {
        return accountRepoProc.findByAccountCode(accountFlowParam.getAccountCode());
    }

    private void checkGenerateAccountFlowParam(AccountFlowParam accountFlowParam) {
        Assert.notNull(accountFlowParam, "参数不能为空");
        Assert.notEmpty(accountFlowParam.getSourceDoc(), "来源单据不能为空");
        Assert.notEmpty(accountFlowParam.getSourceDocType(), "来源单据类型不能为空");
        Assert.notEmpty(accountFlowParam.getSourceDocStatus(), "来源单据状态不能为空");
        Assert.notNull(accountFlowParam.getSourceDocAmount(), "来源单据金额不能为空");
        Assert.notNull(accountFlowParam.getSourceId(), "来源单据id不能为空");
        Assert.notEmpty(accountFlowParam.getSourceNo(), "来源单号不能为空");
        Assert.notEmpty(accountFlowParam.getAccountCode(), "账户编码不能为空");
    }


    @Override
    @SysCodeProc
    public List<AccountFlowVO> selectListByQueryParam(AccountFlowQueryParam queryParam) {
        List<AccountFlowVO> flowVOList = accountFlowRepoProc.selectListByQueryParam(queryParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }

    @Override
    @SysCodeProc
    public List<AccountFlowVO> selectRepairAfterPage(AccountFlowPageParam pageParam) {
        List<AccountFlowVO> flowVOList = accountFlowRepoProc.selectRepairAfterPage(pageParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }

    @Override
    @SysCodeProc
    public List<AccountFlowVO> selectRepairAfter(AccountFlowPageParam pageParam) {
        List<AccountFlowVO> flowVOList = accountFlowRepoProc.selectRepairAfter(pageParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }

}
