package com.elitesland.fin.application.service.flowrepair;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.context.util.DatetimeUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.application.convert.account.AccountConvert;
import com.elitesland.fin.application.convert.creditaccount.CreditAccountRuleConfigDtlConvert;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountRuleConfigDTO;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountRuleConfigDtlDTO;
import com.elitesland.fin.application.facade.dto.flowrepair.AccountFlowProcedureResultDTO;
import com.elitesland.fin.application.facade.dto.flowrepair.AccountFlowRepairHandle20DTO;
import com.elitesland.fin.application.facade.dto.flowrepair.AccountFlowRepairHandleDTO;
import com.elitesland.fin.application.facade.param.account.AccountSnapshotParam;
import com.elitesland.fin.application.facade.param.creditaccountflow.CreditAccountFlowParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowPageParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowQueryParam;
import com.elitesland.fin.application.facade.param.flowrepair.AccountFlowRepairPageParam;
import com.elitesland.fin.application.facade.param.flowrepair.AccountFlowRepairParam;
import com.elitesland.fin.application.facade.param.flowrepair.AccountFlowRepairRedoParam;
import com.elitesland.fin.application.facade.vo.account.AccountRuleConfigQueryVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountRuleConfigPageVO;
import com.elitesland.fin.application.facade.vo.creditaccountflow.CreditAccountFlowVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.facade.vo.flowrepair.AccountFlowRepairVO;
import com.elitesland.fin.application.facade.vo.flowrepair.CreditAccountFlowRepairVO;
import com.elitesland.fin.application.service.account.AccountService;
import com.elitesland.fin.application.service.creditaccountflow.CreditAccountFlowService;
import com.elitesland.fin.application.service.flow.AccountFlowService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinRedisConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.account.AccountDO;
import com.elitesland.fin.entity.account.AccountSnapshotDO;
import com.elitesland.fin.entity.creditaccount.CreditAccountRuleConfigDtlDO;
import com.elitesland.fin.entity.creditaccount.CreditAccountSnapshotDO;
import com.elitesland.fin.entity.flowrepair.AccountFlowRepairDO;
import com.elitesland.fin.repo.account.AccountRepo;
import com.elitesland.fin.repo.account.AccountRepoProc;
import com.elitesland.fin.repo.account.AccountSnapshotRepo;
import com.elitesland.fin.repo.flow.AccountFlowRepoProc;
import com.elitesland.fin.repo.flowrepair.AccountFlowProcedureRepository;
import com.elitesland.fin.repo.flowrepair.AccountFlowRepairRepo;
import com.elitesland.fin.repo.flowrepair.AccountFlowRepairRepoProc;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RBucket;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Example;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementSetter;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import javax.persistence.*;
import java.math.BigDecimal;
import java.sql.*;
import java.sql.Date;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2023/10/25
 * @Version 1.0
 * @Content:
 */
@Service
@Slf4j
@AllArgsConstructor
public class AccountFlowRepairServiceImpl implements AccountFlowRepairService{

    @Autowired
    private AccountFlowRepairRepo accountFlowRepairRepo;
    @Autowired
    private AccountFlowRepairRepoProc accountFlowRepairRepoProc;
    @Autowired
    private AccountFlowService accountFlowService;
    @Autowired
    private AccountFlowRepoProc accountFlowRepoProc;
    @Autowired
    private AccountRepoProc accountRepoProc;
    @Autowired
    private AccountRepo accountRepo;
    @Autowired
    private AccountSnapshotRepo accountSnapshotRepo;
    @Autowired
    private AccountService accountService;
    @Autowired
    private CreditAccountFlowService creditAccountFlowService;
    @Autowired
    private CreditAccountFlowRepairService creditAccountFlowRepairService;
    private final RedissonClient redissonClient;

    //@Autowired
    @PersistenceContext
    private EntityManager entityManager;
    private final JdbcTemplate jdbcTemplate;
    //@Autowired
    private final NamedParameterJdbcTemplate namedTemplate;
    private final TransactionTemplate transactionTemplate;
    @Autowired
    private AccountFlowProcedureRepository accountFlowProcedureRepository;
    //@Value("${spring.datasource.url}")
    private static String dataUrl = "";
    //@Value("${ext.jdbc-username}")
    private static String userName = "";
    //@Value("${ext.jdbc-pwd}")
    private static String pwd = "";

    /**
     * 账户流水修复重算分页查询
     *
     * @param pageParam 入参
     * @return 账户流水修复重算信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountFlowRepairVO> page(AccountFlowRepairPageParam pageParam){

        PagingVO<AccountFlowRepairVO> pagingVO = accountFlowRepairRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AccountFlowRepairVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        return PagingVO.<AccountFlowRepairVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }


    /**
     * 账户流水修复重算保存
     *
     * @param param 入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void save(AccountFlowRepairParam param){
        if (Objects.equals(param.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode())){
            saveFlowRepair10(param);
        }else if (Objects.equals(param.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_20.getValueCode())){
            if (Objects.isNull(param.getId())){
                saveFlowRepair20(param);
            }else {
                saveFlowRepair10(param);
            }
        }else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未匹配到重算原因!");

        }

    }

    private void saveFlowRepair10(AccountFlowRepairParam param){
        if (Objects.equals(param.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode())) {
            if (StringUtils.isBlank(param.getFlowNo())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "流水号不能为空!");
            }
            if (Objects.isNull(param.getActualAmount())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "实际金额不能为空!");
            }
        }

        AccountFlowVO flowVO = selectFlowByFlowNo(param.getFlowNo());

        if (Objects.isNull(param.getId())){
            AccountFlowRepairDO flowRepairDO = new AccountFlowRepairDO();
            assembleFlowRepair(flowRepairDO,param,flowVO);
            accountFlowRepairRepo.save(flowRepairDO);
        }else {
            Optional<AccountFlowRepairDO> flowRepairOptional = accountFlowRepairRepo.findById(param.getId());
            if (flowRepairOptional.isEmpty()){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到此条流水修复数据!");
            }
            AccountFlowRepairDO flowRepairDO = flowRepairOptional.get();
            if (!Objects.equals(flowRepairDO.getRepairStatus(),UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode())){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "草稿状态才可修改!");
            }
            assembleFlowRepair(flowRepairDO,param,flowVO);
            accountFlowRepairRepo.save(flowRepairDO);
        }
    }

    private void assembleFlowRepair(AccountFlowRepairDO flowRepairDO,
                               AccountFlowRepairParam param,AccountFlowVO flowVO){
        if (StringUtils.isNotBlank(param.getFlowNo())) {
            flowRepairDO.setFlowNo(param.getFlowNo());
        }
        if (Objects.nonNull(param.getActualAmount())) {
            flowRepairDO.setActualAmount(param.getActualAmount());
        }
        flowRepairDO.setAmount(flowVO.getAmount());
        flowRepairDO.setOpenAccountEntityCode(null);
        flowRepairDO.setOpenAccountEntityName(flowVO.getAccountHolderName());
        flowRepairDO.setAccountType(flowVO.getAccountType());
        flowRepairDO.setAccountCode(flowVO.getAccountCode());
        flowRepairDO.setAccountName(flowVO.getAccountName());
        flowRepairDO.setSourceNo(flowVO.getSourceNo());
        flowRepairDO.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());

        if (StringUtils.isNotBlank(param.getRecalculationReason())) {
            flowRepairDO.setRecalculationReason(param.getRecalculationReason());
        }
        if (Objects.nonNull(param.getRecalculationDate())){
            flowRepairDO.setRecalculationDate(param.getRecalculationDate());
        }else {
            if (Objects.equals(param.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode())) {
                if (Objects.isNull(flowRepairDO.getRecalculationDate())){
                    flowRepairDO.setRecalculationDate(flowVO.getAuditDate());

                }
            }
        }

        SysUserDTO sysUserDTO = getUser();
        if (StringUtils.isNotBlank(param.getRecalculationUser())) {
            flowRepairDO.setRecalculationUser(param.getRecalculationUser());
        }else {
            flowRepairDO.setRecalculationUser(sysUserDTO.getPrettyName());
        }
        if (Objects.nonNull(param.getRecalculationUserId())){
            flowRepairDO.setRecalculationUserId(param.getRecalculationUserId());
        }else {
            flowRepairDO.setRecalculationUserId(sysUserDTO.getId());
        }

    }

    private SysUserDTO getUser() {
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (userDetails == null) {
            throw new BusinessException("当前登陆人信息获取为空!");
        }
        if (Objects.isNull(userDetails.getUser())){
            throw new BusinessException("当前登陆人信息为空!");
        }
        return userDetails.getUser();
    }

    private AccountFlowVO selectFlowByFlowNo(String flowNo){
        if (StringUtils.isBlank(flowNo)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "流水号为空!");
        }
        AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
        flowQueryParam.setFlowNo(flowNo);
        List<AccountFlowVO> flowVOList = accountFlowService.selectListByQueryParam(flowQueryParam);
        if (CollectionUtils.isEmpty(flowVOList)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到此流水号("+flowNo+")的流水信息!");
        }
        if (CollectionUtils.isNotEmpty(flowVOList) && flowVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "此流水号("+flowNo+")的存在多条流水信息!");
        }

        return flowVOList.get(0);
    }

    /**
     * 根据账户流水修复重算ID批量删除
     *
     * @param ids ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteBatch(List<Long> ids) {
        AccountFlowRepairPageParam flowRepairPageParam = new AccountFlowRepairPageParam();
        flowRepairPageParam.setIds(ids);
        List<AccountFlowRepairVO> respVOList = accountFlowRepairRepoProc.selectListByParam(flowRepairPageParam);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        //针对‘待发布’状态的任务（主表）
        List<AccountFlowRepairVO> voList = respVOList.stream().filter(respVO -> !Objects.equals(respVO.getRepairStatus(), UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(voList)) {
            String checkResult = voList.stream().map(vo ->
                    "流水号:" + vo.getFlowNo() + ""
            ).collect(Collectors.joining(";", "只有草稿可删除,[", "], 请检查"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        //逻辑删除
        accountFlowRepairRepoProc.updateDeleteFlagBatch(1, ids);

    }


    /**
     * 账户流水修复重算
     *
     * @param repairRedoParam 重算入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void repairRedo(AccountFlowRepairRedoParam repairRedoParam){
        //选中或者不选(默认全部)的草稿状态的才可重算
        AccountFlowRepairPageParam flowRepairPageParam = new AccountFlowRepairPageParam();
        flowRepairPageParam.setIds(repairRedoParam.getIds());
        flowRepairPageParam.setFlowNoList(repairRedoParam.getFlowNoList());
        flowRepairPageParam.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());
        List<AccountFlowRepairVO> repairFlowVOList = accountFlowRepairRepoProc.selectListByParam(flowRepairPageParam);
        if (CollectionUtils.isEmpty(repairFlowVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "无符合重算的账户流水数据!");
        }

        val flowNoCheckList = repairFlowVOList.stream().filter(repairParam -> !StringUtils.isBlank(repairParam.getFlowNo()))
                // 获得元素出现频率的 Map，键为元素，值为元素出现的次数
                .map(AccountFlowRepairVO::getFlowNo).collect(Collectors.toMap(e -> e, e -> 1, (a, b) -> a + b))
                // 所有 entry 对应的 Stream
                .entrySet().stream()
                // 过滤出元素出现次数大于 1 的 entry
                .filter(entry -> entry.getValue() > 1)
                // 获得 entry 的键（重复元素）对应的 Stream
                .map(entry -> entry.getKey())
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(flowNoCheckList)) {
            String checkResult = flowNoCheckList.stream().map(Objects::toString).collect(Collectors.joining("、", "账户流水号[", "], 存在重复数据,请检查!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        List<String> repairFlowNoList = repairFlowVOList.stream().map(AccountFlowRepairVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(repairFlowNoList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重算的账户流水数据的流水号为空!");
        }
        /*AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
        flowQueryParam.setFlowNoList(repairFlowNoList);
        log.info("修复重算账户流水查询数据,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowQueryParam));
        List<AccountFlowVO> flowVOList = accountFlowService.selectListByQueryParam(flowQueryParam);
        log.info("修复重算账户流水查询数据,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));

        if (CollectionUtils.isEmpty(flowVOList)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到账户流水信息!");
        }*/

        List<String> repairFlowNoCacheList = new ArrayList<>();
        repairFlowNoList.forEach(flowNo -> {
            if (StringUtils.isNotBlank(getFlowNoCache(flowNo))){
                repairFlowNoCacheList.add(flowNo);
            }else {
                saveFlowNoCache(flowNo);
            }
        });
        if (CollectionUtils.isNotEmpty(repairFlowNoCacheList)) {
            String checkResult = repairFlowNoCacheList.stream().map(Objects::toString).collect(Collectors.joining("、", "账户流水号[", "], 修复重算中,请稍后再次重算!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        if (CollectionUtils.isNotEmpty(repairFlowVOList)){

            //区分10和20
            List<AccountFlowRepairVO> repairFlow10List = repairFlowVOList.stream().filter(accountFlowRepairVO -> Objects.equals(accountFlowRepairVO.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode())).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(repairFlow10List)){
                List<String> repairFlowNo10List = repairFlow10List.stream().map(AccountFlowRepairVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                processingCalculation(repairFlow10List,repairFlowNo10List);
            }

            List<AccountFlowRepairVO> repairFlow20List = repairFlowVOList.stream().filter(accountFlowRepairVO -> Objects.equals(accountFlowRepairVO.getRecalculationReason(),UdcEnum.FLOW_RECALCUL_REASON_20.getValueCode())).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(repairFlow20List)){
                List<String> repairFlowNo20List = repairFlow20List.stream().map(AccountFlowRepairVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                processingCalculation20(repairFlow20List,repairFlowNo20List);
            }

            //processingCalculation(repairFlowVOList,repairFlowNoList);
        }

    }

    @Override
    public void removeFlowNoCacheBatch(List<String> flowNoList) {
        flowNoList.forEach(flowNo -> {
            removeFlowNoCache(flowNo);
        });
    }

    @Override
    public void removeFlowNoCache(String flowNo) {
        RBucket<String> bucket = redissonClient.getBucket(FinRedisConstant.ACCOUNT_FLOW_REPAIR_KEY+flowNo);
        if (bucket.isExists()) {
            bucket.delete();
        }
    }

    @Override
    public String getFlowNoCache(String flowNo) {
        RBucket<String> bucket = redissonClient.getBucket(FinRedisConstant.ACCOUNT_FLOW_REPAIR_KEY+flowNo);
        if (bucket.isExists()) {
            return bucket.get();
        }
        return null;
    }

    @Override
    public void saveFlowNoCache(String flowNo){
        RBucket<String> bucket = redissonClient.getBucket(FinRedisConstant.ACCOUNT_FLOW_REPAIR_KEY+flowNo);
        bucket.set(flowNo);
    }



    private void processingCalculation(List<AccountFlowRepairVO> repairFlowVOList,List<String> repairFlowNoList){

       List<String> accCodeList = new ArrayList<>();

        repairFlowVOList.forEach(flowRepairVO -> {
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            String processingFlowNo = transactionTemplate.execute(transactionStatus -> {
                String repairFlowNo = flowRepairVO.getFlowNo();
                try {
                    AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
                    flowQueryParam.setFlowNo(flowRepairVO.getFlowNo());
                    log.info("修复重算账户流水查询数据,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowQueryParam));
                    List<AccountFlowVO> flowVOList = accountFlowService.selectListByQueryParam(flowQueryParam);
                    log.info("修复重算账户流水查询数据,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));

                    if (CollectionUtils.isNotEmpty(flowVOList) && flowVOList.size() == 1 ){
                        if (Objects.nonNull(flowVOList.get(0))) {
                            AccountFlowVO flowVO = flowVOList.get(0);

                            if (Objects.equals(flowVO.getAccountType(),UdcEnum.ACCOUNT_TYPE_STORE.getValueCode()) &&
                                    FinConstant.ACCOUNT_FLOW_REPAIR_TRANSACTION_TYPE.contains(flowVO.getTransactionType())){
                                repairFlow(flowRepairVO,flowVO,accCodeList);
                            }else if (Objects.equals(flowVO.getAccountType(),UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode()) &&
                                    FinConstant.ACCOUNT_FLOW_REPAIR_TRANSACTION_TYPE.contains(flowVO.getTransactionType())){
                                repairFlow(flowRepairVO,flowVO,accCodeList);

                            }else {
                                log.info("未修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
                            }

                            if (accCodeList.contains(flowVO.getAccountCode())){
                                //更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户金额、账户占用金额、账户可用金额
                                updateAccountAmt(Collections.singletonList(flowVO.getAccountCode()),repairFlowNoList);
                                //更新账户快照表的对应数据
                                //原先流水修复程序的重算日期 取流水的审核日期
                                if (Objects.isNull(flowRepairVO.getRecalculationDate())){
                                    flowRepairVO.setRecalculationDate(flowVO.getAuditDate());
                                }
                                updateAccountSnapshot(Collections.singletonList(flowRepairVO),repairFlowNoList);

                            }

                        } else {
                            log.info("未查询到修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowRepairVO));
                        }
                    }else {
                        log.info("修复账户流水数据为空或者有多条,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));
                    }

                    //释放校验缓存
                    removeFlowNoCache(flowRepairVO.getFlowNo());
                    return flowRepairVO.getFlowNo();
                } catch (Exception e) {
                    log.error("流水号("+repairFlowNo+")账户流水修复重算失败!:{}", e.getMessage());
                    //回滚
                    transactionStatus.setRollbackOnly();
                    repairFlowNoList.forEach(flowNo -> {
                        removeFlowNoCache(flowNo);
                    });
                    //抛出异常，不往下执行
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "流水号("+repairFlowNo+")账户流水修复重算失败!");
                }
            });



        });

       /* //更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户金额、账户占用金额、账户可用金额
        if (CollectionUtils.isNotEmpty(accCodeList)) {
            List<String> accCodes = accCodeList.stream().filter(Objects::nonNull).distinct().collect(Collectors.toList());
            updateAccountAmt(accCodes,repairFlowNoList);
        }*/

        repairFlowNoList.forEach(flowNo -> {
            removeFlowNoCache(flowNo);
        });
    }


    @Transactional(rollbackFor = Exception.class)
    public void repairFlow(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,List<String> accCodeList) {
        repairCurrentFlow(flowRepairVO,flowVO,accCodeList);


    }

    /**
     * 根据账户编码更新账户金额：更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户金额、账户占用金额、账户可用金额
     *
     * @param accCodeList 账户编码
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateAccountAmt(List<String> accCodeList,List<String> repairFlowNoList) {
        accCodeList.forEach(accCode -> {
            //根据账户编码查询,按照审核时间进行倒叙查询并取第一条数据
            AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
            flowQueryParam.setAccountCode(accCode);
            AccountFlowVO flowVO = accountFlowRepoProc.selectPreviousRepairAmtByParam(flowQueryParam);
            //根据编码更新账户金额
            AccountDO accountDO = accountRepoProc.findByAccountCode(accCode);
            if (Objects.nonNull(flowVO) && Objects.nonNull(accountDO)){
                BigDecimal accountAmount = flowVO.getAccountAmount();
                BigDecimal accountOccupancyAmount = flowVO.getAccountOccupancyAmount();

                accountDO.setAccountAmount(accountAmount);
                accountDO.setAccountOccupancyAmount(accountOccupancyAmount);
                accountDO.setAccountAvailableAmount(accountAmount.subtract(accountOccupancyAmount));
                accountRepo.save(accountDO);
            }else if (Objects.isNull(accountDO)){
                repairFlowNoList.forEach(flowNo -> {
                    removeFlowNoCache(flowNo);
                });
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到账户("+accCode+")信息!");

            }


        });
    }

    /**
     * 修复重算当前此条账户流水表的信息和修复流水记录表的信息
     *
     * @param flowRepairVO   修复流水表的此条数据
     * @param flowVO   要修复的此条账户流水数据
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairCurrentFlow(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,List<String> accCodeList) {
        AccountFlowRepairHandleDTO flowRepairHandleDTO = assembleRepairHandle(flowRepairVO,flowVO);
        if (Objects.nonNull(flowRepairHandleDTO)){
            repairCalculationFlow(flowRepairHandleDTO,accCodeList);
            accountFlowRepairRepoProc.updateAmountAndRepairStatusById(
                    flowRepairHandleDTO.getAmount(),UdcEnum.FLOW_REPAIR_STATUS_FIXED.getValueCode(),flowRepairHandleDTO.getFlowRepairId());
            repairAfterFlow(flowRepairVO,flowVO,flowRepairHandleDTO,accCodeList);
        }

    }

    /**
     * 修复重算当前此条之后的账户流水表的信息
     *
     * @param flowRepairVO   修复流水表的此条数据
     * @param flowVO   要修复的此条账户流水数据
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairAfterFlow(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,AccountFlowRepairHandleDTO flowRepairCurrentHandleDTO,List<String> accCodeList) {
        List<AccountFlowRepairHandleDTO> flowRepairHandleList = selectRepairAfterFlow(flowRepairVO,flowVO,flowRepairCurrentHandleDTO);
        if (CollectionUtils.isNotEmpty(flowRepairHandleList)){
            flowRepairHandleList.forEach(flowRepairHandleDTO -> {
                repairCalculationFlowAfter(flowRepairHandleDTO,accCodeList);
            });
        }
    }

    /**
     * 查询某条账户流水以及之后账户流水的数据-相同账户编码的
     *
     * @param flowRepairVO   修复流水表的此条数据
     * @param flowVO   要修复的此条账户流水数据
     * @param flowRepairCurrentHandleDTO   当前修复的此条账户流水的处理数据
     * @return
     */
    public List<AccountFlowRepairHandleDTO> selectRepairAfterFlow(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,AccountFlowRepairHandleDTO flowRepairCurrentHandleDTO) {
        int size = 500;
        int page = 1;

        List<AccountFlowRepairHandleDTO>  flowRepairHandleList= new ArrayList<>();

        //查询要修复的该条账户流水及之后的相同账户编码的全部流水的
        AccountFlowPageParam flowPageParam = new AccountFlowPageParam();
        flowPageParam.setAccountCode(flowVO.getAccountCode());
        flowPageParam.setRepairTime(flowVO.getAuditDate());
        //之后的账户流水不做全部查出，要分批查出循环处理
        while (true) {
            flowPageParam.setCurrent(page++);
            flowPageParam.setSize(size);
            log.info("查询修复账户流水及之后流水,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowPageParam));
            List<AccountFlowVO> flowAfterList = accountFlowService.selectRepairAfterPage(flowPageParam);
            log.info("查询修复账户流水及之后流水,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowAfterList));
            if (CollectionUtils.isEmpty(flowAfterList)) {
                break;
            }

            flowAfterList.forEach(flowAfterVO -> {
                AccountFlowRepairHandleDTO flowRepairHandleDTO = assembleRepairAfterHandle(flowRepairVO,flowAfterVO,flowRepairCurrentHandleDTO);
                if (Objects.nonNull(flowRepairHandleDTO)){
                    flowRepairHandleList.add(flowRepairHandleDTO);
                }else {
                    log.info("按照交易类型未匹配到账户流水,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowAfterList));
                }
            });

            if (flowAfterList.size() < size) {
                // 少于要查询的数量，则说明已无数据
                break;
            }
        }

        return flowRepairHandleList;
    }

    private AccountFlowRepairHandleDTO assembleRepairHandle(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO){
        if (Objects.isNull(flowRepairVO.getActualAmount()) || Objects.isNull(flowVO.getAccountAmount()) ||
                Objects.isNull(flowVO.getAccountOccupancyAmount()) || Objects.isNull(flowVO.getAmount())){
            log.info("未修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
            return null;
        }
        AccountFlowRepairHandleDTO flowRepairHandleDTO = new AccountFlowRepairHandleDTO();
        flowRepairHandleDTO.setFlowRepairId(flowRepairVO.getId());
        flowRepairHandleDTO.setFlowRepairFlowNo(flowRepairVO.getFlowNo());
        flowRepairHandleDTO.setFlowId(flowVO.getId());
        flowRepairHandleDTO.setFlowNo(flowVO.getFlowNo());
        flowRepairHandleDTO.setActualAmount(flowRepairVO.getActualAmount());
        flowRepairHandleDTO.setAmount(flowVO.getAmount());
        //变动金额=实际交易金额-原先错误流水交易金额
        BigDecimal variableAmount = flowRepairVO.getActualAmount().subtract(flowVO.getAmount());
        flowRepairHandleDTO.setVariableAmount(variableAmount);
        flowRepairHandleDTO.setAccountType(flowVO.getAccountType());
        flowRepairHandleDTO.setAccountCode(flowVO.getAccountCode());
        flowRepairHandleDTO.setAccountName(flowVO.getAccountName());
        flowRepairHandleDTO.setTransactionType(flowVO.getTransactionType());

        flowRepairHandleDTO.setAccountAmount(flowVO.getAccountAmount());
        flowRepairHandleDTO.setAccountOccupancyAmount(flowVO.getAccountOccupancyAmount());
        flowRepairHandleDTO.setAccountAvailableAmount(flowVO.getAccountAvailableAmount());
        return flowRepairHandleDTO;
    }

    private AccountFlowRepairHandleDTO assembleRepairAfterHandle(AccountFlowRepairVO flowRepairVO,AccountFlowVO flowVO,AccountFlowRepairHandleDTO flowRepairCurrentHandleDTO){
        if (Objects.isNull(flowRepairVO.getActualAmount()) || Objects.isNull(flowVO.getAccountAmount()) ||
                Objects.isNull(flowVO.getAccountOccupancyAmount()) || Objects.isNull(flowVO.getAmount())){
            log.info("未修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
            return null;
        }
        AccountFlowRepairHandleDTO flowRepairHandleDTO = new AccountFlowRepairHandleDTO();
        flowRepairHandleDTO.setFlowRepairId(flowRepairVO.getId());
        flowRepairHandleDTO.setFlowRepairFlowNo(flowRepairVO.getFlowNo());
        flowRepairHandleDTO.setFlowId(flowVO.getId());
        flowRepairHandleDTO.setFlowNo(flowVO.getFlowNo());
        flowRepairHandleDTO.setActualAmount(flowRepairVO.getActualAmount());
        flowRepairHandleDTO.setAmount(flowVO.getAmount());
        //变动金额=实际交易金额-原先错误流水交易金额
        //BigDecimal variableAmount = flowRepairVO.getActualAmount().subtract(flowVO.getAmount());
        BigDecimal variableAmount = flowRepairCurrentHandleDTO.getVariableAmount();
        flowRepairHandleDTO.setVariableAmount(variableAmount);
        flowRepairHandleDTO.setAccountType(flowVO.getAccountType());
        flowRepairHandleDTO.setAccountCode(flowVO.getAccountCode());
        flowRepairHandleDTO.setAccountName(flowVO.getAccountName());
        flowRepairHandleDTO.setTransactionType(flowRepairCurrentHandleDTO.getTransactionType());

        flowRepairHandleDTO.setAccountAmount(flowVO.getAccountAmount());
        flowRepairHandleDTO.setAccountOccupancyAmount(flowVO.getAccountOccupancyAmount());
        flowRepairHandleDTO.setAccountAvailableAmount(flowVO.getAccountAvailableAmount());
        return flowRepairHandleDTO;
    }
    /**
     * 修复的计算处理逻辑
     *
     * @param flowRepairHandleDTO 账户流水修复重算处理计算DTO
     *
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairCalculationFlow(AccountFlowRepairHandleDTO flowRepairHandleDTO,List<String> accCodeList) {
        //账户金额
        BigDecimal accountAmount = flowRepairHandleDTO.getAccountAmount();
        //账户占用金额
        BigDecimal accountOccupancyAmount = flowRepairHandleDTO.getAccountOccupancyAmount();
        //账户可用金额 = 账户金额-账户占用金额
        BigDecimal accountAvailableAmount = BigDecimal.ZERO;
        //变动金额
        BigDecimal variableAmount = flowRepairHandleDTO.getVariableAmount();
        //实际金额
        BigDecimal actualAmount = flowRepairHandleDTO.getActualAmount();

        if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_R.getValueCode())) {
            //交易类型=释放占用时，该条流水及之后的相同账户编码的全部流水的账户占用金额改成原账户占用金额 - 变动金额，
            accountOccupancyAmount = accountOccupancyAmount.subtract(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);
            accountFlowRepoProc.updateOccupancyAndAvailableAndAmountById(
                    accountOccupancyAmount,accountAvailableAmount,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());

        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_T.getValueCode())) {
            //交易类型=账户占用时，该条流水及之后的相同账户编码的全部流水的账户占用金额改成原账户占用金额 + 变动金额，
            accountOccupancyAmount = accountOccupancyAmount.add(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateOccupancyAndAvailableAndAmountById(
                    accountOccupancyAmount,accountAvailableAmount,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_O.getValueCode())) {
            //交易类型=账户扣减时，该条流水之后的相同账户编码的全部流水的账户金额改成原账户金额 - 变更金额，
            accountAmount = accountAmount.subtract(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateAccountAndAvailableAndAmountById(
                    accountAmount,accountAvailableAmount,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_I.getValueCode())) {
            //交易类型=账户增加时，该条流水之后的相同账户编码的全部流水的账户金额改成原账户金额 + 变更金额，
            accountAmount = accountAmount.add(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateAccountAndAvailableAndAmountById(
                    accountAmount,accountAvailableAmount,actualAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        }

    }

    /**
     * 之后流水的修复的计算处理逻辑
     *
     * @param flowRepairHandleDTO 账户流水修复重算处理计算DTO
     *
     */
    //@Transactional(rollbackFor = Exception.class)
    public void repairCalculationFlowAfter(AccountFlowRepairHandleDTO flowRepairHandleDTO,List<String> accCodeList) {
        //账户金额
        BigDecimal accountAmount = flowRepairHandleDTO.getAccountAmount();
        //账户占用金额
        BigDecimal accountOccupancyAmount = flowRepairHandleDTO.getAccountOccupancyAmount();
        //账户可用金额 = 账户金额-账户占用金额
        BigDecimal accountAvailableAmount = BigDecimal.ZERO;
        //变动金额
        BigDecimal variableAmount = flowRepairHandleDTO.getVariableAmount();
        //实际金额
        //BigDecimal actualAmount = flowRepairHandleDTO.getActualAmount();

        if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_R.getValueCode())) {
            //交易类型=释放占用时，该条流水及之后的相同账户编码的全部流水的账户占用金额改成原账户占用金额 - 变动金额，
            accountOccupancyAmount = accountOccupancyAmount.subtract(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);
            accountFlowRepoProc.updateOccupancyAndAvailableById(
                    accountOccupancyAmount,accountAvailableAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());

        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_T.getValueCode())) {
            //交易类型=账户占用时，该条流水及之后的相同账户编码的全部流水的账户占用金额改成原账户占用金额 + 变动金额，
            accountOccupancyAmount = accountOccupancyAmount.add(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateOccupancyAndAvailableById(
                    accountOccupancyAmount,accountAvailableAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_O.getValueCode())) {
            //交易类型=账户扣减时，该条流水之后的相同账户编码的全部流水的账户金额改成原账户金额 - 变更金额，
            accountAmount = accountAmount.subtract(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateAccountAndAvailableById(
                    accountAmount,accountAvailableAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        } else if (Objects.equals(flowRepairHandleDTO.getTransactionType(), UdcEnum.IO_TYPE_I.getValueCode())) {
            //交易类型=账户增加时，该条流水之后的相同账户编码的全部流水的账户金额改成原账户金额 + 变更金额，
            accountAmount = accountAmount.add(variableAmount);
            accountAvailableAmount = accountAmount.subtract(accountOccupancyAmount);

            accountFlowRepoProc.updateAccountAndAvailableById(
                    accountAmount,accountAvailableAmount,flowRepairHandleDTO.getFlowId());
            accCodeList.add(flowRepairHandleDTO.getAccountCode());


        }

    }


    /*****************************************/
    private void saveFlowRepair20(AccountFlowRepairParam param){
        String accountCode = param.getAccountCode();
        LocalDateTime recalculationDate = param.getRecalculationDate();

        List<AccountFlowRepairDO> repairDoAllList = new ArrayList<>();

        //1.储值账户
        List<AccountFlowRepairHandle20DTO> czRepairHandleAllList = new ArrayList<>();
        //1.1 查询储值账户 交易类型为冻结金额、解除冻结
        List<AccountFlowRepairHandle20DTO> czRepairHandle21List = czAppendSql21(accountCode,recalculationDate);
        if (CollectionUtil.isNotEmpty(czRepairHandle21List)){
            czRepairHandleAllList.addAll(czRepairHandle21List);
        }
        //1.2 查询储值账户 交易类型为余额增加、余额扣减
        List<AccountFlowRepairHandle20DTO> czRepairHandle22List = czAppendSql22(accountCode,recalculationDate);
        if (CollectionUtil.isNotEmpty(czRepairHandle22List)){
            czRepairHandleAllList.addAll(czRepairHandle22List);
        }
        //1.3组装保存储值账户
        assembleHandleFlowRepair20(param,czRepairHandleAllList,repairDoAllList);

        //1.返利账户
        List<AccountFlowRepairHandle20DTO> flRepairHandleAllList = new ArrayList<>();
        //1.1 查询返利账户 交易类型为冻结金额、解除冻结
        List<AccountFlowRepairHandle20DTO> flRepairHandle21List = flAppendSql21(accountCode,recalculationDate);
        if (CollectionUtil.isNotEmpty(flRepairHandle21List)){
            flRepairHandleAllList.addAll(flRepairHandle21List);
        }
        //1.2 查询返利账户 交易类型为余额增加、余额扣减
        List<AccountFlowRepairHandle20DTO> flRepairHandle22List = flAppendSql22(accountCode,recalculationDate);
        if (CollectionUtil.isNotEmpty(flRepairHandle22List)){
            flRepairHandleAllList.addAll(flRepairHandle22List);
        }
        //1.3组装保存返利账户
        assembleHandleFlowRepair20(param,flRepairHandleAllList,repairDoAllList);

        if (CollectionUtil.isNotEmpty(repairDoAllList)){
            accountFlowRepairRepo.saveAll(repairDoAllList);
        }else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未匹配到符合重算的数据!");
        }
    }

    private void assembleHandleFlowRepair20(AccountFlowRepairParam param,List<AccountFlowRepairHandle20DTO> repairHandleAllList,List<AccountFlowRepairDO> repairDoAllList){
        if (CollectionUtil.isNotEmpty(repairHandleAllList)){
            Map<String, List<AccountFlowRepairHandle20DTO>> czRepairHandleAllMap = repairHandleAllList.stream().collect(Collectors.groupingBy(i -> i.getAccountCode()));
            czRepairHandleAllMap.forEach((key, value) -> {
                /*accountRuleConfigPageVOList = value.stream()
                        .sorted(Comparator.comparing(AccountFlowRepairHandle20DTO::getAuditDate))
                        .collect(Collectors.toList());*/

                AccountFlowRepairHandle20DTO repairHandle20DTO = value.stream().min((u1, u2) -> u1.getAuditDate().compareTo(u2.getAuditDate())).orElse(null);
                if (Objects.isNull(repairHandle20DTO)){
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "此账户("+key+")未匹配到最早审核时间的数据!");
                }

                AccountFlowRepairDO flowRepairDO = new AccountFlowRepairDO();
                assembleFlowRepair20(flowRepairDO,param,repairHandle20DTO);
                repairDoAllList.add(flowRepairDO);
                //accountFlowRepairRepo.save(flowRepairDO);
            });

        }
    }

    private void assembleFlowRepair20(AccountFlowRepairDO flowRepairDO,
                                      AccountFlowRepairParam param,
                                      AccountFlowRepairHandle20DTO repairHandle20DTO){
        flowRepairDO.setFlowNo(repairHandle20DTO.getFlowNo());
        flowRepairDO.setActualAmount(repairHandle20DTO.getAmount());
        flowRepairDO.setAmount(repairHandle20DTO.getAmount());
        flowRepairDO.setOpenAccountEntityCode(null);
        flowRepairDO.setOpenAccountEntityName(repairHandle20DTO.getAccountHolderName());
        flowRepairDO.setAccountType(repairHandle20DTO.getAccountType());
        flowRepairDO.setAccountCode(repairHandle20DTO.getAccountCode());
        flowRepairDO.setAccountName(repairHandle20DTO.getAccountName());
        flowRepairDO.setSourceNo(repairHandle20DTO.getSourceNo());
        flowRepairDO.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());

        if (StringUtils.isNotBlank(param.getRecalculationReason())) {
            flowRepairDO.setRecalculationReason(param.getRecalculationReason());
        }
        if (Objects.nonNull(param.getRecalculationDate())){
            flowRepairDO.setRecalculationDate(param.getRecalculationDate());
        }else {
            flowRepairDO.setRecalculationDate(repairHandle20DTO.getPrevAuditDate());
        }

        SysUserDTO sysUserDTO = getUser();
        if (StringUtils.isNotBlank(param.getRecalculationUser())) {
            flowRepairDO.setRecalculationUser(param.getRecalculationUser());
        }else {
            flowRepairDO.setRecalculationUser(sysUserDTO.getPrettyName());
        }
        if (Objects.nonNull(param.getRecalculationUserId())){
            flowRepairDO.setRecalculationUserId(param.getRecalculationUserId());
        }else {
            flowRepairDO.setRecalculationUserId(sysUserDTO.getId());
        }
    }

    /**
     * 查询储值账户 交易类型为冻结金额、解除冻结
     *
     * @param accountCode 账户编码
     * @param recalculationDate 重算日期
     * @return
     */
    private List<AccountFlowRepairHandle20DTO> czAppendSql21(String accountCode,LocalDateTime recalculationDate) {
        StringBuilder sqlWhereBuffer = new StringBuilder();
        Map<String, Object> params = new HashMap<>();
        List<Object> args = new ArrayList<>();
        appendWhere(accountCode,recalculationDate,sqlWhereBuffer,params,args);

        StringBuilder sqlBuffer = new StringBuilder();
        sqlBuffer.append("select b.* from \n" +
                "        (select \n" +
                "        account_code,account_name,account_type,transaction_type,account_holder_name, \n" +
                "        flow_no,amount,account_amount,account_occupancy_amount,account_available_amount,audit_date,source_no, \n" +
                "        prev_account_occupancy_amount, \n" +
                "        prev_flow_no, \n" +
                "        prev_audit_date, \n" +
                "        if(prev_flow_no is null,amount,if(prev_account_occupancy_amount is null,0,prev_account_occupancy_amount  - account_occupancy_amount)) difference_value \n" +
                "        from \n" +
                "                (select A.*, \n" +
                "                        lag(account_occupancy_amount, 1,0) over(partition by account_code order by audit_date) prev_account_occupancy_amount, \n" +
                "                        lag(flow_no, 1) over(partition by account_code order by audit_date) prev_flow_no, \n" +
                "                        lag(audit_date, 1) over(partition by account_code order by audit_date) prev_audit_date \n" +
                "                from \n" +
                "                        account_flow as A \n" +
                "                         where \n" +
                "                                A.delete_flag = 0 \n" +
                "                                and A.transaction_type in ('T','R') \n" +
                "                               and A.account_type in ('STORE') \n" +
                "                               and A.audit_date is not null \n" +
                                                sqlWhereBuffer +
                "                ) T \n" +
                "        ) b \n" +
                "where \n" +
                "        abs(b.difference_value) != abs(b.amount) \n" +
                "       order by b.audit_date ");


        //List<AccountFlowRepairHandle20DTO> accountFlowRepairHandle20DTOList = jdbcTemplate.query(sqlBuffer.toString(), (PreparedStatementSetter) params, new BeanPropertyRowMapper<>(AccountFlowRepairHandle20DTO.class));
        //List<AccountFlowRepairHandle20DTO> accountFlowRepairHandle20DTOList = jdbcTemplate.query(sqlBuffer.toString(),args.toArray(), new BeanPropertyRowMapper<>(AccountFlowRepairHandle20DTO.class));
        List<AccountFlowRepairHandle20DTO> accountFlowRepairHandle20DTOList = namedTemplate.query(sqlBuffer.toString(),params, new BeanPropertyRowMapper<>(AccountFlowRepairHandle20DTO.class));

        return accountFlowRepairHandle20DTOList;
    }

    /**
     * 查询储值账户 交易类型为余额增加、余额扣减
     *
     * @param accountCode 账户编码
     * @param recalculationDate 重算日期
     * @return
     */
    private List<AccountFlowRepairHandle20DTO> czAppendSql22(String accountCode,LocalDateTime recalculationDate) {
        StringBuilder sqlWhereBuffer = new StringBuilder();
        Map<String, Object> params = new HashMap<>();
        List<Object> args = new ArrayList<>();
        appendWhere(accountCode,recalculationDate,sqlWhereBuffer,params,args);

        StringBuilder sqlBuffer = new StringBuilder();
        sqlBuffer.append("select b.* from \n" +
                "        (select \n" +
                "        account_code,account_name,account_type,transaction_type,account_holder_name, \n" +
                "        flow_no,amount,account_amount,account_occupancy_amount,account_available_amount,audit_date,source_no, \n" +
                "        prev_account_amount, \n" +
                "        prev_flow_no, \n" +
                "        prev_audit_date, \n" +
                "        if(prev_flow_no is null,amount,if(prev_account_amount is null,0,prev_account_amount - account_amount)) difference_value \n" +
                "        from \n" +
                "                (select A.*, \n" +
                "                        lag(account_amount, 1,0) over(partition by account_code order by audit_date) prev_account_amount, \n" +
                "                        lag(flow_no, 1) over(partition by account_code order by audit_date) prev_flow_no, \n" +
                "                        lag(audit_date, 1) over(partition by account_code order by audit_date) prev_audit_date \n" +
                "                from \n" +
                "                        account_flow as A \n" +
                "                         where \n" +
                "                                A.delete_flag = 0 \n" +
                "                                and A.transaction_type in ('I','O') \n" +
                "                               and A.account_type in ('STORE') \n" +
                "                               and A.audit_date is not null \n" +
                                                sqlWhereBuffer +
                "                ) T \n" +
                "        ) b \n" +
                "where \n" +
                "        abs(b.difference_value) != abs(b.amount) \n" +
                "       order by b.audit_date");


        List<AccountFlowRepairHandle20DTO> accountFlowRepairHandle20DTOList = namedTemplate.query(sqlBuffer.toString(),params, new BeanPropertyRowMapper<>(AccountFlowRepairHandle20DTO.class));

        return accountFlowRepairHandle20DTOList;
    }


    /**
     * 查询返利账户 交易类型为冻结金额、解除冻结
     *
     * @param accountCode 账户编码
     * @param recalculationDate 重算日期
     * @return
     */
    private List<AccountFlowRepairHandle20DTO> flAppendSql21(String accountCode,LocalDateTime recalculationDate) {
        StringBuilder sqlWhereBuffer = new StringBuilder();
        Map<String, Object> params = new HashMap<>();
        List<Object> args = new ArrayList<>();
        appendWhere(accountCode,recalculationDate,sqlWhereBuffer,params,args);

        StringBuilder sqlBuffer = new StringBuilder();
        sqlBuffer.append("select b.* from \n" +
                "        (select \n" +
                "        account_code,account_name,account_type,transaction_type,account_holder_name, \n" +
                "        flow_no,amount,account_amount,account_occupancy_amount,account_available_amount,audit_date,source_no, \n" +
                "        prev_account_occupancy_amount, \n" +
                "        prev_flow_no, \n" +
                "        prev_audit_date, \n" +
                "        if(prev_flow_no is null,amount,if(prev_account_occupancy_amount is null,0,prev_account_occupancy_amount  - account_occupancy_amount)) difference_value \n" +
                "        from \n" +
                "                (select A.*, \n" +
                "                        lag(account_occupancy_amount, 1,0) over(partition by account_code order by audit_date) prev_account_occupancy_amount, \n" +
                "                        lag(flow_no, 1) over(partition by account_code order by audit_date) prev_flow_no, \n" +
                "                        lag(audit_date, 1) over(partition by account_code order by audit_date) prev_audit_date \n" +
                "                from \n" +
                "                        account_flow as A \n" +
                "                         where \n" +
                "                                A.delete_flag = 0 \n" +
                "                                and A.transaction_type in ('T','R') \n" +
                "                               and A.account_type in ('FLZH') \n" +
                "                               and A.audit_date is not null \n" +
                sqlWhereBuffer +
                "                ) T \n" +
                "        ) b \n" +
                "where \n" +
                "        abs(b.difference_value) != abs(b.amount) \n" +
                "       order by b.audit_date ");


        List<AccountFlowRepairHandle20DTO> accountFlowRepairHandle20DTOList = namedTemplate.query(sqlBuffer.toString(),params, new BeanPropertyRowMapper<>(AccountFlowRepairHandle20DTO.class));

        return accountFlowRepairHandle20DTOList;
    }

    /**
     * 查询返利账户 交易类型为余额增加、余额扣减
     *
     * @param accountCode 账户编码
     * @param recalculationDate 重算日期
     * @return
     */
    private List<AccountFlowRepairHandle20DTO> flAppendSql22(String accountCode,LocalDateTime recalculationDate) {
        StringBuilder sqlWhereBuffer = new StringBuilder();
        Map<String, Object> params = new HashMap<>();
        List<Object> args = new ArrayList<>();
        appendWhere(accountCode,recalculationDate,sqlWhereBuffer,params,args);

        StringBuilder sqlBuffer = new StringBuilder();
        sqlBuffer.append("select b.* from \n" +
                "        (select \n" +
                "        account_code,account_name,account_type,transaction_type,account_holder_name, \n" +
                "        flow_no,amount,account_amount,account_occupancy_amount,account_available_amount,audit_date,source_no, \n" +
                "        prev_account_amount, \n" +
                "        prev_flow_no, \n" +
                "        prev_audit_date, \n" +
                "        if(prev_flow_no is null,amount,if(prev_account_amount is null,0,prev_account_amount - account_amount)) difference_value \n" +
                "        from \n" +
                "                (select A.*, \n" +
                "                        lag(account_amount, 1,0) over(partition by account_code order by audit_date) prev_account_amount, \n" +
                "                        lag(flow_no, 1) over(partition by account_code order by audit_date) prev_flow_no, \n" +
                "                        lag(audit_date, 1) over(partition by account_code order by audit_date) prev_audit_date \n" +
                "                from \n" +
                "                        account_flow as A \n" +
                "                         where \n" +
                "                                A.delete_flag = 0 \n" +
                "                                and A.transaction_type in ('I','O') \n" +
                "                               and A.account_type in ('FLZH') \n" +
                "                               and A.audit_date is not null \n" +
                sqlWhereBuffer +
                "                ) T \n" +
                "        ) b \n" +
                "where \n" +
                "        abs(b.difference_value) != abs(b.amount) \n" +
                "       order by b.audit_date");


        List<AccountFlowRepairHandle20DTO> accountFlowRepairHandle20DTOList = namedTemplate.query(sqlBuffer.toString(),params, new BeanPropertyRowMapper<>(AccountFlowRepairHandle20DTO.class));

        return accountFlowRepairHandle20DTOList;
    }

    private void appendWhere(String accountCode, LocalDateTime recalculationDate, StringBuilder sqlWhereBuffer, Map<String, Object> params, List<Object> args) {
        if (StringUtils.isNotBlank(accountCode)) {
            sqlWhereBuffer.append(" and A.account_code = :accountCode ");
            params.put("accountCode", accountCode);
        }
        if (Objects.nonNull(recalculationDate)) {
            sqlWhereBuffer.append(" and A.audit_date >= :auditDate ");
            params.put("auditDate", recalculationDate);

        }

       /* if (StringUtils.isNotBlank(accountCode)) {
            sqlWhereBuffer.append(" and A.account_code = ? ");
            args.add(accountCode);
        }
        if (Objects.nonNull(recalculationDate)) {
            sqlWhereBuffer.append(" and A.audit_date >= ? ");
            args.add(recalculationDate);
        }*/

    }


    private void processingCalculation20(List<AccountFlowRepairVO> repairFlowVOList,List<String> repairFlowNoList){

        for (AccountFlowRepairVO accountFlowRepairVO : repairFlowVOList) {
            String accountCode = accountFlowRepairVO.getAccountCode();
            LocalDateTime recalculationDate = accountFlowRepairVO.getRecalculationDate();
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            String successCode = transactionTemplate.execute(transactionStatus -> {

                try {
                    //重算 存储过程
                    String successAccountCode = executeAccountFlowProcedure(accountCode,recalculationDate,repairFlowNoList);
                    //String successAccountCode = callableStatementProcedure(accountCode,recalculationDate,repairFlowNoList);
                    if (Objects.nonNull(successAccountCode)){
                        accountFlowRepairRepoProc.updateRepairStatusById(UdcEnum.FLOW_REPAIR_STATUS_FIXED.getValueCode(),accountFlowRepairVO.getId());
                        //更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户金额、账户占用金额、账户可用金额
                        updateAccountAmt(Collections.singletonList(accountCode),repairFlowNoList);

                        //更新账户快照表的对应数据
                        updateAccountSnapshot(Collections.singletonList(accountFlowRepairVO),repairFlowNoList);


                        ///throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "测试存储过程回滚");

                    }else {
                        /*repairFlowNoList.forEach(flowNo -> {
                            removeFlowNoCache(flowNo);
                        });*/
                    }

                    //释放校验缓存
                    removeFlowNoCache(accountFlowRepairVO.getFlowNo());
                    return successAccountCode;
                } catch (Exception e) {

                    log.error("账户("+accountCode+")流水修复重算失败!:{}", e.getMessage());
                    //回滚
                    transactionStatus.setRollbackOnly();
                    repairFlowNoList.forEach(flowNo -> {
                        removeFlowNoCache(flowNo);
                    });
                    //抛出异常，不往下执行
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户("+accountCode+")流水修复重算失败!");
                }
            });

        }

       /* List<String> accCodeList = new ArrayList<>();
        //更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户金额、账户占用金额、账户可用金额
        if (CollectionUtils.isNotEmpty(accCodeList)) {
            List<String> accCodes = accCodeList.stream().filter(Objects::nonNull).distinct().collect(Collectors.toList());
            updateAccountAmt(accCodes,repairFlowNoList);
        }*/
        repairFlowNoList.forEach(flowNo -> {
            removeFlowNoCache(flowNo);
        });




    }

    private void updateAccountSnapshot(List<AccountFlowRepairVO> repairFlowVOList,List<String> repairFlowNoList){
        repairFlowVOList.forEach(accountFlowRepairVO -> {
            //更新账户快照表的对应数据
            //双层循环，快照表外层，流水表内层，当快照时间小于当前轮次的审核时间时返回上一个坐标i-1,若相等则返回本次坐标i
            AccountFlowPageParam accountFlowQueryParam = new AccountFlowPageParam();
            accountFlowQueryParam.setAccountCode(accountFlowRepairVO.getAccountCode());
            accountFlowQueryParam.setRepairTime(accountFlowRepairVO.getRecalculationDate());
            List<AccountFlowVO> accountFlowVOList = accountFlowService.selectRepairAfter(accountFlowQueryParam);
            Map<String, List<AccountFlowVO>> accountFlowAllMap = CollectionUtil.isEmpty(accountFlowVOList) ? new HashMap<>() :
                    accountFlowVOList.stream().collect(Collectors.groupingBy(i -> i.getAccountCode()));


            AccountSnapshotParam accountSnapshotParam = new AccountSnapshotParam();
            accountSnapshotParam.setAccountCode(accountFlowRepairVO.getAccountCode());
            accountSnapshotParam.setRepairTime(accountFlowRepairVO.getRecalculationDate());
            List<AccountSnapshotDO>  accountSnapshotDOList = accountService.selectAccountSnapshotByParam(accountSnapshotParam);
            Map<String, List<AccountSnapshotDO>> accountSnapshotAllMap = CollectionUtil.isEmpty(accountFlowVOList) ? new HashMap<>() :
                    accountSnapshotDOList.stream().collect(Collectors.groupingBy(i -> i.getAccountCode()));


            List<AccountSnapshotDO>  accountSnapshotUpdateList = new ArrayList<>();
            //accountSnapshotAllMap.forEach((key, value) -> {
            for (Map.Entry<String, List<AccountSnapshotDO>> entry : accountSnapshotAllMap.entrySet()) {
                String accountCode = entry.getKey();
                // 根据快照时间排序
                Collections.sort(entry.getValue(), Comparator.comparing(AccountSnapshotDO::getSnapshotTime));

                List<AccountFlowVO> accountFlowList = accountFlowAllMap.get(accountCode);
                // 根据审核时间排序
                Collections.sort(accountFlowList, Comparator.comparing(AccountFlowVO::getAuditDate));

                for (AccountSnapshotDO accountSnapshotDO : entry.getValue()) {

                    //流水坐标
                    Integer flowCoordinate = null;
                    List<AccountFlowVO> accountFlowFiltrationList = new ArrayList<>();
                    for (int i = 0; i < accountFlowList.size(); i++) {
                        AccountFlowVO accountFlowVO = accountFlowList.get(i);

                        //Integer j = i;
                        if (accountFlowVO.getAuditDate().isEqual(accountSnapshotDO.getSnapshotTime())){
                            //flowCoordinate = j;
                            accountFlowFiltrationList.add(accountFlowVO);
                            //break;
                        } else if (accountFlowVO.getAuditDate().isBefore(accountSnapshotDO.getSnapshotTime())) {
                           // flowCoordinate = (j == 0) ? j : j-1;
                            accountFlowFiltrationList.add(accountFlowVO);
                            //break;
                        }else {
                            break;
                        }
                    }

                    AccountFlowVO maxDateAccountFlow = accountFlowFiltrationList.stream().filter(Objects::nonNull)
                            .max(Comparator.comparing(AccountFlowVO::getAuditDate)).orElse(null);

                    if (Objects.nonNull(maxDateAccountFlow)){
                        //AccountFlowVO accountFlowVO = accountFlowList.get(flowCoordinate);
                        accountSnapshotDO.setAccountAmount(maxDateAccountFlow.getAccountAmount());
                        accountSnapshotDO.setAccountAvailableAmount(maxDateAccountFlow.getAccountAvailableAmount());
                        accountSnapshotDO.setAccountOccupancyAmount(maxDateAccountFlow.getAccountOccupancyAmount());
                        //accountSnapshotDO.setSnapshotTime(LocalDateTime.now());
                        accountSnapshotUpdateList.add(accountSnapshotDO);
                    }
                }

                //处理最后一条快照对应最后一条流水
                AccountSnapshotDO lastSnapshot = entry.getValue().get(entry.getValue().size() - 1);
                AccountFlowVO lastFlow = accountFlowList.get(accountFlowList.size() - 1);

               /*Optional<CreditAccountSnapshotDO> lastSnapshotOptional = creditAccountSnapshotUpdateList.stream().filter(creditAccountSnapshotDO -> Objects.equals(creditAccountSnapshotDO.getId(),lastSnapshot.getId())).findFirst();
               if (lastSnapshotOptional.isPresent() || lastSnapshot.getSnapshotTime().isAfter(lastFlow.getAuditDate())){

               }*/

                accountSnapshotUpdateList.forEach(accountSnapshotDO -> {
                    if (Objects.equals(accountSnapshotDO.getId(),lastSnapshot.getId())) {
                        accountSnapshotDO.setAccountAmount(lastFlow.getAccountAmount());
                        accountSnapshotDO.setAccountAvailableAmount(lastFlow.getAccountAvailableAmount());
                        accountSnapshotDO.setAccountOccupancyAmount(lastFlow.getAccountOccupancyAmount());
                    }
                });

            }

            if (CollectionUtil.isNotEmpty(accountSnapshotUpdateList)){
                accountSnapshotRepo.saveAll(accountSnapshotUpdateList);

            }
        });

    }


    public String executeAccountFlowProcedure(String accountCode,LocalDateTime retryTime,List<String> repairFlowNoList) {

        if (StringUtils.isBlank(accountCode)){
            log.info("账户编码为空无法重算");
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户编码为空无法重算!");
        }
        if (Objects.isNull(retryTime)){
            log.info("重算时间为空无法重算");
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重算时间为空无法重算!");
        }
        //LocalDateTime retryTime2 = LocalDateTime.of(retryTime.toLocalDate(),LocalTime.of(retryTime.getHour(),retryTime.getMinute(),retryTime.getSecond()));

        //log.info("重算时间: "+retryTime2);

        //accountFlowProcedureRepository.account_flow_repair(accountCode, retryTime);
        //StoredProcedureQuery storedProcedureQuery = this.entityManager.createNamedStoredProcedureQuery("account_flow_repair");
        //storedProcedureQuery.setParameter("account_code_i", accountCode);
        //storedProcedureQuery.setParameter("audit_date_i", retryTime);
        StoredProcedureQuery storedProcedureQuery = entityManager.createStoredProcedureQuery("account_flow_repair");
        storedProcedureQuery.registerStoredProcedureParameter("account_code_i", String.class, ParameterMode.IN);
        storedProcedureQuery.registerStoredProcedureParameter("audit_date_i", LocalDateTime.class, ParameterMode.IN);
        storedProcedureQuery.setParameter("account_code_i", accountCode);
        storedProcedureQuery.setParameter("audit_date_i", retryTime);
        //storedProcedureQuery.registerStoredProcedureParameter("cur_search_result", Void.class, ParameterMode.REF_CURSOR);
        storedProcedureQuery.execute();

        /*Object rowCnt = storedProcedureQuery.getOutputParameterValue("row_cnt");
        Object accountCodeI = storedProcedureQuery.getOutputParameterValue("account_code_i");
        Object createTimeI = storedProcedureQuery.getOutputParameterValue("audit_date_i");
        Object notfoundmsg = storedProcedureQuery.getOutputParameterValue("notfoundmsg");
        Object cnt = storedProcedureQuery.getOutputParameterValue("cnt");*/

        List  resultList = storedProcedureQuery.getResultList();
        log.info("处理账户流水存储过程的结果:"+resultList.toString());
        if (CollectionUtil.isEmpty(resultList)){
            return null;
        }
        /*List<Object[]> result = query.getResultList();
        for (Object[] row : resultList) {
            // 处理结果集的内容
            log.info("存储过程的结果:"+row.toString());
        }*/

        return accountCode;
    }

    private String callableStatementProcedure(String accountCode,LocalDateTime retryTime,List<String> repairFlowNoList) {
        String successAccountCode = null;
        if (StringUtils.isBlank(accountCode)){
            log.info("账户编码为空无法重算");
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户编码为空无法重算!");
        }
        if (Objects.isNull(retryTime)){
            log.info("重算时间为空无法重算");
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "重算时间为空无法重算!");
        }

        log.info("url: "+dataUrl+" user: "+userName+" password: "+pwd);
        Connection conn = null;
        CallableStatement cstmt = null;
        ResultSet rs = null;
        try {
            conn = DriverManager.getConnection(dataUrl, userName, pwd);
            // 存储过程名称及参数个数
            String procedureCall = "{CALL account_flow_repair(?,?)}";
            cstmt = conn.prepareCall(procedureCall);

            // 设置存储过程的参数
            cstmt.setString(1, accountCode);

            //LocalDateTime localDateTime = LocalDateTime.of(LocalDate.of(2024,01,25),LocalTime.of(00,00,00));
            //Date date = (Date) java.util.Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
            Date date = new java.sql.Date(java.util.Date.from(retryTime.atZone(ZoneId.systemDefault()).toInstant()).getTime());
            cstmt.setDate(2, date);

            // 执行存储过程
            cstmt.execute();

            // 处理存储过程的结果
            rs = cstmt.getResultSet();
            while (rs.next()) {
                // 读取结果集中的数据
                // String result = rs.getString("column_name");
                log.info("结果集中的数据:"+rs);
            }
            successAccountCode = accountCode;

            rs.close();
            cstmt.close();
            conn.close();
        } catch (SQLException e) {
            repairFlowNoList.forEach(flowNo -> {
                removeFlowNoCache(flowNo);
            });
            log.error("账户("+accountCode+")调用存储过程重算失败：" + e.getMessage());
            e.printStackTrace();
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户("+accountCode+")调用存储过程重算失败!");

        } finally {
            try {
                if (rs != null) {
                    rs.close();
                }
                if (cstmt != null) {
                    cstmt.close();
                }
                if (conn != null) {
                    conn.close();
                }
            } catch (SQLException e) {
                log.error("账户("+accountCode+")调用存储过程重算关闭失败：" + e.getMessage());
                e.printStackTrace();
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户("+accountCode+")调用存储过程重算关闭失败!");

            }
        }

        return successAccountCode;
    }

    /**
     * 查询满足自动还款条件的账户
     * 使用纯SQL查询数据
     *
     * @param sql
     * @return
     */
   /* private List<AccountDO> selectAccountData(String sql) {
        Query query = entityManager.createNativeQuery(sql, AccountDO.class);

        // 获取结果集
        List<AccountDO> resultList = query.getResultList();
        return resultList;
    }*/

    /**
     *  联合主键的情况下
     *  分组 后获取创建时间 最新的一条
     */
   /* List<user> collect = new ArrayList<>(infos.stream().collect(Collectors.toMap(
            user::getStationId,
            v -> v,
            (v1, v2) -> {
                Date dataTime = v1.getDataTime();
                Date startDate1 = v2.getDataTime();
                if (dataTime == null || startDate1 == null) {
                    return dataTime != null ? v1 : v2; //拿到不为空的 ，都为空取后一条
                }
                return dataTime.after(startDate1) || dataTime.equals(startDate1) ? v1 : v2;
            }
    )).values());*/



    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void rollbackBySourceNo(String sourceNo){
        AccountFlowQueryParam accountFlowQueryParam = new AccountFlowQueryParam();
        accountFlowQueryParam.setSourceNo(sourceNo);
        List<AccountFlowVO> accountFlowVOList = accountFlowService.selectListByQueryParam(accountFlowQueryParam);

        CreditAccountFlowParam creditAccountFlowParam = new CreditAccountFlowParam();
        creditAccountFlowParam.setSourceNo(sourceNo);
        //creditAccountFlowParam.setSourceNo("ADJ202404070006");
        List<CreditAccountFlowVO> creditAccountFlowVOList = creditAccountFlowService.selectListByQueryParam(creditAccountFlowParam);

        if (CollectionUtils.isEmpty(accountFlowVOList) && CollectionUtils.isEmpty(creditAccountFlowVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到单号("+sourceNo+")对应的流水数据!");
        }

        List<String> accountFlowNoList = CollectionUtils.isEmpty(accountFlowVOList) ? new ArrayList<>() :
                accountFlowVOList.stream().map(AccountFlowVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<String> creditAccountFlowNoList = CollectionUtils.isEmpty(creditAccountFlowVOList) ? new ArrayList<>() :
                creditAccountFlowVOList.stream().map(CreditAccountFlowVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());


        List<String> accountFlowNoCacheList = new ArrayList<>();
        accountFlowNoList.forEach(flowNo -> {
            if (StringUtils.isNotBlank(getFlowNoCache(flowNo))){
                accountFlowNoCacheList.add(flowNo);
            }else {
                saveFlowNoCache(flowNo);
            }
        });

        List<String> creditAccountFlowNoCacheList = new ArrayList<>();
        creditAccountFlowNoList.forEach(flowNo -> {
            if (StringUtils.isNotBlank(getFlowNoCache(flowNo))){
                creditAccountFlowNoCacheList.add(flowNo);
            }else {
                saveFlowNoCache(flowNo);
            }
        });

        if (CollectionUtils.isNotEmpty(accountFlowNoCacheList)) {
            String checkResult = accountFlowNoCacheList.stream().map(Objects::toString).collect(Collectors.joining("、", "账户流水号[", "], 修复重算中,请稍后再次重算!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        if (CollectionUtils.isNotEmpty(creditAccountFlowNoCacheList)) {
            String checkResult = creditAccountFlowNoCacheList.stream().map(Objects::toString).collect(Collectors.joining("、", "信用账户流水号[", "], 修复重算中,请稍后再次重算!"));
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
        }

        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRED);
        String processingFlowNo = transactionTemplate.execute(transactionStatus -> {
            try {
                if (CollectionUtil.isNotEmpty(accountFlowVOList)){
                    List<AccountFlowRepairVO> accountFlowRepairVOList = convertToRepairVO(accountFlowVOList);
                    List<String> repairFlowNo10List = accountFlowRepairVOList.stream().map(AccountFlowRepairVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());

                    accountFlowRepairVOList.forEach(flowRepairVO -> {
                        processingCalculationRollback(flowRepairVO,repairFlowNo10List);
                    });

                    repairFlowNo10List.forEach(flowNo -> {
                        removeFlowNoCache(flowNo);
                    });
                }
                if (CollectionUtil.isNotEmpty(creditAccountFlowVOList)){
                    creditAccountFlowRepairService.rollbackBySourceNo(creditAccountFlowVOList);
                }
                return null;
            } catch (Exception e) {
                log.error("来源单号("+sourceNo+")回滚流水失败!:{}", e.getMessage());
                //回滚
                transactionStatus.setRollbackOnly();
                accountFlowNoList.forEach(flowNo -> {
                    removeFlowNoCache(flowNo);
                });
                creditAccountFlowNoList.forEach(flowNo -> {
                    removeFlowNoCache(flowNo);
                });
                //抛出异常，不往下执行
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "来源单号("+sourceNo+")回滚流水失败!");
            }
        });

    }

    public List<AccountFlowRepairVO> convertToRepairVO(List<AccountFlowVO> accountFlowVOList){
        accountFlowVOList = accountFlowVOList.stream()
                .sorted(Comparator.comparing(AccountFlowVO::getAuditDate))
                .collect(Collectors.toList());
        List<AccountFlowRepairVO> accountFlowRepairVOList = accountFlowVOList.stream().map(flowVO -> {
            AccountFlowRepairVO flowRepairVO = new AccountFlowRepairVO();
            flowRepairVO.setId(-100L);
            flowRepairVO.setFlowNo(flowVO.getFlowNo());
            flowRepairVO.setActualAmount(BigDecimal.ZERO);
            flowRepairVO.setAmount(flowVO.getAmount());
            flowRepairVO.setOpenAccountEntityCode(null);
            flowRepairVO.setOpenAccountEntityName(flowVO.getAccountHolderName());
            flowRepairVO.setAccountType(flowVO.getAccountType());
            flowRepairVO.setAccountCode(flowVO.getAccountCode());
            flowRepairVO.setAccountName(flowVO.getAccountName());
            flowRepairVO.setSourceNo(flowVO.getSourceNo());
            flowRepairVO.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());
            flowRepairVO.setRecalculationReason(UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode());
            flowRepairVO.setRecalculationDate(flowVO.getAuditDate());
            flowRepairVO.setRecalculationUser(null);
            flowRepairVO.setRecalculationUserId(null);
            return flowRepairVO;
        }).collect(Collectors.toList());

        return accountFlowRepairVOList;
    }


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void processingCalculationRollback(AccountFlowRepairVO flowRepairVO,List<String> repairFlowNoList){

        List<String> accCodeList = new ArrayList<>();

        //accountFlowRepairRepoProc.updateRemarkById("测试回滚",749290284927490831L);
        /*repairFlowVOList.forEach(flowRepairVO -> {
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            String processingFlowNo = transactionTemplate.execute(transactionStatus -> {
                String repairFlowNo = flowRepairVO.getFlowNo();
                try {*/
                    AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
                    flowQueryParam.setFlowNo(flowRepairVO.getFlowNo());
                    log.info("修复重算账户流水查询数据,时间：{},入参：{}", LocalDateTime.now(), JSON.toJSONString(flowQueryParam));
                    List<AccountFlowVO> flowVOList = accountFlowService.selectListByQueryParam(flowQueryParam);
                    log.info("修复重算账户流水查询数据,时间：{},出参：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));

                    if (CollectionUtils.isNotEmpty(flowVOList) && flowVOList.size() == 1 ){
                        if (Objects.nonNull(flowVOList.get(0))) {
                            AccountFlowVO flowVO = flowVOList.get(0);

                            if (Objects.equals(flowVO.getAccountType(),UdcEnum.ACCOUNT_TYPE_STORE.getValueCode()) &&
                                    FinConstant.ACCOUNT_FLOW_REPAIR_TRANSACTION_TYPE.contains(flowVO.getTransactionType())){
                                repairFlow(flowRepairVO,flowVO,accCodeList);
                            }else if (Objects.equals(flowVO.getAccountType(),UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode()) &&
                                    FinConstant.ACCOUNT_FLOW_REPAIR_TRANSACTION_TYPE.contains(flowVO.getTransactionType())){
                                repairFlow(flowRepairVO,flowVO,accCodeList);

                            }else {
                                log.info("未修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVO));
                            }

                            if (accCodeList.contains(flowVO.getAccountCode())){
                                //更新完成时同步更新账户列表的金额，用最新一条流水记录的更新同账户编码的账户金额、账户占用金额、账户可用金额
                                updateAccountAmt(Collections.singletonList(flowVO.getAccountCode()),repairFlowNoList);
                               /* //更新账户快照表的对应数据
                                //原先流水修复程序的重算日期 取流水的审核日期
                                if (Objects.isNull(flowRepairVO.getRecalculationDate())){
                                    flowRepairVO.setRecalculationDate(flowVO.getAuditDate());
                                }
                                updateAccountSnapshot(Collections.singletonList(flowRepairVO),repairFlowNoList);*/

                            }

                        } else {
                            log.info("未查询到修复账户流水数据,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowRepairVO));
                        }
                    }else {
                        log.info("修复账户流水数据为空或者有多条,时间：{},参数：{}", LocalDateTime.now(), JSON.toJSONString(flowVOList));
                    }

                    //释放校验缓存
                    removeFlowNoCache(flowRepairVO.getFlowNo());
                    /*return flowRepairVO.getFlowNo();
                } catch (Exception e) {
                    log.error("流水号("+repairFlowNo+")账户流水修复重算失败!:{}", e.getMessage());
                    //回滚
                    ///transactionStatus.setRollbackOnly();
                    repairFlowNoList.forEach(flowNo -> {
                        removeFlowNoCache(flowNo);
                    });
                    //抛出异常，不往下执行
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "流水号("+repairFlowNo+")账户流水修复重算失败!");
                }
            });



        });*/

        /*repairFlowNoList.forEach(flowNo -> {
            removeFlowNoCache(flowNo);
        });*/
    }


}
