package com.elitesland.fin.application.service.accountingengine;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.accountingengine.*;
import com.elitesland.fin.application.facade.param.accountingengine.*;
import com.elitesland.fin.application.facade.vo.accountingengine.*;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.accountingengine.*;
import com.elitesland.fin.repo.accountingengine.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.*;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author gyj
 * @date 2023/10/10
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class FinAccountEngineServiceImpl implements FinAccountEngineService {

    private final FinAccountEngineRepo finAccountEngineRepo;

    private final FinAccountEngineRepoProc finAccountEngineRepoProc;

    private final FinAccountEngineDetailsRepo finAccountEngineDetailsRepo;

    private final FinAccEngDetConditionRepo finAccEngDetConditionRepo;

    private final FinAccEngDetDataRepo finAccEngDetDataRepo;

    private final FinAccEngDetDataLineRepo finAccEngDetDataLineRepo;

    private final FinEventTableRepo finEventTableRepo;

    private final FinEventTableLineRepo finEventTableLineRepo;

    private final FinEventTableConditionRepo finEventTableConditionRepo;

    private final FinJournalLogService finJournalLogService;

    private final FinJournalLogRepo finJournalLogRepo;

    private final FinSetOfBookOuRepo finSetOfBookOuRepo;

    private final FinSetOfBookRepo finSetOfBookRepo;

    private final FinSetOfBookRepoProc finSetOfBookRepoProc;

    private final FinSetOfBookLineRepo finSetOfBookLineRepo;

    private final FinFastCodeRepo finFastCodeRepo;

    private final FinFastCodeLineRepo finFastCodeLineRepo;

    private final FinFastCodeLineRepoProc finFastCodeLineRepoProc;

    private final FinFastCodeRepoProc finFastCodeRepoProc;

    private final FinFlexibleRepo finFlexibleRepo;

    private final FinFlexibleRepoProc finFlexibleRepoProc;

    private final FinFlexibleValueRepo finFlexibleValueRepo;

    private final FinFlexibleValueRepoProc finFlexibleValueRepoProc;

    private final FinSobAccountPeriodRepo finSobAccountPeriodRepo;

    private final FinJournalRepo finJournalRepo;

    private final UdcProvider udcProvider;

    private final TenantClientProvider tenantClientProvider;

    private final FinAccountEngineJdbcService finAccountEngineJdbcService;
    private final FinFlexibleValueService finFlexibleValueService;
    @SysCodeProc
    @Override
    public PagingVO<FinAccountEngineVO> page(FinAccountEngineParam finAccountEngineParam) {
        return FinAccountEngineConvert.INSTANCE.DTOToVO(finAccountEngineRepoProc.page(finAccountEngineParam));
    }


    @SysCodeProc
    @Override
    public FinAccountEngineVO queryFinAccountEngineDetails(FinAccountEngineDetailsParam param) {

        Assert.notNull(param.getId(), "主表id不能为空");
        // 查询表头
        Optional<FinAccountEngineDO> optional = finAccountEngineRepo.findById(param.getId());
        Assert.isTrue(optional.isPresent(), "查询的会计引擎不存在");
        FinAccountEngineVO engineVO = FinAccountEngineConvert.INSTANCE.DOToVo(optional.get());

        // 查询会计引擎详情
        List<FinAccountEngineDetailsDO> detailsDOList = finAccountEngineDetailsRepo.findAllByMasIdIn(List.of(param.getId()));
        List<FinAccountEngineDetailsVO> detailsVOList = FinAccountEngineDetailsConvert.INSTANCE.DOsToVOs(detailsDOList);
        engineVO.setFinAccountEngineDetails(detailsVOList);

        return engineVO;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void enableOrDisable(FinAccountEngineParam finAccountEngineParam) {

        checkEnableOrDisableParam(finAccountEngineParam);

        List<FinAccountEngineDO> finAccountEngineDOList = finAccountEngineRepo.findAllById(finAccountEngineParam.getIds());
        finAccountEngineDOList.stream().forEach(item -> Assert.isFalse(finAccountEngineParam.getStatus().equals(item.getStatus()), "数据已经启用/禁用"));

        finAccountEngineDOList.stream().forEach(item -> item.setStatus(finAccountEngineParam.getStatus()));
    }

    private void checkEnableOrDisableParam(FinAccountEngineParam finAccountEngineParam) {
        Assert.notEmpty(finAccountEngineParam.getIds(), "id必填");
        Assert.notEmpty(finAccountEngineParam.getStatus(), "状态必填");
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveOrUpdateFinAccEngDetCondition(FinAccountEngineDetailsParam param) {

        checkSaveOrUpdateFinAccEngDetConditionParam(param);

        FinAccountEngineDetailsDO finAccountEngineDetailsDO = finAccountEngineDetailsRepo.save(FinAccountEngineDetailsConvert.INSTANCE.paramToDO(param));
        finAccEngDetConditionRepo.deleteAllByMasId(finAccountEngineDetailsDO.getId());

        List<FinAccEngDetConditionParam> finAccEngDetConditionParamList = param.getFinAccEngDetConditionDetailList();
        finAccEngDetConditionParamList.stream().forEach(item -> item.setMasId(finAccountEngineDetailsDO.getId()));
        finAccEngDetConditionRepo.saveAll(FinAccEngDetConditionConvert.INSTANCE.paramToDO(finAccEngDetConditionParamList));

        return finAccountEngineDetailsDO.getId();
    }

    @Override
    public Long saveOrUpdateFinAccEngDetDataLine(FinAccEngDetDataParam finAccEngDetDataParam) {

        finAccEngDetDataLineRepo.deleteAllByMasId(finAccEngDetDataParam.getId());

        List<FinAccEngDetDataLineParam> finAccEngDetDataLineList = finAccEngDetDataParam.getFinAccEngDetDataLineList();
        finAccEngDetDataLineList.stream().forEach(item -> item.setMasId(finAccEngDetDataParam.getId()));
        finAccEngDetDataLineRepo.saveAll(FinAccEngDetDataLineConvert.INSTANCE.param2DO(finAccEngDetDataLineList));

        return finAccEngDetDataParam.getId();
    }

    @Override
    public Long saveOrUpdateFinAccEngDetData(FinAccountEngineDetailsParam finAccountEngineDetailsParam) {
        checkSaveOrUpdateFinAccEngDetDataParam(finAccountEngineDetailsParam);


        List<FinAccEngDetDataDO> finAccEngDetDataDOList = finAccEngDetDataRepo.findAllByMasId(finAccountEngineDetailsParam.getId());
        List<FinAccEngDetDataParam> finAccEngDetDataParamList = finAccountEngineDetailsParam.getFinAccEngDetDataDetailList();

        List<Long> ids = finAccEngDetDataDOList.stream().map(FinAccEngDetDataDO::getId).collect(Collectors.toList());
        List<Long> deleteIds = finAccEngDetDataParamList.stream()
                .filter(item -> item.getId() != null && !ids.contains(item.getId())).map(FinAccEngDetDataParam::getId)
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(deleteIds)) {
            finAccEngDetDataRepo.deleteByIdIn(deleteIds);
        }

        finAccEngDetDataParamList.stream().forEach(item -> item.setMasId(finAccountEngineDetailsParam.getId()));
        finAccEngDetDataRepo.saveAll(FinAccEngDetDataConvert.INSTANCE.param2DO(finAccEngDetDataParamList));

        return finAccountEngineDetailsParam.getId();
    }


    private void checkSaveOrUpdateFinAccEngDetDataParam(FinAccountEngineDetailsParam finAccountEngineDetailsParam) {
        Assert.notNull(finAccountEngineDetailsParam.getId(), "主表id不能为空");

        List<FinAccEngDetDataParam> finAccEngDetDataDetailList = finAccountEngineDetailsParam.getFinAccEngDetDataDetailList();
        Assert.notEmpty(finAccEngDetDataDetailList, "数据行详情不能为空");
        finAccEngDetDataDetailList.stream().forEach(item -> {

        });

    }

    private void checkSaveOrUpdateFinAccEngDetConditionParam(FinAccountEngineDetailsParam param) {

        Assert.notNull(param.getId(), "主表id必填");

        List<FinAccEngDetConditionParam> finAccEngDetConditionParamList = param.getFinAccEngDetConditionDetailList();
        Assert.notEmpty(finAccEngDetConditionParamList, "会计引擎条件必填");

        Set<Integer> serialNums = new HashSet<>();
        finAccEngDetConditionParamList.stream().forEach(item -> {
            Assert.notNull(item.getSerialNum(), "序号必填");
            Assert.notEmpty(item.getColumnName(), "字段名称必填");
            Assert.notEmpty(item.getConditionType(), "条件必填");

            Assert.isFalse(serialNums.contains(item.getSerialNum()), "序号不能重复");
            serialNums.add(item.getSerialNum());
        });
    }

    private void checkFinAccountEngineDetailsParam(FinAccountEngineDetailsParam param) {
        Assert.notNull(param.getSerialNum(), "序号必填");
        Assert.notEmpty(param.getAccountEngine(), "会计引擎名称必填");
        Assert.notNull(param.getActiveFlag(), "是否启用必填");
    }

    @Override
    @SysCodeProc
    public FinAccountEngineVO queryFinAccountEngine(FinAccountEngineParam finAccountEngineParam) {
        FinAccountEngineDO finAccountEngineDO = finAccountEngineRepoProc.findById(finAccountEngineParam.getId());
        FinAccountEngineVO finAccountEngineVO = FinAccountEngineConvert.INSTANCE.DOToVo(finAccountEngineDO);

        List<FinAccountEngineDetailsDO> finAccountEngineDetailsDOList = finAccountEngineDetailsRepo.findAllByMasId(finAccountEngineVO.getId());
        List<FinAccountEngineDetailsVO> finAccountEngineDetailsVOList = FinAccountEngineDetailsConvert.INSTANCE.DOsToVOs(finAccountEngineDetailsDOList);

        if(CollectionUtils.isNotEmpty(finAccountEngineDetailsVOList)){
            finAccountEngineDetailsVOList.stream().forEach(v->{if (v.getSerialNum()==null){
                v.setSerialNum(0);
            }});
            Collections.sort(finAccountEngineDetailsVOList, new Comparator<FinAccountEngineDetailsVO>() {
                @Override
                public int compare(FinAccountEngineDetailsVO o1, FinAccountEngineDetailsVO o2) {
                    return o2.getSerialNum().compareTo(o1.getSerialNum());
                }
            });
        }
        finAccountEngineVO.setFinAccountEngineDetails(finAccountEngineDetailsVOList);

        List<Long> finAccountEngineDetailsIds = finAccountEngineDetailsVOList.stream().map(FinAccountEngineDetailsVO::getId).collect(Collectors.toList());
        List<FinAccEngDetConditionDO> finAccEngDetConditionDOList = finAccEngDetConditionRepo.findAllByMasIdIn(finAccountEngineDetailsIds);
        List<FinAccEngDetConditionVO> finAccEngDetConditionVOList = FinAccEngDetConditionConvert.INSTANCE.DOsToVos(finAccEngDetConditionDOList);

        List<FinAccEngDetDataDO> finAccEngDetDataDOList = finAccEngDetDataRepo.findAllByMasIdIn(finAccountEngineDetailsIds);
        List<FinAccEngDetDataVO> finAccEngDetDataVOList = FinAccEngDetDataConvert.INSTANCE.DOsToVOs(finAccEngDetDataDOList);


        finAccountEngineDetailsVOList.stream().forEach(item -> {
            item.setFinAccEngDetConditionDetailList(finAccEngDetConditionVOList.stream().filter(e -> item.getId().equals(e.getMasId())).collect(Collectors.toList()));
            item.setFinAccEngDetDataDetailList(finAccEngDetDataVOList.stream().filter(e -> item.getId().equals(e.getMasId())).collect(Collectors.toList()));

        });
        List<Long> finAccEngDetDataVOIds = finAccountEngineDetailsVOList.stream().flatMap(item -> item.getFinAccEngDetDataDetailList().stream().map(FinAccEngDetDataVO::getId)).collect(Collectors.toList());
        List<FinAccEngDetDataLineDO> finAccEngDetDataLineDOList = finAccEngDetDataLineRepo.findAllByMasIdIn(finAccEngDetDataVOIds);
        List<FinAccEngDetDataLineVO> finAccEngDetDataLineVOList = FinAccEngDetDataLineConvert.INSTANCE.DOsToVOS(finAccEngDetDataLineDOList);
        //值名称翻译
        List<String> sobCodes = finAccEngDetDataDOList.stream().map(FinAccEngDetDataDO::getSobCode).collect(Collectors.toList());
        List<FinSetOfBookDO> finSetOfBookDOList = finSetOfBookRepo.findAllBySobCodeInAndStatus(sobCodes, UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<Long> finSetOfBookDOIds = finSetOfBookDOList.stream().map(FinSetOfBookDO::getId).collect(Collectors.toList());
        //帐套明细
        Map<String, Map<String, FinSetOfBookLineDO>> setOfBookLineMap = new HashMap<>();
        List<FinSetOfBookLineDO> finSetOfBookLineDOList = finSetOfBookLineRepo.findAllByMasIdIn(finSetOfBookDOIds);
        if(CollectionUtils.isNotEmpty(finSetOfBookLineDOList)){
            Map<Long, List<FinSetOfBookLineDO>> masFinSetOfBookLineMap = finSetOfBookLineDOList.stream().collect(Collectors.groupingBy(v -> v.getMasId(), Collectors.toList()));
            setOfBookLineMap = finSetOfBookDOList.stream().collect(Collectors.toMap(v -> v.getSobCode(), v -> {
                return masFinSetOfBookLineMap.get(v.getId()) != null ? masFinSetOfBookLineMap.get(v.getId()).stream().collect(Collectors.toMap(w ->w.getColumnName(), w -> w, (o, n) -> n)) : new HashMap<>();
            }));
        }
        final Map<String, Map<String, FinSetOfBookLineDO>> finalSetOfBookLineMap = setOfBookLineMap;
        FinFlexibleValueParam finFlexibleValueParam = new FinFlexibleValueParam();
        finFlexibleValueParam.setSobCodes(sobCodes);
        Map<String, Map<String, FinFlexibleValueVO>> flexCodeValueMap = finFlexibleValueService.listBySetOfBook(finFlexibleValueParam);
        finAccountEngineDetailsVOList.stream().forEach(item -> item.getFinAccEngDetDataDetailList().stream().forEach(e -> {
            e.setFinAccEngDetDataLineList(finAccEngDetDataLineVOList.stream().filter(v -> e.getId().equals(v.getMasId())).collect(Collectors.toList()));
            //设置值集名称begin
            String sobCode = e.getSobCode();
            for (FinAccEngDetDataLineVO finAccEngDetDataLineVO : e.getFinAccEngDetDataLineList()) {
                Map<String, FinSetOfBookLineDO> finSetOfBookLineDOMap = finalSetOfBookLineMap.get(sobCode);
                if(finSetOfBookLineDOMap==null){
                    continue;
                }
                String columnName = finAccEngDetDataLineVO.getColumnName();
                FinSetOfBookLineDO finSetOfBookLineDO = finSetOfBookLineDOMap.get(columnName);
                if(finSetOfBookLineDO==null){
                  continue;
                }
                if(!Boolean.TRUE.equals(finSetOfBookLineDO.getFlexibleFlag())){
                    continue;
                }
                String flexibleCode = finSetOfBookLineDO.getFlexibleCode();
                Map<String, FinFlexibleValueVO> flexibleValueDOMap = flexCodeValueMap.get(flexibleCode);
                FinFlexibleValueVO finFlexibleValueVO = flexibleValueDOMap.get(finAccEngDetDataLineVO.getFlexibleCode());
                if (finFlexibleValueVO == null) {
                    continue;
                }
                finAccEngDetDataLineVO.setFlexibleValueName(finFlexibleValueVO.getFlexibleValueName());
            }
            //设置值集名称end
        }));
        //翻译
        Map<String, String> eventTableConditionUdc = udcProvider.getValueMapByUdcCode(UdcEnum.EVENT_TABLE_CONDITION_EQUAL.getModel(), UdcEnum.EVENT_TABLE_CONDITION_EQUAL.getCode());
        Map<String, String> sourceTypeUdc = udcProvider.getValueMapByUdcCode(UdcEnum.SOURCE_TYPE_ACCOUNT_PERIOD.getModel(), UdcEnum.SOURCE_TYPE_ACCOUNT_PERIOD.getCode());
        Map<String, String>  fastCodeOutputUdc = udcProvider.getValueMapByUdcCode(UdcEnum.ACC_ENG_FAST_CODE_OUTPUT_ONE.getModel(), UdcEnum.ACC_ENG_FAST_CODE_OUTPUT_ONE.getCode());

        List<FinAccountEngineDetailsVO> finAccountEngineDetails = finAccountEngineVO.getFinAccountEngineDetails();
        if (CollectionUtils.isNotEmpty(finAccountEngineDetails)) {
            finAccountEngineDetails.stream().forEach(item -> {
                List<FinAccEngDetConditionVO> finAccEngDetConditionDetailList = item.getFinAccEngDetConditionDetailList();
                if (CollectionUtils.isNotEmpty(finAccEngDetConditionDetailList)) {
                    finAccEngDetConditionDetailList.stream().forEach(e -> e.setConditionTypeName(eventTableConditionUdc.get(e.getConditionType())));
                }

                List<FinAccEngDetDataVO> finAccEngDetDataDetailList = item.getFinAccEngDetDataDetailList();
                if (CollectionUtils.isNotEmpty(finAccEngDetDataDetailList)) {
                    finAccEngDetDataDetailList.stream().forEach(e -> {
                        List<FinAccEngDetDataLineVO> finAccEngDetDataLineList = e.getFinAccEngDetDataLineList();
                        if (CollectionUtils.isNotEmpty(finAccEngDetDataLineList)) {
                            finAccEngDetDataLineList.stream().forEach(v -> {
                                v.setSourceTypeName(sourceTypeUdc.get(v.getSourceType()));
                                v.setOutputColumnName(fastCodeOutputUdc.get(v.getOutputColumn()));
                            });
                        }
                    });
                }
            });
        }
        return finAccountEngineVO;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveOrUpdate(FinAccountEngineParam finAccountEngineParam) {
        checkSaveOrUpdateParam(finAccountEngineParam);
        boolean add = finAccountEngineParam.getId() == null ? true : false;
        checkEngineUnique(finAccountEngineParam,add);
        FinAccountEngineDO finAccountEngineDO = finAccountEngineRepo.save(FinAccountEngineConvert.INSTANCE.paramToDO(finAccountEngineParam));
        if (!add) {
            //查询已有的会计明细
            List<FinAccountEngineDetailsDO> finAccountEngineDetailsDOList = finAccountEngineDetailsRepo.findAllByMasId(finAccountEngineParam.getId());
            List<Long> finAccountEngineDetailIds = finAccountEngineDetailsDOList.stream().map(FinAccountEngineDetailsDO::getId).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(finAccountEngineDetailIds)) {
                //删除会计明细条件
                finAccEngDetConditionRepo.deleteByMasIdIn(finAccountEngineDetailIds);
                //查询数据行
                List<FinAccEngDetDataDO> finAccEngDetDataIds = finAccEngDetDataRepo.findAllByMasIdIn(finAccountEngineDetailIds);
                List<Long> collect = finAccEngDetDataIds.stream().map(FinAccEngDetDataDO::getId).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(collect)) {
                    //删除数据行字段
                    finAccEngDetDataLineRepo.deleteByMasIdIn(collect);
                }
                //删除数据行
                finAccEngDetDataRepo.deleteByMasIdIn(finAccountEngineDetailIds);
                //删除会计引擎明细
                finAccountEngineDetailsRepo.deleteByIdIn(finAccountEngineDetailIds);
            }

        }

        //会计引擎详情
        List<FinAccountEngineDetailsParam> finAccountEngineDetailsParamList = finAccountEngineParam.getFinAccountEngineDetails();
        finAccountEngineDetailsParamList.stream().forEach(item -> item.setMasId(finAccountEngineDO.getId()));
        List<FinAccountEngineDetailsDO> finAccountEngineDetailsDOList = finAccountEngineDetailsRepo.saveAll(FinAccountEngineDetailsConvert.INSTANCE.paramToDO(finAccountEngineDetailsParamList));

        finAccountEngineDetailsParamList.stream().forEach(item -> {
            FinAccountEngineDetailsDO result = finAccountEngineDetailsDOList.stream()
                    .filter(e -> e.getSerialNum().equals(item.getSerialNum()))
                    .findFirst()
                    .orElse(null);
            List<FinAccEngDetConditionParam> finAccEngDetConditionDetailList = item.getFinAccEngDetConditionDetailList();
            finAccEngDetConditionDetailList.stream().forEach(e -> e.setMasId(result.getId()));

            List<FinAccEngDetDataParam> finAccEngDetDataDetailList = item.getFinAccEngDetDataDetailList();
            finAccEngDetDataDetailList.stream().forEach(e -> e.setMasId(result.getId()));

        });

        //会计引擎详情条件
        List<FinAccEngDetConditionParam> finAccEngDetConditionParamList = finAccountEngineDetailsParamList.stream()
                .flatMap(item -> item.getFinAccEngDetConditionDetailList().stream())
                .collect(Collectors.toList());

        finAccEngDetConditionRepo.saveAll(FinAccEngDetConditionConvert.INSTANCE.paramToDO(finAccEngDetConditionParamList));

        //数据行详情
        List<FinAccEngDetDataParam> finAccEngDetDataParamList = finAccountEngineDetailsParamList.stream()
                .flatMap(item -> item.getFinAccEngDetDataDetailList().stream())
                .collect(Collectors.toList());

        List<FinAccEngDetDataDO> finAccEngDetDataDOS = finAccEngDetDataRepo.saveAll(FinAccEngDetDataConvert.INSTANCE.param2DO(finAccEngDetDataParamList));

        //数据映射
        finAccountEngineDetailsParamList.stream().forEach(item -> {
            FinAccountEngineDetailsDO result = finAccountEngineDetailsDOList.stream()
                    .filter(e -> e.getSerialNum().equals(item.getSerialNum()))
                    .findFirst()
                    .orElse(null);

            List<FinAccEngDetDataParam> finAccEngDetDataDetailList = item.getFinAccEngDetDataDetailList();
            finAccEngDetDataDetailList.stream().forEach(item_detData -> {
                FinAccEngDetDataDO result_detData = finAccEngDetDataDOS.stream()
                        .filter(e ->e.getMasId().equals(result.getId())&&e.getSerialNum().equals(item_detData.getSerialNum()))
                        .findFirst()
                        .orElse(null);
                List<FinAccEngDetDataLineParam> finAccEngDetDataLineList = item_detData.getFinAccEngDetDataLineList();
                finAccEngDetDataLineList.stream().forEach(e -> e.setMasId(result_detData.getId()));
            });
        });
        List<FinAccEngDetDataLineParam> finAccEngDetDataLineList = finAccEngDetDataParamList.stream()
                .flatMap(item -> item.getFinAccEngDetDataLineList().stream())
                .collect(Collectors.toList());
        finAccEngDetDataLineRepo.saveAll(FinAccEngDetDataLineConvert.INSTANCE.param2DO(finAccEngDetDataLineList));

        return finAccountEngineDO.getId();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveOrUpdateFinAccountEngineDetail(FinAccountEngineParam finAccountEngineParam) {

        checkSaveOrUpdateFinAccountEngineDetailParam(finAccountEngineParam);

        checkIdempotent(finAccountEngineParam);

        List<FinAccountEngineDetailsDO> finAccountEngineDetailsDOList = finAccountEngineDetailsRepo.findAllByMasId(finAccountEngineParam.getId());
        List<FinAccountEngineDetailsParam> finAccountEngineDetailsParamList = finAccountEngineParam.getFinAccountEngineDetails();

        List<Long> ids = finAccountEngineDetailsDOList.stream().map(FinAccountEngineDetailsDO::getId).collect(Collectors.toList());
        List<Long> deleteIds = finAccountEngineDetailsParamList.stream()
                .filter(item -> item.getId() != null && !ids.contains(item.getId())).map(FinAccountEngineDetailsParam::getId)
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(deleteIds)) {
            finAccountEngineDetailsRepo.deleteByIdIn(deleteIds);
        }

        finAccountEngineDetailsParamList.stream().forEach(item -> item.setMasId(finAccountEngineParam.getId()));
        finAccountEngineDetailsRepo.saveAll(FinAccountEngineDetailsConvert.INSTANCE.paramToDO(finAccountEngineDetailsParamList));

        return finAccountEngineParam.getId();
    }

    private void checkSaveOrUpdateFinAccountEngineDetailParam(FinAccountEngineParam finAccountEngineParam) {
        Assert.notNull(finAccountEngineParam.getId(), "主表id不能为空");

        List<FinAccountEngineDetailsParam> finAccountEngineDetailsParamList = finAccountEngineParam.getFinAccountEngineDetails();
        Assert.notEmpty(finAccountEngineDetailsParamList, "会计引擎明细必填");

        Set<Integer> serialNums = new HashSet<>();
        finAccountEngineDetailsParamList.stream().forEach(item -> {

            checkFinAccountEngineDetailsParam(item);

            Assert.isFalse(serialNums.contains(item.getSerialNum()), "序号不能重复");
            serialNums.add(item.getSerialNum());
        });
    }

    private void checkSaveOrUpdateParam(FinAccountEngineParam finAccountEngineParam) {
        Assert.notNull(finAccountEngineParam.getEventTableId(), "事件表单ID必填");
        //Assert.notEmpty(finAccountEngineParam.getEventTable(), "事件表单名称必填");
        Assert.notEmpty(finAccountEngineParam.getColumnCompany(), "公司字段必填");
        Assert.notEmpty(finAccountEngineParam.getJournalHost(), "主机必填");
        Assert.notEmpty(finAccountEngineParam.getJournalPort(), "端口必填");
        Assert.notEmpty(finAccountEngineParam.getJournalUser(), "用户名必填");
        Assert.notEmpty(finAccountEngineParam.getJournalPassword(), "密码必填");
        Assert.notEmpty(finAccountEngineParam.getJournalDatabase(), "数据库名称必填");
        Assert.notEmpty(finAccountEngineParam.getJournalTable(), "主表必填");
        Assert.notEmpty(finAccountEngineParam.getFinAccountEngineDetails(), "会计引擎行明细为空");
        //检查会计明细是都有数据行
        finAccountEngineParam.getFinAccountEngineDetails().forEach(v->{
            Assert.notEmpty(v.getFinAccEngDetDataDetailList(), "会计引擎数据行为空");
            v.getFinAccEngDetDataDetailList().forEach(v1->{
                Assert.notEmpty(v1.getFinAccEngDetDataLineList(), "会计引擎数据行数据映射为空");
            });
        });
    }

    private void checkIdempotent(FinAccountEngineParam finAccountEngineParam) {

        FinAccountEngineDO finAccountEngineDO = finAccountEngineRepoProc.findByEventTable(finAccountEngineParam.getEventTable());

        if (finAccountEngineDO != null) {

            Assert.equals(finAccountEngineParam.getId(), finAccountEngineDO.getId(), "事件表单名称已经存在");
        }
    }
    @SysCodeProc
    @Override
    public List<FinAccEngDetDataLineVO> queryFinAccDetDataLine(FinAccEngDetDataParam param) {
        Assert.notNull(param.getId(), "主表id必填");
        List<FinAccEngDetDataLineDO> dataLineDOList = finAccEngDetDataLineRepo.findAllByMasIdIn(List.of(param.getMasId()));
        List<FinAccEngDetDataLineVO> dataLineVOList = FinAccEngDetDataLineConvert.INSTANCE.DOsToVOS(dataLineDOList);
        return dataLineVOList;
    }


    @Override
    public FinAccountEngineDetailsVO queryFinAccEngDetCondition(FinAccountEngineDetailsParam param) {

        Assert.notNull(param.getId(), "masId is not null");

        // 查询表头
        Optional<FinAccountEngineDetailsDO> optional = finAccountEngineDetailsRepo.findById(param.getId());
        Assert.isTrue(optional.isPresent(), "详情不存在");

        FinAccountEngineDetailsVO detailsVO = FinAccountEngineDetailsConvert.INSTANCE.DOToVO(optional.get());

        // 数据行详情查询
        List<FinAccEngDetDataDO> detDataDOList = finAccEngDetDataRepo.findAllByMasIdIn(List.of(param.getId()));
        List<FinAccEngDetDataVO> detDataVOList = FinAccEngDetDataConvert.INSTANCE.DOsToVOs(detDataDOList);
        detailsVO.setFinAccEngDetDataDetailList(detDataVOList);

        // 会计引擎详情条件查询
        List<FinAccEngDetConditionDO> conditionDOList = finAccEngDetConditionRepo.findAllByMasIdIn(List.of(param.getId()));
        List<FinAccEngDetConditionVO> conditionVOList = FinAccEngDetConditionConvert.INSTANCE.DOsToVos(conditionDOList);
        detailsVO.setFinAccEngDetConditionDetailList(conditionVOList);

        return detailsVO;
    }

    @Override
    public List<FinAccEngDetDataVO> queryFinAccEngDetData(FinAccountEngineDetailsParam finAccountEngineDetailsParam) {
        Assert.notNull(finAccountEngineDetailsParam.getId(), "主表id必填");
        List<FinAccEngDetDataDO> finAccEngDetDataDOList = finAccEngDetDataRepo.findAllByMasId(finAccountEngineDetailsParam.getId());
        return FinAccEngDetDataConvert.INSTANCE.DOsToVOs(finAccEngDetDataDOList);
    }
    @Override
    public List<FinAccEngDetDataLineVO> loadTemplate(FinAccEngDetDataParam finAccEngDetDataParam) {
        checkLoadTemplateParam(finAccEngDetDataParam);
        return loadTemplateFromDB(finAccEngDetDataParam);
    }
    private List<FinAccEngDetDataLineVO> loadTemplateFromDB(FinAccEngDetDataParam finAccEngDetDataParam) {

        //查询分录表字段
        ResultSet resultSet = null;
        Connection connection = null;
        PreparedStatement preparedStatement = null;

        List<FinAccEngDetDataLineVO> finAccEngDetDataLineVOList = new ArrayList<>();

        try {
            Class.forName(FinConstant.DRIVEN_NAME);

            connection = DriverManager.getConnection(String.format(FinConstant.CONNECTION_NAME,
                            finAccEngDetDataParam.getJournalHost(),
                            finAccEngDetDataParam.getJournalPort(),
                            finAccEngDetDataParam.getJournalDatabase()),
                    finAccEngDetDataParam.getJournalUser(),
                    finAccEngDetDataParam.getJournalPassword());

            String sql = "select column_name, data_type, column_key,column_comment,character_maximum_length " +
                    "from information_schema.columns " +
                    "where table_name = '" + finAccEngDetDataParam.getJournalTable() + "' and TABLE_SCHEMA = '" + finAccEngDetDataParam.getJournalDatabase() + "'";

            preparedStatement = connection.prepareStatement(sql);

            resultSet = preparedStatement.executeQuery();

            // 处理查询结果集
            while (resultSet.next()) {
                FinAccEngDetDataLineVO finAccEngDetDataLineVO = new FinAccEngDetDataLineVO();
                int columnCount = resultSet.getMetaData().getColumnCount();
                //屏蔽fin_journal表 segment1~segment18字段
                if (resultSet.getString("column_name").startsWith("segment")) {
                    continue;
                }
                for (int i = 1; i <= columnCount; i++) {
                    if ("column_name".equals(resultSet.getMetaData().getColumnName(i)) ||
                            "COLUMN_NAME".equals(resultSet.getMetaData().getColumnName(i))) {
                        Object columnName = resultSet.getObject(resultSet.getMetaData().getColumnName(i));
                        finAccEngDetDataLineVO.setColumnName(Optional.ofNullable(columnName).map(item -> item.toString()).orElse(null));
                    }

                    if ("data_type".equals(resultSet.getMetaData().getColumnName(i)) ||
                            "DATA_TYPE".equals(resultSet.getMetaData().getColumnName(i))) {
                        Object columnType = resultSet.getObject(resultSet.getMetaData().getColumnName(i));
                        finAccEngDetDataLineVO.setColumnType(Optional.ofNullable(columnType).map(item -> item.toString()).orElse(null));
                    }

                    if ("column_comment".equals(resultSet.getMetaData().getColumnName(i)) ||
                            "COLUMN_COMMENT".equals(resultSet.getMetaData().getColumnName(i))) {
                        Object columnComment = resultSet.getObject(resultSet.getMetaData().getColumnName(i));
                        finAccEngDetDataLineVO.setColumnComment(Optional.ofNullable(columnComment).map(item -> item.toString()).orElse(null));
                    }

                    if ("character_maximum_length".equals(resultSet.getMetaData().getColumnName(i)) ||
                            "CHARACTER_MAXIMUM_LENGTH".equals(resultSet.getMetaData().getColumnName(i))) {
                        Object columnLength = resultSet.getObject(resultSet.getMetaData().getColumnName(i));
                        finAccEngDetDataLineVO.setColumnLength(Optional.ofNullable(columnLength).map(item -> Integer.valueOf(item.toString())).orElse(null));
                    }
                }
                finAccEngDetDataLineVOList.add(finAccEngDetDataLineVO);
            }
        } catch (Exception e) {
            throw new BusinessException("查询数据库字段异常");
        } finally {
            if (connection != null) {
                try {
                    connection.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
            if (resultSet != null) {
                try {
                    resultSet.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
        }
        //查询核算维度字段
        FinSetOfBookDO finSetOfBookDO = finSetOfBookRepoProc.findBySobCodeAndStatus(finAccEngDetDataParam.getSobCode(),
                UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        Assert.notNull(finSetOfBookDO, "账套不存在");

        List<FinSetOfBookLineDO> finSetOfBookLineDOList = finSetOfBookLineRepo.findAllByMasId(finSetOfBookDO.getId());
        Assert.notEmpty(finSetOfBookLineDOList, "账套明细不存在");

        finSetOfBookLineDOList.stream().forEach(item -> {
            FinAccEngDetDataLineVO finAccEngDetDataLineVO = new FinAccEngDetDataLineVO();
            finAccEngDetDataLineVO.setColumnName(item.getColumnName());
            finAccEngDetDataLineVO.setColumnType("varchar");
            finAccEngDetDataLineVO.setColumnComment(null);
            finAccEngDetDataLineVO.setColumnLength(32);
            finAccEngDetDataLineVOList.add(finAccEngDetDataLineVO);
        });
        return finAccEngDetDataLineVOList;
    }

    private void checkLoadTemplateParam(FinAccEngDetDataParam finAccEngDetDataParam) {

        Assert.notEmpty(finAccEngDetDataParam.getJournalHost(), "主机必填");
        Assert.notEmpty(finAccEngDetDataParam.getJournalPort(), "端口号必填");
        Assert.notEmpty(finAccEngDetDataParam.getJournalDatabase(), "数据库名称必填");
        Assert.notEmpty(finAccEngDetDataParam.getJournalUser(), "用户名必填");
        Assert.notEmpty(finAccEngDetDataParam.getJournalPassword(), "密码必填");
        Assert.notEmpty(finAccEngDetDataParam.getJournalTable(), "主表必填");
        Assert.notEmpty(finAccEngDetDataParam.getSobCode(), "账套编码必填");

    }
    private void checkEngineUnique(FinAccountEngineParam finAccountEngineParam,boolean add) {
        //校验事件表单是否被多个会计引擎关联
        List<FinAccountEngineDO> finAccountEngineDOExists= finAccountEngineRepo.findAllByEventTableId(finAccountEngineParam.getEventTableId());
        if(CollectionUtils.isNotEmpty(finAccountEngineDOExists)){
            if(add){
                throw new BusinessException("事件表单：【"+finAccountEngineParam.getEventTable()+"】已被其他会计引擎关联");
            }
            if(finAccountEngineDOExists.size()>1){
                throw new BusinessException("事件表单：【"+finAccountEngineParam.getEventTable()+"】已被其他会计引擎关联");
            }
            if(!finAccountEngineDOExists.get(0).getId().equals(finAccountEngineParam.getId())){
                throw new BusinessException("事件表单：【"+finAccountEngineParam.getEventTable()+"】已被其他会计引擎关联");
            }
        }
    }
}
