package com.elitesland.fin.application.service.unionpay.impl;

import com.chinapay.secss.SecssUtil;
import com.cloudt.apm.common.exception.BusinessException;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.tenant.client.common.TenantProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.fin.application.facade.param.account.AccountStorageFollowReq;
import com.elitesland.fin.application.facade.param.account.AccountStorageQueryReq;
import com.elitesland.fin.application.service.unionpay.UnionPayService;
import com.elitesland.fin.application.service.unionpay.config.UnionPaySecssUtil;
import com.elitesland.fin.application.service.unionpay.entity.constant.Constants;
import com.elitesland.fin.application.service.unionpay.entity.enums.ReceiptStatusEnum;
import com.elitesland.fin.application.service.unionpay.entity.enums.RespCodeEnum;
import com.elitesland.fin.application.service.unionpay.entity.enums.SourceTypeEnum;
import com.elitesland.fin.application.service.unionpay.entity.enums.StoredOrderTypeEnum;
import com.elitesland.fin.application.service.unionpay.entity.req.SendPayReq;
import com.elitesland.fin.application.service.unionpay.util.UnionPayUtil;
import com.elitesland.fin.domain.param.recorder.RecOrderRpcTwoParam;
import com.elitesland.fin.entity.account.AccountStorageDO;
import com.elitesland.fin.entity.account.AccountStorageDetailDO;
import com.elitesland.fin.infinity.http.RestClient;
import com.elitesland.fin.infr.repo.recorder.RecOrderRpcFiledRepo;
import com.elitesland.fin.param.flow.AccountFlowRpcParam;
import com.elitesland.fin.repo.account.AccountStorageDetailRepo;
import com.elitesland.fin.repo.account.AccountStorageRepo;
import com.elitesland.fin.service.flow.AccountFlowRpcService;
import com.elitesland.fin.utils.TimeUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;

/**
 * @author sunxw
 * @description
 * @Date 2023/5/18
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class UnionPayServiceImpl implements UnionPayService {

    @Value("${china.pay.merId}")
    private String merId;
    @Value("${china.pay.refundUrl}")
    private String refundUrl;
    @Value("${china.pay.queryUrl}")
    private String queryUrl;
    @Value("${china.baseBack.pay}")
    private String backPayUrl;
    @Value("${china.baseBack.refund}")
    private String backRefundUrl;
    @Value("${china.pay.remoteAddr}")
    private String remoteAddr;
    @Autowired
    private AccountStorageRepo accountStorageRepo;
    @Autowired
    private RecOrderRpcFiledRepo recOrderRpcFiledRepo;
    @Autowired
    private AccountStorageDetailRepo accountStorageDetailRepo;
    private final AccountFlowRpcService accountFlowRpcService;
    private final RestClient restClient;
    @Autowired
    private SecssUtil secssUtil;
    @Autowired
    private UnionPaySecssUtil unionPaySecssUtil;
    private final TenantProvider tenantProvider;
    private final TenantDataIsolateProvider tenantDataIsolateProvider;
    private static final Random RANDOM = new Random();

    @Transactional
    @Override
    public Map<String,Object> sendPay(SendPayReq payReq) {
        String receiptStatus = accountStorageRepo.getReceiptStatusByOrderId(payReq.getPayOrderId());
        if (!ReceiptStatusEnum.PENDING_PAY.getCode().equals(receiptStatus)){
            log.error("当前储值单单据状态非待支付状态，不能重复发起支付，单据流水号为:[{}]",payReq.getPayOrderId());
            throw new BusinessException("当前储值单单据状态非待支付状态，无法重复发起支付，单据流水号为:"+payReq.getPayOrderId());
        }
        TreeMap<String, Object> data = this.buildPayParams(payReq);
        log.info("4.1支付组装请求报文为 :{}",data);
        //修改储值订单与收款单单据状态为支付中
        AccountStorageDO storageDO = new AccountStorageDO();
        storageDO.setId(payReq.getStorageId());
        storageDO.setReceiptStatus(payReq.getReceiptStatus());
        accountStorageRepo.updateReceiptStatus(payReq.getPayOrderId(),payReq.getReceiptStatus());
        recOrderRpcFiledRepo.updateReceiptStatus(payReq.getPayOrderId(),payReq.getReceiptStatus());
        return data;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void payCallBack(Map<String,String> data) {
        String merOrderNo = data.get(Constants.MER_ORDER_NO);
        log.info("储值订单号[{}]操作成功",merOrderNo);
        AccountStorageQueryReq queryReq = new AccountStorageQueryReq();
        queryReq.setOriTranDate(data.get(Constants.TRAN_DATE));
        queryReq.setPayOrderId(merOrderNo);
        //收到支付成功的异步回调之后再次主动查询订单的支付状态
        String queryPayResult = this.queryPay(queryReq);
        if (StringUtils.isNotEmpty(queryPayResult)){
            String result = queryPayResult.replaceAll("amp;", "");
            HashMap<String, String> queryResult = UnionPayUtil.form2Map(result);
            String receiptStatus;
            RecOrderRpcTwoParam recOrderRpcTwoParam = new RecOrderRpcTwoParam();
            recOrderRpcTwoParam.setSourceNo(merOrderNo);
            String remitterAccount = recOrderRpcFiledRepo.getAccountByOrderId(merOrderNo);
            if (Constants.ORDER_SUCCESS_CODE.equals(data.get(Constants.ORDER_STATUS)) && Constants.ORDER_SUCCESS_CODE.equals(queryResult.get(Constants.ORDER_STATUS))){
                receiptStatus = ReceiptStatusEnum.PAY_SUCCESS.getCode();
                Long storageId = accountStorageRepo.getIdByOrderId(merOrderNo);
                this.generateAccountFlow(receiptStatus,data.get(Constants.ORDER_AMT),storageId,null);
                //修改付款单状态等
                if (StringUtils.isNotEmpty(data.get(Constants.CARD_TRAN_DATA))){
                    String accountData = unionPaySecssUtil.getDecData(data.get(Constants.CARD_TRAN_DATA));
                    log.info("4.1支付异步回调账户信息is {}",accountData);
                    //支付成功，判断回调的汇款银行和汇款账号与付款单是否一致，不一致则需要发起退款
                    if (!remitterAccount.equals(accountData)){
                        AccountStorageFollowReq followReq = new AccountStorageFollowReq();
                        followReq.setPayOrderId(merOrderNo);
                        followReq.setRefundAmt(data.get(Constants.ORDER_AMT));
                        followReq.setOriTranDate(data.get(Constants.TRAN_DATE));
                        this.followPay(followReq);
                    }
                }
            }else {
                //支付失败
                receiptStatus = ReceiptStatusEnum.PAY_FAILED.getCode();
            }
            accountStorageRepo.updateReceiptStatus(merOrderNo,receiptStatus);
            recOrderRpcFiledRepo.updateReceiptStatus(merOrderNo,receiptStatus);
            accountStorageRepo.updateMsg(merOrderNo,data.toString(), RespCodeEnum.find(data.get(Constants.ORDER_STATUS)));
        }
    }

    @Transactional
    @Override
    public String followPay(AccountStorageFollowReq req) {
        TreeMap<String, Object> data = this.buildFollowPayParams(req);
        String result = restClient.postFormData(refundUrl, data);
        log.info("4.15交易后续同步响应信息为 :{}",result);
        AccountStorageDetailDO accountStorageDetailDO = new AccountStorageDetailDO();
        accountStorageDetailDO.setRefundOrderId((String) data.get(Constants.MER_ORDER_NO));
        Long storageId = accountStorageRepo.getIdByOrderId(req.getPayOrderId());
        accountStorageDetailDO.setStorageId(storageId);
        accountStorageDetailDO.setRemitterDate(TimeUtil.string2Date(req.getOriTranDate()));
        accountStorageDetailDO.setOriMsg(req.getOriMsg());
        accountStorageDetailDO.setRefundDate(LocalDate.now());
        accountStorageDetailDO.setRefundReason("实际汇款银行账户与开票主体账户信息不一致");
        BigDecimal refundAmt = new BigDecimal(req.getRefundAmt()).divide(new BigDecimal(100)).setScale(2, RoundingMode.HALF_UP);
        accountStorageDetailDO.setRefundAmt(refundAmt);
        accountStorageDetailRepo.save(accountStorageDetailDO);
        accountStorageRepo.updateReceiptStatus(req.getPayOrderId(), ReceiptStatusEnum.REFUNDING.getCode());
        recOrderRpcFiledRepo.updateReceiptStatus(req.getPayOrderId(), ReceiptStatusEnum.REFUNDING.getCode());
        return result;
    }

    @Transactional
    @Override
    public void followPayCallback(Map<String, String> data) {
        //收到回调时再次主动确认订单支付状态
        AccountStorageQueryReq queryReq = new AccountStorageQueryReq();
        queryReq.setOriTranDate(TimeUtil.date2String(LocalDate.now()));
        queryReq.setPayOrderId(data.get(Constants.ORI_ORDER_NO));
        String queryPayResult = this.queryPay(queryReq);
        String result = queryPayResult.replaceAll("amp;", "");
        Map<String,String> queryResult = UnionPayUtil.form2Map(result);
        if (StringUtils.isNotEmpty(queryPayResult)){
            Long storageId = accountStorageRepo.getIdByOrderId(data.get(Constants.ORI_ORDER_NO));
            String oriPayOrderId = data.get(Constants.ORI_ORDER_NO);
            String receiptStatus;
            if (Constants.ORDER_SUCCESS_CODE.equals(data.get(Constants.ORDER_STATUS)) && Constants.ORDER_SUCCESS_CODE.equals(queryResult.get(Constants.ORDER_STATUS))){
                receiptStatus = ReceiptStatusEnum.REFUND_SUCCESS.getCode();
                Long refundId = accountStorageDetailRepo.selectIdByRefundId(data.get(Constants.MER_ORDER_NO));
                this.generateAccountFlow(receiptStatus,data.get(Constants.REFUND_AMT),storageId,refundId);
            }else {
                receiptStatus = ReceiptStatusEnum.REFUND_FAILED.getCode();
            }
            accountStorageRepo.updateReceiptStatus(oriPayOrderId,receiptStatus);
            recOrderRpcFiledRepo.updateReceiptStatus(oriPayOrderId,receiptStatus);
            accountStorageDetailRepo.updateOriMsg(storageId,data.toString());
        }
    }

    @Override
    public String queryPay(AccountStorageQueryReq req) {
        TreeMap<String, Object> data = this.buildQueryPayParams(req);
        String result = restClient.postFormData(queryUrl, data);
        log.info("4.16交易查询响应信息为 :{}",result);
        return result;
    }

    /**
     * 构建银联商务支付参数
     * @param payReq
     * @return
     */
    private TreeMap<String,Object> buildPayParams(SendPayReq payReq){
        TreeMap<String,Object> req = new TreeMap<>();
        req.put(Constants.VERSION,Constants.REQUEST_VERSION);
        req.put(Constants.MER_ID,merId);
        req.put(Constants.MER_ORDER_NO,payReq.getPayOrderId());
        req.put(Constants.TRAN_DATE, TimeUtil.date2String(LocalDate.now()));
        req.put(Constants.TRAN_TIME,TimeUtil.time2String(LocalTime.now()));
        //元转分
        Long orderAmt = payReq.getRemitterAmt().multiply(new BigDecimal(100)).longValue();
        req.put(Constants.ORDER_AMT,orderAmt.toString());
        req.put(Constants.BUSI_TYPE,Constants.BUSINESS_TYPE);
        req.put(Constants.BANK_INST_NO, payReq.getRemitterBankCode()); // 支付机构号-银联在线支付(这个参数必填，如果是无卡支付前端请求，否则会失败)
        req.put(Constants.TRAN_TYPE,Constants.PAY_TRAN_TYPE);
        //支付接口后端回调地址
        req.put(Constants.MER_BG_URL,backPayUrl);
        //付款支付完成后跳转至的系统前台展示地址
        if (StringUtils.isNotEmpty(payReq.getMerPageUrl())){
            req.put("MerPageUrl",payReq.getMerPageUrl());
        }
        req.put(Constants.REMOTE_ADDR,remoteAddr);
        secssUtil.sign(req);
        log.info("支付交易签名响应is {}，信息 {}",secssUtil.getErrCode(),secssUtil.getErrMsg());
        String sign = secssUtil.getSign();
        req.put(Constants.SIGNATURE,sign);
        return req;
    }

    /**
     * 构建银联商务后续类参数
     * @param param
     * @return
     */
    private TreeMap<String,Object> buildFollowPayParams(AccountStorageFollowReq param){
        String prefix = TimeUtil.datetime2String(LocalDateTime.now());
        Integer suffix = RANDOM.nextInt(99999);
        StringBuilder refundOrderId = new StringBuilder(prefix);
        refundOrderId.append(suffix);
        TreeMap<String,Object> req = new TreeMap<>();
        req.put(Constants.VERSION,Constants.REQUEST_VERSION);
        req.put(Constants.MER_ID,merId);
        req.put(Constants.MER_ORDER_NO,refundOrderId.toString());
        req.put(Constants.TRAN_DATE, TimeUtil.date2String(LocalDate.now()));
        req.put(Constants.TRAN_TIME,TimeUtil.time2String(LocalTime.now()));
        //测试时退三分保证退款成功
        //req.put(Constants.REFUND_AMT,"3");
        req.put(Constants.REFUND_AMT,param.getRefundAmt());
        req.put(Constants.ORI_ORDER_NO,param.getPayOrderId());
        req.put(Constants.ORI_TRAN_DATE, param.getOriTranDate());
        req.put(Constants.TRAN_TYPE,Constants.REFUND_TRAN_TYPE);
        req.put(Constants.BUSI_TYPE,Constants.BUSINESS_TYPE);
        //支付接口回调地址
        req.put(Constants.MER_BG_URL,backRefundUrl);
        secssUtil.sign(req);
        log.info("后续交易签名响应is {}，信息 {}",secssUtil.getErrCode(),secssUtil.getErrMsg());
        String sign = secssUtil.getSign();
        req.put(Constants.SIGNATURE,sign);
        return req;
    }

    /**
     * 构建银联商务交易查询参数
     * @param param
     * @return
     */
    private TreeMap<String,Object> buildQueryPayParams(AccountStorageQueryReq param){
        TreeMap<String,Object> req = new TreeMap<>();
        req.put(Constants.VERSION,Constants.REQUEST_VERSION);
        req.put(Constants.MER_ID,merId);
        req.put(Constants.MER_ORDER_NO,param.getPayOrderId());
        req.put(Constants.TRAN_DATE, param.getOriTranDate());
        req.put(Constants.TRAN_TYPE,Constants.QUERY_TRAN_TYPE);
        req.put(Constants.BUSI_TYPE,Constants.BUSINESS_TYPE);
        secssUtil.sign(req);
        log.info("商务交易签名响应is {}，信息 {}",secssUtil.getErrCode(),secssUtil.getErrMsg());
        String sign = secssUtil.getSign();
        req.put(Constants.SIGNATURE,sign);
        return req;
    }

    private void generateAccountFlow(String sourceDocStatus,String sourceDocAmount,Long storageId,Long key){
        BigDecimal sourceAmt = new BigDecimal(sourceDocAmount).divide(new BigDecimal(100)).setScale(2, RoundingMode.HALF_UP);
        String remitter = accountStorageRepo.getRemitterIdByKey(storageId);
        AccountFlowRpcParam rpcParam = new AccountFlowRpcParam();
        if (ObjectUtils.isNotEmpty(key)){
            rpcParam.setSourceId(key);
        }else {
            rpcParam.setSourceId(storageId);
        }
        rpcParam.setSourceDoc(SourceTypeEnum.SVO.getCode());
        rpcParam.setSourceDocType(StoredOrderTypeEnum.PREPAY.getCode());
        rpcParam.setSourceDocStatus(sourceDocStatus);
        rpcParam.setSourceDocAmount(sourceAmt);
        rpcParam.setSourceNo(storageId.toString());
        rpcParam.setAccountHolderName(remitter);
        try {
            ApiResult<Boolean> apiResult = accountFlowRpcService.generateAccountFlow(rpcParam);
            log.info("储值订单账户流水 req is {},resp is{}",rpcParam,apiResult);
        }catch (Exception e){
            log.error("储值订单调用账户流水逻辑发生错误，req is {}，错误信息 is :{}",rpcParam,e);
        }

    }

}