package com.elitesland.fin.domain.service.paymentperiod;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.paymentperiod.ReceiptPaymentAgreementConvert;
import com.elitesland.fin.application.facade.param.paymentperiod.ReceiptPaymentAgreementPageParam;
import com.elitesland.fin.application.facade.param.paymentperiod.ReceiptPaymentAgreementSaveParam;
import com.elitesland.fin.application.facade.vo.paymentperiod.ReceiptPaymentAgreementPagingVO;
import com.elitesland.fin.application.facade.vo.paymentperiod.ReceiptPaymentAgreementVO;
import com.elitesland.fin.domain.entity.paymentperiod.ReceiptPaymentAgreementDO;
import com.elitesland.fin.infr.repo.paymentperiod.ReceiptPaymentAgreementDtlRepo;
import com.elitesland.fin.infr.repo.paymentperiod.ReceiptPaymentAgreementDtlRepoProc;
import com.elitesland.fin.infr.repo.paymentperiod.ReceiptPaymentAgreementRepo;
import com.elitesland.fin.infr.repo.paymentperiod.ReceiptPaymentAgreementRepoProc;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2024/04/11
 * @Version 1.0
 * @Content:
 */
@Service
@RequiredArgsConstructor
public class ReceiptPaymentAgreementDomainServiceImpl implements ReceiptPaymentAgreementDomainService{

    private final ReceiptPaymentAgreementRepo receiptPaymentAgreementRepo;
    private final ReceiptPaymentAgreementRepoProc receiptPaymentAgreementRepoProc;
    private final ReceiptPaymentAgreementDtlRepo receiptPaymentAgreementDtlRepo;
    private final ReceiptPaymentAgreementDtlRepoProc receiptPaymentAgreementDtlRepoProc;


    /**
     * 收付款协议保存
     *
     * @param saveParam 入参
     * @return收付款协议ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long save(ReceiptPaymentAgreementSaveParam saveParam) {
        ReceiptPaymentAgreementDO receiptPaymentAgreementDO = ReceiptPaymentAgreementConvert.INSTANCE.saveVoToDo(saveParam);
        Long id = receiptPaymentAgreementRepo.save(receiptPaymentAgreementDO).getId();
        return id;
    }


    /**
     * 收付款协议分页查询
     *
     * @param pageParam 入参
     * @return 收付款协议信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<ReceiptPaymentAgreementPagingVO> page(ReceiptPaymentAgreementPageParam pageParam) {

        PagingVO<ReceiptPaymentAgreementPagingVO> pagingVO = receiptPaymentAgreementRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<ReceiptPaymentAgreementPagingVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        return pagingVO;
    }

    /**
     * 根据ID查询主表数据
     *
     * @param id 收付款协议ID
     * @return 收付款协议主表数据
     */
    @Override
    @SysCodeProc
    public ReceiptPaymentAgreementVO findById(Long id) {
        if (Objects.isNull(id)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "ID不能为空!");
        }
        Optional<ReceiptPaymentAgreementDO> optional = receiptPaymentAgreementRepo.findById(id);
        if (optional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        return ReceiptPaymentAgreementConvert.INSTANCE.doToRespVo(optional.get());
    }

    /**
     * 根据收付款协议ID查询收付款协议主表数据
     *
     * @param id 收付款协议ID
     * @return 收付款协议主表数据
     */
    @Override
    @SysCodeProc
    public ReceiptPaymentAgreementVO selectById(Long id) {
        if (Objects.isNull(id)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "ID不能为空!");
        }
        ReceiptPaymentAgreementPageParam queryVO = new ReceiptPaymentAgreementPageParam();
        queryVO.setId(id);
        List<ReceiptPaymentAgreementVO> respVOList = receiptPaymentAgreementRepoProc.selectListByParam(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        if (respVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "查询到多条数据信息!");
        }
        ReceiptPaymentAgreementVO agreementVO = respVOList.get(0);

        return agreementVO;
    }

    /**
     * 根据收付款协议编码查询收付款协议主表数据
     *
     * @param code 根据收付款协议编码
     * @return 收付款协议主表数据
     */
    @Override
    @SysCodeProc
    public ReceiptPaymentAgreementVO selectByCode(String code) {
        if (StringUtils.isBlank(code)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "编码不能为空!");
        }
        ReceiptPaymentAgreementPageParam queryVO = new ReceiptPaymentAgreementPageParam();
        queryVO.setProtocolCode(code);
        List<ReceiptPaymentAgreementVO> respVOList = receiptPaymentAgreementRepoProc.selectListByParam(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        if (respVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "查询到多条数据信息!");
        }
        ReceiptPaymentAgreementVO agreementVO = respVOList.get(0);

        return agreementVO;
    }

    /**
     * 根据入参查询收付款协议数据
     *
     * @param queryVO 入参
     * @return 收付款协议主表数据
     */
    @Override
    @SysCodeProc
    public List<ReceiptPaymentAgreementVO> selectByParam(ReceiptPaymentAgreementPageParam queryVO) {
        List<ReceiptPaymentAgreementVO> agreementVOList = receiptPaymentAgreementRepoProc.selectListByParam(queryVO);
        if (CollectionUtils.isEmpty(agreementVOList)) {
            return Collections.EMPTY_LIST;
        }
        return agreementVOList;
    }

    /**
     * 校验查询编号/名称是否已经存在
     *
     * @param queryVO 入参
     * @return 收付款协议主表数据
     */
    @Override
    @SysCodeProc
    public ReceiptPaymentAgreementVO checkSelectByParam(ReceiptPaymentAgreementPageParam queryVO) {
        ReceiptPaymentAgreementVO agreementVO = receiptPaymentAgreementRepoProc.checkSelectByParam(queryVO);
        return agreementVO;
    }

    /**
     * 根据ID批量更新状态
     *
     * @param status 状态
     * @param ids    ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateStatusBatch(String status, List<Long> ids) {
        receiptPaymentAgreementRepoProc.updateStatusBatch(status, ids);
    }

    /**
     * 根据ID更新状态
     *
     * @param status 状态
     * @param id     ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateStatusById(String status, Long id) {
        receiptPaymentAgreementRepoProc.updateStatusById(status, id);
    }


    /**
     * 根据ID批量更新deleteFlag
     *
     * @param deleteFlag 删除标记
     * @param ids        主表ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDeleteFlagBatch(Integer deleteFlag, List<Long> ids) {
        receiptPaymentAgreementRepoProc.updateDeleteFlagBatch(deleteFlag, ids);
    }

    @Override
    public List<ReceiptPaymentAgreementVO> selectByCodes(List<String> codes) {
        ReceiptPaymentAgreementPageParam queryVO = new ReceiptPaymentAgreementPageParam();
        queryVO.setProtocolCodeList(codes);
        List<ReceiptPaymentAgreementVO> respVOList = receiptPaymentAgreementRepoProc.selectListByParam(queryVO);
        return respVOList;
    }

}
