package com.elitesland.fin.application.service.creditaccount;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.creditaccount.CreditAccountRuleConfigConvert;
import com.elitesland.fin.application.convert.creditaccount.CreditAccountRuleConfigDtlConvert;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountDTO;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountIocDTO;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountRuleConfigDTO;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountRuleConfigDtlDTO;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountParam;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountRuleConfigQueryParam;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountRuleConfigSaveParam;
import com.elitesland.fin.application.facade.vo.account.AccountRuleConfigQueryVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountRuleConfigDtlVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountRuleConfigPageVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountRuleConfigQueryVO;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinRedisConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.service.creditaccount.CreditAccountIocDomainService;
import com.elitesland.fin.domain.service.creditaccount.CreditAccountRuleConfigDomainService;
import com.elitesland.fin.domain.service.creditaccount.CreditAccountRuleConfigDtlDomainService;
import com.elitesland.fin.entity.creditaccount.CreditAccountRuleConfigDO;
import com.elitesland.fin.entity.creditaccount.CreditAccountRuleConfigDtlDO;
import com.elitesland.fin.repo.creditaccount.CreditAccountRuleConfigDtlRepo;
import com.elitesland.fin.repo.creditaccount.CreditAccountRuleConfigRepo;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RBucket;
import org.redisson.api.RedissonClient;
import org.springframework.data.domain.Example;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author gyj
 * @date 2023/05/24 18:23
 */
@Slf4j
@Service
@AllArgsConstructor
@Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
public class CreditAccountRuleConfigServiceImpl implements CreditAccountRuleConfigService {

    private final CreditAccountRuleConfigDomainService accountRuleConfigDomainService;

    private final CreditAccountRuleConfigDtlDomainService accountRuleConfigDtlDomainService;

    private final CreditAccountIocDomainService accountIocDomainService;

    private final RedissonClient redissonClient;

    private final CreditAccountRuleConfigRepo accountRuleConfigRepo;

    private final CreditAccountRuleConfigDtlRepo accountRuleConfigDtlRepo;

    private final UdcProvider udcProvider;

    private final CreditAccountService accountService;

    @Override
    @SysCodeProc
    public PagingVO<CreditAccountRuleConfigPageVO> searchPage(CreditAccountRuleConfigQueryParam param) {
        PagingVO<CreditAccountRuleConfigPageVO> accountRuleConfigPageVOPagingVO = accountRuleConfigDomainService.searchPage(param);
        if (CollectionUtil.isNotEmpty(accountRuleConfigPageVOPagingVO.getRecords())) {
            convertPage(accountRuleConfigPageVOPagingVO.getRecords());
        }

        pageCode2Name(accountRuleConfigPageVOPagingVO.getRecords());
        return accountRuleConfigPageVOPagingVO;
    }

    @Override
    public void saveOrUpdate(CreditAccountRuleConfigSaveParam param) {
        validateSaveOrUpdateParam(param);
        persistence(param);
    }

    private void persistence(CreditAccountRuleConfigSaveParam param) {
        checkRepeatData(param);
        checkExist(param);

        if (param.getId() == null) {
            saveConfigAndDtl(param);
            return;
        }
        updateConfigAndDtl(param);
    }

    private void checkExist(CreditAccountRuleConfigSaveParam param) {
        List<CreditAccountRuleConfigDO> accountRuleConfigDOList;

        accountRuleConfigDOList = accountRuleConfigRepo.findAllByRuleCode(param.getRuleCode());
        if (CollectionUtils.isNotEmpty(accountRuleConfigDOList)) {
            Assert.isTrue(accountRuleConfigDOList.size() == 1, "存在多条相同的规则编码");
            Assert.equals(accountRuleConfigDOList.get(0).getId(), param.getId(), "规则编码已经存在");
        }

        accountRuleConfigDOList = accountRuleConfigRepo.findAllByOptDoc(param.getOptDoc());
        if (CollectionUtils.isNotEmpty(accountRuleConfigDOList)) {
            Assert.isTrue(accountRuleConfigDOList.size() == 1, "存在多条相同的适用单据");
            Assert.equals(accountRuleConfigDOList.get(0).getId(), param.getId(), "适用单据已经存在");
        }
    }

    private void checkRepeatData(CreditAccountRuleConfigSaveParam param) {
        Map<Integer, String> seq = new HashMap();
        param.getAccountRuleConfigDtlDTOList().stream().forEach(item -> {
            Assert.isNull(seq.get(item.getPriorityNo()), "优先级不能重复");
            seq.put(item.getPriorityNo(), "");
        });


        Set<String> repeatSet = new HashSet<>();
        param.getAccountRuleConfigDtlDTOList().stream().forEach(item -> {

            Arrays.stream(item.getOptDocType().split(FinConstant.TRANSLATION_PLUS)).forEach(optDocType -> {
                if (StringUtils.isNotEmpty(item.getSpecificAccount())) {
                    Arrays.stream(item.getSpecificAccount().split(FinConstant.TRANSLATION_PLUS)).forEach(specificAccount -> {
                        String uniqueKey = item.getIoCode().concat(optDocType).concat(item.getOptDocStatus()).concat(specificAccount);
                        Assert.isFalse(repeatSet.contains(uniqueKey), "存在重复规则,事务码、适用单据类型、适用单据状态、指定账户需唯一");
                        repeatSet.add(uniqueKey);

                    });
                } else {
                    String uniqueKey = item.getIoCode().concat(optDocType).concat(item.getOptDocStatus()).concat(item.getOptAccountType());
                    Assert.isFalse(repeatSet.contains(uniqueKey), "存在重复规则,事务码、适用单据类型、适用单据状态、指定账户需唯一");
                    repeatSet.add(uniqueKey);
                }

            });
        });

        //判断空指定用和有指定用户重复的情况
        repeatSet.stream().forEach(item->{
            repeatSet.stream().forEach(item1->{
                if(item.contains(item1) && !StringUtils.equals(item,item1)){
                    Assert.isTrue(false, "存在重复规则,事务码、适用单据类型、适用单据状态、指定账户需唯一");
                }
            });
        });
    }

    private void updateConfigAndDtl(CreditAccountRuleConfigSaveParam param) {
        CreditAccountRuleConfigDO accountRuleConfigDO = CreditAccountRuleConfigConvert.INSTANCE.param2DO(param);

        List<CreditAccountRuleConfigDtlDO> accountRuleConfigDtlDOS = CreditAccountRuleConfigDtlConvert.INSTANCE.param2DOList(param.getAccountRuleConfigDtlDTOList());
        accountRuleConfigDtlDOS.stream().forEach(dtl -> dtl.setMasId(accountRuleConfigDO.getId()));
        accountRuleConfigDomainService.updateDynamically(accountRuleConfigDO);
        accountRuleConfigDtlDomainService.deleteByParam(getQueryBeanByMasId(accountRuleConfigDO.getId()));
        accountRuleConfigDtlDomainService.saveAll(accountRuleConfigDtlDOS);
    }

    private Long saveConfigAndDtl(CreditAccountRuleConfigSaveParam param) {
        CreditAccountRuleConfigDO accountRuleConfigDO = CreditAccountRuleConfigConvert.INSTANCE.param2DO(param);

        List<CreditAccountRuleConfigDtlDO> accountRuleConfigDtlDOS = CreditAccountRuleConfigDtlConvert.INSTANCE.param2DOList(param.getAccountRuleConfigDtlDTOList());
        accountRuleConfigRepo.save(accountRuleConfigDO);
        accountRuleConfigDtlDOS.forEach(dtl -> {
            dtl.setMasId(accountRuleConfigDO.getId());
        });

        accountRuleConfigDtlRepo.saveAll(accountRuleConfigDtlDOS);
        return accountRuleConfigDO.getId();
    }

    private void validateSaveOrUpdateParam(CreditAccountRuleConfigSaveParam param) {
        Assert.notEmpty(param.getRuleCode(), "账户规则编码必填");
        Assert.notEmpty(param.getRuleName(), "账户规则名称必填");
        Assert.notEmpty(param.getOptDoc(), "适用单据必填");
        Assert.notEmpty(param.getStatus(), "启用状态必填");
        Assert.notEmpty(param.getAccountRuleConfigDtlDTOList(), "明细不能为空");

        param.getAccountRuleConfigDtlDTOList().stream().forEach(item -> {
            Assert.notNull(item.getPriorityNo(), "优先级必填");
            Assert.notEmpty(item.getIoCode(), "事务码必填");
            Assert.notEmpty(item.getIoName(), "事务码名称必填");
            Assert.notEmpty(item.getOptDocType(), "适用单据类型必填");
            Assert.notEmpty(item.getOptDocStatus(), "适用单据状态必填");
            Assert.notEmpty(item.getOptAccountType(), "适用账户类型必填");
            Assert.notNull(item.getCalculatePercent(), "计算比例必填");
            Assert.notNull(item.isAutomaticReview(), "流水自动审核必填");
        });
    }

    @Override
    public void del(Long id) {
        Optional<CreditAccountRuleConfigDO> accountRuleConfigDOOptional = accountRuleConfigDomainService.findById(id);

        if (accountRuleConfigDOOptional.isPresent()) {
            CreditAccountRuleConfigDO dbSceneConfig = accountRuleConfigDOOptional.get();
            accountRuleConfigDtlDomainService.deleteByCondition(getQueryBeanByMasId(id));
            accountRuleConfigDomainService.deleteById(id);
            removeCache(dbSceneConfig.getRuleCode());
        }
    }

    private CreditAccountRuleConfigDtlDO getQueryBeanByMasId(Long id) {
        CreditAccountRuleConfigDtlDO whereBean = new CreditAccountRuleConfigDtlDO();
        whereBean.setMasId(id);
        return whereBean;
    }

    @Override
    @SysCodeProc
    public CreditAccountRuleConfigQueryVO detail(Long masId) {
        Optional<CreditAccountRuleConfigDO> dbConfig = accountRuleConfigRepo.findById(masId);
        if (!dbConfig.isPresent()) {
            return null;
        }
        CreditAccountRuleConfigQueryVO queryVO = CreditAccountRuleConfigConvert.INSTANCE.do2VO(dbConfig.get());
        List<CreditAccountRuleConfigDtlDO> all = accountRuleConfigDtlDomainService.findAccountRuleConfigDtlByMasId(masId);
        List<CreditAccountRuleConfigDtlVO> accountRuleConfigDtlVoList = CreditAccountRuleConfigDtlConvert.INSTANCE.do2VOList(all);

        detailCode2Name(accountRuleConfigDtlVoList);
        queryVO.setAccountRuleConfigDtlVoList(accountRuleConfigDtlVoList);

        return queryVO;
    }

    public void detailCode2Name(List<CreditAccountRuleConfigDtlVO> accountRuleConfigDtlVOList) {
        //单据类型
        Map<String, String> docClsUdc = udcProvider.getValueMapByUdcCode(UdcEnum.DOC_CLS_SO_C.getModel(), UdcEnum.DOC_CLS_SO_C.getCode());
        //订单状态
        Map<String, String> docStatusUdc = udcProvider.getValueMapByUdcCode(UdcEnum.DOC_STATUS_RSO_CL.getModel(), UdcEnum.DOC_STATUS_RSO_CL.getCode());
        //适用账户类型
        Map<String, String> creditAccountTypeUdc = udcProvider.getValueMapByUdcCode(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getModel(), UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getCode());

        //根据账户编码获取账户信息
        Set<String> accountCodeSet = new HashSet<>();
        accountRuleConfigDtlVOList.stream().forEach(item -> {
            if(StringUtils.isNotEmpty(item.getSpecificAccount())){
                String[] specificAccountArray = item.getSpecificAccount().split(FinConstant.TRANSLATION_PLUS);
                Arrays.stream(specificAccountArray).forEach(specificAccount -> accountCodeSet.add(specificAccount));
            }
        });

        List<CreditAccountDTO> accountDTOList =new ArrayList<>();
        if(CollectionUtils.isNotEmpty(accountCodeSet)){
            CreditAccountParam accountParam = new CreditAccountParam();
            accountParam.setCreditAccountCodes(accountCodeSet);

            accountDTOList = accountService.getAccountByAccountParam(accountParam);
            Assert.notEmpty(accountDTOList, "查询信用账户名称失败");
        }


        //设置单据类型名称
        //设置指定账户名称
        List<String> optDocTypeList = new ArrayList<>();
        List<String> specificAccountList = new ArrayList<>();
        List<CreditAccountDTO> finalAccountDTOList = accountDTOList;
        accountRuleConfigDtlVOList.stream().forEach(item -> {
            optDocTypeList.clear();
            specificAccountList.clear();

            String[] optDocTypeArray = item.getOptDocType().split(FinConstant.TRANSLATION_PLUS);
            Arrays.stream(optDocTypeArray).forEach(optDocType -> {
                optDocTypeList.add(docClsUdc.get(optDocType));
            });

            if(StringUtils.isNotEmpty(item.getSpecificAccount())){
                String[] specificAccountArray = item.getSpecificAccount().split(FinConstant.TRANSLATION_PLUS);
                Arrays.stream(specificAccountArray).forEach(specificAccount -> {
                    CreditAccountDTO result = finalAccountDTOList.stream()
                            .filter(accountVO -> specificAccount.equals(accountVO.getCreditAccountCode()))
                            .findFirst()
                            .orElse(null);
                    if (result != null) {
                        specificAccountList.add(result.getCreditAccountName());
                    }
                });
            }


            item.setOptDocTypeName(joinedWithPlus(optDocTypeList));

            if(CollectionUtils.isNotEmpty(specificAccountList)){
                item.setSpecificAccountName(joinedWithPlus(specificAccountList));
            }

            item.setOptDocStatusName(docStatusUdc.get(item.getOptDocStatus()));
            item.setOptAccountTypeName(creditAccountTypeUdc.get(item.getOptAccountType()));
        });
    }

    public void pageCode2Name(List<CreditAccountRuleConfigPageVO> accountRuleConfigPageVOList) {
        //单据类型
        Map<String, String> docClsUdc = udcProvider.getValueMapByUdcCode(UdcEnum.DOC_CLS_SO_C.getModel(), UdcEnum.DOC_CLS_SO_C.getCode());
        //订单状态
        Map<String, String> docStatusUdc = udcProvider.getValueMapByUdcCode(UdcEnum.DOC_STATUS_RSO_CL.getModel(), UdcEnum.DOC_STATUS_RSO_CL.getCode());

        //设置单据类型名称
        List<String> optDocTypeList = new ArrayList<>();
        accountRuleConfigPageVOList.stream().forEach(item -> {
            optDocTypeList.clear();

            String[] optDocTypeArray = item.getOptDocType().split(FinConstant.TRANSLATION_PLUS);
            Arrays.stream(optDocTypeArray).forEach(optDocType -> {
                optDocTypeList.add(docClsUdc.get(optDocType));
            });

            item.setOptDocTypeName(joinedWithPlus(optDocTypeList));
            item.setOptDocStatusName(docStatusUdc.get(item.getOptDocStatus()));
        });
    }

    private String joinedWithPlus(List<String> param) {
        return StringUtils.join(param, FinConstant.PLUS);
    }

    private void removeCache(String ruleCode) {
        RBucket<AccountRuleConfigQueryVO> bucket = redissonClient.getBucket(FinRedisConstant.RULE_CONFIG_KEY + ruleCode);
        if (bucket.isExists()) {
            bucket.delete();
        }
    }

    @Override
    public CreditAccountRuleConfigDTO getByRuleCode(String ruleCode) {
        RBucket<CreditAccountRuleConfigDTO> bucket = redissonClient.getBucket(FinRedisConstant.RULE_CONFIG_KEY + ruleCode);
        if (bucket.isExists()) {
            return bucket.get();
        }
        var dbConfig = accountRuleConfigDomainService.findByRuleCode(ruleCode);
        if (dbConfig != null) {
            Example<CreditAccountRuleConfigDtlDO> dtlExample = Example.of(getQueryBeanByMasId(dbConfig.getId()));
            List<CreditAccountRuleConfigDtlDO> all = accountRuleConfigDtlDomainService.findAll(dtlExample);
            List<CreditAccountRuleConfigDtlDTO> accountRuleConfigDtls = CreditAccountRuleConfigDtlConvert.INSTANCE.do2DTOList(all);
            dbConfig.setAccountRuleConfigDtlDTOList(accountRuleConfigDtls);
            bucket.set(dbConfig);
            return dbConfig;
        }
        return null;
    }

    @Override
    public PagingVO<CreditAccountRuleConfigPageVO> searchPageWhihoutName(CreditAccountRuleConfigQueryParam param) {
        PagingVO<CreditAccountRuleConfigPageVO> accountRuleConfigPageVOPagingVO = accountRuleConfigDomainService.searchPage(param);
        /*if (CollectionUtil.isNotEmpty(accountRuleConfigPageVOPagingVO.getRecords())) {
            convertPage(accountRuleConfigPageVOPagingVO.getRecords());
        }
        pageCode2Name(accountRuleConfigPageVOPagingVO.getRecords());*/
        return accountRuleConfigPageVOPagingVO;
    }

    private void convertPage(List<CreditAccountRuleConfigPageVO> accountRuleConfigDtls) {
        List<CreditAccountIocDTO> accountIocNameList = accountIocDomainService.findByCodeBatch(accountRuleConfigDtls.stream().map(CreditAccountRuleConfigPageVO::getIoCode).collect(Collectors.toList()));
        Map<String, String> ioCodeMap = new HashMap<>();
        Map<String, String> ioTypeMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(accountIocNameList)) {
            ioTypeMap = accountIocNameList.stream().collect(Collectors.toMap(CreditAccountIocDTO::getIoType, i -> i.getIoTypeName() == null ? "" : i.getIoTypeName(), (key1, key2) -> key2));
            ioCodeMap = accountIocNameList.stream().collect(Collectors.toMap(CreditAccountIocDTO::getIoCode, i -> i.getIoName() == null ? "" : i.getIoName(), (key1, key2) -> key2));
        }
        for (CreditAccountRuleConfigPageVO configDtl : accountRuleConfigDtls) {
            if (StringUtils.isNotBlank(configDtl.getIoCode())) {
                configDtl.setIoName(ioCodeMap.get(configDtl.getIoCode()));
            }
        }
    }

}
