package com.elitesland.fin.application.service.paymentperiod;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.paymentperiod.ReceiptPaymentAgreementConvert;
import com.elitesland.fin.application.convert.paymentperiod.ReceiptPaymentAgreementDtlConvert;
import com.elitesland.fin.application.facade.param.paymentperiod.ReceiptPaymentAgreementComPageParam;
import com.elitesland.fin.application.facade.param.paymentperiod.ReceiptPaymentAgreementDtlSaveParam;
import com.elitesland.fin.application.facade.param.paymentperiod.ReceiptPaymentAgreementPageParam;
import com.elitesland.fin.application.facade.param.paymentperiod.ReceiptPaymentAgreementSaveParam;
import com.elitesland.fin.application.facade.vo.artype.ArTypeDtlVO;
import com.elitesland.fin.application.facade.vo.paymentperiod.ReceiptPaymentAgreementComPagingVO;
import com.elitesland.fin.application.facade.vo.paymentperiod.ReceiptPaymentAgreementDtlVO;
import com.elitesland.fin.application.facade.vo.paymentperiod.ReceiptPaymentAgreementPagingVO;
import com.elitesland.fin.application.facade.vo.paymentperiod.ReceiptPaymentAgreementVO;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.SysNumberGenerator;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.paymentperiod.ReceiptPaymentAgreementDtlDO;
import com.elitesland.fin.domain.service.paymentperiod.ReceiptPaymentAgreementDomainService;
import com.elitesland.fin.domain.service.paymentperiod.ReceiptPaymentAgreementDtlDomainService;
import com.elitesland.fin.infr.repo.paymentperiod.ReceiptPaymentAgreementDtlRepo;
import com.elitesland.fin.infr.repo.paymentperiod.ReceiptPaymentAgreementDtlRepoProc;
import com.elitesland.fin.infr.repo.paymentperiod.ReceiptPaymentAgreementRepo;
import com.elitesland.fin.infr.repo.paymentperiod.ReceiptPaymentAgreementRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;


/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2024/04/11
 * @Version 1.0
 * @Content:
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ReceiptPaymentAgreementServiceImpl implements ReceiptPaymentAgreementService {

    private final ReceiptPaymentAgreementDomainService domainService;
    private final ReceiptPaymentAgreementDtlDomainService dtlDomainService;
    private final ReceiptPaymentAgreementRepo receiptPaymentAgreementRepo;
    private final ReceiptPaymentAgreementRepoProc receiptPaymentAgreementRepoProc;
    private final ReceiptPaymentAgreementDtlRepo receiptPaymentAgreementDtlRepo;
    private final ReceiptPaymentAgreementDtlRepoProc receiptPaymentAgreementDtlRepoProc;
    private final SysNumberGenerator sysNumberGeneratorWrapper;


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long save(ReceiptPaymentAgreementSaveParam saveParam){
        List<ReceiptPaymentAgreementDtlSaveParam> dtlSaveParamList = saveParam.getDtlSaveParamList();
        //校验必填项
        checkMandatoryField(saveParam);
        checkMandatoryFieldDtlList(dtlSaveParamList);

        //更新时校验ID是否正确
        ReceiptPaymentAgreementVO receiptPaymentAgreementVO = null;
        Boolean flag = true;
        if (Objects.nonNull(saveParam.getId())) {
            receiptPaymentAgreementVO = domainService.findById(saveParam.getId());
            //修改时的业务校验逻辑
            checkUpdateBusiness(receiptPaymentAgreementVO);
            flag = false;
        }

        //业务校验逻辑
        checkAllowShipItemGroup(saveParam, flag);
        checkAllowShipItemGroupDtl(saveParam, dtlSaveParamList, flag);

        //新增/修改保存时默认值
        saveDefaultValue(flag, saveParam, receiptPaymentAgreementVO);

        // ACTIVE_STATUS 启用停用
        //保存任务主表
        Long id = domainService.save(saveParam);
        saveParam.setId(id);

        //明细全删全插,先删除
        dtlDomainService.deleteByMasId(id);
        //明细保存时默认值
        saveDtlDefaultValue(dtlSaveParamList, saveParam);
        //保存明细
        dtlDomainService.saveDtl(dtlSaveParamList);

        return id;
    }

    /**
     * 新增/修改保存时默认值
     *
     * @param flag   新增/修改标识 true 新增 false 修改
     * @param saveVO 入参
     * @param respVO 数据库旧数据
     * @return
     */
    private void saveDefaultValue(Boolean flag, ReceiptPaymentAgreementSaveParam saveVO, ReceiptPaymentAgreementVO respVO) {

        if (flag) {
            //新增时-发号器新增编码
            if (StringUtils.isBlank(saveVO.getProtocolCode())){
                saveVO.setProtocolCode(getProtocolCode());
            }
        } else {
            //修改 协议类型和协议编码不允许编辑
            if (StringUtils.isBlank(saveVO.getProtocolCode())) {
                saveVO.setProtocolCode(respVO.getProtocolCode());
            }
            if (StringUtils.isBlank(saveVO.getProtocolType())) {
                saveVO.setProtocolType(respVO.getProtocolType());
            }
        }
       /* if (Objects.isNull(saveVO.getDeleteFlag())) {
            saveVO.setDeleteFlag(0);
        }*/
        //状态:默认为启用
        if (StringUtils.isBlank(saveVO.getStatus())) {
            saveVO.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        }


    }

    /**
     * 明细新增/修改时默认值
     *
     * @param dtlSaveVOList 明细入参
     * @param saveVO        主表信息
     * @return
     */
    private void saveDtlDefaultValue(List<ReceiptPaymentAgreementDtlSaveParam> dtlSaveVOList, ReceiptPaymentAgreementSaveParam saveVO) {
        dtlSaveVOList.forEach(dtlSaveVO -> {
            dtlSaveVO.setMasId(saveVO.getId());
        });
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long update(ReceiptPaymentAgreementSaveParam saveParam){

        return null;
    }



    /**
     * 收付款协议主表的业务校验逻辑
     *
     * @param saveVO 入参
     * @param flag   新增/修改标识 true 新增 false 修改
     * @return
     */
    private void checkAllowShipItemGroup(ReceiptPaymentAgreementSaveParam saveVO, Boolean flag) {
        ReceiptPaymentAgreementPageParam pageParam1 = new ReceiptPaymentAgreementPageParam();
        pageParam1.setProtocolCode(saveVO.getProtocolCode());
        pageParam1.setProtocolType(saveVO.getProtocolType());
        ReceiptPaymentAgreementVO agreementVO1 = domainService.checkSelectByParam(pageParam1);
        if (flag) {
            if (Objects.nonNull(agreementVO1) && Objects.equals(agreementVO1.getProtocolCode(), saveVO.getProtocolCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, judgmentPrompt(saveVO.getProtocolType())+"编码已存在!");
            }
        } else {
            if (Objects.nonNull(agreementVO1) && Objects.equals(agreementVO1.getProtocolCode(), saveVO.getProtocolCode())
                    && !Objects.equals(saveVO.getId(), agreementVO1.getId())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, judgmentPrompt(saveVO.getProtocolType())+"编码已存在!");
            }
        }

        ReceiptPaymentAgreementPageParam pageParam2 = new ReceiptPaymentAgreementPageParam();
        pageParam2.setProtocolName(saveVO.getProtocolName());
        pageParam2.setProtocolType(saveVO.getProtocolType());
        ReceiptPaymentAgreementVO agreementVO2 = domainService.checkSelectByParam(pageParam2);
        if (flag) {
            if (Objects.nonNull(agreementVO2) && Objects.equals(agreementVO2.getProtocolName(), saveVO.getProtocolName())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, judgmentPrompt(saveVO.getProtocolType())+"名称已存在!");
            }
        } else {
            if (Objects.nonNull(agreementVO2) && Objects.equals(agreementVO2.getProtocolName(), saveVO.getProtocolName())
                    && !Objects.equals(saveVO.getId(), agreementVO2.getId())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, judgmentPrompt(saveVO.getProtocolType())+"名称已存在!");
            }
        }
    }

    /**
     * 收付款协议明细的业务校验逻辑
     *
     * @param saveVO        主表入参
     * @param dtlSaveVOList 明细入参
     * @param flag          新增/修改标识 true 新增 false 修改
     * @return
     */
    private void checkAllowShipItemGroupDtl(ReceiptPaymentAgreementSaveParam saveVO, List<ReceiptPaymentAgreementDtlSaveParam> dtlSaveVOList, Boolean flag) {


    }

    private String judgmentPrompt(String type){
        if (Objects.equals(type,UdcEnum.AGREEMENT_PROTOCOL_TYPE_RECEIPT.getValueCode())){
            return UdcEnum.AGREEMENT_PROTOCOL_TYPE_RECEIPT.getValueCodeName();
        } else if (Objects.equals(type,UdcEnum.AGREEMENT_PROTOCOL_TYPE_PAYMENT.getValueCode())){
            return UdcEnum.AGREEMENT_PROTOCOL_TYPE_PAYMENT.getValueCodeName();
        }

        return "";
    }



    /**
     * 修改时的业务校验逻辑
     *
     * @param respVO 数据库旧数据信息
     * @return
     */
    private void checkUpdateBusiness(ReceiptPaymentAgreementVO respVO) {
        //启用状态，支持编辑修改相关信息。
        if (!Objects.equals(respVO.getStatus(), UdcEnum.FIN_ACTIVE_STATUS_CLOSED.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只有禁用状态的才可编辑修改!");
        }

    }


    /**
     * 校验必填项:收付款主表
     *
     * @param saveVO 入参
     */
    private void checkMandatoryField(ReceiptPaymentAgreementSaveParam saveVO) {
        if (StringUtils.isBlank(saveVO.getProtocolName())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "协议名称为空!");
        }
        if (StringUtils.isBlank(saveVO.getProtocolType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "协议类型为空!");
        }
    }

    /**
     * 校验必填项:收付款明细
     *
     * @param saveVOList 入参
     */
    private void checkMandatoryFieldDtlList(List<ReceiptPaymentAgreementDtlSaveParam> saveVOList) {
        /*if (CollectionUtils.isEmpty(saveVOList)) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "明细数据为空!");
        }*/

        if (!CollectionUtils.isEmpty(saveVOList)) {
            saveVOList.forEach(saveVO -> {
                checkMandatoryFieldDtl(saveVO);
            });
        }

    }

    /**
     * 校验必填项:收付款明细
     *
     * @param saveVO 入参
     */
    private void checkMandatoryFieldDtl(ReceiptPaymentAgreementDtlSaveParam saveVO) {
        if (StringUtils.isBlank(saveVO.getPeriodType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账期类型为空!");
        }
        if (StringUtils.isBlank(saveVO.getStartEffectiveDate())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "起效日期为空!");
        }

        if (Objects.equals(saveVO.getPeriodType(), UdcEnum.AGREEMENT_PERIOD_TYPE_IMMOBILIZATION.getValueCode())){

            if (Objects.isNull(saveVO.getEffectiveAdditionalMonth())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账期类型=固定时,生效附加月必填!");
            }
            if (Objects.isNull(saveVO.getEffectiveDate())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账期类型=固定时,生效日必填!");
            }
        }else if(Objects.equals(saveVO.getPeriodType(), UdcEnum.AGREEMENT_PERIOD_TYPE_ROUTINE.getValueCode())){
            if (Objects.isNull(saveVO.getValidityDateDelayDays())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账期类型=常规时,起效日期延迟天数必填!");
            }
            if (Objects.isNull(saveVO.getPeriodDays())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账期类型=常规时,账期天数必填!");
            }
        }

    }

    /**
     * 根据发号器规则自动生成协议编码
     *
     * @param
     * @return 协议编码
     */
    private String getProtocolCode() {
        String code = sysNumberGeneratorWrapper.generate(FinConstant.AGREEMENT_PROTOCOL_CODE);
        if (StringUtils.isBlank(code)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "生成收付款协议编码失败!");
        }
        return code;
    }



    /**
     * 收付款协议分页查询
     *
     * @param pageParam 入参
     * @return 收付款协议信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<ReceiptPaymentAgreementPagingVO> page(ReceiptPaymentAgreementPageParam pageParam) {

        return domainService.page(pageParam);
    }

    /**
     * 根据收付款协议ID查询收付款协议详情数据
     *
     * @param id 收付款协议ID
     * @return 收付款协议详情数据
     */
    @Override
    @SysCodeProc
    public ReceiptPaymentAgreementVO findIdOne(Long id) {
        ReceiptPaymentAgreementVO agreementVO = domainService.selectById(id);
        List<ReceiptPaymentAgreementDtlVO> agreementDtlVOList = dtlDomainService.selectByMasId(agreementVO.getId());

        if (CollectionUtils.isEmpty(agreementDtlVOList)) {
            return agreementVO;
        }
        translateDtl(agreementDtlVOList);

        agreementVO.setDtlVOList(agreementDtlVOList);

        return agreementVO;
    }

    /**
     * 根据收付款协议编码查询收付款协议详情数据
     *
     * @param code 收付款协议编码
     * @return 收付款协议详情数据
     */
    @Override
    @SysCodeProc
    public ReceiptPaymentAgreementVO findCodeOne(String code) {
        ReceiptPaymentAgreementVO agreementVO = domainService.selectByCode(code);
        List<ReceiptPaymentAgreementDtlVO> agreementDtlVOList = dtlDomainService.selectByMasId(agreementVO.getId());

        if (CollectionUtils.isEmpty(agreementDtlVOList)) {
            return agreementVO;
        }
        translateDtl(agreementDtlVOList);

        agreementVO.setDtlVOList(agreementDtlVOList);

        return agreementVO;
    }

    /**
     * 组装填充收付款协议明细的相关关联字段信息
     *
     * @param respVOList 收付款协议数据信息
     * @return
     */
    private void translateDtl(List<ReceiptPaymentAgreementDtlVO> respVOList) {

    }

    /**
     * 根据收付款协议ID批量删除
     *
     * @param ids 收付款协议ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteBatch(List<Long> ids) {

        ReceiptPaymentAgreementPageParam queryVO = new ReceiptPaymentAgreementPageParam();
        queryVO.setIdList(ids);
        List<ReceiptPaymentAgreementVO> respVOList = domainService.selectByParam(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        //TODO 禁用状态批量删除，删除时检查 收款协议是否被订单/对账单/应收单引用；付款协议是否被采购订单/对账单/应付单引用，若被引用不允许删除
        List<Long> idList = respVOList.stream().map(ReceiptPaymentAgreementVO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        //逻辑删除主表
        domainService.updateDeleteFlagBatch(1, idList);
        //逻辑删除明细
        dtlDomainService.updateDeleteFlagBatch(1, idList);
    }


    /**
     * 根据收付款协议ID批量启用
     *
     * @param ids 收付款协议ID集合
     * @return
     */
    @Override
    public void enableBatch(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择需要启用的数据!");
        }
        ReceiptPaymentAgreementPageParam checkEqualParam = new ReceiptPaymentAgreementPageParam();
        checkEqualParam.setIdList(ids);
        checkEqualParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<ReceiptPaymentAgreementVO> checkEqualList = domainService.selectByParam(checkEqualParam);
        if (!CollectionUtils.isEmpty(checkEqualList)) {
            String checkItemResult = checkEqualList.stream().map(checkVO ->
                    judgmentPrompt(checkVO.getProtocolType())+":(" + checkVO.getProtocolName() + ")"
            ).collect(Collectors.joining(";", "已启用的不可重复启用.[", "],请检查!"));
            throw new BusinessException(ApiCode.FAIL, checkItemResult);
        }

        domainService.updateStatusBatch(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), ids);

    }


    /**
     * 根据收付款协议ID批量停用
     *
     * @param ids 收付款协议ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void disableBatch(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择需要禁用的数据!");
        }
        ReceiptPaymentAgreementPageParam checkEqualParam = new ReceiptPaymentAgreementPageParam();
        checkEqualParam.setIdList(ids);
        checkEqualParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_CLOSED.getValueCode());
        List<ReceiptPaymentAgreementVO> checkEqualList = domainService.selectByParam(checkEqualParam);
        if (!CollectionUtils.isEmpty(checkEqualList)) {
            String checkItemResult = checkEqualList.stream().map(checkVO ->
                    judgmentPrompt(checkVO.getProtocolType())+":(" + checkVO.getProtocolName() + ")"
            ).collect(Collectors.joining(";", "已禁用的不可重复禁用.[", "],请检查!"));
            throw new BusinessException(ApiCode.FAIL, checkItemResult);
        }
        domainService.updateStatusBatch(UdcEnum.FIN_ACTIVE_STATUS_CLOSED.getValueCode(), ids);
    }



    @Override
    @SysCodeProc
    public PagingVO<ReceiptPaymentAgreementComPagingVO> receiptCommonPage(ReceiptPaymentAgreementComPageParam comPageParam) {
        ReceiptPaymentAgreementPageParam pageParam = ReceiptPaymentAgreementConvert.INSTANCE.comToPageParam(comPageParam);
        pageParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        pageParam.setProtocolType(UdcEnum.AGREEMENT_PROTOCOL_TYPE_RECEIPT.getValueCode());
        PagingVO<ReceiptPaymentAgreementPagingVO> pagingVO = domainService.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<ReceiptPaymentAgreementComPagingVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        List<ReceiptPaymentAgreementComPagingVO> comPagingVOList = pagingVO.getRecords().stream().map(ReceiptPaymentAgreementConvert.INSTANCE::voToComPagingVo).collect(Collectors.toList());


        return PagingVO.<ReceiptPaymentAgreementComPagingVO>builder()
                .total(pagingVO.getTotal())
                .records(comPagingVOList)
                .build();
    }


    @Override
    @SysCodeProc
    public PagingVO<ReceiptPaymentAgreementComPagingVO> paymentCommonPage(ReceiptPaymentAgreementComPageParam comPageParam) {

        ReceiptPaymentAgreementPageParam pageParam = ReceiptPaymentAgreementConvert.INSTANCE.comToPageParam(comPageParam);
        pageParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        pageParam.setProtocolType(UdcEnum.AGREEMENT_PROTOCOL_TYPE_PAYMENT.getValueCode());
        PagingVO<ReceiptPaymentAgreementPagingVO> pagingVO = domainService.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<ReceiptPaymentAgreementComPagingVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        List<ReceiptPaymentAgreementComPagingVO> comPagingVOList = pagingVO.getRecords().stream().map(ReceiptPaymentAgreementConvert.INSTANCE::voToComPagingVo).collect(Collectors.toList());


        return PagingVO.<ReceiptPaymentAgreementComPagingVO>builder()
                .total(pagingVO.getTotal())
                .records(comPagingVOList)
                .build();
    }

    @Override
    @SysCodeProc
    public PagingVO<ReceiptPaymentAgreementComPagingVO> commonPage(ReceiptPaymentAgreementComPageParam comPageParam){
        ReceiptPaymentAgreementPageParam pageParam = ReceiptPaymentAgreementConvert.INSTANCE.comToPageParam(comPageParam);

        PagingVO<ReceiptPaymentAgreementPagingVO> pagingVO = domainService.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<ReceiptPaymentAgreementComPagingVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        List<ReceiptPaymentAgreementComPagingVO> comPagingVOList = pagingVO.getRecords().stream().map(ReceiptPaymentAgreementConvert.INSTANCE::voToComPagingVo).collect(Collectors.toList());


        return PagingVO.<ReceiptPaymentAgreementComPagingVO>builder()
                .total(pagingVO.getTotal())
                .records(comPagingVOList)
                .build();
    }
}
