package com.elitesland.fin.domain.service.artype;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitesland.fin.application.convert.artype.ArTypeConvert;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.domain.entity.artype.ArType;
import com.elitesland.fin.domain.entity.artype.ArTypeDO;
import com.elitesland.fin.domain.entity.artype.ArTypeDtl;
import com.elitesland.fin.domain.param.artype.ArTypeDtlPageParam;
import com.elitesland.fin.domain.param.artype.ArTypePageParam;
import com.elitesland.fin.infr.dto.artype.ArTypeDTO;
import com.elitesland.fin.infr.dto.artype.ArTypeDtlDTO;
import com.elitesland.fin.infr.repo.artype.ArTypeOuRepoProc;
import com.elitesland.fin.infr.repo.artype.ArTypeRepo;
import com.elitesland.fin.infr.repo.artype.ArTypeRepoProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitesland.inv.dto.invstk.InvIoFinReceiptRpcDTO;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.domain.service.artype
 * @date 2022/4/11 13:42
 */
@Service
@RequiredArgsConstructor
public class ArTypeDomainServiceImpl implements ArTypeDomainService {

    private final ArTypeRepo arTypeRepo;
    private final ArTypeRepoProc arTypeRepoProc;
    private final ArTypeOuRepoProc arTypeOuRepoProc;
    private final ArTypeDtlDomainService arTypeDtlDomainService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long save(ArType arType) {
        arType.checkNotNull();
        arType.setDefaultValue();
        checkArType(arType);
        // 编码和名称是否存在
        Boolean isExistName = arTypeRepoProc.existsName(arType.getArTypeName());
        if (isExistName) {
            throw new BusinessException("应收单类型名称已存在");
        }
        Boolean isExistCode = arTypeRepoProc.existsCode(arType.getArTypeCode());
        if (isExistCode) {
            throw new BusinessException("应收单类型编码已存在");
        }

        // 如果第一条新增，则为默认
        Boolean isFir = arTypeRepoProc.isFir();
        if (isFir) {
            arType.setDefaultFlag(true);
        }
        ArTypeDO res = arTypeRepo.save(ArTypeConvert.INSTANCE.convert(arType));
        Long masId = res.getId();

        //明细全删全插
        if (Objects.nonNull(arType.getId())){
            arTypeDtlDomainService.deleteByIdBatch(Collections.singletonList(arType.getId()));
        }

        if (CollectionUtil.isNotEmpty(arType.getDtlList())){
            List<ArTypeDtl> arTypeDtlList = arType.getDtlList();
            arTypeDtlList.forEach(arTypeDtl -> {
                arTypeDtl.setMasId(masId);
            });
            arTypeDtlDomainService.save(arTypeDtlList);
        }

        return masId;
    }

    private void checkArType(ArType arType){
        //保存校验时  来源单据+来源单据类型+来源单据状态+分配的公司都重复时才报错
        List<ArTypeDtl> dtlList = arType.getDtlList();
        if (CollectionUtil.isEmpty(dtlList)){
            return;
        }
        List<String> sourceDocList = dtlList.stream().map(ArTypeDtl::getSourceDoc).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<String> sourceDocTypeList = dtlList.stream().map(ArTypeDtl::getSourceDocType).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<String> sourceDocStatusList = dtlList.stream().map(ArTypeDtl::getSourceDocStatus).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        ArTypeDtlPageParam arTypeDtlPageParam = new ArTypeDtlPageParam();
        arTypeDtlPageParam.setSourceDocList(sourceDocList);
        arTypeDtlPageParam.setSourceDocTypeList(sourceDocTypeList);
        arTypeDtlPageParam.setSourceDocStatusList(sourceDocStatusList);
        List<ArTypeDtlDTO> arTypeDtlDTOList = arTypeDtlDomainService.selectListByParam(arTypeDtlPageParam);
        if (CollectionUtil.isEmpty(arTypeDtlDTOList)){
            return;
        }

        Map<String, List<ArTypeDtlDTO>> arTypeDtlListMap = arTypeDtlDTOList.stream().collect(Collectors.groupingBy(i -> arSpliceGroupKey(
                i.getSourceDoc(),i.getSourceDocType(),i.getSourceDocStatus())));

        //查公司,公司是保存后分配的，去掉
        List<Long> masIdList = arTypeDtlDTOList.stream().map(ArTypeDtlDTO::getMasId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<ArTypeDTO> arTypeDTOList = arTypeRepoProc.findByIds(masIdList);

        dtlList.forEach(arTypeDtl -> {
            String spliceGroupKey = arSpliceGroupKey(arTypeDtl.getSourceDoc(),arTypeDtl.getSourceDocType(),arTypeDtl.getSourceDocStatus());
            List<ArTypeDtlDTO> spliceGroupList = arTypeDtlListMap.get(spliceGroupKey);
            if (CollectionUtil.isNotEmpty(spliceGroupList)){
                List<Long> spliceGroupMasIdList = spliceGroupList.stream().map(ArTypeDtlDTO::getMasId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                Optional<ArTypeDTO> arTypeDTOOptional = arTypeDTOList.stream().filter(arTypeDTO -> spliceGroupMasIdList.contains(arTypeDTO.getId())
                        && !Objects.equals(arTypeDTO.getId(),arType.getId())).findFirst();
                if (arTypeDTOOptional.isPresent()){
                    throw new BusinessException("来源单据("+arTypeDtl.getSourceDoc()+")-" +
                            "来源单据类型("+arTypeDtl.getSourceDocType()+")-" +
                            "来源单据状态("+arTypeDtl.getSourceDocStatus()+")与" +
                            "应收单类型配置("+arTypeDTOOptional.get().getArTypeCode()+")重复");
                }
            }
        });

    }

    private String arSpliceGroupKey(String sourceDoc,String sourceDocType,String sourceDocStatus){
        String spliceGroupKey = sourceDoc + FinConstant.LINE_SPLIT +
                sourceDocType + FinConstant.LINE_SPLIT +
                sourceDocStatus;
        return spliceGroupKey;
    }

    @Override
    @SysCodeProc
    public ArTypeDTO get(Long id) {
        ArTypeDTO res = arTypeRepoProc.get(id);
        return res;
    }

    @Override
    @SysCodeProc
    public PagingVO<ArTypeDTO> page(ArTypePageParam arTypePageParam) {

        List<Long> artypeIds = new ArrayList<>();
        if (null != arTypePageParam.getOuId()) {
            artypeIds = arTypeOuRepoProc.queryByOuId(arTypePageParam.getOuId());
        }
        PagingVO<ArTypeDTO> res = arTypeRepoProc.page(arTypePageParam, artypeIds);
        return res;
    }

    @Override
    @SysCodeProc
    public List<ArTypeDTO> findByIds(List<Long> ids) {
        List<ArTypeDTO> res = arTypeRepoProc.findByIds(ids);
        return res;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean updateEnableFlag(List<Long> ids, boolean enable) {
        arTypeRepoProc.updateEnableFlag(ids, enable);
        return true;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long updateDef(Long id) {
        arTypeRepoProc.updateDef();
        Long res = arTypeRepoProc.updateDefById(id);
        return res;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long del(Long id) {
        Assert.notNull(id, "id不能为空");
        // 停用状态和默认状态不能被删除
        ArTypeDTO arTypeDTO = arTypeRepoProc.findById(id);
        if (arTypeDTO.getEnableFlag() || arTypeDTO.getDefaultFlag()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能删除未启用的数据和非默认的数据");
        }
        Long res = arTypeRepoProc.del(id);
        return res;
    }

    @Override
    @SysCodeProc
    public List<ArTypeDTO> getList() {
        List<ArTypeDTO> res = arTypeRepoProc.getList();
        return res;
    }

    @Override
    @SysCodeProc
    public ArTypeDTO getDef() {
        ArTypeDTO res = arTypeRepoProc.getDef();
        return res;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long update(ArType arType) {
        arType.checkNotNull();
        checkArType(arType);
        Assert.notNull(arType.getDefaultFlag(), "defaultFlag is null");
        Assert.notNull(arType.getAutoAudit(), "autoAudit is null");
        Assert.notNull(arType.getEnableFlag(), "enableFlag is null");
        ArTypeDO res = arTypeRepo.save(ArTypeConvert.INSTANCE.convert(arType));

        Long masId = res.getId();

        //明细全删全插
        if (Objects.nonNull(arType.getId())){
            arTypeDtlDomainService.deleteByIdBatch(Collections.singletonList(arType.getId()));
        }

        if (CollectionUtil.isNotEmpty(arType.getDtlList())){
            List<ArTypeDtl> arTypeDtlList = arType.getDtlList();
            arTypeDtlList.forEach(arTypeDtl -> {
                arTypeDtl.setMasId(masId);
            });
            arTypeDtlDomainService.save(arTypeDtlList);
        }

        return masId;
    }

    @Override
    public Boolean queryIsAuto(Long id) {
        return arTypeRepoProc.queryIsAutoAudit(id);
    }

    @Override
    @SysCodeProc
    public List<ArTypeDTO> arOrderAutoSelectMatchByParam(ArTypePageParam arTypePageParam) {
        return arTypeRepoProc.arOrderAutoSelectMatchByParam(arTypePageParam);
    }

}
