package com.elitesland.fin.domain.service.recorder;

import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.core.seq.SeqNumProvider;
import com.elitesland.fin.application.convert.recorder.RecOrderConvert;
import com.elitesland.fin.application.convert.recorder.RecOrderDtlConvert;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinFlexFieldCodeConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.recorder.RecOrder;
import com.elitesland.fin.domain.entity.recorder.RecOrderDO;
import com.elitesland.fin.domain.entity.recorder.RecOrderDtl;
import com.elitesland.fin.domain.entity.recorder.RecOrderDtlDO;
import com.elitesland.fin.domain.param.recorder.RecOrderPageParam;
import com.elitesland.fin.domain.param.recorder.RecOrderParam;
import com.elitesland.fin.infr.dto.common.ArVerDTO;
import com.elitesland.fin.infr.dto.recorder.RecOrderDTO;
import com.elitesland.fin.infr.dto.recorder.RecOrderDtlDTO;
import com.elitesland.fin.infr.factory.recorder.RecOrderFactory;
import com.elitesland.fin.infr.repo.recorder.RecOrderDtlRepo;
import com.elitesland.fin.infr.repo.recorder.RecOrderDtlRepoProc;
import com.elitesland.fin.infr.repo.recorder.RecOrderRepo;
import com.elitesland.fin.infr.repo.recorder.RecOrderRepoProc;
import com.elitesland.fin.utils.BeanUtils;
import com.elitesland.fin.utils.BigDecimalUtils;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.workflow.ProcessInfo;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author zhiyu.he
 * @date 2022/4/13 10:32
 */
@Service
@RequiredArgsConstructor
public class RecOrderDomainServiceImpl implements RecOrderDomainService {

    private final RecOrderRepoProc recOrderRepoProc;

    private final RecOrderFactory recOrderFactory;

    private final RecOrderRepo recOrderRepo;

    private final RecOrderDtlRepoProc recOrderDtlRepoProc;

    private final RecOrderDtlRepo recOrderDtlRepo;

    //发号器生成付款单号
    private final SeqNumProvider sysNumberRuleService;

    private final FlexFieldUtilService flexFieldUtilService;

    @Override
    public PagingVO<RecOrderDTO> page(RecOrderPageParam param) {
        if (param.getReDateStart() == null || param.getReDateEnd() == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择收款日期!");
        }
        return recOrderFactory.recOrderPage(param);
    }
    @Override
    public PagingVO<RecOrderDTO> writeoffPage(RecOrderPageParam param) {
       /* if (param.getReDateStart() == null || param.getReDateEnd() == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择收款日期!");
        }*/
        //查询待核销
        return recOrderFactory.writeoffPage(param);
    }
    @Override
    public RecOrderDTO queryById(Long id, Boolean flag) {
        if (flag) {
            RecOrderDTO payOrderDTO = recOrderRepoProc.queryById(id);
            List<RecOrderDtlDTO> recOrderDtlDTOS = recOrderDtlRepoProc.queryByMasId(id);
            payOrderDTO.setDtlDTOS(recOrderDtlDTOS);
            return payOrderDTO;
        } else {
            return recOrderRepoProc.queryById(id);
        }
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> deleteByIds(List<Long> ids) {
        List<RecOrderDTO> recOrderDTOS = recOrderRepoProc.queryByIds(ids);
        recOrderDTOS.forEach(dto -> {
            if (!dto.getOrderState().equals(UdcEnum.APPLY_STATUS_DRAFT.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能删除状态为草稿的收款单!");
            }
            //红冲原单处理
            if (Objects.nonNull(dto.getRedSourceNo())) {
                recOrderRepoProc.setRedFlag(dto.getRedSourceId(),false);
            }
        });
        //删除主表数据
        recOrderRepoProc.deleteById(ids);
        recOrderDtlRepoProc.deleteByMasId(ids);
        return ids;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long save(RecOrder recOrder) {
        //保存逻辑
        //无ID则为新增
        if (recOrder.getId() != null) {
            recOrder.checkOrderState();
        } else {
            //新增数据才会生成付款单号
            String recOrderNo = sysNumberRuleService.generateCode(FinConstant.FIN,FinConstant.SKD, null);
            recOrder.setRecOrderNo(recOrderNo);
        }
        //默认核销状态
        if (recOrder.getVerAmt() == null) {
            recOrder.defaultVer();
        }
        //校验非空
        recOrder.checkNotNull(recOrder.getCreateMode());
        // 非手工
        if (!recOrder.getCreateMode().equals(UdcEnum.FIN_REC_DOC_CLS_MANU.getValueCode())) {
            //校验金额不能为0
            if (recOrder.getId() == null) {
                recOrder.setSoDef();
                recOrder.checkDtl(false);
                recOrder.countBySo();
                recOrder.checkTotalMoney();
            }
        } else {
            //校验明细是否为空  金额是否为0
            recOrder.checkTotalMoney();
        }
        recOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        //如果来源是应付单走应付单逻辑
        return updateOrSave(recOrder).getId();
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long submit(RecOrder recOrder, Boolean auto) {
        //无ID则为新增  校验订单状态
        if (recOrder.getId() != null) {
            recOrder.checkOrderState();
        } else {
            //新增数据才会生成付款单号
            String payOrderNo = sysNumberRuleService.generateCode(FinConstant.FIN,FinConstant.SKD, null);
            recOrder.setRecOrderNo(payOrderNo);
        }
        //校验表头非空项
        recOrder.checkNotNull(recOrder.getCreateMode());
        //无来源都要检验是否为0金额
        if (!recOrder.getCreateMode().equals(UdcEnum.FIN_PAY_DOC_CLS_MANU.getValueCode())) {
            //校验金额不能为0
            recOrder.checkTotalMoney();
        }
        //校验所有金额
        recOrder.checkMoney();
        //校验明细是否为空
        recOrder.checkDtl(true);
        if (recOrder.getVerAmt() == null) {
            //默认核销状态
            recOrder.defaultVer();
        }
        if (auto) {
            recOrder.setProposedStatus(UdcEnum.DOC_PROPOSED_STATUS_DRAFT.getValueCode());
            recOrder.setOrderState(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
        } else {
            recOrder.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
        }
        RecOrderDO recOrderDO = updateOrSave(recOrder);
        return recOrderDO.getId();
    }

    @Override
    public List<RecOrderDTO> queryByRecTypeId(List<Long> recTypeId) {
        return recOrderRepoProc.queryByRecTypeId(recTypeId);
    }

    @Override
    public List<RecOrderDTO> queryByParam(RecOrderParam recOrderParam) {
        return recOrderRepoProc.getRecOrderList(recOrderParam);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void updateVerAmt(Long id, BigDecimal amt) {
        ArVerDTO arVerDTO = recOrderRepoProc.queryVerAmtById(id);
        BigDecimal totalAmt = arVerDTO.getTotalAmt();
        if (arVerDTO.getVerAmt().add(amt).compareTo(totalAmt) > 0) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前核销金额大于收款单金额，请刷新页面重新输入");
        }
        arVerDTO.setAmtAndVerState(amt);
        long rows = recOrderRepoProc.updateVerAmt(arVerDTO);
        if (rows == 0) {
            throw new BusinessException("收款单单更新已核销金额失败");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateWorkInfo(ProcessInfo processInfo, Long id) {
        recOrderRepoProc.updateWorkInfo(processInfo, id);
    }

    @Override
    public RecOrder redPunchCreate(Long id) {
        RecOrderDO recOrderDO = recOrderRepo.getReferenceById(id);
        recOrderDO.setRedState(Boolean.TRUE);
        recOrderRepo.save(recOrderDO);
        RecOrder newRecOrder = BeanUtils.toBean(recOrderDO, RecOrder.class);

        newRecOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        newRecOrder.setProcInstId(null);
        newRecOrder.setProcInstStatus(null);
        newRecOrder.setRedState(null);
        newRecOrder.setRedSourceNo(recOrderDO.getRecOrderNo());
        newRecOrder.setRedSourceId(recOrderDO.getId());
        newRecOrder.setTaxAmt(BigDecimalUtils.negate(newRecOrder.getTaxAmt()));
        newRecOrder.setTotalAmt(BigDecimalUtils.negate(newRecOrder.getTotalAmt()));
        newRecOrder.setRealRecAmt(BigDecimalUtils.negate(newRecOrder.getRealRecAmt()));
        newRecOrder.setTotalCurAmt(BigDecimalUtils.negate(newRecOrder.getTotalCurAmt()));
        newRecOrder.setTaxCurAmt(BigDecimalUtils.negate(newRecOrder.getTaxCurAmt()));
        newRecOrder.setRealRecCurAmt(BigDecimalUtils.negate(newRecOrder.getRealRecCurAmt()));
        newRecOrder.setId(null);
        newRecOrder.setRecOrderNo(null);
        newRecOrder.setProposedStatus(null);
        newRecOrder.setAuditUser(null);
        newRecOrder.setApprovedTime(null);
        newRecOrder.setAuditUserId(null);
        newRecOrder.setRemark(null);

        List<RecOrderDtlDO> dtls = recOrderDtlRepo.findAllByMasId(id);
        List<RecOrderDtl> newDtls = dtls.stream().map(dtl -> {

            RecOrderDtl dtlDO = BeanUtils.toBean(dtl, RecOrderDtl.class);
            dtlDO.setId(null);
            dtlDO.setTaxAmt(BigDecimalUtils.negate(dtl.getTaxAmt()));
            dtlDO.setTotalAmt(BigDecimalUtils.negate(dtl.getTotalAmt()));
            dtlDO.setRealRecAmt(BigDecimalUtils.negate(dtl.getRealRecAmt()));
            dtlDO.setTotalCurAmt(BigDecimalUtils.negate(dtl.getTotalCurAmt()));
            dtlDO.setTaxCurAmt(BigDecimalUtils.negate(dtl.getTaxCurAmt()));
            dtlDO.setRealRecCurAmt(BigDecimalUtils.negate(dtl.getRealRecCurAmt()));
            return dtlDO;
        }).collect(Collectors.toList());
        newRecOrder.setDtlList(newDtls);
        newRecOrder.setCheck(false);
        return newRecOrder;
    }


    private RecOrderDO updateOrSave(RecOrder recOrder) {
        //更新字段 主表信息
        RecOrderDO recOrderDO = RecOrderConvert.INSTANCE.convertToDo(recOrder);
        //先删除之前明细信息
        if (recOrder.getId() != null) {
            recOrderDtlRepoProc.deleteByMasIds(List.of(recOrder.getId()));
        } else {
            recOrderDO.setAuditDataVersion(0);
        }
        flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.REC_ORDER, recOrderDO);
        RecOrderDO save = recOrderRepo.save(recOrderDO);
        //细单信息
        List<RecOrderDtlDO> recOrderDtlDOS = RecOrderDtlConvert.INSTANCE.convertToDO(recOrder.getDtlList());

        //设置主键关联ID的值
        recOrderDtlDOS.forEach(dtl -> {
            //获取新增后的主表ID
            dtl.setMasId(save.getId());
            if(dtl.getId()==null){
                dtl.setAuditDataVersion(0);
            }
            dtl.setUnVerAmt(dtl.getTotalAmt());
            dtl.setVerAmt(BigDecimal.ZERO);
            dtl.setApplyVerAmTing(BigDecimal.ZERO);
        });
        recOrderDtlRepo.saveAll(recOrderDtlDOS);
        return save;
    }
}
