package com.elitesland.fin.repo.writeoff;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.facade.dto.writeoff.RecOrderAmtUpdateDTO;
import com.elitesland.fin.application.facade.param.recorder.FinRecOrderDetailQuery;
import com.elitesland.fin.application.facade.param.recorder.RecOrderDtlExPageParam;
import com.elitesland.fin.application.facade.vo.recorder.RecOrderDtlExVo;
import com.elitesland.fin.domain.entity.recorder.*;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Repository;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * 收款单.
 *
 * @author shihao.ma
 * @since 2023/9/14
 */
@Repository
public class RecOrderDetailRepoProc extends BaseRepoProc<RecOrderExDtlDo> {

    private static final QRecOrderExDtlDo QDO = QRecOrderExDtlDo.recOrderExDtlDo;
    private static final QRecOrderExDo QMasExt = QRecOrderExDo.recOrderExDo;
    private static final QRecOrderDO QMas = QRecOrderDO.recOrderDO;
    private static final QRecOrderDtlDO QDetail = QRecOrderDtlDO.recOrderDtlDO;

    protected RecOrderDetailRepoProc() {
        super(QDO);
    }

    public List<RecOrderExDtlDo> listByMasIds(Collection<Long> masIds) {
        return super.getListByValue(QDO.masId, masIds);
    }

    /**
     * 查询应收单明细、明细扩展集合.
     *
     * @param query 查询条件
     * @return 查询结果集
     */
    public List<RecOrderDtlExVo> listRecOrderDetail(FinRecOrderDetailQuery query) {

        JPAQuery<RecOrderDtlExVo> select = jpaQueryFactory.select(Projections.bean(RecOrderDtlExVo.class,
                QDetail.id,
                QDetail.masId,
                QDetail.recType,
                QDetail.recKind,
                QDetail.recBank,
                QDetail.recAccount,
                QDetail.recFlow,
                QDetail.sourceNo,
                QDetail.sourceId,
                QDetail.sourceLine,
                QDetail.sourceLineId,
                QDetail.realRecAmt,
                QDetail.realRecCurAmt,
                QDetail.taxAmt,
                QDetail.taxCurAmt,
                QDetail.taxRate,
                QDetail.totalAmt,
                QDetail.totalCurAmt,
                QDetail.expensesType,

                QDO.thirdOrderDtId,
                QDO.custCode,
                QDO.naturePayment,
                QDO.buCode,
                QDO.businessCode,
                QDO.currCode,
                QDO.exchangeRate,
                QDO.recBank,
                QDO.payBank,
                QDO.unVerAmt,
                QDO.verAmt,
                QDO.applyVerAmTing
        ))
                .from(QDetail)
                .leftJoin(QDO).on(QDO.relateId.eq(QDetail.relateId));

        List<Predicate> predicates = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(query.getRecDIds())) {
            predicates.add(QDetail.id.in(query.getRecDIds()));
        }

        if (CollectionUtils.isNotEmpty(query.getMasIds())) {
            predicates.add(QDetail.masId.in(query.getMasIds()));
        }

        select.where(ExpressionUtils.allOf(predicates));
        return select.fetch();
    }

    /**
     * 查询收款单明细金额.
     *
     * @param recDId 明细ID
     * @return 查询结果
     */
    public RecOrderExDtlDo getRecOrderDetailAmt(Long recDId) {
        JPAQuery<RecOrderExDtlDo> select = jpaQueryFactory.select(Projections.bean(RecOrderExDtlDo.class,
                QDO.id.as("extId"),
                QDetail.totalAmt,
                QDO.verAmt,
                QDO.applyVerAmTing,
                QDO.auditDataVersion.as("version"),
                QDO.unVerAmt
        ))
                .from(QDetail)
                .leftJoin(QDO).on(QDO.relateId.eq(QDetail.relateId))
                .where(QDetail.id.eq(recDId));
        return select.fetchOne();
    }
    public RecOrderDtlDO getExtRecOrderDetailAmt(Long recDId) {
        JPAQuery<RecOrderDtlDO> select = jpaQueryFactory.select(Projections.fields(RecOrderDtlDO.class,
                        QDetail.id,
                        QDetail.totalAmt,
                        QDetail.verAmt,
                        QDetail.applyVerAmTing,
                        QDetail.auditDataVersion,
                        QDetail.unVerAmt
                ))
                .from(QDetail)
                .where(QDetail.id.eq(recDId));
        return select.fetchOne();
    }
    public List<RecOrderExDtlDo> getRecOrderDetailAmt(Collection<Long> recDIds) {
        JPAQuery<RecOrderExDtlDo> select = jpaQueryFactory.select(Projections.bean(RecOrderExDtlDo.class,
                QDO.id.as("extId"),
                QDetail.totalAmt,
                QDetail.id.as("recDId"),
                QDO.verAmt,
                QDO.applyVerAmTing,
                QDO.auditDataVersion.as("version"),
                QDO.unVerAmt
        ))
                .from(QDetail)
                .leftJoin(QDO).on(QDO.relateId.eq(QDetail.relateId))
                .where(QDetail.id.in(recDIds));
        return select.fetch();
    }

    /**
     * 更新核销金额.
     *
     * @param update 更新结果
     * @return 更新条数
     */
    public long updateVerAmt(RecOrderAmtUpdateDTO update) {
        return jpaQueryFactory.update(QDO)
                .set(QDO.verAmt, update.getVerAmt())
                .set(QDO.applyVerAmTing, update.getVerAmting())
                .set(QDO.unVerAmt, update.getUnVerAmt())
                .set(QDO.auditDataVersion, update.getVersion() + 1)
                .where(QDO.id.eq(update.getArDId()).and(QDO.auditDataVersion.eq(update.getVersion())))
                .execute();
    }
    public long updateExtVerAmt(RecOrderAmtUpdateDTO update) {
        return jpaQueryFactory.update(QDetail)
                .set(QDetail.verAmt, update.getVerAmt())
                .set(QDetail.applyVerAmTing, update.getVerAmting())
                .set(QDetail.unVerAmt, update.getUnVerAmt())
                .set(QDetail.auditDataVersion, update.getVersion() + 1)
                .where(QDetail.id.eq(update.getArDId()).and(QDetail.auditDataVersion.eq(update.getVersion())))
                .execute();
    }

    public PagingVO<RecOrderDtlExVo> getDtlPage(RecOrderDtlExPageParam query) {
        PredicateBuilder predicateBuilder = PredicateBuilder.builder()
                .andIn(QMas.id, query.getMasIds())
                .andEq(QMas.id, query.getMasId())
                .andLike(QMas.sourceNo, query.getThirdOrderNum())
                .andEq(QMas.ouCode, query.getRecOuCode())
                .andEq(QMas.saleUserId, query.getSaleUser())
                .andEq(QDO.businessCode, query.getBusinessCode())
                .andLike(QMasExt.invoiceNumber, query.getInvoiceNumber())
                .andEq(QDO.custCode, query.getCustCode())
                .andEq(QDetail.recKind, query.getRecKind())
                .andEq(QDO.naturePayment, query.getNaturePayment())
                .andLike(QDO.recBank, query.getRecBank())
                .andLike(QDO.payBank, query.getPayBank())
                .andBetween(QDO.createTime, query.getCreateTimeStart(), query.getCreateTimeEnd())
                .andEq(QDO.remark, query.getRemark());

        if (CollectionUtils.isEmpty(query.getMasIds()) && query.getMasId() == null) {
            predicateBuilder.andNe(QDO.unVerAmt, BigDecimal.ZERO);
        }

        Predicate predicate = predicateBuilder.build();

        JPAQuery<RecOrderDtlExVo> select = jpaQueryFactory.select(Projections.bean(RecOrderDtlExVo.class,
                QDO.id,
                QDO.relateId,
                QDO.thirdOrderDtId,
                QDO.currCode,
                QDO.custCode,
                QDO.naturePayment,
                QDO.buCode,
                QDO.recBank,
                QDO.payBank,
                QDO.businessCode,
                QDO.exchangeRate,
                QDO.deleteFlag,
                QDO.tenantId,
                QDO.belongOrgId,
                QDO.tenantOrgId,
                QDO.createTime,
                QDO.updater,
                QDO.modifyTime,
                QDO.creator,
                QDO.remark,
                QDO.unVerAmt,
                QDO.applyVerAmTing,
                QDO.verAmt,
                QMasExt.invoiceNumber,
                QMas.recOuCode,
                QMas.sourceNo,
                QMas.id.as("recOrderId"),
                QMas.recOrderNo,
                QMas.totalAmt.as("mainTotalAmt"),
                QDetail.id.as("recDId"),
                QDetail.masId,
                QDetail.recType,
                QDetail.recKind,
                QDetail.recAccount,
                QDetail.recFlow,
                QDetail.sourceId,
                QDetail.sourceLine,
                QDetail.sourceLineId,
                QDetail.realRecAmt,
                QDetail.realRecCurAmt,
                QDetail.taxAmt,
                QDetail.taxRate,
                QDetail.totalAmt,
                QDetail.totalCurAmt,
                QDetail.expensesType,
                QDetail.taxCurAmt
        ))
                .from(QDO)
                .leftJoin(QMasExt).on(QDO.masId.eq(QMasExt.id))
                .leftJoin(QDetail).on(QDetail.relateId.eq(QDO.relateId))
                .where(predicate)
                .groupBy(QDO.id);
        return super.queryByPage(select, query.getPageRequest(), QDO.createTime.desc());
    }
}
