package com.elitesland.fin.application.service.flow;

import com.elitesland.fin.application.facade.param.flow.AccountFlowAccAmtParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowPageParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowParam;
import com.elitesland.fin.application.facade.vo.account.AccountVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowAccAmtVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.entity.flow.AccountFlowDO;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;
import java.util.Optional;

/**
 * <p>
 * 功能说明:账户流水共同的Service
 * </p>
 *
 * @Author Darren
 * @Date 2023/02/27
 * @Version 1.0
 * @Content:
 */
public interface AccountFlowCommonService {

    /**
     * 根据账户编码查询上一条流水的信息:只查询数据库数据
     * <p>
     * 按照创建时间进行倒叙查询并取第一条数据
     *
     * @param accCode 账户编码
     * @return 账户流水信息
     */
    Optional<AccountFlowVO> selectByAccCode(String accCode);

    /**
     * 查询上一条流水的信息:查询数据库或查询账户里的数据
     * <p>
     * 初始化第一次查不到流水记录的时候需要查询账户里的信息
     * 查到上一条流水数据时使用上一条的数据
     *
     * @param accCode 账户编码
     * @return 组装后的账户流水账户金额、账户占用金额计算处理返回共用对象
     */
    AccountFlowAccAmtVO selectPreviousData(String accCode);

    /**
     * 根据账户编码查询账户信息
     *
     * @param accCode 账户编码
     * @return 账户信息对象
     */
    AccountVO selectAccountByAccCode(String accCode);

    /**
     * 共用入参对象赋值、传值
     *
     * @param amount           本条发生金额
     * @param accAmt           上一条的账户金额
     * @param accOccAmt        上一条的账户占用金额
     * @param accHandleType    账户金额处理类型
     * @param accOccHandleType 账户占用金额处理类型
     * @return 共用入参对象
     */
    AccountFlowAccAmtParam accAmtParamPassValue(BigDecimal amount, BigDecimal accAmt, BigDecimal accOccAmt
            , String accHandleType, String accOccHandleType);

    /**
     * 根据账户处理类型处理填充账户金额、账户占用金额,进行计算赋值：
     * <p>
     * 本条账户金额 = 上一条的账户金额
     * 本条账户金额 = 上一条的账户金额 + 本条发生金额
     * 本条账户金额 = 上一条的账户金额 - 本条发生金额
     * 本条账户金额 = 赋值为空
     * <p>
     * 本条账户占用金额 = 上一条的账户占用金额
     * 本条账户占用金额 = 上一条的账户占用金额 + 本条发生金额
     * 本条账户占用金额 = 上一条的账户占用金额 - 本条发生金额
     * 本条账户占用金额 = 赋值为空
     * <p>
     * 账户流水账户金额、账户占用金额处理类型-相加、相减、相同、值为空
     *
     * @param accAmtParam 账户金额、账户占用金额计算处理入参共用对象
     * @return 账户金额、账户占用金额计算处理返回共用对象
     */
    AccountFlowAccAmtVO handleAccAmt(AccountFlowAccAmtParam accAmtParam);

    /**
     * 账户流水批量保存
     *
     * @param paramList 账户流水入参
     * @return 账户流水ID
     */
    List<Long> saveAccountFlowBatch(List<AccountFlowDO> paramList);

    /**
     * 账户流水保存：
     *
     * @param accountFlowDO 账户流水入参
     * @return 账户流水ID
     */
    Long saveAccountFlow(AccountFlowDO accountFlowDO);

    /**
     * 账户流水保存并更新账户金额信息：
     *
     * @param accountFlowParam 账户流水入参
     * @return 账户流水ID
     */
    Long saveAccountAndFlow(AccountFlowParam accountFlowParam);

    /**
     * 账户流水Param批量保存：对Param进行转换DO并保存
     *
     * @param paramList 入参
     * @return 账户流水ID
     */
    List<Long> saveParamBatch(List<AccountFlowParam> paramList);

    /**
     * 默认赋值
     *
     * @param accountFlowParam 入参
     */
    void defaultAssignment(AccountFlowParam accountFlowParam);

    /**
     * 根据账户进出方式对发生金额进行正负转换
     *
     * @param accIoType 账户进出方式
     * @return -1 或 1
     */
    BigDecimal plusOrMinus(String accIoType);

    /**
     * 根据账户进出方式判断是否为负数(扣减)
     *
     * @param accIoType 账户进出方式
     * @return true 或 false
     */
    Boolean plusOrMinus02(String accIoType);

    /**
     * 判断过滤出账户类型为整车的
     *
     * @param accType 账户类型
     * @return 属于 true 否则 false
     */
    boolean judgeZcAccType(String accType);

    /**
     * 判断过滤出账户类型为配件的
     *
     * @param accType 账户类型
     * @return 属于 true 否则 false
     */
    boolean judgePjAccType(String accType);

    /**
     * 判断过滤出账户类型为附件包的
     *
     * @param accType 账户类型
     * @return 属于 true 否则 false
     */
    boolean judgeFjbAccType(String accType);

    /**
     * 判断过滤出账户类型为统筹的
     *
     * @param accType 账户类型
     * @return 属于 true 否则 false
     */
    boolean judgeTcAccType(String accType);

    /**
     * 把年月格式yyyy-MM字符串转换拼接成年月日格式的日期
     *
     * @param yearMonthStr 年月格式yyyy-MM字符串入参
     * @return 年月日格式的日期
     */
    LocalDate transitionYearMonthStr(String yearMonthStr);

    /**
     * 根据年月日格式的日期拼装入参里的交易起始时间、交易截止时间
     * 根据传入的当前年月日获取其所在月份的第一天最小值和最后一天最大值
     *
     * @param localDate 年月日格式的日期
     * @param pageParam 查询入参
     */
    void splicTransactionTimeSection(LocalDate localDate, AccountFlowPageParam pageParam);

    /**
     * 调剂单审核通过时对入参的拆分类型做判断：必须有一条，是表明是从账户的，用于拆分使用
     *
     * @param paramList 入参
     */
    void checkAdjustApprove(List<AccountFlowParam> paramList);

    /**
     * 调剂单审核通过时:判断是否需要拆分一条交易类型为调剂单释放的数据
     *
     * @param birthOrderType 生单类型
     * @return true 是  false 否
     */
    boolean judgeAdjustRelease(String birthOrderType);

}
