package com.elitesland.fin.infr.repo.expense;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.domain.expense.QExpTypeDO;
import com.elitesland.fin.domain.expense.QExpTypeDtlDO;
import com.elitesland.fin.domain.param.expense.ExpTypePageParam;
import com.elitesland.fin.infr.dto.expesne.ExpTypeDTO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author wang.xl
 * @version V1.0
 * @Package
 * @date 2022/4/11 13:53
 */
@Component
@RequiredArgsConstructor
public class ExpTypeRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private final QExpTypeDO qExpTypeDO = QExpTypeDO.expTypeDO;
    private final QExpTypeDtlDO qExpTypeDtlDO = QExpTypeDtlDO.expTypeDtlDO;

    public <T> JPAQuery<T> select(Class<T> T) {
        return jpaQueryFactory.select(Projections.bean(T,
                        qExpTypeDO.id,
                        qExpTypeDO.itemCode,
                        qExpTypeDO.itemName,
                        qExpTypeDO.expTypeCode,
                        qExpTypeDO.expTypeName,
                        qExpTypeDO.enableFlag,
                        qExpTypeDO.sourceDoc,
                        qExpTypeDO.sourceDocType,
                        qExpTypeDO.sourceDocStatus,
                        qExpTypeDO.remark,
                        qExpTypeDO.deleteFlag,
                        qExpTypeDO.createTime,
                        qExpTypeDO.creator,
                        qExpTypeDO.createUserId,
                        qExpTypeDO.modifyTime,
                        qExpTypeDO.modifyUserId,
                        qExpTypeDO.updater
                ))
                .from(qExpTypeDO);
    }

    public Boolean existsName(String expTypeName) {

        long res = select(ExpTypeDTO.class)
                .where(qExpTypeDO.expTypeName.like("%" + expTypeName + "%"))
                .fetchCount();
        return res > 0;
    }

    public Boolean existsCode(String expTypeCode) {

        long res = select(ExpTypeDTO.class)
                .where(qExpTypeDO.expTypeCode.like("%" + expTypeCode + "%"))
                .fetchCount();
        return res > 0;
    }

    public ExpTypeDTO get(Long id) {
        return select(ExpTypeDTO.class)
                .where(qExpTypeDO.id.eq(id))
                .fetchOne();
    }

    public PagingVO<ExpTypeDTO> page(ExpTypePageParam expTypePageParam, List<Long> exptypeIds) {
        List<Predicate> predicates = new ArrayList<>();
        if (CharSequenceUtil.isNotBlank(expTypePageParam.getExpTypeName())) {
            predicates.add(qExpTypeDO.expTypeName.like("%" + expTypePageParam.getExpTypeName() + "%"));
        }
        if (CharSequenceUtil.isNotBlank(expTypePageParam.getExpTypeCode())) {
            predicates.add(qExpTypeDO.expTypeCode.like("%" + expTypePageParam.getExpTypeCode() + "%"));
        }
        if (expTypePageParam.getEnableFlag() != null) {
            predicates.add(qExpTypeDO.enableFlag.eq(expTypePageParam.getEnableFlag()));
        }
        if (null != expTypePageParam.getOuId()) {
            if (CollUtil.isNotEmpty(exptypeIds)) {
                predicates.add(qExpTypeDO.id.in(exptypeIds));
            } else {
                return new PagingVO<>();
            }
        }


        if (CollectionUtil.isNotEmpty(expTypePageParam.getIdList())) {
            predicates.add(qExpTypeDO.id.in(expTypePageParam.getIdList()));
        }
        predicates.add(qExpTypeDO.deleteFlag.eq(0));
        JPAQuery<ExpTypeDTO> query = select(ExpTypeDTO.class)
                .where(ExpressionUtils.allOf(predicates));
        expTypePageParam.setPaging(query);
        expTypePageParam.fillOrders(query, qExpTypeDO);
        return new PagingVO<>(query.fetchCount(), query.fetch());
    }

    public List<ExpTypeDTO> findByIds(List<Long> ids) {
        return select(ExpTypeDTO.class)
                .where(qExpTypeDO.id.in(ids))
                .fetch();
    }

    public void updateEnableFlag(List<Long> ids, boolean enable) {
        jpaQueryFactory.update(qExpTypeDO)
                .set(qExpTypeDO.enableFlag, enable)
                .where(qExpTypeDO.id.in(ids))
                .execute();
    }

    public Boolean isFir() {
        long res = jpaQueryFactory.select(qExpTypeDO.id)
                .from(qExpTypeDO)
                .fetchCount();
        return res == 0;

    }

    public ExpTypeDTO findById(Long id) {
        return select(ExpTypeDTO.class)
                .where(qExpTypeDO.id.eq(id))
                .fetchOne();
    }

    public ExpTypeDTO findByExpTypeCode(String expTypeCode) {
        return select(ExpTypeDTO.class)
                .where(qExpTypeDO.expTypeCode.eq(expTypeCode).and(qExpTypeDO.enableFlag.eq(true)))
                .fetchOne();
    }

    public Long del(Long id) {
        return jpaQueryFactory.delete(qExpTypeDO)
                .where(qExpTypeDO.id.eq(id))
                .execute();
    }

    public List<ExpTypeDTO> getList() {
        return select(ExpTypeDTO.class)
                .where(qExpTypeDO.deleteFlag.eq(0))
                .where(qExpTypeDO.enableFlag.eq(true))
                .fetch();
    }

    public List<ExpTypeDTO> arOrderAutoSelectMatchByParam(ExpTypePageParam expTypePageParam) {
        List<Predicate> predicates = new ArrayList<>();

        if (CharSequenceUtil.isNotBlank(expTypePageParam.getSourceDoc())) {
            predicates.add(qExpTypeDtlDO.sourceDoc.eq(expTypePageParam.getSourceDoc()));
        }
        if (CharSequenceUtil.isNotBlank(expTypePageParam.getSourceDocType())) {
            predicates.add(qExpTypeDtlDO.sourceDocType.eq(expTypePageParam.getSourceDocType()));
        }
        if (CharSequenceUtil.isNotBlank(expTypePageParam.getSourceDocStatus())) {
            predicates.add(qExpTypeDtlDO.sourceDocStatus.eq(expTypePageParam.getSourceDocStatus()));
        }
        predicates.add(qExpTypeDO.enableFlag.eq(true));
        predicates.add(qExpTypeDO.deleteFlag.eq(0));

        JPAQuery<ExpTypeDTO> query = jpaQueryFactory.select(Projections.bean(
                        ExpTypeDTO.class,
                        qExpTypeDO.id,
                        qExpTypeDO.expTypeCode,
                        qExpTypeDO.expTypeName,
                        qExpTypeDO.enableFlag,
                        qExpTypeDO.createTime
                )).from(qExpTypeDO)
                .leftJoin(qExpTypeDtlDO)
                .on(qExpTypeDO.id.eq(qExpTypeDtlDO.masId))
                .where(ExpressionUtils.allOf(predicates))
                .groupBy(qExpTypeDO.id);

        return query.fetch();
    }

}
