package com.elitesland.fin.application.service.workflow.saleinv;

import com.elitescloud.boot.mq.MessageQueueTemplate;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.system.param.SysUserIdFlowRoleRpcParam;
import com.elitescloud.cloudt.system.service.SysUserFlowRoleRpcService;
import com.elitesland.fin.Application;
import com.elitesland.fin.application.facade.dto.mq.DemoMqMessageDTO;
import com.elitesland.fin.application.facade.param.saleinv.SaleInvStatusParam;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.saleinv.QSaleInvDO;
import com.elitesland.fin.infr.dto.saleinv.SaleInvDTO;
import com.elitesland.fin.infr.repo.saleinv.SaleInvRepoProc;
import com.elitesland.fin.repo.invoice.InvoiceAwaitRepoProc;
import com.elitesland.workflow.CommentInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @date 2022/5/11 10:22
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class SaleInvProcessServiceImpl implements SaleInvProcessService {

    private final SysUserFlowRoleRpcService sysUserFlowRoleRpcService;

    private final SaleInvRepoProc saleInvRepoProc;
    private final JPAQueryFactory jpaQueryFactory;

    private final MessageQueueTemplate messageQueueTemplate;

    private final InvoiceAwaitRepoProc invoiceAwaitRepoProc;

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void processStatusChange(long id, ProcInstStatus procInstStatus, CommentInfo commentInfo) {
        SaleInvDTO saleInvDTO = saleInvRepoProc.get(id);
        if (saleInvDTO == null) {
            throw new BusinessException("单据(id:" + id + ")不存在,无法审批");
        }
        QSaleInvDO saleInvDO = QSaleInvDO.saleInvDO;
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(saleInvDO)
                .set(saleInvDO.procInstStatus, procInstStatus)
                .set(saleInvDO.auditDate, LocalDateTime.now())
                .set(saleInvDO.auditUserId, Long.valueOf(commentInfo.getUserId()))
                .set(saleInvDO.auditRejection, commentInfo.getComment())
                .set(saleInvDO.auditUser, commentInfo.getUserName())
                .where(saleInvDO.id.eq(id));

        //单据状态
        switch (procInstStatus) {
            case NOTSUBMIT: // 未提交
                //一般情况将单据状态变成"草稿"
                jpaUpdateClause.set(saleInvDO.orderState, UdcEnum.APPLY_STATUS_REJECTED.getValueCode());
                jpaUpdateClause.set(saleInvDO.procInstStatus,procInstStatus.REJECTED);
                updateInvoiceAwaitStatus(saleInvDTO.getApplyNo(),UdcEnum.INV_STATE_WAIT.getValueCode());
                break;
            case INTERRUPT: // 中断执行
                //一般情况将单据状态变成"草稿",并且将单据上的"流程实例状态"，"流程实例ID"清成null(不是空字符串)
                jpaUpdateClause.set(saleInvDO.orderState, UdcEnum.APPLY_STATUS_EXPIRED.getValueCode())
                        .set(saleInvDO.procInstId, (String) null);
                break;
            case REJECTED: // 审批拒绝
                //一般情况将单据状态变成"草稿"或"拒绝"
                jpaUpdateClause.set(saleInvDO.orderState, UdcEnum.APPLY_STATUS_REJECTED.getValueCode());
                // 更新待开票列表的数据为待开票
                updateInvoiceAwaitStatus(saleInvDTO.getApplyNo(),UdcEnum.INV_STATE_WAIT.getValueCode());

                break;
            case INVALID: // 作废
                //一般情况将单据状态变成"作废" ，或直接删除单据
                jpaUpdateClause.set(saleInvDO.orderState, UdcEnum.APPLY_STATUS_VOID.getValueCode())
                        .set(saleInvDO.procInstId, (String) null);
                break;
            case APPROVING: // 审批中
                //一般情况将单据状态变成"审批中
                jpaUpdateClause.set(saleInvDO.orderState, UdcEnum.APPLY_STATUS_DOING.getValueCode());
                updateInvoiceAwaitStatus(saleInvDTO.getApplyNo(),UdcEnum.INV_STATE_ING.getValueCode());
                break;
            case APPROVED: // 最终节点审批通过
                //一般情况将单据状态变成"审批通过"
                jpaUpdateClause.set(saleInvDO.orderState, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode())
                        .set(saleInvDO.approvedTime, LocalDateTime.now());
                // 审批通过 同步NC
                sendMessage(saleInvDTO);
                break;
            default:
                break;
        }
        //执行
        jpaUpdateClause.execute();
    }

    private void updateInvoiceAwaitStatus(String applyNo, String invState) {
        SaleInvStatusParam saleInvStatusParam = new SaleInvStatusParam();
        saleInvStatusParam.setApplyNo(applyNo);
        saleInvStatusParam.setInvState(invState);
        invoiceAwaitRepoProc.updateInvState(saleInvStatusParam);
    }

    @Override
    public ArrayList<String> taskAssignee(String businessKey, String customParams) {
        Long businessId = Long.valueOf(businessKey);
        SaleInvDTO saleInvDTO = saleInvRepoProc.get(businessId);
        if (saleInvDTO == null) {
            throw new BusinessException("单据(id:" + businessId + ")不存在,无法审批");
        }
        SysUserIdFlowRoleRpcParam param = SysUserIdFlowRoleRpcParam.builder()
                .ouIds(Collections.singletonList(saleInvDTO.getOuId()))
                .flowRoleCodes(Collections.singletonList(customParams))
                .build();
        List<Long> userIdsByFlowRoles = sysUserFlowRoleRpcService.findUserIdsByFlowRoles(param);
        return userIdsByFlowRoles.stream().map(e -> e + "").collect(Collectors.toCollection(ArrayList::new));

    }

    private void sendMessage(SaleInvDTO saleInvDTO) {
        // 发送消息给WMS
        DemoMqMessageDTO messageDTO = new DemoMqMessageDTO();
        messageDTO.setId(saleInvDTO.getId());
        messageDTO.setCode(saleInvDTO.getApplyNo());
        messageDTO.setInterfaceType(FinConstant.ZT_TO_NC_SALINV);
        messageDTO.setDomainCode(FinConstant.FIN);
        try {
            messageQueueTemplate.publishMessage(Application.NAME, "yst-fin", messageDTO);
        } catch (Exception e) {
            log.error("发送mq消息，同步销售开票信息到NC失败,同步内容{}, 失败原因：{}", messageDTO, e.getMessage());
            throw new com.elitescloud.boot.exception.BusinessException("同步销售开票信息到NC失败");
        }
    }
}
