package com.elitesland.fin.application.service.accountingengine;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.accountingengine.*;
import com.elitesland.fin.application.facade.param.accountingengine.FinAccEngDetConditionParam;
import com.elitesland.fin.application.facade.param.accountingengine.FinAccountEngineDetailsParam;
import com.elitesland.fin.application.facade.param.accountingengine.FinAccountEngineParam;
import com.elitesland.fin.application.facade.param.accountingengine.FinJournalLogParam;
import com.elitesland.fin.application.facade.vo.accountingengine.*;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountRuleConfigPageVO;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.accountingengine.*;
import com.elitesland.fin.repo.accountingengine.*;
import com.elitesland.fin.utils.StringUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.jdt.internal.compiler.impl.Constant;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.sql.*;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.elitesland.fin.domain.entity.accountingengine.QFinAccEngDetDataLineDO.finAccEngDetDataLineDO;

/**
 * @author gyj
 * @date 2023/10/10
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class FinAccountEngineServiceImpl implements FinAccountEngineService {

    private final FinAccountEngineRepo finAccountEngineRepo;

    private final FinAccountEngineRepoProc finAccountEngineRepoProc;

    private final FinAccountEngineDetailsRepo finAccountEngineDetailsRepo;

    private final FinAccEngDetConditionRepo finAccEngDetConditionRepo;

    private final FinAccEngDetDataRepo finAccEngDetDataRepo;

    private final FinAccEngDetDataLineRepo finAccEngDetDataLineRepo;

    private final FinEventTableRepo finEventTableRepo;

    private final FinEventTableLineRepo finEventTableLineRepo;

    private final FinEventTableConditionRepo finEventTableConditionRepo;

    private final FinJournalLogService finJournalLogService;

    private final FinJournalLogRepo finJournalLogRepo;

    private final FinSetOfBookOuRepo finSetOfBookOuRepo;

    private final FinSetOfBookRepo finSetOfBookRepo;

    private final FinSetOfBookRepoProc finSetOfBookRepoProc;

    private final FinSetOfBookLineRepo finSetOfBookLineRepo;

    private final FinFastCodeRepo finFastCodeRepo;

    private final FinFastCodeLineRepo finFastCodeLineRepo;

    private final FinFastCodeLineRepoProc finFastCodeLineRepoProc;

    private final FinFastCodeRepoProc finFastCodeRepoProc;

    private final FinFlexibleRepo finFlexibleRepo;

    private final FinFlexibleRepoProc finFlexibleRepoProc;

    private final FinFlexibleValueRepo finFlexibleValueRepo;

    private final FinFlexibleValueRepoProc finFlexibleValueRepoProc;

    private final FinSobAccountPeriodRepo finSobAccountPeriodRepo;

    private final FinJournalRepo finJournalRepo;


    @SysCodeProc
    @Override
    public PagingVO<FinAccountEngineVO> page(FinAccountEngineParam finAccountEngineParam) {
        return FinAccountEngineConvert.INSTANCE.DTOToVO(finAccountEngineRepoProc.page(finAccountEngineParam));
    }

    @SysCodeProc
    @Override
    public FinAccountEngineVO queryFinAccountEngine(FinAccountEngineDetailsParam param) {

        Assert.notNull(param.getMasId(), "masId不能为空");
        // 查询表头
        Optional<FinAccountEngineDO> optional = finAccountEngineRepo.findById(param.getMasId());
        Assert.isTrue(optional.isPresent(), "查询的会计引擎不存在");
        FinAccountEngineVO engineVO = FinAccountEngineConvert.INSTANCE.DOToVo(optional.get());

        // 查询会计引擎详情
        List<FinAccountEngineDetailsDO> detailsDOList = finAccountEngineDetailsRepo.findAllByMasIdIn(List.of(param.getMasId()));
        List<FinAccountEngineDetailsVO> detailsVOList = FinAccountEngineDetailsConvert.INSTANCE.DOsToVOs(detailsDOList);
        engineVO.setFinAccountEngineDetails(detailsVOList);

        return engineVO;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void enableOrDisable(FinAccountEngineParam finAccountEngineParam) {

        checkEnableOrDisableParam(finAccountEngineParam);

        List<FinAccountEngineDO> finAccountEngineDOList = finAccountEngineRepo.findAllById(finAccountEngineParam.getIds());
        finAccountEngineDOList.stream().forEach(item -> Assert.isFalse(finAccountEngineParam.getStatus().equals(item.getStatus()), "数据已经启用/禁用"));

        finAccountEngineDOList.stream().forEach(item -> item.setStatus(finAccountEngineParam.getStatus()));
    }

    private void checkEnableOrDisableParam(FinAccountEngineParam finAccountEngineParam) {
        Assert.notEmpty(finAccountEngineParam.getIds(), "id必填");
        Assert.notEmpty(finAccountEngineParam.getStatus(), "状态必填");
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveOrUpdateFinAccountEngineDetails(FinAccountEngineDetailsParam param) {

        checkSaveOrUpdateFinAccountEngineDetails(param);

        FinAccountEngineDetailsDO finAccountEngineDetailsDO = finAccountEngineDetailsRepo.save(FinAccountEngineDetailsConvert.INSTANCE.paramToDO(param));
        finAccEngDetConditionRepo.deleteAllByMasId(finAccountEngineDetailsDO.getId());

        List<FinAccEngDetConditionParam> finAccEngDetConditionParamList = param.getFinAccEngDetConditionDetailList();
        finAccEngDetConditionParamList.stream().forEach(item -> item.setMasId(finAccountEngineDetailsDO.getId()));
        finAccEngDetConditionRepo.saveAll(FinAccEngDetConditionConvert.INSTANCE.paramToDO(finAccEngDetConditionParamList));

        return finAccountEngineDetailsDO.getId();
    }

    private void checkSaveOrUpdateFinAccountEngineDetails(FinAccountEngineDetailsParam param) {
        checkFinAccountEngineDetailsParam(param);

        List<FinAccEngDetConditionParam> finAccEngDetConditionParamList = param.getFinAccEngDetConditionDetailList();
        Assert.notEmpty(finAccEngDetConditionParamList, "会计引擎条件必填");

        Set<Integer> serialNums = new HashSet<>();
        finAccEngDetConditionParamList.stream().forEach(item -> {

            Assert.notNull(item.getSerialNum(), "序号必填");
            Assert.notEmpty(item.getColumnName(), "字段名称必填");
            Assert.notEmpty(item.getConditionType(), "条件必填");
            Assert.notEmpty(item.getValueFrom(), "从值内容必填");

            Assert.isFalse(serialNums.contains(item.getSerialNum()), "序号不能重复");
            serialNums.add(item.getSerialNum());
        });
    }

    private void checkFinAccountEngineDetailsParam(FinAccountEngineDetailsParam param) {
        Assert.notNull(param.getSerialNum(), "序号必填");
        Assert.notEmpty(param.getAccountEngine(), "会计引擎名称必填");
        Assert.notNull(param.getActiveFlag(), "是否启用必填");
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveOrUpdate(FinAccountEngineParam finAccountEngineParam) {

        checkSaveOrUpdateParam(finAccountEngineParam);

        checkIdempotent(finAccountEngineParam);

        FinAccountEngineDO finAccountEngineDO = finAccountEngineRepo.save(FinAccountEngineConvert.INSTANCE.paramToDO(finAccountEngineParam));
        finAccountEngineDetailsRepo.deleteAllByMasId(finAccountEngineDO.getId());

        List<FinAccountEngineDetailsParam> finAccountEngineDetailsParamList = finAccountEngineParam.getFinAccountEngineDetails();
        finAccountEngineDetailsParamList.stream().forEach(item -> item.setMasId(finAccountEngineDO.getId()));
        finAccountEngineDetailsRepo.saveAll(FinAccountEngineDetailsConvert.INSTANCE.paramToDO(finAccountEngineDetailsParamList));

        return finAccountEngineDO.getId();
    }

    private void checkSaveOrUpdateParam(FinAccountEngineParam finAccountEngineParam) {
        Assert.notEmpty(finAccountEngineParam.getEventTable(), "事件表单名称必填");
        Assert.notEmpty(finAccountEngineParam.getColumnCompany(), "公司字段必填");
        Assert.notEmpty(finAccountEngineParam.getJournalHost(), "主机必填");
        Assert.notEmpty(finAccountEngineParam.getJournalPort(), "端口必填");
        Assert.notEmpty(finAccountEngineParam.getJournalUser(), "用户名必填");
        Assert.notEmpty(finAccountEngineParam.getJournalPassword(), "密码必填");
        Assert.notEmpty(finAccountEngineParam.getJournalDatabase(), "数据库名称必填");
        Assert.notEmpty(finAccountEngineParam.getJournalTable(), "主表必填");

        List<FinAccountEngineDetailsParam> finAccountEngineDetailsParamList = finAccountEngineParam.getFinAccountEngineDetails();
        Assert.notEmpty(finAccountEngineDetailsParamList, "会计引擎明细必填");

        Set<Integer> serialNums = new HashSet<>();
        finAccountEngineDetailsParamList.stream().forEach(item -> {

            checkFinAccountEngineDetailsParam(item);

            Assert.isFalse(serialNums.contains(item.getSerialNum()), "序号不能重复");
            serialNums.add(item.getSerialNum());
        });
    }

    private void checkIdempotent(FinAccountEngineParam finAccountEngineParam) {

        FinAccountEngineDO finAccountEngineDO = finAccountEngineRepoProc.findByEventTable(finAccountEngineParam.getEventTable());

        if (finAccountEngineDO != null) {

            Assert.equals(finAccountEngineParam.getId(), finAccountEngineDO.getId(), "事件表单名称已经存在");
        }
    }

    @Override
    public List<FinAccEngDetDataLineVO> queryFinAccDetDataLine(FinAccountEngineDetailsParam param) {
        Assert.notNull(param.getMasId(), "masId is not null");
        List<FinAccEngDetDataLineDO> dataLineDOList = finAccEngDetDataLineRepo.findAllByMasIdIn(List.of(param.getMasId()));
        List<FinAccEngDetDataLineVO> dataLineVOList = FinAccEngDetDataLineConvert.INSTANCE.DOsToVOS(dataLineDOList);
        return dataLineVOList;
    }

    @Override
    public FinAccountEngineDetailsVO queryAccountEngineDetails(FinAccountEngineDetailsParam param) {

        Assert.notNull(param.getMasId(), "masId is not null");

        // 查询表头
        Optional<FinAccountEngineDetailsDO> optional = finAccountEngineDetailsRepo.findById(param.getMasId());
        Assert.isTrue(optional.isPresent(), "详情不存在");

        FinAccountEngineDetailsVO detailsVO = FinAccountEngineDetailsConvert.INSTANCE.DOToVO(optional.get());

        // 数据行详情查询
        List<FinAccEngDetDataDO> detDataDOList = finAccEngDetDataRepo.findAllByMasIdIn(List.of(param.getMasId()));
        List<FinAccEngDetDataVO> detDataVOList = FinAccEngDetDataConvert.INSTANCE.DOsToVOs(detDataDOList);
        detailsVO.setFinAccEngDetDataVOList(detDataVOList);

        // 会计引擎详情条件查询
        List<FinAccEngDetConditionDO> conditionDOList = finAccEngDetConditionRepo.findAllByMasIdIn(List.of(param.getMasId()));
        List<FinAccEngDetConditionVO> conditionVOList = FinAccEngDetConditionConvert.INSTANCE.DOsToVos(conditionDOList);
        detailsVO.setFinAccEngDetConditionDetailList(conditionVOList);

        return detailsVO;
    }

    private String buildWhere(List<FinEventTableConditionDO> finEventTableConditionDOList,
                              List<FinAccEngDetConditionDO> finAccEngDetConditionDOList) {

        if (CollectionUtils.isEmpty(finEventTableConditionDOList) && CollectionUtils.isEmpty(finAccEngDetConditionDOList)) {
            return "1=1";
        }

        List<String> wheres = finEventTableConditionDOList.stream().map(item ->
                buildWhere(item.getConditionType(),
                        item.getTableName(),
                        item.getColumnName(),
                        item.getValueFrom(),
                        item.getValueTo()))
                .collect(Collectors.toList());

        wheres.addAll(finAccEngDetConditionDOList.stream().map(item ->
                buildWhere(item.getConditionType(),
                        item.getTableName(),
                        item.getColumnName(),
                        item.getValueFrom(),
                        item.getValueTo()))
                .collect(Collectors.toList()));

        return StringUtils.join(wheres, FinConstant.BLANK.concat(FinConstant.AND));
    }

    private String buildWhere(String conditionType, String tableName, String columnName, String valueFrom, String valueTo) {

        String concat = tableName
                .concat(FinConstant.POINT)
                .concat(columnName)
                .concat(FinConstant.BLANK);

        if (UdcEnum.EVENT_TABLE_CONDITION_BETWEEN_AND.getValueCode().equals(conditionType)) {
            return concat
                    .concat(FinConstant.BETWEEN)
                    .concat(FinConstant.BLANK)
                    .concat(valueFrom)
                    .concat(FinConstant.BLANK)
                    .concat(FinConstant.AND)
                    .concat(FinConstant.BLANK)
                    .concat(valueTo);
        }

        return concat
                .concat(conditionType)
                .concat(FinConstant.BLANK)
                .concat(valueFrom);
    }

    private String buildFields(List<FinEventTableLineDO> finEventTableLineDOList) {
        List<String> fieldList = finEventTableLineDOList.stream()
                .map(item -> item.getTableName().concat(FinConstant.POINT).concat(item.getColumnName()))
                .collect(Collectors.toList());

        return StringUtils.join(fieldList, FinConstant.COMMA);
    }

    private String buildSql(FinAccountEngineCustomDO finAccountEngineCustomDO,
                            List<FinAccEngDetConditionDO> finAccEngDetConditionDOList) {

        if (StringUtils.isNotEmpty(finAccountEngineCustomDO.getFinEventTableDO().getTable())) {
            return FinConstant.SELECT
                    .concat(FinConstant.BLANK)
                    .concat(buildFields(finAccountEngineCustomDO.getFinEventTableLineDOList()))
                    .concat(FinConstant.BLANK)
                    .concat(FinConstant.FROM)
                    .concat(FinConstant.BLANK)
                    .concat(finAccountEngineCustomDO.getFinEventTableDO().getMasTable())
                    .concat(FinConstant.BLANK)
                    .concat(FinConstant.INNER)
                    .concat(FinConstant.BLANK)
                    .concat(FinConstant.JOIN)
                    .concat(FinConstant.BLANK)
                    .concat(finAccountEngineCustomDO.getFinEventTableDO().getTable())
                    .concat(FinConstant.BLANK)
                    .concat(FinConstant.WHERE)
                    .concat(FinConstant.BLANK)
                    .concat(finAccountEngineCustomDO.getFinEventTableDO().getMasTable())
                    .concat(FinConstant.POINT)
                    .concat(FinConstant.ID)
                    .concat(FinConstant.BLANK)
                    .concat(FinConstant.EQUAL)
                    .concat(FinConstant.BLANK)
                    .concat(finAccountEngineCustomDO.getFinEventTableDO().getTable())
                    .concat(FinConstant.POINT)
                    .concat(FinConstant.MAS_ID)
                    .concat(FinConstant.BLANK)
                    .concat(FinConstant.AND)
                    .concat(FinConstant.BLANK)
                    .concat(buildWhere(finAccountEngineCustomDO.getFinEventTableConditionDOList(), finAccEngDetConditionDOList));
        }

        return FinConstant.SELECT
                .concat(FinConstant.BLANK)
                .concat(buildFields(finAccountEngineCustomDO.getFinEventTableLineDOList()))
                .concat(FinConstant.BLANK)
                .concat(FinConstant.FROM)
                .concat(FinConstant.BLANK)
                .concat(finAccountEngineCustomDO.getFinEventTableDO().getMasTable())
                .concat(FinConstant.BLANK)
                .concat(FinConstant.WHERE)
                .concat(FinConstant.BLANK)
                .concat(FinConstant.ONE)
                .concat(FinConstant.EQUAL)
                .concat(FinConstant.ONE)
                .concat(FinConstant.BLANK)
                .concat(FinConstant.AND)
                .concat(FinConstant.BLANK)
                .concat(buildWhere(finAccountEngineCustomDO.getFinEventTableConditionDOList(), finAccEngDetConditionDOList));
    }

    @Override
    public void generateJournal(FinAccountEngineParam finAccountEngineParam) {
        checkGenerateJournalParam(finAccountEngineParam);

        List<Map> saveMapList = new ArrayList<>();

        buildSaveMapList(finAccountEngineParam, saveMapList);

        saveFinJournal(finAccountEngineParam, saveMapList);
    }

    public void buildSaveMapList(FinAccountEngineParam finAccountEngineParam, List<Map> saveMapList) {

        FinAccountEngineCustomDO finAccountEngineCustomDO = queryData(finAccountEngineParam);

        ResultSet resultSet = null;
        Connection connection = null;
        PreparedStatement preparedStatement = null;
        String docNum = null;
        try {
            Class.forName(FinConstant.DRIVEN_NAME);

            connection = DriverManager.getConnection(String.format(FinConstant.CONNECTION_NAME,
                    finAccountEngineCustomDO.getFinEventTableDO().getHost(),
                    finAccountEngineCustomDO.getFinEventTableDO().getPort(),
                    finAccountEngineCustomDO.getFinEventTableDO().getDatabase()),
                    finAccountEngineCustomDO.getFinEventTableDO().getUser(),
                    finAccountEngineCustomDO.getFinEventTableDO().getPassword());

            //会计引擎详情
            for (FinAccountEngineDetailsDO finAccountEngineDetailsDO : finAccountEngineCustomDO.getFinAccountEngineDetailsDOList()) {

                //会计引擎详情条件
                List<FinAccEngDetConditionDO> finAccEngDetConditionDOListItem = finAccountEngineCustomDO.getFinAccEngDetConditionDOList().stream()
                        .filter(finAccEngDetDataDO -> finAccountEngineDetailsDO.getId().equals(finAccEngDetDataDO.getMasId()))
                        .collect(Collectors.toList());

                String sql = buildSql(finAccountEngineCustomDO, finAccEngDetConditionDOListItem);

                preparedStatement = connection.prepareStatement(sql);

                resultSet = preparedStatement.executeQuery();

                List<Map> resultList = new ArrayList<>();
                // 处理查询结果集
                while (resultSet.next()) {
                    Map<String, Object> map = new HashMap();

                    int columnCount = resultSet.getMetaData().getColumnCount();
                    for (int i = 1; i <= columnCount; i++) {
                        map.put(resultSet.getMetaData().getColumnName(i), resultSet.getObject(resultSet.getMetaData().getColumnName(i)));
                    }

                    resultList.add(map);
                }

                if (CollectionUtils.isEmpty(resultList)) {
                    saveFinJournalLog(null, String.format("{},{}的条件未命中",
                            finAccountEngineParam.getEventTable(),
                            finAccountEngineDetailsDO.getAccountEngine()));
                    return;
                }

                //数据行详情
                List<FinAccEngDetDataDO> finAccEngDetDataDOListItem = finAccountEngineCustomDO.getFinAccEngDetDataDOList()
                        .stream()
                        .filter(finAccEngDetDataDO -> finAccountEngineDetailsDO.getId().equals(finAccEngDetDataDO.getMasId()))
                        .collect(Collectors.toList());
                Assert.notEmpty(finAccEngDetDataDOListItem, "查不到数据行详情");

                //会计引擎数据行
                for (FinAccEngDetDataDO finAccEngDetDataDO : finAccEngDetDataDOListItem) {

                    for (Map result : resultList) {

                        Assert.notNull(result.get(finAccountEngineParam.getColumnCompany()), "公司编码为空");

                        String ouCode = String.valueOf(result.get(finAccountEngineParam.getColumnCompany()));

                        Assert.notNull(result.get(finAccountEngineParam.getColumnDocNum()), "单据编码为空");

                        docNum = String.valueOf(result.get(finAccountEngineParam.getColumnDocNum()));

                        if (!checkFinSetOfBookOu(finAccountEngineCustomDO, ouCode, docNum, finAccountEngineParam, finAccountEngineDetailsDO, finAccEngDetDataDO)) {
                            return;
                        }

                        //数据映射
                        List<FinAccEngDetDataLineDO> finAccEngDetDataLineDOListItem = finAccountEngineCustomDO.getFinAccEngDetDataLineDOList().stream()
                                .filter(finAccEngDetDataLineDO -> finAccEngDetDataDO.getId().equals(finAccEngDetDataLineDO.getMasId()))
                                .collect(Collectors.toList());
                        Assert.notEmpty(finAccEngDetDataLineDOListItem, "查不到数据映射");

                        Map saveMap = new HashMap();

                        for (FinAccEngDetDataLineDO finAccEngDetDataLineDO : finAccEngDetDataLineDOListItem) {

                            if (!checkConstant(finAccEngDetDataLineDO, docNum, finAccountEngineParam, finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap)) {
                                return;
                            }
                            if (!checkColumn(finAccEngDetDataLineDO, result, docNum, finAccountEngineParam, finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap)) {
                                return;
                            }
                            if (!checkFastCode(finAccEngDetDataLineDO, finAccountEngineCustomDO, result, docNum, finAccountEngineParam, finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap)) {
                                return;
                            }
                            if (!checkFlexible(finAccEngDetDataLineDO, finAccountEngineCustomDO, docNum, finAccountEngineParam, finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap)) {
                                return;
                            }
                            if (!checkAccountPeriod(finAccEngDetDataLineDO, result, finAccountEngineCustomDO, ouCode, docNum, finAccountEngineParam, finAccountEngineDetailsDO, finAccEngDetDataDO, saveMap)) {
                                return;
                            }
                        }
                        saveMapList.add(saveMap);
                    }
                }
            }
        } catch (Exception e) {
            saveFinJournalLog(docNum, e.getMessage());
        } finally {
            if (connection != null) {
                try {
                    connection.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
            if (resultSet != null) {
                try {
                    resultSet.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private boolean checkFinSetOfBookOu(FinAccountEngineCustomDO finAccountEngineCustomDO,
                                        String ouCode,
                                        String docNum,
                                        FinAccountEngineParam finAccountEngineParam,
                                        FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                        FinAccEngDetDataDO finAccEngDetDataDO) {


        String sobCode = finAccEngDetDataDO.getSobCode();
        FinSetOfBookDO finSetOfBookDOResult = finAccountEngineCustomDO.getFinSetOfBookDOList().stream()
                .filter(finSetOfBookDO -> finSetOfBookDO.getSobCode().equals(sobCode))
                .findFirst()
                .orElse(null);
        Assert.notNull(finSetOfBookDOResult, "查不到启用状态的账套");

        //帐套公司
        List<FinSetOfBookOuDO> finSetOfBookOuDOListItem = finAccountEngineCustomDO.getFinSetOfBookOuDOList().stream()
                .filter(finSetOfBookOuDO -> finSetOfBookDOResult.getId().equals(finSetOfBookOuDO.getMasId()))
                .collect(Collectors.toList());
        Assert.notNull(finSetOfBookOuDOListItem, "查不到账套公司");

        FinSetOfBookOuDO finSetOfBookOuDOResult = finSetOfBookOuDOListItem.stream()
                .filter(finSetOfBookOuDO -> ouCode.equals(finSetOfBookOuDO.getOuCode()))
                .findFirst()
                .orElse(null);

        if (finSetOfBookOuDOResult == null) {
            saveFinJournalLog(docNum,
                    String.format("{}，{},{},{}无分配此账套",
                            finAccountEngineParam.getEventTable(),
                            finAccountEngineDetailsDO.getAccountEngine(),
                            finAccEngDetDataDO.getName(),
                            ouCode));
            return false;
        }
        return true;
    }

    private FinAccountEngineCustomDO queryData(FinAccountEngineParam finAccountEngineParam) {

        //事件表单
        FinEventTableDO finEventTableDO = finEventTableRepo.findByEventTableAndStatus(finAccountEngineParam.getEventTable(),
                UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        Assert.notNull(finEventTableDO, "查不到启用状态的事件表单");

        //事件表单明细
        List<FinEventTableLineDO> finEventTableLineDOList = finEventTableLineRepo.findAllByMasId(finEventTableDO.getId());
        Assert.notEmpty(finEventTableLineDOList, "查不到事件表单明细");

        //事件表单条件
        List<FinEventTableConditionDO> finEventTableConditionDOList = finEventTableConditionRepo.findAllByMasId(finEventTableDO.getId());

        //会计引擎
        FinAccountEngineDO finAccountEngineDO = finAccountEngineRepo.findByIdAndStatus(finAccountEngineParam.getId(),
                UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        Assert.notNull(finAccountEngineDO, "查不到启用状态的会计引擎");

        //会计引擎明细
        List<FinAccountEngineDetailsDO> finAccountEngineDetailsDOList = finAccountEngineDetailsRepo
                .findAllByMasIdAndActiveFlag(finAccountEngineParam.getId(), Boolean.TRUE)
                .stream()
                .sorted(Comparator.comparing(FinAccountEngineDetailsDO::getSerialNum).reversed())
                .collect(Collectors.toList());
        Assert.notEmpty(finAccountEngineDetailsDOList, "查不到启用的会计引擎明细");

        List<Long> finAccountEngineDetailsDOIds = finAccountEngineDetailsDOList.stream().map(FinAccountEngineDetailsDO::getId).collect(Collectors.toList());

        //会计引擎条件
        List<FinAccEngDetConditionDO> finAccEngDetConditionDOList = finAccEngDetConditionRepo.findAllByMasIdIn(finAccountEngineDetailsDOIds);

        //数据行
        List<FinAccEngDetDataDO> finAccEngDetDataDOList = finAccEngDetDataRepo.findAllByMasIdIn(finAccountEngineDetailsDOIds).stream()
                .sorted(Comparator.comparing(FinAccEngDetDataDO::getSerialNum).reversed())
                .collect(Collectors.toList());
        Assert.notEmpty(finAccEngDetDataDOList, "查不到数据行详情");

        //数据映射
        List<Long> finAccEngDetDataDOIds = finAccEngDetDataDOList.stream().map(FinAccEngDetDataDO::getId).collect(Collectors.toList());
        List<FinAccEngDetDataLineDO> finAccEngDetDataLineDOList = finAccEngDetDataLineRepo.findAllByMasIdIn(finAccEngDetDataDOIds);
        Assert.notEmpty(finAccEngDetDataLineDOList, "查不到数据映射");

        //快码
        List<String> fastCodes = finAccEngDetDataLineDOList.stream().map(FinAccEngDetDataLineDO::getFastCode).collect(Collectors.toList());
        List<FinFastCodeDO> finFastCodeDOList = finFastCodeRepoProc.findAllByFastCodeInAndStatus(fastCodes, UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        //快码明细
        List<Long> finFastCodeDOIds = finFastCodeDOList.stream().map(FinFastCodeDO::getId).collect(Collectors.toList());
        List<FinFastCodeLineDO> finFastCodeLineDOList = finFastCodeLineRepoProc.findByMasIdIn(finFastCodeDOIds);

        //帐套
        List<String> sobCodes = finAccEngDetDataDOList.stream().map(FinAccEngDetDataDO::getSobCode).collect(Collectors.toList());
        List<FinSetOfBookDO> finSetOfBookDOList = finSetOfBookRepo.findAllBySobCodeInAndStatus(sobCodes, UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        Assert.notEmpty(finSetOfBookDOList, "查不到启用状态的帐套");

        //帐套公司
        List<Long> finSetOfBookDOIds = finSetOfBookDOList.stream().map(FinSetOfBookDO::getId).collect(Collectors.toList());
        List<FinSetOfBookOuDO> finSetOfBookOuDOList = finSetOfBookOuRepo.findAllByMasIdIn(finSetOfBookDOIds);
        Assert.notEmpty(finSetOfBookOuDOList, "查不到帐套帐套公司");

        //帐套明细
        List<FinSetOfBookLineDO> finSetOfBookLineDOList = finSetOfBookLineRepo.findAllByMasIdIn(finSetOfBookDOIds);
        Assert.notEmpty(finSetOfBookLineDOList, "查不到帐套明细");

        //会计期间
        List<FinSobAccountPeriodDO> finSobAccountPeriodDOList = finSobAccountPeriodRepo.findAllByStatus(UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_OPEN.getValueCode());

        List<String> flexibleCodes = finSetOfBookLineDOList.stream()
                .filter(finSetOfBookLineDO -> StringUtils.isNotEmpty(finSetOfBookLineDO.getFlexibleCode()))
                .map(FinSetOfBookLineDO::getFlexibleCode)
                .collect(Collectors.toList());

        //集值
        List<FinFlexibleDO> finFlexibleDOList = finFlexibleRepoProc.findAllByFlexibleCodeInAndStatus(flexibleCodes, UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        //集值明细
        List<Long> finSetOfBookLineDOIds = finFlexibleDOList.stream()
                .map(FinFlexibleDO::getId)
                .collect(Collectors.toList());

        List<FinFlexibleValueDO> finFlexibleValueDOList = finFlexibleValueRepoProc.findByMasIdInAndValidTime(finSetOfBookLineDOIds, LocalDateTime.now());


        FinAccountEngineCustomDO finAccountEngineCustomDO = new FinAccountEngineCustomDO();

        finAccountEngineCustomDO.setFinEventTableDO(finEventTableDO);
        finAccountEngineCustomDO.setFinEventTableLineDOList(finEventTableLineDOList);
        finAccountEngineCustomDO.setFinEventTableConditionDOList(finEventTableConditionDOList);
        finAccountEngineCustomDO.setFinAccountEngineDetailsDOList(finAccountEngineDetailsDOList);
        finAccountEngineCustomDO.setFinAccEngDetConditionDOList(finAccEngDetConditionDOList);
        finAccountEngineCustomDO.setFinAccEngDetDataDOList(finAccEngDetDataDOList);
        finAccountEngineCustomDO.setFinAccEngDetDataLineDOList(finAccEngDetDataLineDOList);
        finAccountEngineCustomDO.setFinFastCodeDOList(finFastCodeDOList);
        finAccountEngineCustomDO.setFinFastCodeLineDOList(finFastCodeLineDOList);
        finAccountEngineCustomDO.setFinSetOfBookDOList(finSetOfBookDOList);
        finAccountEngineCustomDO.setFinSetOfBookOuDOList(finSetOfBookOuDOList);
        finAccountEngineCustomDO.setFinSetOfBookLineDOList(finSetOfBookLineDOList);
        finAccountEngineCustomDO.setFinSobAccountPeriodDOList(finSobAccountPeriodDOList);
        finAccountEngineCustomDO.setFinFlexibleDOList(finFlexibleDOList);
        finAccountEngineCustomDO.setFinFlexibleValueDOList(finFlexibleValueDOList);
        return finAccountEngineCustomDO;
    }


    private boolean checkConstant(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                  String docNum,
                                  FinAccountEngineParam finAccountEngineParam,
                                  FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                  FinAccEngDetDataDO finAccEngDetDataDO,
                                  Map saveMap) {
        if (UdcEnum.SOURCE_TYPE_CONSTANT.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {
            if (StringUtils.isEmpty(finAccEngDetDataLineDO.getConstant())) {
                saveFinJournalLog(docNum,
                        String.format("{},{},{},{}常数出错",
                                finAccountEngineParam.getEventTable(),
                                finAccountEngineDetailsDO.getAccountEngine(),
                                finAccEngDetDataDO.getName(),
                                finAccEngDetDataLineDO.getColumnName()));
                return false;
            }

            put(saveMap, finAccEngDetDataLineDO.getColumnName(), finAccEngDetDataLineDO.getConstant());
        }
        return true;
    }

    private boolean checkColumn(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                Map result,
                                String docNum,
                                FinAccountEngineParam finAccountEngineParam,
                                FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                FinAccEngDetDataDO finAccEngDetDataDO,
                                Map saveMap) {
        if (UdcEnum.SOURCE_TYPE_COLUMN.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {
            if (result.get(finAccEngDetDataLineDO.getColumnSource()) == null) {

                saveFinJournalLog(docNum,
                        String.format("{},{},{},{}列来源字段出错",
                                finAccountEngineParam.getEventTable(),
                                finAccountEngineDetailsDO.getAccountEngine(),
                                finAccEngDetDataDO.getName(),
                                finAccEngDetDataLineDO.getColumnName()));
                return false;
            }
            put(saveMap, finAccEngDetDataLineDO.getColumnName(), finAccEngDetDataLineDO.getColumnSource());
        }
        return true;
    }

    private boolean checkFastCode(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                  FinAccountEngineCustomDO finAccountEngineCustomDO,
                                  Map result,
                                  String docNum,
                                  FinAccountEngineParam finAccountEngineParam,
                                  FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                  FinAccEngDetDataDO finAccEngDetDataDO,
                                  Map saveMap) {
        if (UdcEnum.SOURCE_TYPE_FAST_CODE.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {

            FinFastCodeDO finFastCodeDOResult = finAccountEngineCustomDO.getFinFastCodeDOList().stream()
                    .filter(finFastCodeDO -> finFastCodeDO.getFastCode().equals(finAccEngDetDataLineDO.getFastCode()))
                    .findFirst()
                    .orElse(null);

            Assert.notNull(finFastCodeDOResult, "查不到启用状态的快码");

            List<FinFastCodeLineDO> finFastCodeLineDOListItem = finAccountEngineCustomDO.getFinFastCodeLineDOList().stream()
                    .filter(finFastCodeLineDO -> finFastCodeDOResult.getId().equals(finFastCodeLineDO.getMasId()))
                    .collect(Collectors.toList());
            Assert.notNull(finFastCodeDOResult, "没查到快码明细");

            FinFastCodeLineDO finFastCodeLineDOResult = finFastCodeLineDOListItem.stream().filter(finFastCodeLineDO -> {

                if (StringUtils.isNotEmpty(finFastCodeLineDO.getConditionOne())) {
                    if (!finFastCodeLineDO.getConditionOneValue().equals(result.get(finFastCodeLineDO.getConditionOne()))) {
                        return false;
                    }
                }


                if (StringUtils.isNotEmpty(finFastCodeLineDO.getConditionTwo())) {
                    if (!finFastCodeLineDO.getConditionTwoValue().equals(result.get(finFastCodeLineDO.getConditionTwo()))) {
                        return false;
                    }
                }

                if (StringUtils.isNotEmpty(finFastCodeLineDO.getConditionThree())) {
                    if (!finFastCodeLineDO.getConditionThreeValue().equals(result.get(finFastCodeLineDO.getConditionThree()))) {
                        return false;
                    }
                }
                return true;
            }).findFirst().orElse(null);

            if (finFastCodeLineDOResult == null) {

                saveFinJournalLog(docNum,
                        String.format("{},{},{},{}快码出错",
                                finAccountEngineParam.getEventTable(),
                                finAccountEngineDetailsDO.getAccountEngine(),
                                finAccEngDetDataDO.getName(),
                                finAccEngDetDataLineDO.getColumnName()));
                return false;
            }

            if (FinConstant.OUT_PUT_ONE.equals(finAccEngDetDataLineDO.getOutputColumn())) {
                saveMap.put(StringUtil.convertToCamelCase(finAccEngDetDataLineDO.getColumnName()), finFastCodeLineDOResult.getOutputOne());
            }

            if (FinConstant.OUT_PUT_TWO.equals(finAccEngDetDataLineDO.getOutputColumn())) {
                saveMap.put(StringUtil.convertToCamelCase(finAccEngDetDataLineDO.getColumnName()), finFastCodeLineDOResult.getOutputTwo());
            }

            if (FinConstant.OUT_PUT_THREE.equals(finAccEngDetDataLineDO.getOutputColumn())) {
                saveMap.put(StringUtil.convertToCamelCase(finAccEngDetDataLineDO.getColumnName()), finFastCodeLineDOResult.getOutputThree());
            }
        }
        return true;
    }

    private boolean checkFlexible(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                  FinAccountEngineCustomDO finAccountEngineCustomDO,
                                  String docNum,
                                  FinAccountEngineParam finAccountEngineParam,
                                  FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                  FinAccEngDetDataDO finAccEngDetDataDO,
                                  Map saveMap) {
        if (UdcEnum.SOURCE_TYPE_FLEXIBLE.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {

            FinFlexibleValueDO finFlexibleValueDOResult = finAccountEngineCustomDO.getFinFlexibleValueDOList().stream()
                    .filter(finFlexibleValueDO -> finFlexibleValueDO.getFlexibleValueCode().equals(finAccEngDetDataLineDO.getFlexibleCode()))
                    .findFirst()
                    .orElse(null);

            Assert.notNull(finFlexibleValueDOResult, "查不到有效的值集明细");

            if (finFlexibleValueDOResult == null) {

                saveFinJournalLog(docNum,
                        String.format("{},{},{},{}值集出错",
                                finAccountEngineParam.getEventTable(),
                                finAccountEngineDetailsDO.getAccountEngine(),
                                finAccEngDetDataDO.getName(),
                                finAccEngDetDataLineDO.getColumnName()));
                return false;
            }

            put(saveMap, finAccEngDetDataLineDO.getColumnName(), finFlexibleValueDOResult.getFlexibleValueName());
        }
        return true;
    }

    private boolean checkAccountPeriod(FinAccEngDetDataLineDO finAccEngDetDataLineDO,
                                       Map result,
                                       FinAccountEngineCustomDO finAccountEngineCustomDO,
                                       String ouCode,
                                       String docNum,
                                       FinAccountEngineParam finAccountEngineParam,
                                       FinAccountEngineDetailsDO finAccountEngineDetailsDO,
                                       FinAccEngDetDataDO finAccEngDetDataDO,
                                       Map saveMap) {
        if (UdcEnum.SOURCE_TYPE_ACCOUNT_PERIOD.getValueCode().equals(finAccEngDetDataLineDO.getSourceType())) {


            Assert.notNull(result.get(finAccEngDetDataLineDO.getColumnSource()), "会计期间为空");

            LocalDateTime localDateTime = (LocalDateTime) result.get(finAccEngDetDataLineDO.getColumnSource());

            String sobCode = finAccEngDetDataDO.getSobCode();
            FinSetOfBookDO finSetOfBookDOResult = finAccountEngineCustomDO.getFinSetOfBookDOList().stream()
                    .filter(finSetOfBookDO -> finSetOfBookDO.getSobCode().equals(sobCode))
                    .findFirst()
                    .orElse(null);
            Assert.notNull(finSetOfBookDOResult, "查不到启用状态的账套");

            FinSobAccountPeriodDO finSobAccountPeriodDOResult = finAccountEngineCustomDO.getFinSobAccountPeriodDOList().stream().filter(finSobAccountPeriodDO ->

                    finSetOfBookDOResult.getSobCode().equals(finSobAccountPeriodDO.getSobCode()) &&
                            ouCode.equals(finSobAccountPeriodDO.getOuCode()) &&
                            UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_OPEN.getValueCode().equals(finSobAccountPeriodDO.getStatus()) &&
                            localDateTime.compareTo(finSobAccountPeriodDO.getActiveStartTime()) >= 0 &&
                            localDateTime.compareTo(finSobAccountPeriodDO.getActiveEndTime()) <= 0)
                    .findFirst().orElse(null);

            if (finSobAccountPeriodDOResult == null) {

                saveFinJournalLog(docNum,
                        String.format("{},{},{},{}会计期间出错",
                                finAccountEngineParam.getEventTable(),
                                finAccountEngineDetailsDO.getAccountEngine(),
                                finAccEngDetDataDO.getName(),
                                finAccEngDetDataLineDO.getColumnName()));
                return false;
            }
            put(saveMap, finAccEngDetDataLineDO.getColumnName(), finSobAccountPeriodDOResult.getPeriodStyle());
        }
        return true;
    }

    private void checkGenerateJournalParam(FinAccountEngineParam finAccountEngineParam) {
        Assert.notEmpty(finAccountEngineParam.getEventTable(), "事件表单名称不能为空");
        Assert.notNull(finAccountEngineParam.getId(), "id不能为空");
    }

    private void put(Map map, String key, String value) {
        map.put(StringUtil.convertToCamelCase(key), value);
    }

    private void saveFinJournal(FinAccountEngineParam finAccountEngineParam, List<Map> saveMapList) {

        if (CollectionUtils.isEmpty(saveMapList)) {
            return;
        }

        //会计引擎
        FinAccountEngineDO finAccountEngineDO = finAccountEngineRepo.findAllById(finAccountEngineParam.getId());
        Assert.notNull(finAccountEngineDO, "查不到会计引擎");

        Connection connection = null;
        PreparedStatement preparedStatement = null;

        try {
            Class.forName(FinConstant.DRIVEN_NAME);

            connection = DriverManager.getConnection(String.format(FinConstant.CONNECTION_NAME,
                    finAccountEngineDO.getJournalHost(),
                    finAccountEngineDO.getJournalPort(),
                    finAccountEngineDO.getJournalDatabase()),
                    finAccountEngineDO.getJournalUser(),
                    finAccountEngineDO.getJournalPassword());

            String sql = buildSql(finAccountEngineDO, saveMapList);

            preparedStatement = connection.prepareStatement(sql);
            preparedStatement.execute();

        } catch (Exception e) {
            throw new BusinessException(e);
        } finally {
            if (connection != null) {
                try {
                    connection.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
            if (preparedStatement != null) {
                try {
                    preparedStatement.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private String buildSql(FinAccountEngineDO finAccountEngineDO, List<Map> saveMapList) {


        List<String> values = saveMapList.stream().map(map -> FinConstant.LEFT_PARENTHESIS
                .concat(StringUtils.join(map.values(), FinConstant.COMMA))
                .concat(FinConstant.RIGHT_PARENTHESIS)).collect(Collectors.toList());

        String sql = FinConstant.INSERT
                .concat(FinConstant.BLANK)
                .concat(FinConstant.INTO)
                .concat(FinConstant.BLANK)
                .concat(finAccountEngineDO.getJournalTable())
                .concat(FinConstant.BLANK)
                .concat(FinConstant.LEFT_PARENTHESIS)
                .concat(StringUtils.join(saveMapList.get(0).keySet(), FinConstant.COMMA))
                .concat(FinConstant.RIGHT_PARENTHESIS)
                .concat(FinConstant.BLANK)
                .concat(FinConstant.VALUES)
                .concat(StringUtils.join(values, FinConstant.COMMA));

        return sql;
    }


    private void saveFinJournalLog(String docNum, String log) {
        FinJournalLogDO finJournalLogDo = new FinJournalLogDO();
        finJournalLogDo.setDocNum(docNum);
        finJournalLogDo.setLog(log);
        finJournalLogRepo.save(finJournalLogDo);
    }
}
