package com.elitesland.fin.application.service.creditaccount;


import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountInitialLimitParam;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountParam;
import com.elitesland.fin.application.facade.vo.creditaccount.InitialLimitPageVO;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.creditaccount.CreditAccountInitialLimitDO;
import com.elitesland.fin.repo.creditaccount.CreditAccountInitialLimitRepo;
import com.elitesland.fin.repo.creditaccount.CreditAccountInitialLimitRepoProc;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepoProc;
import com.elitesland.fin.repo.creditaccountflow.CreditAccountFlowRepo;
import com.elitesland.fin.repo.creditaccountflow.CreditAccountFlowRepoProc;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service
@AllArgsConstructor
public class CreditAccountInitialLimitServiceImpl implements CreditAccountInitialLimitService {

    private final CreditAccountInitialLimitRepo creditAccountInitialLimitRepo;
    private final CreditAccountInitialLimitRepoProc creditAccountInitialLimitRepoProc;
    private final CreditAccountRepoProc creditAccountRepoProc;

    private final CreditAccountFlowRepoProc creditAccountFlowRepoProc;
    private final CreditAccountFlowRepo creditAccountFlowRepo;

    @Override
    @SysCodeProc
    public PagingVO<InitialLimitPageVO> pageSearch(CreditAccountInitialLimitParam pageParam) {
        return creditAccountInitialLimitRepoProc.pageSearch(pageParam);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void finishedInitial(List<Long> ids) {
        Assert.notEmpty(ids, "请选择需要结束初始化的数据");

        // 查找期初余额
        List<CreditAccountInitialLimitDO> initialLimitDOList = creditAccountInitialLimitRepo.findAllById(ids);
        Assert.notEmpty(initialLimitDOList, "没有查询到需要初始化的数据");

        List<CreditAccountInitialLimitDO> finishedDOS = initialLimitDOList.stream().filter(item -> StrUtil.equals(item.getStatus(), UdcEnum.INITIAL_STATUS_FINISHED.getValueCode())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.isEmpty(finishedDOS), "请选择没有初始化的数据");

        // 更新账户初始余额
        List<CreditAccountParam> params = new ArrayList<>();
        initialLimitDOList.stream().forEach(item -> {

            Assert.isFalse(creditAccountFlowRepo.existsByCreditAccountCode(item.getCreditAccountCode()), "账户{}已经生成过流水了,不能再结束初始化", item.getCreditAccountCode());

            CreditAccountParam param = new CreditAccountParam();
            param.setCreditAccountCodeName(item.getCreditAccountCode());
            param.setCreditAccountLimit(ObjectUtil.isNotNull(item.getInitialAccountLimit()) ? item.getInitialAccountLimit() : BigDecimal.ZERO);
            param.setCreditAccountOccupancyLimit(ObjectUtil.isNotNull(item.getInitialAccountOccupancyLimit()) ? item.getInitialAccountOccupancyLimit() : BigDecimal.ZERO);
            param.setCreditAccountUsedLimit(ObjectUtil.isNotNull(item.getInitialAccountUsedLimit()) ? item.getInitialAccountUsedLimit() : BigDecimal.ZERO);
            param.setCreditAccountAvailableLimit(NumberUtil.sub(item.getInitialAccountLimit(), item.getInitialAccountOccupancyLimit(), item.getInitialAccountUsedLimit()));
            params.add(param);
        });
        creditAccountRepoProc.updateByParam(params);

        // 更新期初余额列表状态
        initialLimitDOList.stream().forEach(item -> {
            item.setStatus(UdcEnum.INITIAL_STATUS_FINISHED.getValueCode());
            item.setFinishedTime(LocalDateTime.now());
            item.setInitialAccountAvailableLimit(NumberUtil.sub(item.getInitialAccountLimit(), item.getInitialAccountOccupancyLimit(), item.getInitialAccountUsedLimit()));
        });
        creditAccountInitialLimitRepo.saveAll(initialLimitDOList);

    }
}
