package com.elitesland.fin.application.service.creditaccount;


import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.creditaccount.CreditAccountIocConvert;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountIocDTO;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountIocParam;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountIocParamVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountIocRespVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountIocSaveVO;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.service.creditaccount.CreditAccountIocDomainService;
import com.elitesland.fin.entity.creditaccount.CreditAccountIoc;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author gyj
 * @date 2023/05/24 18:23
 */
@Slf4j
@Service
@AllArgsConstructor
public class CreditAccountIocServiceImpl implements CreditAccountIocService {

    private final CreditAccountIocDomainService creditAccountIocDomainService;


    /**
     * 新增账户事务码数据
     *
     * @param saveVo 新增账户事务码数据
     * @return 新增账户事务码ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> createOne(CreditAccountIocSaveVO saveVo) {

        CreditAccountIoc accountIoc = CreditAccountIocConvert.INSTANCE.saveVoToEn(saveVo);
        //新增时默认启用
        if (StringUtils.isEmpty(accountIoc.getStatus())) {
            accountIoc.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        }
        //检查必填项
        checkIocMandatory(accountIoc);

        //校验编码和名称是否已存在
        checkIocCodeAndName(accountIoc);
        //保存
        Long id = creditAccountIocDomainService.saveAccountIoc(accountIoc, Optional.empty());

        return ApiResult.ok(id);
    }

    /**
     * 修改账户事务码数据
     *
     * @param saveVo 修改账户事务码对象
     * @return 账户事务码ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> update(CreditAccountIocSaveVO saveVo) {

        val iocDTOOptional = creditAccountIocDomainService.findById(saveVo.getId());
        if (iocDTOOptional.isEmpty()) {
            return ApiResult.fail("未查询到账户事务码信息!");
        }

        CreditAccountIoc accountIoc = CreditAccountIocConvert.INSTANCE.saveVoToEn(saveVo);
        //检查必填项
        checkIocMandatory(accountIoc);
        //校验编码和名称是否已存在
        checkIocCodeAndName(accountIoc);
        //保存
        Long id = creditAccountIocDomainService.saveAccountIoc(accountIoc, iocDTOOptional);

        return ApiResult.ok(id);
    }


    /**
     * 检查必填项
     *
     * @param accountIoc 账户事务码对象
     */
    private void checkIocMandatory(CreditAccountIoc accountIoc) {
        if (StringUtils.isEmpty(accountIoc.getIoCode())) {
            throw new BusinessException(ApiCode.FAIL, "事务码不能为空!");
        }
        if (StringUtils.isEmpty(accountIoc.getIoName())) {
            throw new BusinessException(ApiCode.FAIL, "事务码名称不能为空!");
        }
        if (StringUtils.isEmpty(accountIoc.getIoTypeName())) {
            throw new BusinessException(ApiCode.FAIL, "事务类型名称不能为空!");
        }
        if (StringUtils.isEmpty(accountIoc.getIoType())) {
            throw new BusinessException(ApiCode.FAIL, "事务类型编码不能为空!");
        }
        if (StringUtils.isEmpty(accountIoc.getStatus())) {
            throw new BusinessException(ApiCode.FAIL, "是否启用不能为空!");
        }


    }

    /**
     * 校验编码和名称是否已存在
     *
     * @param accountIoc 账户事务码对象
     */
    private void checkIocCodeAndName(CreditAccountIoc accountIoc) {

        Set<String> ioCodeSet = new HashSet<>();
        ioCodeSet.add(accountIoc.getIoCode());

        CreditAccountIocParam accountIocParam = new CreditAccountIocParam();
        accountIocParam.setIoCodeSet(ioCodeSet);

        List<CreditAccountIocDTO> accountIocDTOList = creditAccountIocDomainService.queryByAccountIocParam(accountIocParam);


        CreditAccountIocDTO accountIocDTO = accountIocDTOList.stream().filter(dto -> StringUtils.equals(accountIoc.getIoCode(), dto.getIoCode()))
                .findFirst()
                .orElse(null);
        if (accountIocDTO != null) {
            Assert.equals(accountIoc.getId(), accountIocDTO.getId(), "账户事务码已存在");
        }

        Set<String> ioNameSet = new HashSet<>();
        ioNameSet.add(accountIoc.getIoName());

        accountIocParam = new CreditAccountIocParam();
        accountIocParam.setIoNameSet(ioNameSet);

        List<CreditAccountIocDTO> accountIocNameList = creditAccountIocDomainService.queryByAccountIocParam(accountIocParam);
        accountIocDTO = accountIocNameList.stream().filter(dto -> StringUtils.equals(accountIoc.getIoName(), dto.getIoName()))
                .findFirst()
                .orElse(null);
        if (accountIocDTO != null) {
            Assert.equals(accountIoc.getId(), accountIocDTO.getId(), "事务码名称已存在");
        }
    }


    /**
     * 通过账户事务码ID查询单条数据
     *
     * @param id 账户事务码ID
     * @return 单条账户事务码数据
     */
    @Override
    @SysCodeProc
    public ApiResult<CreditAccountIocRespVO> findIdOne(Long id) {
        val accountIocDTOOptional = creditAccountIocDomainService.findById(id);
        if (accountIocDTOOptional.isEmpty()) {
            return ApiResult.fail("未查询到账户事务码信息!");
        }
        CreditAccountIocDTO accountIocDTO = accountIocDTOOptional.get();

        //转换RespVO
        CreditAccountIocRespVO accountIocRespVO = CreditAccountIocConvert.INSTANCE.dtoToRespVo(accountIocDTO);

        return ApiResult.ok(accountIocRespVO);
    }

    /**
     * 通过账户事务码查询单条数据
     *
     * @param code 账户事务码编码
     * @return 单条账户事务码数据
     */
    @Override
    @SysCodeProc
    public Optional<CreditAccountIocRespVO> findByCode(String code) {
        val accountDTOOptional = creditAccountIocDomainService.findByCode(code);
        if (accountDTOOptional.isEmpty()) {
            return Optional.empty();
        }
        CreditAccountIocDTO accountIocDTO = accountDTOOptional.get();

        //转换RespVO
        CreditAccountIocRespVO accountIocRespVO = CreditAccountIocConvert.INSTANCE.dtoToRespVo(accountIocDTO);

        return Optional.of(accountIocRespVO);
    }

    /**
     * 通过账户事务码集合查询数据
     *
     * @param codeList 账户事务码编码集合
     * @return 账户事务码数据集合
     */
    @Override
    @SysCodeProc
    public List<CreditAccountIocRespVO> findByCodeBatch(List<String> codeList) {
        List<CreditAccountIocDTO> accountIocDTOList = creditAccountIocDomainService.findByCodeBatch(codeList);
        if (CollectionUtils.isEmpty(accountIocDTOList)) {
            return Collections.EMPTY_LIST;
        }

        //转换RespVO
        List<CreditAccountIocRespVO> accountIocRespVOList = accountIocDTOList.stream().map(
                CreditAccountIocConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());

        return accountIocRespVOList;
    }

    /**
     * 分页查询账户事务码
     *
     * @param param 分类查询、分页和排序对象
     * @return 多条账户事务码数据
     */
    @Override
    @SysCodeProc
    public PagingVO<CreditAccountIocRespVO> search(CreditAccountIocParamVO param) {

        val pagingVo = creditAccountIocDomainService.search(param);
        if (CollectionUtils.isEmpty(pagingVo.getRecords())) {
            return PagingVO.<CreditAccountIocRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        val vos = pagingVo.getRecords();


        //转换RespVO
        List<CreditAccountIocRespVO> whAreaSettingRespVOList = vos.stream().map(
                CreditAccountIocConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());


        return PagingVO.<CreditAccountIocRespVO>builder()
                .total(pagingVo.getTotal())
                .records(whAreaSettingRespVOList)
                .build();
    }

    /**
     * 根据入参查询账户事务码
     *
     * @param param 分类查询、分页和排序对象
     * @return 多条账户事务码数据
     */
    @Override
    @SysCodeProc
    public List<CreditAccountIocRespVO> select(CreditAccountIocParamVO param) {
        List<CreditAccountIocDTO> accountIocDTOList = creditAccountIocDomainService.selectByParam(param);

        if (CollectionUtils.isEmpty(accountIocDTOList)) {
            return Collections.EMPTY_LIST;
        }

        //转换RespVO
        List<CreditAccountIocRespVO> accountIocRespVOList = accountIocDTOList.stream().map(
                CreditAccountIocConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());

        return accountIocRespVOList;
    }

    @Override
    public List<CreditAccountIocDTO> queryByAccountIocParam(CreditAccountIocParam accountIocParam) {
        return creditAccountIocDomainService.queryByAccountIocParam(accountIocParam);
    }


    /**
     * 根据账户事务码ID批量逻辑删除
     *
     * @param ids 账户事务码ID集合
     * @return 账户事务码ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<String> deleteBatch(List<Long> ids) {

        creditAccountIocDomainService.updateDeleteFlagByIds(1, ids);
        return ApiResult.ok();
    }

}
