package com.elitesland.fin.application.service.excel.imp;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitesland.fin.application.facade.excel.flowrepair.CreditAccountFlowRepairImportEntity;
import com.elitesland.fin.application.facade.param.creditaccountflow.CreditAccountFlowParam;
import com.elitesland.fin.application.facade.vo.creditaccountflow.CreditAccountFlowVO;
import com.elitesland.fin.application.service.creditaccountflow.CreditAccountFlowService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.flowrepair.CreditAccountFlowRepairDO;
import com.elitesland.fin.repo.flowrepair.CreditAccountFlowRepairRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明: 信用账户流水修复重算
 * </p>
 *
 * @Author Darren
 * @Date 2023/10/26
 * @Version 1.0
 * @Content:
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class CreditAccountFlowRepairImportServiceImpl implements DataImport<CreditAccountFlowRepairImportEntity> {
    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    private final CreditAccountFlowService creditAccountFlowService;
    private final CreditAccountFlowRepairRepo creditAccountFlowRepairRepo;

    @Override
    public String getTmplCode() {
        return "credit_acc_flow_repair_import";

    }

    @Override
    public List<String> executeImport(List<CreditAccountFlowRepairImportEntity> dataList, int rowId) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        //导入结果集
        List<CreditAccountFlowRepairImportEntity> flowRepairImportList = new ArrayList<>();

        List<String> repairFlowNoList = dataList.stream().map(CreditAccountFlowRepairImportEntity::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());

        List<CreditAccountFlowVO> flowVOList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(repairFlowNoList)){
            CreditAccountFlowParam flowQueryParam = new CreditAccountFlowParam();
            flowQueryParam.setFlowNoList(repairFlowNoList);
            flowVOList = creditAccountFlowService.selectListByQueryParam(flowQueryParam);
        }

        List<String> flowNoList = flowVOList.stream().map(CreditAccountFlowVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());


        List<String> errorResult = new ArrayList<>();

        for (CreditAccountFlowRepairImportEntity data : dataList) {
            List<String> errorList = new ArrayList<>();

            //导入数据是否存在重复
            int finalRowId = rowId;
            data.setRowId(rowId);

            String flowNo = data.getFlowNo();
            if (StringUtils.isBlank(flowNo)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "流水号", "不能为空"));
            } else {
                // 如果不包含流水号， 抛出错误
                if (!flowNoList.contains(flowNo)) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "流水号", "流水号不存在，请输入正确流水号"));
                }
            }

            // 获取实际金额
            BigDecimal actualAmount = data.getActualAmount();
            if (Objects.isNull(actualAmount)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "实际金额", "不能为空"));
            }

            if (StringUtils.isNotBlank(flowNo) && Objects.nonNull(actualAmount)){
                flowRepairImportList.stream().filter(v -> Objects.equals(flowNo, v.getFlowNo())
                        && (actualAmount.compareTo(v.getActualAmount()) == 0) ).findFirst().ifPresent(flowExcel ->
                        errorList.add(MessageFormat.format(ERROR_TEMPLATE, finalRowId, "整行", "导入重复数据,与第" + flowExcel.getRowId() + "行重复"))
                );
            }

            // 如果没有错误信息 则将数据加入到结果集合中
            if (CollectionUtils.isEmpty(errorList)) {
                flowRepairImportList.add(data);
                errorResult.add(null);
            } else {
                errorResult.add(StringUtils.join(errorList, FinConstant.WRAP));
            }
            rowId++;

        }


        if (CollectionUtils.isNotEmpty(flowRepairImportList)) {

            saveData(flowRepairImportList, flowVOList);

            return CollectionUtil.isNotEmpty(errorResult) ? errorResult : null;
        } else {
            return errorResult;
        }

    }

    private void saveData(List<CreditAccountFlowRepairImportEntity> saveList, List<CreditAccountFlowVO> flowVOList) {

        List<CreditAccountFlowRepairDO> flowRepairDOList = new ArrayList<>();
        Map<String, List<CreditAccountFlowRepairImportEntity>> saveListMap = saveList.stream().collect(Collectors.groupingBy(CreditAccountFlowRepairImportEntity::getFlowNo));
        for (String flowNo : saveListMap.keySet()) {
            List<CreditAccountFlowRepairImportEntity> repairFlowList = saveListMap.get(flowNo);
            CreditAccountFlowVO creditAccountFlowVO = flowVOList.stream().filter(flowVO -> Objects.equals(flowVO.getFlowNo(), flowNo)).findFirst().get();

            List<CreditAccountFlowRepairDO> creditAccountFlowRepairDOList = repairFlowList.stream().map(flowExcel -> {
                CreditAccountFlowRepairDO flowRepairDO = new CreditAccountFlowRepairDO();
                flowRepairDO.setFlowNo(flowExcel.getFlowNo());
                flowRepairDO.setActualAmount(flowExcel.getActualAmount());
                flowRepairDO.setAmount(creditAccountFlowVO.getAmount());
                flowRepairDO.setOpenAccountEntityCode(creditAccountFlowVO.getObjectCode());
                flowRepairDO.setOpenAccountEntityName(creditAccountFlowVO.getObjectName());
                flowRepairDO.setAccountType(creditAccountFlowVO.getCreditAccountType());
                flowRepairDO.setAccountCode(creditAccountFlowVO.getCreditAccountCode());
                flowRepairDO.setAccountName(creditAccountFlowVO.getCreditAccountName());
                flowRepairDO.setSourceNo(creditAccountFlowVO.getSourceNo());
                flowRepairDO.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());
                return flowRepairDO;
            }).collect(Collectors.toList());

            flowRepairDOList.addAll(creditAccountFlowRepairDOList);
        }

        // 保存数据
        creditAccountFlowRepairRepo.saveAll(flowRepairDOList);

    }


}
