package com.elitesland.fin.application.service.payment;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.payment.PaymentRuleConfigConvert;
import com.elitesland.fin.application.convert.payment.PaymentRuleConfigDtlConvert;
import com.elitesland.fin.application.facade.dto.payment.PaymentRuleConfigDTO;
import com.elitesland.fin.application.facade.dto.payment.PaymentRuleConfigDtlDTO;
import com.elitesland.fin.application.facade.param.payment.PaymentRuleConfigParam;
import com.elitesland.fin.application.facade.vo.payment.PaymentRuleConfigDtlVO;
import com.elitesland.fin.application.facade.vo.payment.PaymentRuleConfigPageVO;
import com.elitesland.fin.application.facade.vo.payment.PaymentRuleConfigQueryVO;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinRedisConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.service.payment.PaymentRuleConfigDomainService;
import com.elitesland.fin.domain.service.payment.PaymentRuleConfigDtlDomainService;
import com.elitesland.fin.entity.payment.PaymentRuleConfigDO;
import com.elitesland.fin.entity.payment.PaymentRuleConfigDtlDO;
import com.elitesland.fin.repo.payment.PaymentRuleConfigDtlRepo;
import com.elitesland.fin.repo.payment.PaymentRuleConfigRepo;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RBucket;
import org.redisson.api.RedissonClient;
import org.springframework.data.domain.Example;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * @author gyj
 * @date 2023/05/24 18:23
 */
@Slf4j
@Service
@AllArgsConstructor
@Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
public class PaymentRuleConfigServiceImpl implements PaymentRuleConfigService {

    private final PaymentRuleConfigDomainService paymentRuleConfigDomainService;

    private final PaymentRuleConfigDtlDomainService paymentRuleConfigDtlDomainService;

    private final RedissonClient redissonClient;

    private final PaymentRuleConfigRepo paymentRuleConfigRepo;

    private final PaymentRuleConfigDtlRepo paymentRuleConfigDtlRepo;

    private final UdcProvider udcProvider;

    @Override
    @SysCodeProc
    public PagingVO<PaymentRuleConfigPageVO> searchPage(PaymentRuleConfigParam param) {
        PagingVO<PaymentRuleConfigPageVO> paymentRuleConfigPageVOPagingVO = paymentRuleConfigDomainService.searchPage(param);
        pageCode2Name(paymentRuleConfigPageVOPagingVO.getRecords());
        return paymentRuleConfigPageVOPagingVO;
    }

    @Override
    public void saveOrUpdate(PaymentRuleConfigParam param) {
        validateSaveOrUpdateParam(param);
        persistence(param);
    }

    private void persistence(PaymentRuleConfigParam param) {
        checkRepeatData(param);
        checkExist(param);

        if (param.getId() == null) {
            saveConfigAndDtl(param);
            return;
        }
        updateConfigAndDtl(param);
    }

    private void checkExist(PaymentRuleConfigParam param) {
        List<PaymentRuleConfigDO> paymentRuleConfigDOList;

        paymentRuleConfigDOList = paymentRuleConfigRepo.findAllByRuleCode(param.getRuleCode());
        if (CollectionUtils.isNotEmpty(paymentRuleConfigDOList)) {
            Assert.isTrue(paymentRuleConfigDOList.size() == 1, "存在多条相同的规则编码");
            Assert.equals(paymentRuleConfigDOList.get(0).getId(), param.getId(), "规则编码已经存在");
        }

        paymentRuleConfigDOList = paymentRuleConfigRepo.findAllByOptDoc(param.getOptDoc());
        if (CollectionUtils.isNotEmpty(paymentRuleConfigDOList)) {
            Assert.isTrue(paymentRuleConfigDOList.size() == 1, "存在多条相同的适用单据");
            Assert.equals(paymentRuleConfigDOList.get(0).getId(), param.getId(), "适用单据已经存在");
        }
    }

    private void checkRepeatData(PaymentRuleConfigParam param) {
        Map<Integer, String> seq = new HashMap();
        param.getPaymentRuleConfigDtlDTOList().stream().forEach(item -> {
            Assert.isNull(seq.get(item.getPriorityNo()), "优先级不能重复");
            seq.put(item.getPriorityNo(), "");
        });


        Set<String> repeatSet = new HashSet<>();
        param.getPaymentRuleConfigDtlDTOList().stream().forEach(item -> {

            Arrays.stream(item.getOptDocType().split(FinConstant.TRANSLATION_PLUS)).forEach(optDocType -> {
                String uniqueKey = optDocType.concat(item.getOptDocStatus()).concat(item.getOptAccountType());
                Assert.isFalse(repeatSet.contains(uniqueKey), "存在重复规则,扣款单据类型、扣款单据状态、账户类型需唯一");
                repeatSet.add(uniqueKey);
            });
        });

        //判断空指定用和有指定用户重复的情况
        repeatSet.stream().forEach(item -> {
            repeatSet.stream().forEach(item1 -> {
                if (item.contains(item1) && !StringUtils.equals(item, item1)) {
                    Assert.isTrue(false, "存在重复规则,扣款单据类型、扣款单据状态、账户类型需唯一");
                }
            });
        });
    }

    private void updateConfigAndDtl(PaymentRuleConfigParam param) {
        PaymentRuleConfigDO paymentRuleConfigDO = PaymentRuleConfigConvert.INSTANCE.param2DO(param);

        List<PaymentRuleConfigDtlDO> paymentRuleConfigDtlDOS = PaymentRuleConfigDtlConvert.INSTANCE.param2DOList(param.getPaymentRuleConfigDtlDTOList());
        paymentRuleConfigDtlDOS.stream().forEach(dtl -> dtl.setMasId(paymentRuleConfigDO.getId()));
        paymentRuleConfigDomainService.updateDynamically(paymentRuleConfigDO);
        paymentRuleConfigDtlDomainService.deleteByParam(getQueryBeanByMasId(paymentRuleConfigDO.getId()));
        paymentRuleConfigDtlDomainService.saveAll(paymentRuleConfigDtlDOS);
    }

    private Long saveConfigAndDtl(PaymentRuleConfigParam param) {
        PaymentRuleConfigDO paymentRuleConfigDO = PaymentRuleConfigConvert.INSTANCE.param2DO(param);

        List<PaymentRuleConfigDtlDO> paymentRuleConfigDtlDOS = PaymentRuleConfigDtlConvert.INSTANCE.param2DOList(param.getPaymentRuleConfigDtlDTOList());
        paymentRuleConfigRepo.save(paymentRuleConfigDO);
        paymentRuleConfigDtlDOS.forEach(dtl -> {
            dtl.setMasId(paymentRuleConfigDO.getId());
        });

        paymentRuleConfigDtlRepo.saveAll(paymentRuleConfigDtlDOS);
        return paymentRuleConfigDO.getId();
    }

    private void validateSaveOrUpdateParam(PaymentRuleConfigParam param) {
        Assert.notEmpty(param.getRuleCode(), "账户规则编码必填");
        Assert.notEmpty(param.getRuleName(), "账户规则名称必填");
        Assert.notEmpty(param.getOptDoc(), "扣款单据必填");
        Assert.notEmpty(param.getStatus(), "启用状态必填");
        Assert.notEmpty(param.getPaymentRuleConfigDtlDTOList(), "明细不能为空");

        param.getPaymentRuleConfigDtlDTOList().stream().forEach(item -> {
            Assert.notNull(item.getPriorityNo(), "优先级必填");
            Assert.notEmpty(item.getOptDocType(), "扣款单据类型必填");
            Assert.notEmpty(item.getOptDocStatus(), "扣款单据状态必填");
            Assert.notEmpty(item.getOptAccountType(), "扣款账户类型必填");
//            Assert.notNull(item.getCalculatePercent(), "最高扣款比例必填");
        });
    }


    @Override
    public void del(Long id) {
        Optional<PaymentRuleConfigDO> paymentRuleConfigDOOptional = paymentRuleConfigDomainService.findById(id);

        if (paymentRuleConfigDOOptional.isPresent()) {
            PaymentRuleConfigDO dbSceneConfig = paymentRuleConfigDOOptional.get();
            paymentRuleConfigDtlDomainService.deleteByCondition(getQueryBeanByMasId(id));
            paymentRuleConfigDomainService.deleteById(id);
            removeCache(dbSceneConfig.getRuleCode());
        }
    }

    private PaymentRuleConfigDtlDO getQueryBeanByMasId(Long id) {
        PaymentRuleConfigDtlDO whereBean = new PaymentRuleConfigDtlDO();
        whereBean.setMasId(id);
        return whereBean;
    }

    @Override
    @SysCodeProc
    public PaymentRuleConfigQueryVO detail(Long masId) {
        Optional<PaymentRuleConfigDO> dbConfig = paymentRuleConfigRepo.findById(masId);
        if (!dbConfig.isPresent()) {
            return null;
        }
        PaymentRuleConfigQueryVO queryVO = PaymentRuleConfigConvert.INSTANCE.do2VO(dbConfig.get());
        List<PaymentRuleConfigDtlDO> all = paymentRuleConfigDtlDomainService.findPaymentRuleConfigDtlByMasId(masId);
        List<PaymentRuleConfigDtlVO> paymentRuleConfigDtlVoList = PaymentRuleConfigDtlConvert.INSTANCE.do2VOList(all);

        detailCode2Name(paymentRuleConfigDtlVoList);
        queryVO.setPaymentRuleConfigDtlVoList(paymentRuleConfigDtlVoList);

        return queryVO;
    }

    public void detailCode2Name(List<PaymentRuleConfigDtlVO> paymentRuleConfigDtlVOList) {
        // 单据类型
        Map<String, String> docClsUdc = udcProvider.getValueMapByUdcCode(UdcEnum.DOC_CLS_SO_C.getModel(), UdcEnum.DOC_CLS_SO_C.getCode());
        // 订单状态
        Map<String, String> docStatusUdc = udcProvider.getValueMapByUdcCode(UdcEnum.DOC_STATUS_RSO_CL.getModel(), UdcEnum.DOC_STATUS_RSO_CL.getCode());
        // 适用账户类型
        Map<String, String> accountTypeUdc = udcProvider.getValueMapByUdcCode(UdcEnum.ACCOUNT_TYPE_STORE.getModel(), UdcEnum.ACCOUNT_TYPE_STORE.getCode());
        Map<String, String> creditAccountTypeUdc = udcProvider.getValueMapByUdcCode(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getModel(), UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getCode());

//        paymentRuleConfigDtlVOList.stream().forEach(item -> {
//            String type = item.getOptAccountType();
//            if (accountTypeUdc.get(type) != null) {
//                item.setOptAccountTypeName(accountTypeUdc.get(type));
//                return;
//            }
//            item.setOptAccountTypeName(creditAccountTypeUdc.get(type));
//        });

        //设置单据类型名称
        //设置指定账户名称
        List<String> optDocTypeList = new ArrayList<>();
        paymentRuleConfigDtlVOList.stream().forEach(item -> {
            optDocTypeList.clear();
            String[] optDocTypeArray = item.getOptDocType().split(FinConstant.TRANSLATION_PLUS);
            Arrays.stream(optDocTypeArray).forEach(optDocType -> {
                optDocTypeList.add(docClsUdc.get(optDocType));
            });

            item.setOptDocTypeName(joinedWithPlus(optDocTypeList));

            item.setOptDocStatusName(docStatusUdc.get(item.getOptDocStatus()));
            String type = item.getOptAccountType();
            if (accountTypeUdc.get(type) != null) {
                item.setOptAccountTypeName(accountTypeUdc.get(type));
                return;
            }
            item.setOptAccountTypeName(creditAccountTypeUdc.get(type));
        });
    }

    public void pageCode2Name(List<PaymentRuleConfigPageVO> paymentRuleConfigPageVOList) {
        //单据类型
        Map<String, String> docClsUdc = udcProvider.getValueMapByUdcCode(UdcEnum.DOC_CLS_SO_C.getModel(), UdcEnum.DOC_CLS_SO_C.getCode());
        //订单状态
        Map<String, String> docStatusUdc = udcProvider.getValueMapByUdcCode(UdcEnum.DOC_STATUS_RSO_CL.getModel(), UdcEnum.DOC_STATUS_RSO_CL.getCode());

        //设置单据类型名称
        List<String> optDocTypeList = new ArrayList<>();
        paymentRuleConfigPageVOList.stream().forEach(item -> {
            optDocTypeList.clear();

            String[] optDocTypeArray = item.getOptDocType().split(FinConstant.TRANSLATION_PLUS);
            Arrays.stream(optDocTypeArray).forEach(optDocType -> {
                optDocTypeList.add(docClsUdc.get(optDocType));
            });

            item.setOptDocTypeName(joinedWithPlus(optDocTypeList));
            item.setOptDocStatusName(docStatusUdc.get(item.getOptDocStatus()));
        });
    }

    private String joinedWithPlus(List<String> param) {
        return StringUtils.join(param, FinConstant.PLUS);
    }

    private void removeCache(String ruleCode) {
        RBucket<PaymentRuleConfigQueryVO> bucket = redissonClient.getBucket(FinRedisConstant.RULE_CONFIG_KEY + ruleCode);
        if (bucket.isExists()) {
            bucket.delete();
        }
    }

    @Override
    public PaymentRuleConfigDTO getByRuleCode(String ruleCode) {
        RBucket<PaymentRuleConfigDTO> bucket = redissonClient.getBucket(FinRedisConstant.RULE_CONFIG_KEY + ruleCode);
        if (bucket.isExists()) {
            return bucket.get();
        }
        var dbConfig = paymentRuleConfigDomainService.findByRuleCode(ruleCode);
        if (dbConfig != null) {
            Example<PaymentRuleConfigDtlDO> dtlExample = Example.of(getQueryBeanByMasId(dbConfig.getId()));
            List<PaymentRuleConfigDtlDO> all = paymentRuleConfigDtlDomainService.findAll(dtlExample);
            List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtls = PaymentRuleConfigDtlConvert.INSTANCE.do2DTOList(all);
            dbConfig.setPaymentRuleConfigDtlDTOList(paymentRuleConfigDtls);
            bucket.set(dbConfig);
            return dbConfig;
        }
        return null;
    }



//    private void convertPage(List<PaymentRuleConfigPageVO> paymentRuleConfigDtls) {
//        List<AccountIocDTO> accountIocNameList = accountIocDomainService.findByCodeBatch(paymentRuleConfigDtls.stream().map(PaymentRuleConfigPageVO::getIoCode).collect(Collectors.toList()));
//        Map<String, String> ioCodeMap = new HashMap<>();
//        Map<String, String> ioTypeMap = new HashMap<>();
//        if (CollectionUtils.isNotEmpty(accountIocNameList)) {
//            ioTypeMap = accountIocNameList.stream().collect(Collectors.toMap(AccountIocDTO::getIoType, i -> i.getIoTypeName() == null ? "" : i.getIoTypeName(), (key1, key2) -> key2));
//            ioCodeMap = accountIocNameList.stream().collect(Collectors.toMap(AccountIocDTO::getIoCode, i -> i.getIoName() == null ? "" : i.getIoName(), (key1, key2) -> key2));
//        }
//        for (PaymentRuleConfigPageVO configDtl : paymentRuleConfigDtls) {
//            if (StringUtils.isNotBlank(configDtl.getIoCode())) {
//                configDtl.setIoName(ioCodeMap.get(configDtl.getIoCode()));
//            }
//        }
//    }

    @Override
    public List<PaymentRuleConfigDtlDTO> queryAccountAmount(PaymentRuleConfigParam param) {
        return paymentRuleConfigDomainService.queryAccountAmount(param);
    }

    @Override
    public List<PaymentRuleConfigDtlDTO> pay(PaymentRuleConfigParam param) {
        return paymentRuleConfigDomainService.pay(param);
    }

}
